/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#include "Asn1Backup.h"
#include <PKI_ERR.h>
#include <openssl/asn1t.h>

ASN1_SEQUENCE(NEWPKI_BACKUP_CERT) = {
	ASN1_SIMPLE(NEWPKI_BACKUP_CERT, cert, X509),
	ASN1_SIMPLE(NEWPKI_BACKUP_CERT, privkey, ASN1_OCTET_STRING),
}ASN1_SEQUENCE_END(NEWPKI_BACKUP_CERT)
ASN1_SEQUENCE(INTERNAL_CA_BACKUP) = {
	ASN1_SIMPLE(INTERNAL_CA_BACKUP, ca_type, ASN1_INTEGER),
	ASN1_SIMPLE(INTERNAL_CA_BACKUP, internal_Ca, INTERNAL_CA),
}ASN1_SEQUENCE_END(INTERNAL_CA_BACKUP)
ASN1_SEQUENCE(NEWPKI_BACKUP_RA) = {
	ASN1_SIMPLE(NEWPKI_BACKUP_RA, entity_cert, NEWPKI_BACKUP_CERT),
	ASN1_SEQUENCE_OF(NEWPKI_BACKUP_RA, profiles, NEWPKI_PROFILE_DATAS),
	ASN1_SEQUENCE_OF(NEWPKI_BACKUP_RA, requests, CRYPTED_NEWPKI_REQUEST),
}ASN1_SEQUENCE_END(NEWPKI_BACKUP_RA)
ASN1_SEQUENCE(NEWPKI_BACKUP_CA) = {
	ASN1_SIMPLE(NEWPKI_BACKUP_CA, entity_cert, NEWPKI_BACKUP_CERT),
	ASN1_SEQUENCE_OF(NEWPKI_BACKUP_CA, certs, INTERNAL_CA_CERT),
	ASN1_SIMPLE(NEWPKI_BACKUP_CA, internal_Cas, INTERNAL_CA_BACKUP),
	ASN1_SEQUENCE_OF(NEWPKI_BACKUP_CA, requests, CRYPTED_NEWPKI_REQUEST),
}ASN1_SEQUENCE_END(NEWPKI_BACKUP_CA)
ASN1_SEQUENCE(NEWPKI_BACKUP_PKI) = {
	ASN1_SIMPLE(NEWPKI_BACKUP_PKI, entity_cert, NEWPKI_BACKUP_CERT),
	ASN1_SEQUENCE_OF(NEWPKI_BACKUP_PKI, certs, INTERNAL_CA_CERT),
	ASN1_SEQUENCE_OF(NEWPKI_BACKUP_PKI, internal_Cas, INTERNAL_CA_BACKUP),
}ASN1_SEQUENCE_END(NEWPKI_BACKUP_PKI)
ASN1_SEQUENCE(NEWPKI_BACKUP_REP) = {
	ASN1_SIMPLE(NEWPKI_BACKUP_REP, entity_cert, NEWPKI_BACKUP_CERT),
}ASN1_SEQUENCE_END(NEWPKI_BACKUP_REP)
ASN1_SEQUENCE(NEWPKI_BACKUP_PUB) = {
	ASN1_SIMPLE(NEWPKI_BACKUP_PUB, entity_cert, NEWPKI_BACKUP_CERT),
}ASN1_SEQUENCE_END(NEWPKI_BACKUP_PUB)
ASN1_SEQUENCE(NEWPKI_BACKUP_KEY) = {
	ASN1_SIMPLE(NEWPKI_BACKUP_KEY, entity_cert, NEWPKI_BACKUP_CERT),
}ASN1_SEQUENCE_END(NEWPKI_BACKUP_KEY)
ASN1_SEQUENCE(NEWPKI_BACKUP_EE) = {
	ASN1_SIMPLE(NEWPKI_BACKUP_EE, entity_cert, NEWPKI_BACKUP_CERT),
}ASN1_SEQUENCE_END(NEWPKI_BACKUP_EE)
ASN1_SEQUENCE(NEWPKI_BACKUP_BACK) = {
	ASN1_SIMPLE(NEWPKI_BACKUP_BACK, entity_cert, NEWPKI_BACKUP_CERT),
}ASN1_SEQUENCE_END(NEWPKI_BACKUP_BACK)
ASN1_CHOICE(NEWPKI_BACKUP) = {
	ASN1_EXP(NEWPKI_BACKUP, d.ra_backup, NEWPKI_BACKUP_RA, ENTITY_TYPE_RA),
	ASN1_EXP(NEWPKI_BACKUP, d.ca_backup, NEWPKI_BACKUP_CA, ENTITY_TYPE_CA),
	ASN1_EXP(NEWPKI_BACKUP, d.rep_backup, NEWPKI_BACKUP_REP, ENTITY_TYPE_REPOSITORY),
	ASN1_EXP(NEWPKI_BACKUP, d.pub_backup, NEWPKI_BACKUP_PUB, ENTITY_TYPE_PUBLICATION),
	ASN1_EXP(NEWPKI_BACKUP, d.key_backup, NEWPKI_BACKUP_KEY, ENTITY_TYPE_KEY_STORE),
	ASN1_EXP(NEWPKI_BACKUP, d.pki_backup, NEWPKI_BACKUP_PKI, ENTITY_TYPE_PKI),
	ASN1_EXP(NEWPKI_BACKUP, d.back_backup, NEWPKI_BACKUP_BACK, ENTITY_TYPE_BACKUP),
	ASN1_EXP(NEWPKI_BACKUP, d.ee_backup, NEWPKI_BACKUP_EE, ENTITY_TYPE_EE),
} ASN1_CHOICE_END(NEWPKI_BACKUP)
NewpkiBackupCert NewpkiBackupCert::EmptyInstance;
bool NewpkiBackupCert::set_cert(const PKI_CERT & c_cert)
{
	m_cert = c_cert;
	return true;
}

const PKI_CERT & NewpkiBackupCert::get_cert() const
{
	return m_cert;
}

PKI_CERT & NewpkiBackupCert::get_cert()
{
	return m_cert;
}

bool NewpkiBackupCert::set_privkey(const Asn1OctetString & c_privkey)
{
	m_privkey = c_privkey;
	return true;
}

const Asn1OctetString & NewpkiBackupCert::get_privkey() const
{
	return m_privkey;
}

Asn1OctetString & NewpkiBackupCert::get_privkey()
{
	return m_privkey;
}

NewpkiBackupCert::NewpkiBackupCert():NewPKIObject()
{
	resetAll();
}

NewpkiBackupCert::NewpkiBackupCert(const NewpkiBackupCert & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

NewpkiBackupCert::~NewpkiBackupCert()
{
	Clear();
}

void NewpkiBackupCert::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void NewpkiBackupCert::freeAll()
{
}

void NewpkiBackupCert::resetAll()
{
	m_cert.Clear();
	m_privkey.Clear();
}

bool NewpkiBackupCert::load_Datas(const NEWPKI_BACKUP_CERT * Datas)
{
	Clear();
	if(Datas->cert)
	{
		if(!m_cert.load_Datas(Datas->cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->privkey)
	{
		if(!m_privkey.load_Datas(Datas->privkey))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool NewpkiBackupCert::give_Datas(NEWPKI_BACKUP_CERT ** Datas) const
{
	if(!(*Datas) && !(*Datas = (NEWPKI_BACKUP_CERT*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->cert && !((*Datas)->cert = (X509*)ASN1_item_new(ASN1_ITEM_rptr(X509))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_cert.give_Datas(&(*Datas)->cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->cert, ASN1_ITEM_rptr(X509));
		(*Datas)->cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->privkey && !((*Datas)->privkey = (ASN1_OCTET_STRING*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_OCTET_STRING))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_privkey.give_Datas(&(*Datas)->privkey))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->privkey, ASN1_ITEM_rptr(ASN1_OCTET_STRING));
		(*Datas)->privkey = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool NewpkiBackupCert::operator=(const NewpkiBackupCert & other)
{
	Clear();
	m_cert = other.m_cert;
	m_privkey = other.m_privkey;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * NewpkiBackupCert::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT);
}
InternalCaBackup InternalCaBackup::EmptyInstance;
bool InternalCaBackup::set_caType(unsigned long c_caType)
{
	m_caType = c_caType;
	return true;
}

unsigned long InternalCaBackup::get_caType() const
{
	return m_caType;
}

bool InternalCaBackup::set_internalCa(const InternalCa & c_internalCa)
{
	m_internalCa = c_internalCa;
	return true;
}

const InternalCa & InternalCaBackup::get_internalCa() const
{
	return m_internalCa;
}

InternalCa & InternalCaBackup::get_internalCa()
{
	return m_internalCa;
}

InternalCaBackup::InternalCaBackup():NewPKIObject()
{
	resetAll();
}

InternalCaBackup::InternalCaBackup(const InternalCaBackup & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

InternalCaBackup::~InternalCaBackup()
{
	Clear();
}

void InternalCaBackup::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void InternalCaBackup::freeAll()
{
}

void InternalCaBackup::resetAll()
{
	m_caType = 0;
	m_internalCa.Clear();
}

bool InternalCaBackup::load_Datas(const INTERNAL_CA_BACKUP * Datas)
{
	Clear();
	if(Datas->ca_type)
	{
		m_caType = ASN1_INTEGER_GET(Datas->ca_type);
	}
	if(Datas->internal_Ca)
	{
		if(!m_internalCa.load_Datas(Datas->internal_Ca))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool InternalCaBackup::give_Datas(INTERNAL_CA_BACKUP ** Datas) const
{
	if(!(*Datas) && !(*Datas = (INTERNAL_CA_BACKUP*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->ca_type && !((*Datas)->ca_type = (ASN1_INTEGER*)ASN1_item_new(ASN1_ITEM_rptr(ASN1_INTEGER))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(ASN1_INTEGER_set((*Datas)->ca_type, m_caType) <= 0)
	{
		ASN1_INTEGER_free((*Datas)->ca_type);
		(*Datas)->ca_type = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
	if(!(*Datas)->internal_Ca && !((*Datas)->internal_Ca = (INTERNAL_CA*)ASN1_item_new(ASN1_ITEM_rptr(INTERNAL_CA))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_internalCa.give_Datas(&(*Datas)->internal_Ca))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->internal_Ca, ASN1_ITEM_rptr(INTERNAL_CA));
		(*Datas)->internal_Ca = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool InternalCaBackup::operator=(const InternalCaBackup & other)
{
	Clear();
	m_caType = other.m_caType;
	m_internalCa = other.m_internalCa;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * InternalCaBackup::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(INTERNAL_CA_BACKUP);
}
NewpkiBackupRa NewpkiBackupRa::EmptyInstance;
bool NewpkiBackupRa::set_entityCert(const NewpkiBackupCert & c_entityCert)
{
	m_entityCert = c_entityCert;
	return true;
}

const NewpkiBackupCert & NewpkiBackupRa::get_entityCert() const
{
	return m_entityCert;
}

NewpkiBackupCert & NewpkiBackupRa::get_entityCert()
{
	return m_entityCert;
}

bool NewpkiBackupRa::set_profiles(const mVector< NewpkiProfileDatas > & c_profiles)
{
	m_profiles = c_profiles;
	return true;
}

const mVector< NewpkiProfileDatas > & NewpkiBackupRa::get_profiles() const
{
	return m_profiles;
}

mVector< NewpkiProfileDatas > & NewpkiBackupRa::get_profiles()
{
	return m_profiles;
}

bool NewpkiBackupRa::set_requests(const mVector< CryptedNewpkiRequest > & c_requests)
{
	m_requests = c_requests;
	return true;
}

const mVector< CryptedNewpkiRequest > & NewpkiBackupRa::get_requests() const
{
	return m_requests;
}

mVector< CryptedNewpkiRequest > & NewpkiBackupRa::get_requests()
{
	return m_requests;
}

NewpkiBackupRa::NewpkiBackupRa():NewPKIObject()
{
	resetAll();
}

NewpkiBackupRa::NewpkiBackupRa(const NewpkiBackupRa & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

NewpkiBackupRa::~NewpkiBackupRa()
{
	Clear();
}

void NewpkiBackupRa::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void NewpkiBackupRa::freeAll()
{
}

void NewpkiBackupRa::resetAll()
{
	m_entityCert.Clear();
	m_profiles.clear();
	m_requests.clear();
}

bool NewpkiBackupRa::load_Datas(const NEWPKI_BACKUP_RA * Datas)
{
	Clear();
	CRYPTED_NEWPKI_REQUEST * currrequests;
	int i;
	NEWPKI_PROFILE_DATAS * currprofiles;
	if(Datas->entity_cert)
	{
		if(!m_entityCert.load_Datas(Datas->entity_cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->profiles)
	{
		for(i=0; i<SKM_sk_num(NEWPKI_PROFILE_DATAS, Datas->profiles); i++)
		{
			currprofiles = SKM_sk_value(NEWPKI_PROFILE_DATAS, Datas->profiles, i);
			if(!currprofiles)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_profiles.insert(m_profiles.begin() + i);
			if(!m_profiles[i].load_Datas(currprofiles))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	if(Datas->requests)
	{
		for(i=0; i<SKM_sk_num(CRYPTED_NEWPKI_REQUEST, Datas->requests); i++)
		{
			currrequests = SKM_sk_value(CRYPTED_NEWPKI_REQUEST, Datas->requests, i);
			if(!currrequests)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_requests.insert(m_requests.begin() + i);
			if(!m_requests[i].load_Datas(currrequests))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	m_isOk=true;
	return true;
}

bool NewpkiBackupRa::give_Datas(NEWPKI_BACKUP_RA ** Datas) const
{
	if(!(*Datas) && !(*Datas = (NEWPKI_BACKUP_RA*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	CRYPTED_NEWPKI_REQUEST * currrequests;
	NEWPKI_PROFILE_DATAS * currprofiles;
	size_t i;
	if(!(*Datas)->entity_cert && !((*Datas)->entity_cert = (NEWPKI_BACKUP_CERT*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entityCert.give_Datas(&(*Datas)->entity_cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_cert, ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT));
		(*Datas)->entity_cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->profiles && !((*Datas)->profiles = SKM_sk_new_null(NEWPKI_PROFILE_DATAS)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_profiles.size() ; i++)
	{
		currprofiles = NULL;
		if(!m_profiles[i].give_Datas(&currprofiles))
		{
			ASN1_item_free((ASN1_VALUE*)currprofiles, ASN1_ITEM_rptr(NEWPKI_PROFILE_DATAS));
			currprofiles = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(NEWPKI_PROFILE_DATAS, (*Datas)->profiles, currprofiles) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currprofiles, ASN1_ITEM_rptr(NEWPKI_PROFILE_DATAS));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->requests && !((*Datas)->requests = SKM_sk_new_null(CRYPTED_NEWPKI_REQUEST)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_requests.size() ; i++)
	{
		currrequests = NULL;
		if(!m_requests[i].give_Datas(&currrequests))
		{
			ASN1_item_free((ASN1_VALUE*)currrequests, ASN1_ITEM_rptr(CRYPTED_NEWPKI_REQUEST));
			currrequests = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(CRYPTED_NEWPKI_REQUEST, (*Datas)->requests, currrequests) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currrequests, ASN1_ITEM_rptr(CRYPTED_NEWPKI_REQUEST));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	return true;
}

bool NewpkiBackupRa::operator=(const NewpkiBackupRa & other)
{
	Clear();
	m_entityCert = other.m_entityCert;
	m_profiles = other.m_profiles;
	m_requests = other.m_requests;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * NewpkiBackupRa::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(NEWPKI_BACKUP_RA);
}
NewpkiBackupCa NewpkiBackupCa::EmptyInstance;
bool NewpkiBackupCa::set_certs(const mVector< InternalCaCert > & c_certs)
{
	m_certs = c_certs;
	return true;
}

const mVector< InternalCaCert > & NewpkiBackupCa::get_certs() const
{
	return m_certs;
}

mVector< InternalCaCert > & NewpkiBackupCa::get_certs()
{
	return m_certs;
}

bool NewpkiBackupCa::set_entityCert(const NewpkiBackupCert & c_entityCert)
{
	m_entityCert = c_entityCert;
	return true;
}

const NewpkiBackupCert & NewpkiBackupCa::get_entityCert() const
{
	return m_entityCert;
}

NewpkiBackupCert & NewpkiBackupCa::get_entityCert()
{
	return m_entityCert;
}

bool NewpkiBackupCa::set_internalCas(const InternalCaBackup & c_internalCas)
{
	m_internalCas = c_internalCas;
	return true;
}

const InternalCaBackup & NewpkiBackupCa::get_internalCas() const
{
	return m_internalCas;
}

InternalCaBackup & NewpkiBackupCa::get_internalCas()
{
	return m_internalCas;
}

bool NewpkiBackupCa::set_requests(const mVector< CryptedNewpkiRequest > & c_requests)
{
	m_requests = c_requests;
	return true;
}

const mVector< CryptedNewpkiRequest > & NewpkiBackupCa::get_requests() const
{
	return m_requests;
}

mVector< CryptedNewpkiRequest > & NewpkiBackupCa::get_requests()
{
	return m_requests;
}

NewpkiBackupCa::NewpkiBackupCa():NewPKIObject()
{
	resetAll();
}

NewpkiBackupCa::NewpkiBackupCa(const NewpkiBackupCa & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

NewpkiBackupCa::~NewpkiBackupCa()
{
	Clear();
}

void NewpkiBackupCa::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void NewpkiBackupCa::freeAll()
{
}

void NewpkiBackupCa::resetAll()
{
	m_certs.clear();
	m_entityCert.Clear();
	m_internalCas.Clear();
	m_requests.clear();
}

bool NewpkiBackupCa::load_Datas(const NEWPKI_BACKUP_CA * Datas)
{
	Clear();
	CRYPTED_NEWPKI_REQUEST * currrequests;
	int i;
	INTERNAL_CA_CERT * currcerts;
	if(Datas->certs)
	{
		for(i=0; i<SKM_sk_num(INTERNAL_CA_CERT, Datas->certs); i++)
		{
			currcerts = SKM_sk_value(INTERNAL_CA_CERT, Datas->certs, i);
			if(!currcerts)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_certs.insert(m_certs.begin() + i);
			if(!m_certs[i].load_Datas(currcerts))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	if(Datas->entity_cert)
	{
		if(!m_entityCert.load_Datas(Datas->entity_cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->internal_Cas)
	{
		if(!m_internalCas.load_Datas(Datas->internal_Cas))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->requests)
	{
		for(i=0; i<SKM_sk_num(CRYPTED_NEWPKI_REQUEST, Datas->requests); i++)
		{
			currrequests = SKM_sk_value(CRYPTED_NEWPKI_REQUEST, Datas->requests, i);
			if(!currrequests)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_requests.insert(m_requests.begin() + i);
			if(!m_requests[i].load_Datas(currrequests))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	m_isOk=true;
	return true;
}

bool NewpkiBackupCa::give_Datas(NEWPKI_BACKUP_CA ** Datas) const
{
	if(!(*Datas) && !(*Datas = (NEWPKI_BACKUP_CA*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	CRYPTED_NEWPKI_REQUEST * currrequests;
	INTERNAL_CA_CERT * currcerts;
	size_t i;
	if(!(*Datas)->certs && !((*Datas)->certs = SKM_sk_new_null(INTERNAL_CA_CERT)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_certs.size() ; i++)
	{
		currcerts = NULL;
		if(!m_certs[i].give_Datas(&currcerts))
		{
			ASN1_item_free((ASN1_VALUE*)currcerts, ASN1_ITEM_rptr(INTERNAL_CA_CERT));
			currcerts = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(INTERNAL_CA_CERT, (*Datas)->certs, currcerts) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currcerts, ASN1_ITEM_rptr(INTERNAL_CA_CERT));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->entity_cert && !((*Datas)->entity_cert = (NEWPKI_BACKUP_CERT*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entityCert.give_Datas(&(*Datas)->entity_cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_cert, ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT));
		(*Datas)->entity_cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->internal_Cas && !((*Datas)->internal_Cas = (INTERNAL_CA_BACKUP*)ASN1_item_new(ASN1_ITEM_rptr(INTERNAL_CA_BACKUP))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_internalCas.give_Datas(&(*Datas)->internal_Cas))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->internal_Cas, ASN1_ITEM_rptr(INTERNAL_CA_BACKUP));
		(*Datas)->internal_Cas = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->requests && !((*Datas)->requests = SKM_sk_new_null(CRYPTED_NEWPKI_REQUEST)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_requests.size() ; i++)
	{
		currrequests = NULL;
		if(!m_requests[i].give_Datas(&currrequests))
		{
			ASN1_item_free((ASN1_VALUE*)currrequests, ASN1_ITEM_rptr(CRYPTED_NEWPKI_REQUEST));
			currrequests = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(CRYPTED_NEWPKI_REQUEST, (*Datas)->requests, currrequests) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currrequests, ASN1_ITEM_rptr(CRYPTED_NEWPKI_REQUEST));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	return true;
}

bool NewpkiBackupCa::operator=(const NewpkiBackupCa & other)
{
	Clear();
	m_certs = other.m_certs;
	m_entityCert = other.m_entityCert;
	m_internalCas = other.m_internalCas;
	m_requests = other.m_requests;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * NewpkiBackupCa::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(NEWPKI_BACKUP_CA);
}
NewpkiBackupPki NewpkiBackupPki::EmptyInstance;
bool NewpkiBackupPki::set_certs(const mVector< InternalCaCert > & c_certs)
{
	m_certs = c_certs;
	return true;
}

const mVector< InternalCaCert > & NewpkiBackupPki::get_certs() const
{
	return m_certs;
}

mVector< InternalCaCert > & NewpkiBackupPki::get_certs()
{
	return m_certs;
}

bool NewpkiBackupPki::set_entityCert(const NewpkiBackupCert & c_entityCert)
{
	m_entityCert = c_entityCert;
	return true;
}

const NewpkiBackupCert & NewpkiBackupPki::get_entityCert() const
{
	return m_entityCert;
}

NewpkiBackupCert & NewpkiBackupPki::get_entityCert()
{
	return m_entityCert;
}

bool NewpkiBackupPki::set_internalCas(const mVector< InternalCaBackup > & c_internalCas)
{
	m_internalCas = c_internalCas;
	return true;
}

const mVector< InternalCaBackup > & NewpkiBackupPki::get_internalCas() const
{
	return m_internalCas;
}

mVector< InternalCaBackup > & NewpkiBackupPki::get_internalCas()
{
	return m_internalCas;
}

NewpkiBackupPki::NewpkiBackupPki():NewPKIObject()
{
	resetAll();
}

NewpkiBackupPki::NewpkiBackupPki(const NewpkiBackupPki & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

NewpkiBackupPki::~NewpkiBackupPki()
{
	Clear();
}

void NewpkiBackupPki::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void NewpkiBackupPki::freeAll()
{
}

void NewpkiBackupPki::resetAll()
{
	m_certs.clear();
	m_entityCert.Clear();
	m_internalCas.clear();
}

bool NewpkiBackupPki::load_Datas(const NEWPKI_BACKUP_PKI * Datas)
{
	Clear();
	INTERNAL_CA_BACKUP * currinternalCas;
	int i;
	INTERNAL_CA_CERT * currcerts;
	if(Datas->certs)
	{
		for(i=0; i<SKM_sk_num(INTERNAL_CA_CERT, Datas->certs); i++)
		{
			currcerts = SKM_sk_value(INTERNAL_CA_CERT, Datas->certs, i);
			if(!currcerts)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_certs.insert(m_certs.begin() + i);
			if(!m_certs[i].load_Datas(currcerts))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	if(Datas->entity_cert)
	{
		if(!m_entityCert.load_Datas(Datas->entity_cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	if(Datas->internal_Cas)
	{
		for(i=0; i<SKM_sk_num(INTERNAL_CA_BACKUP, Datas->internal_Cas); i++)
		{
			currinternalCas = SKM_sk_value(INTERNAL_CA_BACKUP, Datas->internal_Cas, i);
			if(!currinternalCas)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			m_internalCas.insert(m_internalCas.begin() + i);
			if(!m_internalCas[i].load_Datas(currinternalCas))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
		}
	}
	m_isOk=true;
	return true;
}

bool NewpkiBackupPki::give_Datas(NEWPKI_BACKUP_PKI ** Datas) const
{
	if(!(*Datas) && !(*Datas = (NEWPKI_BACKUP_PKI*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	INTERNAL_CA_BACKUP * currinternalCas;
	INTERNAL_CA_CERT * currcerts;
	size_t i;
	if(!(*Datas)->certs && !((*Datas)->certs = SKM_sk_new_null(INTERNAL_CA_CERT)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_certs.size() ; i++)
	{
		currcerts = NULL;
		if(!m_certs[i].give_Datas(&currcerts))
		{
			ASN1_item_free((ASN1_VALUE*)currcerts, ASN1_ITEM_rptr(INTERNAL_CA_CERT));
			currcerts = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(INTERNAL_CA_CERT, (*Datas)->certs, currcerts) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currcerts, ASN1_ITEM_rptr(INTERNAL_CA_CERT));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	if(!(*Datas)->entity_cert && !((*Datas)->entity_cert = (NEWPKI_BACKUP_CERT*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entityCert.give_Datas(&(*Datas)->entity_cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_cert, ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT));
		(*Datas)->entity_cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!(*Datas)->internal_Cas && !((*Datas)->internal_Cas = SKM_sk_new_null(INTERNAL_CA_BACKUP)))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	for(i = 0 ; i < m_internalCas.size() ; i++)
	{
		currinternalCas = NULL;
		if(!m_internalCas[i].give_Datas(&currinternalCas))
		{
			ASN1_item_free((ASN1_VALUE*)currinternalCas, ASN1_ITEM_rptr(INTERNAL_CA_BACKUP));
			currinternalCas = NULL;
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(SKM_sk_push(INTERNAL_CA_BACKUP, (*Datas)->internal_Cas, currinternalCas) < 0)
		{
			ASN1_item_free((ASN1_VALUE*)currinternalCas, ASN1_ITEM_rptr(INTERNAL_CA_BACKUP));
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_UNKNOWN);
			return false;
		}
	}
	return true;
}

bool NewpkiBackupPki::operator=(const NewpkiBackupPki & other)
{
	Clear();
	m_certs = other.m_certs;
	m_entityCert = other.m_entityCert;
	m_internalCas = other.m_internalCas;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * NewpkiBackupPki::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(NEWPKI_BACKUP_PKI);
}
NewpkiBackupRep NewpkiBackupRep::EmptyInstance;
bool NewpkiBackupRep::set_entityCert(const NewpkiBackupCert & c_entityCert)
{
	m_entityCert = c_entityCert;
	return true;
}

const NewpkiBackupCert & NewpkiBackupRep::get_entityCert() const
{
	return m_entityCert;
}

NewpkiBackupCert & NewpkiBackupRep::get_entityCert()
{
	return m_entityCert;
}

NewpkiBackupRep::NewpkiBackupRep():NewPKIObject()
{
	resetAll();
}

NewpkiBackupRep::NewpkiBackupRep(const NewpkiBackupRep & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

NewpkiBackupRep::~NewpkiBackupRep()
{
	Clear();
}

void NewpkiBackupRep::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void NewpkiBackupRep::freeAll()
{
}

void NewpkiBackupRep::resetAll()
{
	m_entityCert.Clear();
}

bool NewpkiBackupRep::load_Datas(const NEWPKI_BACKUP_REP * Datas)
{
	Clear();
	if(Datas->entity_cert)
	{
		if(!m_entityCert.load_Datas(Datas->entity_cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool NewpkiBackupRep::give_Datas(NEWPKI_BACKUP_REP ** Datas) const
{
	if(!(*Datas) && !(*Datas = (NEWPKI_BACKUP_REP*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->entity_cert && !((*Datas)->entity_cert = (NEWPKI_BACKUP_CERT*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entityCert.give_Datas(&(*Datas)->entity_cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_cert, ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT));
		(*Datas)->entity_cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool NewpkiBackupRep::operator=(const NewpkiBackupRep & other)
{
	Clear();
	m_entityCert = other.m_entityCert;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * NewpkiBackupRep::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(NEWPKI_BACKUP_REP);
}
NewpkiBackupPub NewpkiBackupPub::EmptyInstance;
bool NewpkiBackupPub::set_entityCert(const NewpkiBackupCert & c_entityCert)
{
	m_entityCert = c_entityCert;
	return true;
}

const NewpkiBackupCert & NewpkiBackupPub::get_entityCert() const
{
	return m_entityCert;
}

NewpkiBackupCert & NewpkiBackupPub::get_entityCert()
{
	return m_entityCert;
}

NewpkiBackupPub::NewpkiBackupPub():NewPKIObject()
{
	resetAll();
}

NewpkiBackupPub::NewpkiBackupPub(const NewpkiBackupPub & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

NewpkiBackupPub::~NewpkiBackupPub()
{
	Clear();
}

void NewpkiBackupPub::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void NewpkiBackupPub::freeAll()
{
}

void NewpkiBackupPub::resetAll()
{
	m_entityCert.Clear();
}

bool NewpkiBackupPub::load_Datas(const NEWPKI_BACKUP_PUB * Datas)
{
	Clear();
	if(Datas->entity_cert)
	{
		if(!m_entityCert.load_Datas(Datas->entity_cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool NewpkiBackupPub::give_Datas(NEWPKI_BACKUP_PUB ** Datas) const
{
	if(!(*Datas) && !(*Datas = (NEWPKI_BACKUP_PUB*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->entity_cert && !((*Datas)->entity_cert = (NEWPKI_BACKUP_CERT*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entityCert.give_Datas(&(*Datas)->entity_cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_cert, ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT));
		(*Datas)->entity_cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool NewpkiBackupPub::operator=(const NewpkiBackupPub & other)
{
	Clear();
	m_entityCert = other.m_entityCert;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * NewpkiBackupPub::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(NEWPKI_BACKUP_PUB);
}
NewpkiBackupKey NewpkiBackupKey::EmptyInstance;
bool NewpkiBackupKey::set_entityCert(const NewpkiBackupCert & c_entityCert)
{
	m_entityCert = c_entityCert;
	return true;
}

const NewpkiBackupCert & NewpkiBackupKey::get_entityCert() const
{
	return m_entityCert;
}

NewpkiBackupCert & NewpkiBackupKey::get_entityCert()
{
	return m_entityCert;
}

NewpkiBackupKey::NewpkiBackupKey():NewPKIObject()
{
	resetAll();
}

NewpkiBackupKey::NewpkiBackupKey(const NewpkiBackupKey & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

NewpkiBackupKey::~NewpkiBackupKey()
{
	Clear();
}

void NewpkiBackupKey::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void NewpkiBackupKey::freeAll()
{
}

void NewpkiBackupKey::resetAll()
{
	m_entityCert.Clear();
}

bool NewpkiBackupKey::load_Datas(const NEWPKI_BACKUP_KEY * Datas)
{
	Clear();
	if(Datas->entity_cert)
	{
		if(!m_entityCert.load_Datas(Datas->entity_cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool NewpkiBackupKey::give_Datas(NEWPKI_BACKUP_KEY ** Datas) const
{
	if(!(*Datas) && !(*Datas = (NEWPKI_BACKUP_KEY*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->entity_cert && !((*Datas)->entity_cert = (NEWPKI_BACKUP_CERT*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entityCert.give_Datas(&(*Datas)->entity_cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_cert, ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT));
		(*Datas)->entity_cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool NewpkiBackupKey::operator=(const NewpkiBackupKey & other)
{
	Clear();
	m_entityCert = other.m_entityCert;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * NewpkiBackupKey::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(NEWPKI_BACKUP_KEY);
}
NewpkiBackupEe NewpkiBackupEe::EmptyInstance;
bool NewpkiBackupEe::set_entityCert(const NewpkiBackupCert & c_entityCert)
{
	m_entityCert = c_entityCert;
	return true;
}

const NewpkiBackupCert & NewpkiBackupEe::get_entityCert() const
{
	return m_entityCert;
}

NewpkiBackupCert & NewpkiBackupEe::get_entityCert()
{
	return m_entityCert;
}

NewpkiBackupEe::NewpkiBackupEe():NewPKIObject()
{
	resetAll();
}

NewpkiBackupEe::NewpkiBackupEe(const NewpkiBackupEe & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

NewpkiBackupEe::~NewpkiBackupEe()
{
	Clear();
}

void NewpkiBackupEe::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void NewpkiBackupEe::freeAll()
{
}

void NewpkiBackupEe::resetAll()
{
	m_entityCert.Clear();
}

bool NewpkiBackupEe::load_Datas(const NEWPKI_BACKUP_EE * Datas)
{
	Clear();
	if(Datas->entity_cert)
	{
		if(!m_entityCert.load_Datas(Datas->entity_cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool NewpkiBackupEe::give_Datas(NEWPKI_BACKUP_EE ** Datas) const
{
	if(!(*Datas) && !(*Datas = (NEWPKI_BACKUP_EE*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->entity_cert && !((*Datas)->entity_cert = (NEWPKI_BACKUP_CERT*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entityCert.give_Datas(&(*Datas)->entity_cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_cert, ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT));
		(*Datas)->entity_cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool NewpkiBackupEe::operator=(const NewpkiBackupEe & other)
{
	Clear();
	m_entityCert = other.m_entityCert;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * NewpkiBackupEe::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(NEWPKI_BACKUP_EE);
}
NewpkiBackupBack NewpkiBackupBack::EmptyInstance;
bool NewpkiBackupBack::set_entityCert(const NewpkiBackupCert & c_entityCert)
{
	m_entityCert = c_entityCert;
	return true;
}

const NewpkiBackupCert & NewpkiBackupBack::get_entityCert() const
{
	return m_entityCert;
}

NewpkiBackupCert & NewpkiBackupBack::get_entityCert()
{
	return m_entityCert;
}

NewpkiBackupBack::NewpkiBackupBack():NewPKIObject()
{
	resetAll();
}

NewpkiBackupBack::NewpkiBackupBack(const NewpkiBackupBack & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

NewpkiBackupBack::~NewpkiBackupBack()
{
	Clear();
}

void NewpkiBackupBack::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void NewpkiBackupBack::freeAll()
{
}

void NewpkiBackupBack::resetAll()
{
	m_entityCert.Clear();
}

bool NewpkiBackupBack::load_Datas(const NEWPKI_BACKUP_BACK * Datas)
{
	Clear();
	if(Datas->entity_cert)
	{
		if(!m_entityCert.load_Datas(Datas->entity_cert))
		{
			NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	m_isOk=true;
	return true;
}

bool NewpkiBackupBack::give_Datas(NEWPKI_BACKUP_BACK ** Datas) const
{
	if(!(*Datas) && !(*Datas = (NEWPKI_BACKUP_BACK*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!(*Datas)->entity_cert && !((*Datas)->entity_cert = (NEWPKI_BACKUP_CERT*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT))))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	if(!m_entityCert.give_Datas(&(*Datas)->entity_cert))
	{
		ASN1_item_free((ASN1_VALUE*)(*Datas)->entity_cert, ASN1_ITEM_rptr(NEWPKI_BACKUP_CERT));
		(*Datas)->entity_cert = NULL;
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool NewpkiBackupBack::operator=(const NewpkiBackupBack & other)
{
	Clear();
	m_entityCert = other.m_entityCert;
	m_isOk=true;
	return true;
}



const ASN1_ITEM * NewpkiBackupBack::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(NEWPKI_BACKUP_BACK);
}
NewpkiBackup NewpkiBackup::EmptyInstance;
bool NewpkiBackup::set_type(int c_type)
{
	Clear();
	m_type = c_type;
	if(!malloc_byType(m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

int NewpkiBackup::get_type() const
{
	 return m_type;
}

bool NewpkiBackup::set_backBackup(const NewpkiBackupBack & c_backBackup)
{
	if(m_type != 6)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_backBackup) = c_backBackup;
	m_isOk=true;
	return true;
}

const NewpkiBackupBack & NewpkiBackup::get_backBackup() const
{
	if((m_type != 6) || !m_backBackup)
	{
		return NewpkiBackupBack::EmptyInstance;
	}
	return (*m_backBackup);
}

NewpkiBackupBack & NewpkiBackup::get_backBackup()
{
	if((m_type != 6) || !m_backBackup)
	{
		return NewpkiBackupBack::EmptyInstance;
	}
	return (*m_backBackup);
}

bool NewpkiBackup::set_caBackup(const NewpkiBackupCa & c_caBackup)
{
	if(m_type != 1)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_caBackup) = c_caBackup;
	m_isOk=true;
	return true;
}

const NewpkiBackupCa & NewpkiBackup::get_caBackup() const
{
	if((m_type != 1) || !m_caBackup)
	{
		return NewpkiBackupCa::EmptyInstance;
	}
	return (*m_caBackup);
}

NewpkiBackupCa & NewpkiBackup::get_caBackup()
{
	if((m_type != 1) || !m_caBackup)
	{
		return NewpkiBackupCa::EmptyInstance;
	}
	return (*m_caBackup);
}

bool NewpkiBackup::set_eeBackup(const NewpkiBackupEe & c_eeBackup)
{
	if(m_type != 7)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_eeBackup) = c_eeBackup;
	m_isOk=true;
	return true;
}

const NewpkiBackupEe & NewpkiBackup::get_eeBackup() const
{
	if((m_type != 7) || !m_eeBackup)
	{
		return NewpkiBackupEe::EmptyInstance;
	}
	return (*m_eeBackup);
}

NewpkiBackupEe & NewpkiBackup::get_eeBackup()
{
	if((m_type != 7) || !m_eeBackup)
	{
		return NewpkiBackupEe::EmptyInstance;
	}
	return (*m_eeBackup);
}

bool NewpkiBackup::set_keyBackup(const NewpkiBackupKey & c_keyBackup)
{
	if(m_type != 4)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_keyBackup) = c_keyBackup;
	m_isOk=true;
	return true;
}

const NewpkiBackupKey & NewpkiBackup::get_keyBackup() const
{
	if((m_type != 4) || !m_keyBackup)
	{
		return NewpkiBackupKey::EmptyInstance;
	}
	return (*m_keyBackup);
}

NewpkiBackupKey & NewpkiBackup::get_keyBackup()
{
	if((m_type != 4) || !m_keyBackup)
	{
		return NewpkiBackupKey::EmptyInstance;
	}
	return (*m_keyBackup);
}

bool NewpkiBackup::set_pkiBackup(const NewpkiBackupPki & c_pkiBackup)
{
	if(m_type != 5)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_pkiBackup) = c_pkiBackup;
	m_isOk=true;
	return true;
}

const NewpkiBackupPki & NewpkiBackup::get_pkiBackup() const
{
	if((m_type != 5) || !m_pkiBackup)
	{
		return NewpkiBackupPki::EmptyInstance;
	}
	return (*m_pkiBackup);
}

NewpkiBackupPki & NewpkiBackup::get_pkiBackup()
{
	if((m_type != 5) || !m_pkiBackup)
	{
		return NewpkiBackupPki::EmptyInstance;
	}
	return (*m_pkiBackup);
}

bool NewpkiBackup::set_pubBackup(const NewpkiBackupPub & c_pubBackup)
{
	if(m_type != 3)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_pubBackup) = c_pubBackup;
	m_isOk=true;
	return true;
}

const NewpkiBackupPub & NewpkiBackup::get_pubBackup() const
{
	if((m_type != 3) || !m_pubBackup)
	{
		return NewpkiBackupPub::EmptyInstance;
	}
	return (*m_pubBackup);
}

NewpkiBackupPub & NewpkiBackup::get_pubBackup()
{
	if((m_type != 3) || !m_pubBackup)
	{
		return NewpkiBackupPub::EmptyInstance;
	}
	return (*m_pubBackup);
}

bool NewpkiBackup::set_raBackup(const NewpkiBackupRa & c_raBackup)
{
	if(m_type != 0)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_raBackup) = c_raBackup;
	m_isOk=true;
	return true;
}

const NewpkiBackupRa & NewpkiBackup::get_raBackup() const
{
	if((m_type != 0) || !m_raBackup)
	{
		return NewpkiBackupRa::EmptyInstance;
	}
	return (*m_raBackup);
}

NewpkiBackupRa & NewpkiBackup::get_raBackup()
{
	if((m_type != 0) || !m_raBackup)
	{
		return NewpkiBackupRa::EmptyInstance;
	}
	return (*m_raBackup);
}

bool NewpkiBackup::set_repBackup(const NewpkiBackupRep & c_repBackup)
{
	if(m_type != 2)
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	(*m_repBackup) = c_repBackup;
	m_isOk=true;
	return true;
}

const NewpkiBackupRep & NewpkiBackup::get_repBackup() const
{
	if((m_type != 2) || !m_repBackup)
	{
		return NewpkiBackupRep::EmptyInstance;
	}
	return (*m_repBackup);
}

NewpkiBackupRep & NewpkiBackup::get_repBackup()
{
	if((m_type != 2) || !m_repBackup)
	{
		return NewpkiBackupRep::EmptyInstance;
	}
	return (*m_repBackup);
}

bool NewpkiBackup::malloc_byType (int c_type)
{
	switch(m_type)
	{
		case 6:
			m_backBackup = new NewpkiBackupBack();
			if(!m_backBackup)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 1:
			m_caBackup = new NewpkiBackupCa();
			if(!m_caBackup)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 7:
			m_eeBackup = new NewpkiBackupEe();
			if(!m_eeBackup)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 4:
			m_keyBackup = new NewpkiBackupKey();
			if(!m_keyBackup)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 5:
			m_pkiBackup = new NewpkiBackupPki();
			if(!m_pkiBackup)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 3:
			m_pubBackup = new NewpkiBackupPub();
			if(!m_pubBackup)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 0:
			m_raBackup = new NewpkiBackupRa();
			if(!m_raBackup)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
		case 2:
			m_repBackup = new NewpkiBackupRep();
			if(!m_repBackup)
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			break;
	}

	return true;
}

NewpkiBackup::NewpkiBackup():NewPKIObject()
{
	resetAll();
}

NewpkiBackup::NewpkiBackup(const NewpkiBackup & other):NewPKIObject()
{
	resetAll();
	*this = other;
}

NewpkiBackup::~NewpkiBackup()
{
	Clear();
}

void NewpkiBackup::Clear()
{
	freeAll();
	resetAll();
	m_isOk=false;
}

void NewpkiBackup::freeAll()
{
	if(m_backBackup)
	{
		delete m_backBackup;
	}
	if(m_caBackup)
	{
		delete m_caBackup;
	}
	if(m_eeBackup)
	{
		delete m_eeBackup;
	}
	if(m_keyBackup)
	{
		delete m_keyBackup;
	}
	if(m_pkiBackup)
	{
		delete m_pkiBackup;
	}
	if(m_pubBackup)
	{
		delete m_pubBackup;
	}
	if(m_raBackup)
	{
		delete m_raBackup;
	}
	if(m_repBackup)
	{
		delete m_repBackup;
	}
}

void NewpkiBackup::resetAll()
{
	m_type = -1;
	m_backBackup = NULL;
	m_caBackup = NULL;
	m_eeBackup = NULL;
	m_keyBackup = NULL;
	m_pkiBackup = NULL;
	m_pubBackup = NULL;
	m_raBackup = NULL;
	m_repBackup = NULL;
}

bool NewpkiBackup::load_Datas(const NEWPKI_BACKUP * Datas)
{
	Clear();
	if(!set_type(Datas->type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(Datas->type)
	{
		case 6:
			if(Datas->d.back_backup)
			{
				if(!(*m_backBackup).load_Datas(Datas->d.back_backup))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 1:
			if(Datas->d.ca_backup)
			{
				if(!(*m_caBackup).load_Datas(Datas->d.ca_backup))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 7:
			if(Datas->d.ee_backup)
			{
				if(!(*m_eeBackup).load_Datas(Datas->d.ee_backup))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 4:
			if(Datas->d.key_backup)
			{
				if(!(*m_keyBackup).load_Datas(Datas->d.key_backup))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 5:
			if(Datas->d.pki_backup)
			{
				if(!(*m_pkiBackup).load_Datas(Datas->d.pki_backup))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 3:
			if(Datas->d.pub_backup)
			{
				if(!(*m_pubBackup).load_Datas(Datas->d.pub_backup))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 0:
			if(Datas->d.ra_backup)
			{
				if(!(*m_raBackup).load_Datas(Datas->d.ra_backup))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
		case 2:
			if(Datas->d.rep_backup)
			{
				if(!(*m_repBackup).load_Datas(Datas->d.rep_backup))
				{
					NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			break;
	}
	m_isOk=true;
	return true;
}

bool NewpkiBackup::give_Datas(NEWPKI_BACKUP ** Datas) const
{
	if(!(*Datas) && !(*Datas = (NEWPKI_BACKUP*)ASN1_item_new(get_ASN1_ITEM())))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	(*Datas)->type = m_type;
	switch(m_type)
	{
		case 6:
			if(!((*Datas)->d.back_backup = (NEWPKI_BACKUP_BACK*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_BACK))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_backBackup).give_Datas(&(*Datas)->d.back_backup))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.back_backup, ASN1_ITEM_rptr(NEWPKI_BACKUP_BACK));
				(*Datas)->d.back_backup = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 1:
			if(!((*Datas)->d.ca_backup = (NEWPKI_BACKUP_CA*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_CA))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_caBackup).give_Datas(&(*Datas)->d.ca_backup))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.ca_backup, ASN1_ITEM_rptr(NEWPKI_BACKUP_CA));
				(*Datas)->d.ca_backup = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 7:
			if(!((*Datas)->d.ee_backup = (NEWPKI_BACKUP_EE*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_EE))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_eeBackup).give_Datas(&(*Datas)->d.ee_backup))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.ee_backup, ASN1_ITEM_rptr(NEWPKI_BACKUP_EE));
				(*Datas)->d.ee_backup = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 4:
			if(!((*Datas)->d.key_backup = (NEWPKI_BACKUP_KEY*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_KEY))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_keyBackup).give_Datas(&(*Datas)->d.key_backup))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.key_backup, ASN1_ITEM_rptr(NEWPKI_BACKUP_KEY));
				(*Datas)->d.key_backup = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 5:
			if(!((*Datas)->d.pki_backup = (NEWPKI_BACKUP_PKI*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_PKI))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_pkiBackup).give_Datas(&(*Datas)->d.pki_backup))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.pki_backup, ASN1_ITEM_rptr(NEWPKI_BACKUP_PKI));
				(*Datas)->d.pki_backup = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 3:
			if(!((*Datas)->d.pub_backup = (NEWPKI_BACKUP_PUB*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_PUB))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_pubBackup).give_Datas(&(*Datas)->d.pub_backup))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.pub_backup, ASN1_ITEM_rptr(NEWPKI_BACKUP_PUB));
				(*Datas)->d.pub_backup = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 0:
			if(!((*Datas)->d.ra_backup = (NEWPKI_BACKUP_RA*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_RA))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_raBackup).give_Datas(&(*Datas)->d.ra_backup))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.ra_backup, ASN1_ITEM_rptr(NEWPKI_BACKUP_RA));
				(*Datas)->d.ra_backup = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case 2:
			if(!((*Datas)->d.rep_backup = (NEWPKI_BACKUP_REP*)ASN1_item_new(ASN1_ITEM_rptr(NEWPKI_BACKUP_REP))))
			{
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_MALLOC);
				return false;
			}
			if(!(*m_repBackup).give_Datas(&(*Datas)->d.rep_backup))
			{
				ASN1_item_free((ASN1_VALUE*)(*Datas)->d.rep_backup, ASN1_ITEM_rptr(NEWPKI_BACKUP_REP));
				(*Datas)->d.rep_backup = NULL;
				NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
	}
	return true;
}

bool NewpkiBackup::operator=(const NewpkiBackup & other)
{
	Clear();
	if(!set_type(other.m_type))
	{
		NEWPKIerr(CRYPTO_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	switch(other.m_type)
	{
		case 6:
			if(!other.m_backBackup)
				break;
			(*m_backBackup) = *(other.m_backBackup);
			break;
		case 1:
			if(!other.m_caBackup)
				break;
			(*m_caBackup) = *(other.m_caBackup);
			break;
		case 7:
			if(!other.m_eeBackup)
				break;
			(*m_eeBackup) = *(other.m_eeBackup);
			break;
		case 4:
			if(!other.m_keyBackup)
				break;
			(*m_keyBackup) = *(other.m_keyBackup);
			break;
		case 5:
			if(!other.m_pkiBackup)
				break;
			(*m_pkiBackup) = *(other.m_pkiBackup);
			break;
		case 3:
			if(!other.m_pubBackup)
				break;
			(*m_pubBackup) = *(other.m_pubBackup);
			break;
		case 0:
			if(!other.m_raBackup)
				break;
			(*m_raBackup) = *(other.m_raBackup);
			break;
		case 2:
			if(!other.m_repBackup)
				break;
			(*m_repBackup) = *(other.m_repBackup);
			break;
	}
	m_isOk=true;
	return true;
}



const ASN1_ITEM * NewpkiBackup::get_ASN1_ITEM()
{
	return ASN1_ITEM_rptr(NEWPKI_BACKUP);
}
