/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// DlgCsrProperties.cpp: implementation of the DlgCsrProperties class.
//
//////////////////////////////////////////////////////////////////////

#include "DlgCsrProperties.h"
#include "dlgs_wdr.h"
#include "clintl.h"


//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
BEGIN_EVENT_TABLE(DlgCsrPropertiesListCtrl, wxListCtrl)
    EVT_LIST_ITEM_SELECTED(IDC_LIST_FIELDS, DlgCsrPropertiesListCtrl::OnItemClick)
END_EVENT_TABLE()


BEGIN_EVENT_TABLE(DlgCsrProperties, wxDialog)
    EVT_BUTTON(IDC_OK, DlgCsrProperties::OnOk)
    EVT_BUTTON(IDC_SAVE_CSR, DlgCsrProperties::OnSaveCsr)
END_EVENT_TABLE()

void DlgCsrPropertiesListCtrl::OnItemClick(wxListEvent& event)
{
	DlgCsrProperties * wParent = (DlgCsrProperties *)GetParent();
	wParent->OnItemClick(event);
}

DlgCsrProperties::DlgCsrProperties(wxWindow * wParent, const PKI_CSR & Csr):
		wxDialog(wParent, (wxWindowID)-1, _("CSR"), wxDefaultPosition, wxDefaultSize, wxSYSTEM_MENU | wxMINIMIZE_BOX | wxCAPTION),
		m_csr(Csr)
{
	m_imageListSmall = NULL;
	InitDatas();
}

DlgCsrProperties::~DlgCsrProperties()
{
	void * value;
	long SelectedItem = -1;
	while( (SelectedItem = m_listCtrl->GetNextItem(SelectedItem, wxLIST_NEXT_ALL, wxLIST_STATE_DONTCARE)) != -1)
	{
		value = (void*)m_listCtrl->GetItemData(SelectedItem);
		if(value)
		{
			free(value);
		}
	}
	if(m_imageListSmall)
		delete m_imageListSmall;
}

void DlgCsrProperties::InitDatas()
{
	int IconIdValue;
	int IconIdExtCrit;
	int IconIdExtNorm;

	mString datas;
	mString strDatas1;
	mString strDatas2;



	DlgCsrProperties_SetWindow(this);
	((wxTextCtrl *)FindWindow(IDC_VALUE))->SetBackgroundColour(GetBackgroundColour());

	m_listCtrl = ((DlgCsrPropertiesListCtrl *)FindWindow(IDC_LIST_FIELDS));
	m_imageListSmall = new wxImageList(16, 16, TRUE);

	wxIcon ico;
	ico.CopyFromBitmap(DlgCertProperties_GetBitmap(IDB_CERT_FIELD_VALUE));
	IconIdValue = m_imageListSmall->Add( ico );
	
	ico.CopyFromBitmap(DlgCertProperties_GetBitmap(IDB_CERT_FIELD_EXT_CRIT));
	IconIdExtCrit = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgCertProperties_GetBitmap(IDB_CERT_FIELD_EXT));
	IconIdExtNorm = m_imageListSmall->Add( ico );

	m_listCtrl->SetImageList(m_imageListSmall, wxIMAGE_LIST_SMALL);

	m_listCtrl->InsertColumn(0, _("Field"), wxLIST_FORMAT_LEFT, 135);
	m_listCtrl->InsertColumn(1, _("Value"), wxLIST_FORMAT_LEFT, 198);




	// The version
	datas.sprintf("V%ld", m_csr.GetVersion() + 1);
	InserItem(_("Version"), datas.c_str(), datas.c_str(), IconIdValue);
	
	//The signature algo
	datas = m_csr.GetSignatureAlg();
	InserItem(_("Signature algorithm"), datas.c_str(), datas.c_str(), IconIdValue);


	//The subject
	strDatas1 = FormatDN(m_csr.GetRequestDN(), true);
	strDatas2 = FormatDN(m_csr.GetRequestDN(), false);
	InserItem(_("Subject"), strDatas1.c_str(), strDatas2.c_str(), IconIdValue);

	//The key len
	datas.sprintf("%ld Bits", m_csr.GetKeyLen());
	strDatas1 = GetPubKey(m_csr.GetPublicKey());
	InserItem(_("Public key"), datas.c_str(), strDatas1.c_str(), IconIdValue);


	const char * name;
	const char * value;
	long i;
	int IconId;

	for(i=0; i < m_csr.GetExtensions().EntriesCount(); i++)
	{
		name = m_csr.GetExtensions().GetName(i);
		value = m_csr.GetExtensions().Get(i);
		if(!name || ! value) continue;

		if(strstr(value, "critical,"))
		{
			IconId = IconIdExtCrit;
			value += strlen("critical,");
			while(isspace(*value)) value++;
		}
		else
		{
			IconId = IconIdExtNorm;
		}

		InserItem(name, value, value, IconId);
	}
	CenterOnScreen();
	ShowModal();
}


mString DlgCsrProperties::GetPubKey(const EVP_PKEY * pkey)
{
	mString retValue;
	char * datas;
	BIO * membio = BIO_new(BIO_s_mem());
	if(!membio) return "";
	int KeyLen;
	BIGNUM * PubKey = NULL;;
	int i;


	if (pkey->type == EVP_PKEY_RSA)
	{
		PubKey = pkey->pkey.rsa->n;
	}
	if(!PubKey)
	{
		BIO_free_all(membio);
		return retValue;
	}

	KeyLen = BN_num_bytes(PubKey);

	datas = (char *)malloc(KeyLen + 1);
	if(!datas)
	{
		BIO_free_all(membio);
		return retValue;
	}

	if(BN_bn2bin(PubKey, (unsigned char *)datas) <= 0)
	{
		BIO_free_all(membio);
		return retValue;
	}



	for (i=0; i<KeyLen; i++)
	{
		if ((i%18) == 0 && i)
			BIO_write(membio,"\n",2);
	
		BIO_printf(membio,"%02x%s",(unsigned char)datas[i], ((i+1) == KeyLen)?"":":");
	}
	free(datas);
	

	datas = (char *)malloc(BIO_number_written(membio) + 1);
	if(!datas)
	{
		BIO_free_all(membio);
		return retValue;
	}

	BIO_read(membio, datas, BIO_number_written(membio));
	datas[BIO_number_written(membio)]=0;
	BIO_free_all(membio);

	retValue = datas;
	free(datas);

	return retValue;
}

void DlgCsrProperties::OnItemClick(wxListEvent& event)
{
	char * value;
	long SelectedItem;

	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;
	value = (char*)m_listCtrl->GetItemData(SelectedItem);

	((wxTextCtrl *)FindWindow(IDC_VALUE))->SetValue(value);
}

void DlgCsrProperties::OnOk(wxCommandEvent& event)
{
	Close(TRUE);
}

void DlgCsrProperties::OnSaveCsr(wxCommandEvent& event)
{
	long pos;
	const char * cn;
	FILE * fd;

	pos = m_csr.GetRequestDN().SeekEntryName("commonName", HASHTABLE_NOT_FOUND);
	if(pos != HASHTABLE_NOT_FOUND)
	{
		cn = m_csr.GetRequestDN().Get(pos);
		if(!cn)
			cn = "";
	}
	else
	{
		cn = "";
	}
	wxFileDialog dialog(this, _("Save CSR"), "", cn, _("CSR File (*.csr)|*.csr"), wxSAVE|wxOVERWRITE_PROMPT);

	if (dialog.ShowModal() != wxID_OK)
	{
		return;
	}
	fd = fopen(dialog.GetPath() , "wb");
	if(!fd)
	{
		HandleErrorResult(NULL, this, 0);
		return;
	}
	if (PEM_write_X509_REQ(fd, m_csr.GetX509_REQ()) <= 0)
	{
		HandleError(NULL , this);
	}
	fclose(fd);
}

void DlgCsrProperties::InserItem(const char *name, const char *value, const char *datas, int icon_id)
{
	int RowId = m_listCtrl->GetItemCount();

	m_listCtrl->InsertItem(RowId, name, icon_id);
	m_listCtrl->SetItemData(RowId, (long)strdup(datas));
	m_listCtrl->SetItem(RowId, 1, value);
}
