/* tcl_numspec.c
 
   The "numspec" Tcl command and the "numspec" Tcl object type are
   implemented here.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <tcl.h>
#include <dnet.h>

#include "missing.h"
#include "util.h"
#include "util-tcl.h"
#include "numbers.h"
#include "nexp_commands.h"

/********************************************************************
 *		     The "numspec" Tcl object type                  *
 ********************************************************************/

/* Forward declarations */
static void update_string(Tcl_Obj *);
static void free_int_rep(Tcl_Obj *);
static int set_from_any(Tcl_Interp *interp, Tcl_Obj *objPtr);
static void dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr);

Tcl_ObjType tclNumspecType = {
    .name = "numspec",
    .freeIntRepProc = &free_int_rep,
    .dupIntRepProc = &dup_int_rep,
    .updateStringProc = &update_string,
    .setFromAnyProc = &set_from_any
};

static int
set_from_any(Tcl_Interp *interp, Tcl_Obj *objPtr _U_)
{
    if (interp)
	Tcl_SetResult(interp,
		      "numspec object internal representation can't be created "
		      "from string", TCL_VOLATILE);

    return TCL_ERROR;
}

static void
dup_int_rep(Tcl_Obj *srcPtr, Tcl_Obj *dupPtr)
{
    numspec_t *old, *new;

    old = srcPtr->internalRep.otherValuePtr;
    new = num_init(num_spec(old) );

    dupPtr->typePtr = &tclNumspecType;
    dupPtr->internalRep.otherValuePtr = new;
}


static void
update_string(Tcl_Obj *numspec_obj)
{
    numspec_t *n;
    size_t len;
    const char *string_rep;

    n = numspec_obj->internalRep.otherValuePtr;

    string_rep = num_spec(n);
    len = strlen(string_rep);

    numspec_obj->bytes = ckalloc(len + 1);
    if (!numspec_obj->bytes)
	return;

    strlcpy(numspec_obj->bytes, string_rep, len + 1);

    numspec_obj->length = len;
}

static void
free_int_rep(Tcl_Obj *obj)
{
    numspec_t *n;

    n = obj->internalRep.otherValuePtr;
    num_destroy(n);
    /* XXX - need to free string representation? */
}

Tcl_Obj *
Tcl_NewNumspecObj(const char *number_spec, int hint)
{
    Tcl_Obj *obj;
    numspec_t *n;

    n = hint == HINT_NUM ? num_init(number_spec) : ip_init(number_spec);
    if (n == NULL)
	return NULL;

    obj = Tcl_NewObj();

    obj->bytes = NULL; /* Mark as invalid the string representation */
    obj->typePtr = &tclNumspecType;
    obj->internalRep.otherValuePtr = n;

    return obj;
}

/********************************************************************
 *				 num                                *
 ********************************************************************/

static int
NExp_NumObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
	       Tcl_Obj * const *objv)
{
    int index;
    Tcl_Obj *obj;
    char *numspec_def;
    static const char *subcmds[] = {
	"exists", "show", "new", "next", "count", NULL
    };
    enum subcmds {
	SUBCMD_EXISTS, SUBCMD_SHOW, SUBCMD_NEW, SUBCMD_NEXT, SUBCMD_COUNT
    };

    if (objc < 2) {
	Tcl_WrongNumArgs(interp, 1, objv, "subcmd numName ?arg ...?");
	return TCL_ERROR;
    }

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    switch ( (enum subcmds) index) {
    case SUBCMD_EXISTS: {
	Tcl_Obj *obj;

	if (objc != 3) {
	    Tcl_WrongNumArgs(interp, 2, objv, "numName");
	    return TCL_ERROR;
	}

	obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
	if (!obj || obj->typePtr != &tclNumspecType) {
	    /* Variable does not exist or it's not of our type */
	    obj = Tcl_NewBooleanObj(0);
	} else {
	    obj = Tcl_NewBooleanObj(1);
	}

	Tcl_SetObjResult(interp, obj);
	return TCL_OK;
    }
    case SUBCMD_SHOW:
	if (objc != 3) {
	    Tcl_SetResult(interp, "usage: num show <num variable>",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
	if (!obj)
	    return TCL_ERROR;

	if (obj->typePtr != &tclNumspecType) {
	    Tcl_AppendResult(interp, "\"", Tcl_GetString(objv[2]),
			     "\" is not a numspec object", NULL);
	    return TCL_ERROR;
	}

	printf("%s\n", num_spec(obj->internalRep.otherValuePtr) );
	break;
    case SUBCMD_NEW:
	if (objc < 3) {
	    Tcl_SetResult(interp, "usage: num new <numspec def>",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	numspec_def = copy_objv(objc - 2, &objv[2]);
	if (!numspec_def) {
	    Tcl_SetResult(interp, "usage: num new <numspec def>",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	obj = Tcl_NewNumspecObj(numspec_def, HINT_NUM);
	if (!obj) {
	    Tcl_SetResult(interp, "can't create numspec object",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	Tcl_SetObjResult(interp, obj);
	break;
    case SUBCMD_NEXT:
	if (objc != 3) {
	    Tcl_SetResult(interp, "usage: num next <num variable>",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
	if (!obj)
	    return TCL_ERROR;

	if (obj->typePtr != &tclNumspecType) {
	    Tcl_AppendResult(interp, "\"", Tcl_GetString(objv[2]),
			     "\" is not a numspec object", NULL);
	    return TCL_ERROR;
	}

	obj = Tcl_NewIntObj(num_next(obj->internalRep.otherValuePtr) );

	Tcl_SetObjResult(interp, obj);
	break;
    case SUBCMD_COUNT:
	if (objc != 3) {
	    Tcl_SetResult(interp, "usage: num count <num variable>",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
	if (!obj)
	    return TCL_ERROR;

	if (obj->typePtr != &tclNumspecType) {
	    Tcl_AppendResult(interp, "\"", Tcl_GetString(objv[2]),
			     "\" is not a numspec object", NULL);
	    return TCL_ERROR;
	}

	obj = Tcl_NewIntObj(num_nvalues(obj->internalRep.otherValuePtr) );
	Tcl_SetObjResult(interp, obj);
	break;
    }

    return TCL_OK;
}

/********************************************************************
 *				ipaddr                              *
 ********************************************************************/

static int
NExp_IPAddrObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int objc,
		  Tcl_Obj * const *objv)
{
    int index;
    static const char *subcmds[] = {
	"show", "new", "next", "count", NULL
    };
    enum subcmds {
	SUBCMD_SHOW, SUBCMD_NEW, SUBCMD_NEXT, SUBCMD_COUNT
    };
    Tcl_Obj *obj;
    char *numspec_def;
    char *ipaddr;

    if (Tcl_GetIndexFromObj(interp, objv[1], subcmds, "subcmd", 0, &index)
	!= TCL_OK)
	return TCL_ERROR;

    switch ( (enum subcmds) index) {
    case SUBCMD_SHOW:
	if (objc != 3) {
	    Tcl_SetResult(interp, "usage: ipaddr show <num variable>",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
	if (!obj)
	    return TCL_ERROR;

	if (obj->typePtr != &tclNumspecType) {
	    Tcl_AppendResult(interp, "\"", Tcl_GetString(objv[2]),
			     "\" is not a numspec object", NULL);
	    return TCL_ERROR;
	}

	printf("%s\n", num_spec(obj->internalRep.otherValuePtr) );
	break;
    case SUBCMD_NEW:
	if (objc < 3) {
	    Tcl_SetResult(interp, "usage: ipaddr new <numspec def>",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	numspec_def = copy_objv(objc - 2, &objv[2]);
	if (!numspec_def) {
	    Tcl_SetResult(interp, "usage: ipaddr new <numspec def>",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	obj = Tcl_NewNumspecObj(numspec_def, HINT_IP);
	if (!obj) {
	    Tcl_SetResult(interp, "can't create numspec object",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	Tcl_SetObjResult(interp, obj);
	break;
    case SUBCMD_NEXT:
	if (objc != 3) {
	    Tcl_SetResult(interp, "usage: ipaddr next <num variable>",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
	if (!obj)
	    return TCL_ERROR;

	if (obj->typePtr != &tclNumspecType) {
	    Tcl_AppendResult(interp, "\"", Tcl_GetString(objv[2]),
			     "\" is not a numspec object", NULL);
	    return TCL_ERROR;
	}

	ipaddr = ipaddr2str(ip_next(obj->internalRep.otherValuePtr) );

	obj = Tcl_NewStringObj(ipaddr, strlen(ipaddr) );

	Tcl_SetObjResult(interp, obj);
	break;
    case SUBCMD_COUNT:
	if (objc != 3) {
	    Tcl_SetResult(interp, "usage: ipaddr count <num variable>",
			  TCL_VOLATILE);
	    return TCL_ERROR;
	}

	obj = Tcl_ObjGetVar2(interp, objv[2], NULL, TCL_LEAVE_ERR_MSG);
	if (!obj)
	    return TCL_ERROR;

	if (obj->typePtr != &tclNumspecType) {
	    Tcl_AppendResult(interp, "\"", Tcl_GetString(objv[2]),
			     "\" is not a numspec object", NULL);
	    return TCL_ERROR;
	}

	obj = Tcl_NewIntObj(num_nvalues(obj->internalRep.otherValuePtr) );
	Tcl_SetObjResult(interp, obj);
	break;
    }

    return TCL_OK;
}

static struct nexp_cmd_data cmd_data[] = {
    {"num", NExp_NumObjCmd, NULL, 0, 0},
    {"ipaddr", NExp_IPAddrObjCmd, NULL, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_numspec_cmds(Tcl_Interp *interp)
{
    Tcl_RegisterObjType(&tclNumspecType);

    nexp_create_commands(interp, cmd_data);
}
