/* expect_network.c
  
   Expect_network commands

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris
   Copyright (C) 1987-2002 Don Libes <libes@cme.nist.gov>,
	National Institute of Standards and Technology <http://www.nist.gov/>

   This is part of Network Expect (nexp)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

#include <epan/packet.h> /* Apparently only for CHAR_ASCII */
#include <wiretap/libpcap.h> /* For wtap_pcap_encap_to_wtap_encap() */

#include "util-tcl.h"

/*
 * This differs from regular Expect, where the default timeout is 10
 * seconds. The rationale is that things happen at a faster speed
 * when dealing with a network than when dealing with interactive
 * processes.
 */
#define DEFAULT_TIMEOUT 0.5

/*
 * Used to pass data to the PCAP handler (pcap_handler type) invoked
 * by pcap_dispatch().
 */
struct pcap_handler_data {
    Tcl_Interp *interp;
    struct nexp_listener *listener;
    int passed_rfilter;
};

#if __FreeBSD__ || __APPLE__
#define SELECT_LIES
#endif

static struct nexp_cmd_descriptor nexp_cmd_before = {
    .cmdtype = NEXP_CMD_BEFORE,
    .duration = NEXP_PERMANENT,
    .ncases = 0,
    .cases = NULL
};

static struct nexp_cmd_descriptor nexp_cmd_after = {
    .cmdtype = NEXP_CMD_AFTER,
    .duration = NEXP_PERMANENT,
    .ncases = 0,
    .cases = NULL
};

static void
fill_in_framedata(frame_data *fdata, struct nexp_listener *l,
		  const struct pcap_pkthdr *h)
{
    int ll_type;

    nstime_t first_ts, prev_cap_ts; /* For "struct timeval" -> nstime_t
				       temporary conversions */

    fdata->next = NULL;
    fdata->prev = NULL;
    fdata->pfd = NULL;
    fdata->num = l->nreads; /* Frame number is number of times we've read
			       from this listener */
    fdata->pkt_len = h->len;
    fdata->cum_bytes  = 0; /* XXX - can I set this to 0? EP.- */
    fdata->cap_len = h->caplen;
    fdata->file_off = 0; /* XXX - can I set this to 0? EP.- */

    ll_type = l->type == LISTENER_LIVE
	      ? l->_live.datalink_type : l->_pcap.datalink_type;
    fdata->lnk_t = wtap_pcap_encap_to_wtap_encap(ll_type);

    fdata->abs_ts.secs = h->ts.tv_sec;
    fdata->abs_ts.nsecs = h->ts.tv_usec*1000;
    fdata->flags.passed_dfilter = 0;
    fdata->flags.encoding = CHAR_ASCII;
    fdata->flags.visited = 0;
    fdata->flags.marked = 0;
    fdata->flags.ref_time = 0;
    fdata->color_filter = NULL;

    /*
     * If we don't have the timestamp of the first packet provided by the
     * listener it's because this is the first packet. Save the timestamp of
     * this packet as the timestamp of the first packet.
     */
    if (!timerisset(&l->first_ts) )
	l->first_ts = h->ts;

    /*
     * Listeners keep time at microseconds resolution. Wireshark stores
     * time with nanoseconds resolution. Therefore we need to do some
     * conversions before calculating time deltas.
     */
    first_ts.secs = l->first_ts.tv_sec;
    first_ts.nsecs = l->first_ts.tv_usec*1000;

    /* Get the time elapsed between the first packet and this packet. */
    nstime_delta(&fdata->rel_ts, &fdata->abs_ts, &first_ts);

    /*
     * If we don't have the timestamp of the previous packet read from
     * this listener it's because this is the first packet. Save the
     * timestamp of this packet as the timestamp of the previous read
     * packet.
     */
    if (!timerisset(&l->prev_cap_ts) )
	l->prev_cap_ts = h->ts;

    /*
     * Listeners keep time at microseconds resolution. Wireshark stores
     * time with nanoseconds resolution. Therefore we need to do some
     * conversions before calculating time deltas.
     */
    prev_cap_ts.secs = l->prev_cap_ts.tv_sec;
    prev_cap_ts.nsecs = l->prev_cap_ts.tv_usec*1000;

    /*
     * Get the time elapsed between the previous captured packet and
     * this packet.
     */
    nstime_delta(&fdata->del_cap_ts, &fdata->abs_ts, &prev_cap_ts);

    /*
     * For us, delta between this packet and the previous captured packet
     * and delta between this packet and the previous displayed packet
     * are the same.
     */
    fdata->del_dis_ts = fdata->del_cap_ts;

    l->prev_cap_ts = h->ts;
}

/* Free up all data attached to a "frame_data" structure. */
static void
clear_fdata(frame_data *fdata)
{
    if (fdata->pfd)
	g_slist_free(fdata->pfd);
}

/*
 * Callback function for pcap_dispatch(). This function gets called by
 * pcap_dispatch() when there is a new packet that needs to be processed.
 * This function is the one that actually processes the new packet.
 */
static void
process_packet(u_char *user, const struct pcap_pkthdr *h, const u_char *bytes)
{
    struct pcap_handler_data *hd;
    frame_data fdata;
    dfilter_t *rfcode;
    const char *ns_name;

    hd = (struct pcap_handler_data *) user;

    hd->listener->nreads++; /* Increment number of times we're read from
			       this listener */

    fill_in_framedata(&fdata, hd->listener, h);

    rfcode = hd->listener->type == LISTENER_LIVE ? hd->listener->_live.rfcode
						 : hd->listener->_pcap.rfcode;

    ns_name = Tcl_GetVar(hd->interp, DISSECTION_NS_VARNAME, TCL_GLOBAL_ONLY);

    hd->passed_rfilter = pkt_dissect_tcl(hd->interp, bytes, &fdata, rfcode,
					 ns_name, 1);

    clear_fdata(&fdata);

    /*
     * We rely on the current time in some operations (when reading packets
     * from a savefile while respecting the arrival time.)
     */
    gettimeofday(&hd->listener->prev_cap_ts_real, NULL);

    if (hd->listener->type == LISTENER_SAVEFILE)
	/*
	 * We've read a packet from the save file so we need to increment
	 * the index into the array of delays (listener->_pcap.delays[]).
	 */
	hd->listener->_pcap.curr_pkt++;
}

/*
 * Some debugging code...
 */
static void
dump_nexp_cmd_descriptor(struct nexp_cmd_descriptor *ecd)
{
    int i;

    char *cmdtype;
    static char *cmdtypes[] = {
	[NEXP_CMD_BEFORE] = "NEXP_CMD_BEFORE",
	[NEXP_CMD_AFTER] = "NEXP_CMD_AFTER",
	[NEXP_CMD_BG] = "NEXP_CMD_BG",
	[NEXP_CMD_FG] = "NEXP_CMD_FG"
    };

    char *duration;
    static char *durations[] = {
	[NEXP_TEMPORARY] = "NEXP_TEMPORARY",
	[NEXP_PERMANENT] = "NEXP_PERMANENT"
    };

    char *use;
    static char *uses[] = {
	[PAT_EOF] = "PAT_EOF",
	[PAT_TIMEOUT] = "PAT_TIMEOUT",
	[PAT_DEFAULT] = "PAT_DEFAULT",
	[PAT_FULLBUFFER] = "PAT_FULLBUFFER",
	[PAT_EXPR] = "PAT_EXPR"
    };

    printf("Dumping Network Expect Command Descriptor at %p\n", ecd);

    if (ecd->cmdtype < 0 || ecd->cmdtype
	> (int) (sizeof(cmdtypes)/sizeof(cmdtypes[0]) ) )
	cmdtype = "Invalid";
    else
	cmdtype = cmdtypes[ecd->cmdtype];
    printf("  cmdtype: %s (%d)\n", cmdtype, ecd->cmdtype);

    if (ecd->duration < 0
	|| ecd->duration > (int) (sizeof(durations)/sizeof(durations[0]) ) )
	duration = "Invalid";
    else
	duration = durations[ecd->duration];
    printf("  duration: %s (%d)\n", duration, ecd->duration);

    printf("  Timeout specified by -timeout: %s\n",
	   ecd->timeout_specified_by_flag ? "yes" : "no");

    printf("  Timeout value: %f\n",
	   ecd->timeout_specified_by_flag ? ecd->timeout : 0);

    printf("  Expect Cases Descriptor:\n");
    printf("    count: %d\n", ecd->ncases);

    for (i = 0; i < ecd->ncases; i++) {
	struct ecase *ec;

	ec = ecd->cases[i];

	printf("    Case #%d: (at %p)\n", i, ecd->cases[i]);

	printf("      Listener: '%s'\n", ecd->cases[i]->listener->name);

	if (ecd->cases[i]->pat)
	    printf("      Expression: '%s' (at %p)\n",
		   Tcl_GetString(ecd->cases[i]->pat), ecd->cases[i]->pat);
	else
	    printf("      Expression: none\n");

	if (ecd->cases[i]->body)
	    printf("      Body: %s (at %p)\n",
		   Tcl_GetString(ecd->cases[i]->body), ecd->cases[i]->body);
	else
	    printf("      Body: none\n");

	if (ecd->cases[i]->use > (sizeof(uses)/sizeof(uses[0]) ) )
	    use = "Invalid";
	else
	    use = uses[ecd->cases[i]->use];
	printf("      use: %s (%d)\n", use, ecd->cases[i]->use);

	printf("      Write indices: %s\n",
	       ecd->cases[i]->indices ? "yes" : "no");
	printf("      Re-read indirects: %s\n",
	       ecd->cases[i]->iread ? "yes" : "no");
	printf("      Write timestamps: %s\n",
	       ecd->cases[i]->timestamp ? "yes" : "no");
    }

    printf("\n");
}

/*
 * Returns a pointer to an array of pointers to nexp_listener structures.
 * The members of the array are the listeners in the network expect
 * command descriptor's ecases. The last element of the list is a NULL
 * pointer and the list must be free()'ed.
 */
static struct nexp_listener **
nexp_cmd_descriptor_listeners(struct nexp_cmd_descriptor *ecmd)
{
    int i, j, inlist;
    struct nexp_listener **listeners, **ptr;

    ptr = listeners = xmalloc(sizeof(*listeners)
			      *(ecmd->ncases + 1 /* For terminating NULL */) );
    *listeners = NULL;

    for (i = 0; i < ecmd->ncases; i++) {
	/* This is a lame linear search */
	for (inlist = j = 0; listeners[j]; j++)
	    if (listeners[j] == ecmd->cases[i]->listener) {
		inlist = 1;
		break;
	    }

	if (!inlist)
	    *ptr++ = ecmd->cases[i]->listener;
    }

    *ptr = NULL;

    return listeners;
}

/*
    Return TRUE if string appears to be a set of arguments
    The intent of this test is to support the ability of commands to have
    all their args braced as one.  This conflicts with the possibility of
    actually intending to have a single argument.
    The bad case is in expect which can have a single argument with embedded
    \n's although it's rare.  Examples that this code should handle:
    \n		FALSE (pattern)
    \n\n		FALSE
    \n  \n \n	FALSE
    foo		FALSE
    foo\n	FALSE
    \nfoo\n	TRUE  (set of args)
    \nfoo\nbar	TRUE

    Current test is very cheap and almost always right :-)
*/
static int 
nexp_one_arg_braced(Tcl_Obj *objPtr)
{
    int seen_nl = 0;
    char *p = Tcl_GetString(objPtr);

    for (; *p; p++) {
	if (*p == '\n') {
	    seen_nl = 1;
	    continue;
	}

	if (!isspace(*p) ) { /* INTL: ISO space */
	    return(seen_nl);
	}
    }

    return 0;
}

/*
    Called to execute a command of only one argument - a hack to commands
    to be called with all args surrounded by an outer set of braces returns
    TCL_whatever.
*/
static int
nexp_eval_with_one_arg(ClientData clientData _U_, Tcl_Interp *interp,
		      Tcl_Obj *CONST objv[])
{
#define NUM_STATIC_OBJS 20
    Tcl_Obj *staticObjArray[NUM_STATIC_OBJS];
    int maxobjs = NUM_STATIC_OBJS;
    Tcl_Token *tokenPtr;
    char *p, *next;
    int rc;
    Tcl_Obj **objs = staticObjArray;
    int objc, bytesLeft, numWords, i;
    Tcl_Parse parse;

    /*
     * Prepend the command name and the -nobrace switch so we can
     * reinvoke without recursing.
     */
    objc = 2;
    objs[0] = objv[0];
    objs[1] = Tcl_NewStringObj("-nobrace", -1);
    Tcl_IncrRefCount(objs[0]);
    Tcl_IncrRefCount(objs[1]);

    p = Tcl_GetStringFromObj(objv[1], &bytesLeft);

    /*
     * Treat the pattern/action block like a series of Tcl commands.
     * For each command, parse the command words, perform substititions
     * on each word, and add the words to an array of values.  We don't
     * actually evaluate the individual commands, just the substitutions.
     */

    do {
	if (Tcl_ParseCommand(interp, p, bytesLeft, 0, &parse) != TCL_OK) {
	    rc = TCL_ERROR;
	    goto done;
	}
	numWords = parse.numWords;
 	if (numWords > 0) {
	    /*
	     * Generate an array of objects for the words of the command.
	     */
    
	    if (objc + numWords > maxobjs) {
		Tcl_Obj ** newobjs;
		maxobjs = (objc + numWords) * 2;
		newobjs = (Tcl_Obj **)ckalloc(maxobjs * sizeof (Tcl_Obj *));
		memcpy(newobjs, objs, objc*sizeof(Tcl_Obj *));
		if (objs != staticObjArray) {
		    ckfree((char*)objs);
		}
		objs = newobjs;   
	    }

	    /*
	     * For each word, perform substitutions then store the
	     * result in the objs array.
	     */
	    
	    for (tokenPtr = parse.tokenPtr;
		 numWords > 0;
		 numWords--, tokenPtr += (tokenPtr->numComponents + 1) ) {
		objs[objc] = Tcl_EvalTokens(interp, tokenPtr + 1,
					    tokenPtr->numComponents);
		if (objs[objc] == NULL) {
		    rc = TCL_ERROR;
		    goto done;
		}
		objc++;
	    }
	}

	/*
	 * Advance to the next command in the script.
	 */
	next = (char *) parse.commandStart + parse.commandSize;
	bytesLeft -= next - p;
	p = next;
	Tcl_FreeParse(&parse);
    } while (bytesLeft > 0);

    /*
     * Now evaluate the entire command with no further substitutions.
     */

    rc = Tcl_EvalObjv(interp, objc, objs, 0);

 done:

    for (i = 0; i < objc; i++) {
	Tcl_DecrRefCount(objs[i]);
    }

    if (objs != staticObjArray) {
	ckfree((char *) objs);
    }

    return(rc);
#undef NUM_STATIC_OBJS
}

static void
ecase_clear(struct ecase *ec)
{
    ec->listener = NULL;
    ec->pat = NULL;
    ec->body = NULL;
    ec->transfer = 1;
    ec->indices = 0;
    ec->iread = 0;
    ec->timestamp = 0;
    ec->use = PAT_EXPR;
}

static struct ecase *
ecase_new(void)
{
    struct ecase *ec;
    
    ec = (struct ecase *) ckalloc(sizeof(struct ecase) );
    ecase_clear(ec);

    return ec;
}

/* free up everything in ecase */
static void
free_ecase(struct ecase *ec, int duration)
{
    if (duration == NEXP_PERMANENT) {
	if (ec->pat)
	    Tcl_DecrRefCount(ec->pat);

	if (ec->body)
	    Tcl_DecrRefCount(ec->body);
    }

    ckfree( (char *) ec);
}

/* free up any argv structures in the ecases */
static void
free_ecases(struct nexp_cmd_descriptor *eg)
{
    int i;

    if (!eg->cases)
	return;

    for (i = 0; i < eg->ncases; i++)
	free_ecase(eg->cases[i], eg->duration);

    ckfree( (char *) eg->cases);

    eg->ncases = 0;
    eg->cases = NULL;
}

/*
    parse_expect_args parses the arguments to expect or its variants. It
    normally returns TCL_OK, and returns TCL_ERROR for failure. (It can't
    return i_list directly because there is no way to differentiate between
    clearing, say, expect_before and signalling an error.)

    eg (expect_global) is initialized to reflect the arguments parsed
    eg->ecd.cases is an array of ecases
    eg->ecd.count is the # of ecases
    eg->i_list is a linked list of exp_i's which represent the -i info

    Each exp_i is chained to the next so that they can be easily free'd if
    necessary.  Each exp_i has a reference count.  If the -i is not used
    (e.g., has no following patterns), the ref count will be 0.

    Each ecase points to an exp_i.  Several ecases may point to the same exp_i.
    Variables named by indirect exp_i's are read for the direct values.

    If called from a foreground expect and no patterns or -i are given, a
    default exp_i is forced so that the command "expect" works right.

    The exp_i chain can be broken by the caller if desired.

    default_esPtr: suggested ExpState if called as expect_user or _tty
*/

static int
parse_expect_args(Tcl_Interp *interp, struct nexp_cmd_descriptor *eg,
		  int objc, Tcl_Obj *CONST objv[])
{
    int i;
    char *string;
    struct ecase ec;	/* temporary to collect args */
    const char *previous_listener;
    int debug = 0;

    previous_listener = nexp_get_var(interp, LISTENER_SPAWN_ID_VARNAME);
    if (!previous_listener) {
	nexp_error(interp, "Use spawn_network to spawn a listener first.");
	return TCL_ERROR;
    }

    eg->timeout_specified_by_flag = 0;

    ecase_clear(&ec);

    /*
     * Allocate an array to store the ecases.  Force array even if 0
     * cases.  This will often be too large (i.e., if there are flags)
     * but won't affect anything.
     */
    eg->cases = (struct ecase **) ckalloc(sizeof(struct ecase *)
					  * (1 + objc/2) );
    eg->ncases = 0;

    for (i = 1; i < objc; i++) {
	int index;
	/*
	 * Some of these options are not being used in NetExpect. They
	 * are just leftovers from the Expect starting point. We need
	 * to make sure some of these are not used at all and remove them.
	 */
	static const char *flags[] = {
	    "-expr", "-notransfer", "-i", "-indices", "-iread",
	    "-timestamp", "-timeout", "-nobrace", "-debug", NULL
	};
	enum flags {
	    NEXP_ARG_EXPR, NEXP_ARG_NOTRANSFER, NEXP_ARG_SPAWN_ID,
	    NEXP_ARG_INDICES, NEXP_ARG_IREAD, NEXP_ARG_TIMESTAMP,
	    NEXP_ARG_DASH_TIMEOUT, NEXP_ARG_NOBRACE, NEXP_ARG_DEBUG
	};

	string = Tcl_GetString(objv[i]);

	if (string[0] == '-') {
	    /* Argument starts with '-', therefore it's a switch */

	    /*
	     * Allow abbreviations of switches and report an error if we
	     * get an invalid switch.
	     */
	    if (Tcl_GetIndexFromObj(interp, objv[i], flags, "flag", 0, &index)
		!= TCL_OK)
		return TCL_ERROR;

	    switch ( (enum flags) index) {
	    case NEXP_ARG_EXPR:
		if (++i >= objc) {
		    Tcl_WrongNumArgs(interp, 1, objv, "-expr expression");
		    return TCL_ERROR;
		}
		ec.use = PAT_EXPR;
		goto pattern;
	    case NEXP_ARG_NOTRANSFER:
		ec.transfer = 0;
		break;
	    case NEXP_ARG_SPAWN_ID:
		if (++i >= objc) {
		    Tcl_WrongNumArgs(interp, 1, objv, "-i spawn_id");
		    goto error;
		}

		previous_listener = Tcl_GetString(objv[i]);

		ec.listener = lookup_listener(previous_listener);
		if (!ec.listener) {
		    nexp_error(interp, "No listener named \"%s\". Use "
			       "\"spawn_network -info\" to find out existing "
			       "listeners.", previous_listener);
		    goto error;
		}

		break;
	    case NEXP_ARG_INDICES:
		ec.indices = 1;
		break;
	    case NEXP_ARG_IREAD:
		ec.iread = 1;
		break;
	    case NEXP_ARG_TIMESTAMP:
		ec.timestamp = 1;
		break;
	    case NEXP_ARG_DASH_TIMEOUT:
		if (++i >= objc) {
		    Tcl_WrongNumArgs(interp, 1, objv, "-timeout seconds");
		    goto error;
		}

		if (Tcl_GetDoubleFromObj(interp, objv[i], &eg->timeout)
		    != TCL_OK) {
		    goto error;
		}
		eg->timeout_specified_by_flag = 1;
		break;
	    case NEXP_ARG_NOBRACE:
		/*
		 * nobrace does nothing but take up space on the command line
		 * which prevents us from re-expanding any command lines of one
		 * argument that looks like it should be expanded to multiple
		 * arguments.
		 */
		break;
	    case NEXP_ARG_DEBUG:
		debug = 1;
		break;
	    }
	} else {
	    /*
	     * Argument doesn't start with '-', therefore we have a pattern
	     * or keyword.
	     */

	    static const char *keywords[] = {
		"timeout", "eof", "full_buffer", "default", NULL
	    };
	    enum keywords {
		NEXP_ARG_TIMEOUT, NEXP_ARG_EOF, NEXP_ARG_FULL_BUFFER,
		NEXP_ARG_DEFAULT
	    };

	    /*
	     * Match keywords exactly, otherwise they are patterns.
	     */

	    if (Tcl_GetIndexFromObj(interp, objv[i], keywords, "keyword",
				    TCL_EXACT, &index) != TCL_OK) {
		Tcl_ResetResult(interp);
		goto pattern;
	    }

	    switch ( (enum keywords) index) {
	    case NEXP_ARG_TIMEOUT:
		ec.use = PAT_TIMEOUT;
		break;
	    case NEXP_ARG_EOF:
		ec.use = PAT_EOF;
		break;
	    case NEXP_ARG_FULL_BUFFER:
		ec.use = PAT_FULLBUFFER;
		break;
	    case NEXP_ARG_DEFAULT:
		ec.use = PAT_DEFAULT;
		break;
	    }

pattern:
	    if (!ec.listener) {
		/* If no -i flag has occurred yet, use previous listener
		 * ID. */
		ec.listener = lookup_listener(previous_listener);
		if (!ec.listener) {
		    nexp_error(interp, "No listener named \"%s\".",
			       previous_listener);
		    goto error;
		}
	    }

	    /* save original pattern spec */
	    /* keywords such as "-timeout" are saved as patterns here */
	    /* useful for debugging but not otherwise used */
	    ec.pat = objv[i];
	    if (eg->duration == NEXP_PERMANENT)
		Tcl_IncrRefCount(ec.pat);

	    if (++i < objc) {
		ec.body = objv[i];
		if (eg->duration == NEXP_PERMANENT) Tcl_IncrRefCount(ec.body);
	    } else {
		ec.body = NULL;
	    }

	    /*
	     * Finally put the new ecase in the right place within the list
	     * of ecases.
	     */
	    *(eg->cases[eg->ncases] = ecase_new() ) = ec;

	    /* clear out for next set */
	    ecase_clear(&ec);

	    eg->ncases++;
	}
    }

    if (debug)
	dump_nexp_cmd_descriptor(eg);

    return(TCL_OK);

error:
    free_ecases(eg);

    return(TCL_ERROR);
}

/*
 * This function is called after some data from the network has arrived,
 * or after there has been a timeout.
 *
 * It sets *eval_out if an ecase evaluates to true or if it matches the
 * special "timeout" or "eof" patterns.
 *
 * Returns original status arg or NEXP_TCLERROR.
 */
static int
eval_cases(Tcl_Interp *interp, struct nexp_cmd_descriptor *eg,
	   struct ecase **eval_out, fd_set *rfds, int status)
{
    int i, value, fd;
    int retval;
    struct pcap_handler_data hd;
    fd_set fds_with_data; /* Not used in a select(); just used as flags */
    struct timeval tv;
    int curr_pkt;
    uint8_t buffer[1024];
    Tcl_Obj *barray;
    struct pcap_pkthdr h;

    if (*eval_out || status == NEXP_TCLERROR)
	return status;

    if (status == NEXP_TIMEOUT) {
	for (i = 0; i < eg->ncases; i++) {
	    if (eg->cases[i]->use == PAT_TIMEOUT) {
		*eval_out = eg->cases[i];
		break;
	    }
	}

	return status;
    }

    FD_ZERO(&fds_with_data);

    for (i = 0; i < eg->ncases; i++) {
	fd = listener_fd(eg->cases[i]->listener);

	/*
	 * First we read data from file descriptors that have data.
	 */
	if (FD_ISSET(fd, rfds) ) {
	    /*
	     * This file descriptor has data that we can read without
	     * blocking...
	     */

	    if (eg->cases[i]->listener->type == LISTENER_SAVEFILE
		&& !eg->cases[i]->listener->_pcap.fullspeed) {
		/*
		 * We're dealing with a listener that reads from a PCAP
		 * savefile and the "fullspeed" flag is not set so we must
		 * respect the time at which packets arrived.
		 */

		/*
		 * Calculate the time that has elapsed since the previous
		 * packet in the savefile. Leave the result in tv and use that
		 * in the timercmp() below.
		 */
		gettimeofday(&tv, NULL);

		timersub(&tv, &eg->cases[i]->listener->prev_cap_ts_real, &tv);

		curr_pkt = eg->cases[i]->listener->_pcap.curr_pkt;

		if (timercmp(&tv,
			     &eg->cases[i]->listener->_pcap.delays[curr_pkt],
			     <) ) {
		    /*
		     * The elapsed time is less than the elapsed time between
		     * packets in the savefile, so we can't process the next
		     * packet yet.
		     *
		     * It is important to reset the file descriptor associated
		     * with this ecase so a "race" condition doesn't happen and
		     * the ecases are evaluated in order.
		     */
		    FD_CLR(fd, rfds);
		    continue;
		}
	    }

	    /*
	     * Read from the listener. Where we read from depends on the
	     * listener type.
	     */
	    switch (eg->cases[i]->listener->type) {
	    case LISTENER_LIVE:
		hd.interp = interp;
		hd.listener = eg->cases[i]->listener;
		hd.passed_rfilter = 0;
		retval = pcap_dispatch(eg->cases[i]->listener->_live.pd, 1,
				       &process_packet, (u_char *) &hd);
		break;
	    case LISTENER_SAVEFILE:
		/*
		 * Before blindly reading and processing the next packet
		 * from the save file we need to determine whether the
		 * user has supplied a list of packets that need to be
		 * read, and whether the next packet to read is in that
		 * list. If the next packet to read is in that list then
		 * we read and process it. If it is not, we read it and
		 * throw it out (ignore it.)
		 */

		if (eg->cases[i]->listener->_pcap.packet_list) {
		    if (!bsearch(&eg->cases[i]->listener->_pcap.curr_pkt,
				 eg->cases[i]->listener->_pcap.packet_list,
				 eg->cases[i]->listener->_pcap.npacketsl,
				 sizeof(int), compare_integers) ) {
			/*
			 * The next packet to be read is not in the list
			 * of packets provided by the user, so it needs
			 * to be read but not processed, i.e. ignored.
			 */
			if (!pcap_next(eg->cases[i]->listener->_pcap.pd,
				       &h) ) {
			    /*
			     * We probably reached end-of-file. In this case
			     * we force processing of this packet as an
			     * EOF.
			     */
			    retval = 0;
			    break;
			}

			/*
			 * It is important to reset the file descriptor
			 * associated with this ecase so a "race" condition
			 * doesn't happen and the ecases are evaluated in
			 * order.
			 */
			FD_CLR(fd, rfds);

			/*
			 * We read one packet, so make sure we update
			 * the current packet counter.
			 */
			eg->cases[i]->listener->_pcap.curr_pkt++;
			continue;
		    }
		}

		/* Read and process the next packet. No ignoring. */
		hd.interp = interp;
		hd.listener = eg->cases[i]->listener;
		hd.passed_rfilter = 0;
		retval = pcap_dispatch(eg->cases[i]->listener->_pcap.pd, 1,
				       &process_packet, (u_char *) &hd);
		break;
	    case LISTENER_STREAM:
	    case LISTENER_DGRAM:
		retval = read(eg->cases[i]->listener->_socket.fd, buffer,
			      sizeof(buffer) );

		barray = Tcl_NewBArrayObj(buffer, retval);
		Tcl_SetVar2Ex(interp, "data", NULL, barray, 0);
		break;
	    }

	    if (retval == 0) {
		/*
		 * Okay, pcap_dispatch() returned 0. There are three
		 * possibilities here: 1) we're reading from a savefile, in
		 * which case the 0 means "EOF", 2) we ended up calling
		 * pcap_dispatch() after select() told us that a file
		 * descriptor was ready for reading but in reality the file
		 * descriptor is being spuriously reported as ready, or 3) we
		 * called this function without being sure that a file
		 * descriptor was ready for reading, i.e. before calling
		 * select.
		 *
		 * For case (1) we really need to return NEXP_EOF and do
		 * nothing else. For (2) and (3) *must* keep looking for other
		 * file descriptors that may have data. Otherwise, in some
		 * platforms, we will not process packets in cases where
		 * select() will not return "file ready for reading" when there
		 * is more than one packet in the queue.
		 */
		if (eg->cases[i]->listener->type == LISTENER_SAVEFILE) {
		    for (i = 0; i < eg->ncases; i++) {
			if (eg->cases[i]->use == PAT_EOF) {
			    *eval_out = eg->cases[i];
			    break;
			}
		    }

		    return NEXP_EOF;
		} else
		    continue;
	    } else if (retval == -1) {
		/*
		 * Something bad happened; handle the error. For all listener
		 * types but LISTENER_SAVEFILE we return NEXP_TCLERROR, which
		 * causes execution of the "expect_network" command to be
		 * terminated and TCL_ERROR to be returned to the Tcl
		 * interpreter. However, for listeners of type
		 * LISTENER_SAVEFILE we return NEXP_EOF to treat the error as
		 * an end-of-file situation. This allows us to correctly
		 * process a PCAP file that was cut short in the middle of a
		 * packet.
		 */

		switch (eg->cases[i]->listener->type) {
		case LISTENER_LIVE:
		    nexp_error(interp, "pcap_dispatch() error: %s",
			       pcap_geterr(eg->cases[i]->listener->_live.pd) );
		    break;
		case LISTENER_SAVEFILE:
		    nexp_error(interp,
			       "pcap_dispatch() error while reading from "
			       "PCAP savefile: %s. Handling as EOF.\n",
			       pcap_geterr(eg->cases[i]->listener->_pcap.pd) );

		    for (i = 0; i < eg->ncases; i++) {
			if (eg->cases[i]->use == PAT_EOF) {
			    *eval_out = eg->cases[i];
			    break;
			}
		    }

		    return NEXP_EOF;
		case LISTENER_STREAM:
		case LISTENER_DGRAM:
		    nexp_error(interp, "read() error: %s", strerror(errno) );
		    break;
		}

		return NEXP_TCLERROR;
	    }

	    /*
	     * I don't see a reason to check whether retval > 0 since
	     * if we reach this point it is because there was some
	     * data ready to be read.
	     *
	     * The only thing that needs to be done is to make
	     * sure we don't call pcap_dispatch() again for this
	     * same file descriptor. This would happen (if we don't
	     * prevent it) if more than one expect case references the
	     * same file descriptor.
	     */
	    FD_CLR(fd, rfds);

	    /*
	     * Set semaphore so we know below that we need to evaluate
	     * ecases corresponding to this FD.
	     */
	    FD_SET(fd, &fds_with_data);
	}

	/*
	 * If the dissected packet did not pass the wireshark display filter
	 * then we ignore it, and don't even bother evaluating the expect
	 * case. It's just like if the packet didn't pass the PCAP capture
	 * filter.
	 */
	if (  (eg->cases[i]->listener->type == LISTENER_LIVE
	    || eg->cases[i]->listener->type == LISTENER_SAVEFILE)
	    && !hd.passed_rfilter)
	    continue;

	/*
	 * Now we evaluate the expect case. Note that we only evaluate
	 * the ecase *if* we previously read data for the ecase's file
	 * descriptor and that we do *not* evaluate the ecase if it is
	 * not of type PAT_EXPR.
	 */
	if (FD_ISSET(fd, &fds_with_data) && eg->cases[i]->use == PAT_EXPR) {
	    retval = Tcl_ExprBooleanObj(interp, eg->cases[i]->pat, &value);
	    if (retval != TCL_OK)
		return NEXP_TCLERROR;

	    if (value) {
		*eval_out = eg->cases[i];
		return NEXP_MATCH;
	    }
	}
    }

    return NEXP_NOMATCH;
}

/*
eo = final case of interest
*/
static int
process_match(Tcl_Interp *interp, struct ecase *eo)
{
    int retval = TCL_OK;

    /*
     * eo could be NULL (if there was a timeout or reached EOF, and there
     * isn't a "timeout" or "eof" expect case.)
     */
    if (eo) {
	/*
	 * Do not evaluate empty bodies. This allows for
	 * expect_network commands that only have an expression,
	 * i.e. "expect_network {expression}".
	 */
	if (eo->body)
	    retval = Tcl_EvalObjEx(interp, eo->body, 0);
    }

    return retval;
}

#ifndef __APPLE__

/*
 * The main expect and match loop. This function is called by
 * NExp_ExpectNetworkObjCmd() and NExp_ExpectNetworkBackgroundObjCmd().
 */
static int
expect_network(Tcl_Interp *interp, struct nexp_cmd_descriptor *eg)
{
    int i, retval, fd;
    struct timeval tv, *tvptr;
    fd_set rfds;
    int highest_fd;
    struct ecase *eo;
    double timeout;
    Tcl_Obj *obj;

restart: /* We jump here if a body ends with the "nexp_continue" command. */

    /*
     * This is important; otherwise we'd end up processing a match
     * that has not occurred at the end of the following loop and
     * when the nexp_continue command is used.
     */
    eo = NULL;

    /*
     * We do not leave this loop until we timeout waiting for an
     * ecase's expression to be true, or one ecase's expression is
     * true.
     */
    for (;;) {

#ifdef SELECT_LIES
	/*
	 * In some platforms select() returns "file descriptor ready for
	 * reading" only *once* when there is more than one packet ready to be
	 * read. For instance, if there are two packets ready to be read, the
	 * first call to select() will return "file descriptor ready for
	 * reading". Then the packet is read, and then select() gets called
	 * again. This second call to select() will just sit there, waiting for
	 * a new packet to arrive even though there's one packet in the queue
	 * ready to be read. On Linux this doesn't happen and select() will
	 * always return "file descriptor ready for reading" whenever there are
	 * packets ready to be read.
	 *
	 * The following code tries to read from all file descriptors in an
	 * attempt to detect which ones have data. All reads must be done on
	 * file descriptors that are in non-blocking mode (non-blocking mode is
	 * set during listener creation) to avoid blocking. If no file
	 * descriptor has data to be read then we proceed to wait for data
	 * using select().
	 *
	 * The way we accomplish this is by setting all file descriptors in the
	 * file descriptor set used by eval_cases() to "fake" that they all are
	 * ready for reading.
	 *
	 * To confirm whether a certain platform is affected by this you can
	 * run the example icmp.nexp (in the examples/ directory) and suspend
	 * it immediately. From another machine run "ping -c 2 <IP that
	 * icmp.nexp is faking>" and then restart icmp.nexp.  If the platform
	 * is affected by this issue then you'll see only one ICMP echo reply
	 * come back because only the first one will be processed and then
	 * icmp.nexp will block in select() waiting for new packets to arrive.
	 * If the platform is not affected then you will just see an increased
	 * round-trip time. Instead of adding symbols to the pre-processor
	 * conditional we should really write an autoconf test for this. Maybe
	 * one day; patches welcome.
	 */

	FD_ZERO(&rfds);

	for (i = 0; i < nexp_cmd_before.ncases; i++)
	    FD_SET(listener_fd(nexp_cmd_before.cases[i]->listener), &rfds);

	for (i = 0; i < eg->ncases; i++)
	    FD_SET(listener_fd(eg->cases[i]->listener), &rfds);

	for (i = 0; i < nexp_cmd_after.ncases; i++)
	    FD_SET(listener_fd(nexp_cmd_after.cases[i]->listener), &rfds);

	retval = eval_cases(interp, &nexp_cmd_before, &eo, &rfds, 1);
	retval = eval_cases(interp, eg, &eo, &rfds, retval);
	retval = eval_cases(interp, &nexp_cmd_after, &eo, &rfds, retval);

	if (retval == NEXP_TCLERROR)
	    return TCL_ERROR;

	if (retval == NEXP_TIMEOUT || retval == NEXP_EOF || eo)
	    break;

	/*
	 * So, we tried to read from all file descriptors and we didn't
	 * find any with data, or those with data didn't match any
	 * ecase. Now we can call select() and block inside it. The
	 * important thing is that we are now sure that there is no
	 * data ready to be read in some file descriptor that select()
	 * will not tell us anything about, which could be the case in
	 * some platforms, as mentioned above.
	 */
#endif /* SELECT_LIES */

	FD_ZERO(&rfds);

	/*
	 * Arm file descriptor sets. We arm file descriptors sets for
	 * the expect_before and expect_after commands in addition to
	 * the file descriptors of this expect command.
	 */

	/* Arm file descriptor sets for expect_before ecases */
	for (i = highest_fd = 0; i < nexp_cmd_before.ncases; i++) {
	    fd = listener_fd(nexp_cmd_before.cases[i]->listener);

	    /*
	     * We may set a file descriptor in rfds more than once (if more
	     * than one expect case references the same listener), but that's
	     * okay.
	     */
	    FD_SET(fd, &rfds);
	    if (fd > highest_fd)
		highest_fd = fd;
	}

	/* Arm file descriptor sets for this expect command's ecases */
	for (i = 0; i < eg->ncases; i++) {
	    fd = listener_fd(eg->cases[i]->listener);

	    /*
	     * We may set a file descriptor in rfds more than once (if more
	     * than one expect case references the same listener), but that's
	     * okay.
	     */
	    FD_SET(fd, &rfds);
	    if (fd > highest_fd)
		highest_fd = fd;
	}

	/* Arm file descriptor sets for expect_after ecases */
	for (i = 0; i < nexp_cmd_after.ncases; i++) {
	    fd = listener_fd(nexp_cmd_after.cases[i]->listener);

	    /*
	     * We may set a file descriptor in rfds more than once (if more
	     * than one expect case references the same listener), but that's
	     * okay.
	     */
	    FD_SET(fd, &rfds);
	    if (fd > highest_fd)
		highest_fd = fd;
	}

	/*
	 * Timeout is calculated inside the loop so we can change the
	 * timeout inside the expect body and then use nexp_continue
	 * to run expect again with the new timeout.
	 */
	if (eg->timeout_specified_by_flag)
	    timeout = eg->timeout;
	else {
	    obj = nexp_get_var2ex(interp, TIMEOUT_VARNAME);
	    if (obj)
		Tcl_GetDoubleFromObj(interp, obj, &timeout);
	    else
		timeout = DEFAULT_TIMEOUT;
	}

	/*
	 * No timeout if "timeout" is set to -1 or if the expect command
	 * is expect_network_background.
	 */
	if (timeout == -1.0 || eg->cmdtype == NEXP_CMD_BG)
	    tvptr = NULL;
	else {
	    tv.tv_sec = timeout;
	    tv.tv_usec = (timeout - tv.tv_sec) * 1000000UL;
	    tvptr = &tv;
	}

	/*
	 * Here we go; the most important system call...
	 */
	retval = select(highest_fd + 1, &rfds, NULL, NULL, tvptr);
	if (retval >= 0) {
	    /*
	     * Either data is now available, or there was a timeout before
	     * anything interesting happened.
	     */
	    if (retval == 0)
		retval = NEXP_TIMEOUT;

	    /*
	     * Keep in mind that all the NEXP_* values (NEXP_EOF,
	     * NEXP_TIMEOUT, NEXP_TCLERROR, NEXP_MATCH, and NEXP_NOMATCH)
	     * are negative. So, if retval > 0 after select() then we'll
	     * be fine in eval_cases() because eval_cases() only checks for two
	     * specific status codes, NEXP_TCLERROR and NEXP_TIMEOUT. In other
	     * words, it is okay to *only* initialize retval to NEXP_TIMEOUT if
	     * select() returns 0.
	     */

	    retval = eval_cases(interp, &nexp_cmd_before, &eo, &rfds, retval);
	    retval = eval_cases(interp, eg, &eo, &rfds, retval);
	    retval = eval_cases(interp, &nexp_cmd_after, &eo, &rfds, retval);

	    if (retval == NEXP_TCLERROR)
		return TCL_ERROR;

	    if (retval == NEXP_TIMEOUT || retval == NEXP_EOF || eo)
		break;

	    /*
	     * No match was made with current data, go back to the top
	     * of the loop and wait for more data.
	     */
	} else {
	    nexp_error(interp, "select() error. retval = %d, errno = %d",
		       retval, errno);
	    return TCL_ERROR;
	}
    }

    retval = process_match(interp, eo);
    if (retval == NEXP_CONTINUE)
	/*
	 * Last command in body was "nexp_continue" - restart
	 * the expect_network command.
	 */
	goto restart;

    return retval;
}

#else /* ifndef __APPLE__ */

/*
 * The main expect and match loop. This function is called by
 * NExp_ExpectNetworkObjCmd() and NExp_ExpectNetworkBackgroundObjCmd().
 */
static int
expect_network(Tcl_Interp *interp, struct nexp_cmd_descriptor *eg)
{
    int i, retval;
    struct timeval start_time, curr_time, tv;
    fd_set rfds;
    struct ecase *eo;
    double timeout;
    Tcl_Obj *obj;

restart: /* We jump here if a body ends with the "nexp_continue" command. */

    /*
     * This is important; otherwise we'd end up processing a match
     * that has not occurred at the end of the following loop and
     * when the nexp_continue command is used.
     */
    eo = NULL;

    gettimeofday(&start_time, NULL);

    /*
     * We do not leave this loop until we timeout waiting for an
     * ecase's expression to be true, or one ecase's expression is
     * true.
     */
    for (;;) {
	FD_ZERO(&rfds);

	for (i = 0; i < nexp_cmd_before.ncases; i++)
	    FD_SET(listener_fd(nexp_cmd_before.cases[i]->listener), &rfds);

	for (i = 0; i < eg->ncases; i++)
	    FD_SET(listener_fd(eg->cases[i]->listener), &rfds);

	for (i = 0; i < nexp_cmd_after.ncases; i++)
	    FD_SET(listener_fd(nexp_cmd_after.cases[i]->listener), &rfds);

	retval = eval_cases(interp, &nexp_cmd_before, &eo, &rfds, 1);
	retval = eval_cases(interp, eg, &eo, &rfds, retval);
	retval = eval_cases(interp, &nexp_cmd_after, &eo, &rfds, retval);

	if (retval == NEXP_TCLERROR)
	    return TCL_ERROR;

	if (retval == NEXP_TIMEOUT || retval == NEXP_EOF || eo)
	    break;

	/*
	 * Timeout is calculated inside the loop so we can change the
	 * timeout inside the expect body and then use nexp_continue
	 * to run expect again with the new timeout.
	 */
	if (eg->timeout_specified_by_flag)
	    timeout = eg->timeout;
	else {
	    obj = nexp_get_var2ex(interp, TIMEOUT_VARNAME);
	    if (obj)
		Tcl_GetDoubleFromObj(interp, obj, &timeout);
	    else
		timeout = DEFAULT_TIMEOUT;
	}

	/*
	 * No timeout if "timeout" is set to -1 or if the expect command
	 * is expect_network_background.
	 */
	if (timeout == -1.0 || eg->cmdtype == NEXP_CMD_BG)
	    continue;

	tv.tv_sec = timeout;
	tv.tv_usec = (timeout - tv.tv_sec) * 1000000UL;

	timeradd(&start_time, &tv, &tv);

	gettimeofday(&curr_time, NULL);

	if (!timercmp(&curr_time, &tv, <) ) {
	    retval = NEXP_TIMEOUT;
	    break;
	}
    }

    retval = process_match(interp, eo);
    if (retval == NEXP_CONTINUE)
	/*
	 * Last command in body was "nexp_continue" - restart
	 * the expect_network command.
	 */
	goto restart;

    return retval;
}
#endif /* __APPLE__ */

static int
NExp_ExpectNetworkObjCmd(ClientData clientData, Tcl_Interp *interp,
			 int objc, Tcl_Obj *CONST objv[])
{
    int retval;
    struct nexp_cmd_descriptor eg;

    if (objc == 2 && nexp_one_arg_braced(objv[1]) )
	return nexp_eval_with_one_arg(clientData, interp, objv);
    else if (objc == 3 && !strcmp(Tcl_GetString(objv[1]), "-brace") ) {
	Tcl_Obj *new_objv[2];

	new_objv[0] = objv[0];
	new_objv[1] = objv[2];

	return nexp_eval_with_one_arg(clientData, interp, new_objv);
    }

    /*
     * make arg list for processing cases
     * do it dynamically, since expect can be called recursively
     */

    eg.duration = NEXP_TEMPORARY;
    eg.cmdtype = NEXP_CMD_FG;
    eg.ncases = 0;

    if (parse_expect_args(interp, &eg, objc, objv) == TCL_ERROR)
	return TCL_ERROR;

    retval = expect_network(interp, &eg);

    free_ecases(&eg);

    return retval;
}

static void
nexp_listener_append(Tcl_Interp *interp, struct nexp_listener *l)
{
    Tcl_AppendElement(interp, "-i");
    Tcl_AppendElement(interp, l->name);
}

static void
ecase_append(Tcl_Interp *interp, struct ecase *ec)
{
    if (!ec->transfer) Tcl_AppendElement(interp,"-notransfer");
    if (ec->indices) Tcl_AppendElement(interp,"-indices");

    if (ec->use == PAT_EXPR) Tcl_AppendElement(interp,"-expr");

    Tcl_AppendElement(interp,Tcl_GetString(ec->pat));
    Tcl_AppendElement(interp,ec->body?Tcl_GetString(ec->body):"");
}

/* return current setting of the permanent expect_before/after/bg */
static int
netexpect_info(Tcl_Interp *interp, struct nexp_cmd_descriptor *ecmd, int objc,
	       Tcl_Obj *CONST objv[])
{
    int i;
    int all = 0;	/* report on all fds */
    const char *spawn_id = NULL;

    /* start with 2 to skip over "cmdname -info" */
    for (i = 2; i < objc; i++) {
	int index;
	static const char *flags[] = {"-i", "-all", NULL};
	enum flags {NEXP_ARG_I, NEXP_ARG_ALL};

	/*
	 * Allow abbreviations of switches and report an error if we
	 * get an invalid switch.
	 */
	if (Tcl_GetIndexFromObj(interp, objv[i], flags, "flag", 0, &index)
	    != TCL_OK)
	    return TCL_ERROR;

	switch ( (enum flags) index) {
	case NEXP_ARG_I:
	    i++;
	    if (i >= objc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-i spawn_id");
		return TCL_ERROR;
	    }
	    spawn_id = Tcl_GetString(objv[i]);
	    break;
	case NEXP_ARG_ALL:
	    all = 1;
	    break;
	}
    }

    if (all) {
	/* avoid printing out -i when redundant */
	struct nexp_listener *previous = NULL;

	for (i = 0; i < ecmd->ncases; i++) {
	    if (previous != ecmd->cases[i]->listener) {
		nexp_listener_append(interp, ecmd->cases[i]->listener);
		previous = ecmd->cases[i]->listener;
	    }
	    ecase_append(interp, ecmd->cases[i]);
	}

	return TCL_OK;
    }

    if (!spawn_id) {
	/* Use current spawn ID (in the $spawn_id variable) */
	spawn_id = nexp_get_var(interp, LISTENER_SPAWN_ID_VARNAME);
	if (!spawn_id)
	    return TCL_ERROR;
    }

    for (i = 0; i < ecmd->ncases; i++) {
	if (!strcmp(ecmd->cases[i]->listener->name, spawn_id) )
	    ecase_append(interp, ecmd->cases[i]);
    }

    return TCL_OK;
}

/*
 * The expect_network_before and expect_network_after commands.
 */
static int
NExp_ExpectGlobalObjCmd(ClientData clientData, Tcl_Interp *interp,
			 int objc, Tcl_Obj *CONST objv[])
{
    struct nexp_cmd_descriptor *ecmd, eg;
    struct nexp_listener **listeners;
    int i, j, new_count;
    int start_index; /* where to add new ecases in old list */

    ecmd = (struct nexp_cmd_descriptor *) clientData;

    if (objc == 2 && nexp_one_arg_braced(objv[1]) )
	return nexp_eval_with_one_arg(clientData, interp, objv);
    else if (objc == 3 && !strcmp(Tcl_GetString(objv[1]), "-brace") ) {
	Tcl_Obj *new_objv[2];

	new_objv[0] = objv[0];
	new_objv[1] = objv[2];

	return nexp_eval_with_one_arg(clientData, interp, new_objv);
    }

    if (objc > 1 && !strcmp(Tcl_GetString(objv[1]), "-info") )
	return netexpect_info(interp, ecmd, objc, objv);

    /*
     * Use the old expect command descriptor as a template to initialize
     * the new expect command descriptor. This is important because
     * some Tcl objects are locked (via refcounts) depending on the
     * value of certain fields in the expect command descriptor,
     * especifically ecmd->duration.
     */
    eg = *ecmd;

    if (parse_expect_args(interp, &eg, objc, objv) == TCL_ERROR)
	return TCL_ERROR;

    /*
     * We now have the new expect network command descriptor in eg.
     *
     * The next step is to remove all the ecases from the old expect network
     * command descriptor (which we received as a parameter) that are
     * associated with listeners specified in the new expect command. To do
     * this we first build a list of all the listeners in the ecases of the new
     * expect command, and then go over the ecases in the old expect network
     * command descriptor.
     */

    listeners = nexp_cmd_descriptor_listeners(&eg);

    /*
     * We've got the list of listeners used by the new ecases. Now traverse
     * the array of ecases in the old expect network command descriptor
     * to see what needs to be deleted.
     */
    for (i = 0; i < ecmd->ncases; i++)
	for (j = 0; listeners[j]; j++)
	    if (ecmd->cases[i]->listener == listeners[j]) {
		/*
		 * This ecase (ecmd->cases[i]) is associated with a listener
		 * that has been specified in the new expect statement, so
		 * free it and mark it for removal.
		 */
		free_ecase(ecmd->cases[i], ecmd->duration);
		ecmd->cases[i] = NULL;
		break;
	    }

    free(listeners);

    /*
     * The ecases in the old network expect command that are associated with
     * listeners used by ecases in the new network expect command are now
     * gone. Now we need to append the new ecases in the new network expect
     * command to the list of ecases in the old network expect command.
     */

    if (eg.ncases == 0)
	return TCL_OK;

    new_count = ecmd->ncases + eg.ncases;

    /*
     * Determine where to place the new ecases and allocate memory for
     * them.
     */
    if (ecmd->ncases > 0) {
	/* Append to the end */
	ecmd->cases = (struct ecase **) ckrealloc( (char *) ecmd->cases,
						  new_count
						  *sizeof(struct ecase *) );
	start_index = ecmd->ncases;
    } else {
	/* Insert at the beginning */
	ecmd->cases = (struct ecase **) ckalloc(eg.ncases
						*sizeof(struct ecase *) );
	start_index = 0;
    }

    /* Put in place the new ecases */
    memcpy(&ecmd->cases[start_index], eg.cases,
	   eg.ncases*sizeof(struct ecase *) );

    /*
     * ecmd->cases[] now contains all the ecases we need to have. However,
     * there may be gaps marked by NULL pointers if we deleted some ecases
     * above. We now need to compress ecmd->cases[] by removing these gaps.
     */
    for (j = i = 0; i < new_count; i++) {
	if (ecmd->cases[i] == NULL)
	    continue;

	ecmd->cases[j++] = ecmd->cases[i];
    }

    /*
     * Finally, adjust the size of ecmd->cases[] and the number of ecases
     * in the old expect network command.
     */
    ecmd->ncases = j;
    ecmd->cases = (struct ecase **) ckrealloc( (char *) ecmd->cases,
					      j*sizeof(struct ecase *) );

    return TCL_OK;
}

/********************************************************************
 *		      expect_network_background                     *
 ********************************************************************/

static pid_t pgid = -1; /* Process group ID of all expect_network_background
			   processes. Initialized after execution of the first
			  expect_network_background command. */

static int
NExp_ExpectNetworkBackgroundObjCmd(ClientData clientData, Tcl_Interp *interp,
				   int objc, Tcl_Obj *CONST objv[])
{
    int pid, retval;
    struct nexp_cmd_descriptor eg;
    Tcl_Obj *obj;

    if (objc == 2 && nexp_one_arg_braced(objv[1]) )
	return nexp_eval_with_one_arg(clientData, interp, objv);
    else if (objc == 3 && !strcmp(Tcl_GetString(objv[1]), "-brace") ) {
	Tcl_Obj *new_objv[2];

	new_objv[0] = objv[0];
	new_objv[1] = objv[2];

	return nexp_eval_with_one_arg(clientData, interp, new_objv);
    } else if (objc == 2 && !strcmp(Tcl_GetString(objv[1]), "-cancel") ) {
	/*
	 * User requested termination of all expect_network_background
	 * processes. Send TERM signal to the process group.
	 */
	if (pgid != -1)
	    kill(-pgid, SIGTERM);

	return TCL_OK;
    }

    /*
     * make arg list for processing cases
     * do it dynamically, since expect can be called recursively
     */

    eg.duration = NEXP_PERMANENT;
    eg.cmdtype = NEXP_CMD_BG;
    eg.ncases = 0;

    if (parse_expect_args(interp, &eg, objc, objv) == TCL_ERROR)
	return TCL_ERROR;

    /*
     * Go into the background. The child process will be the one doing
     * all the work. The parent process will just continue execution of
     * the main thread and go back immediately.
     */
    pid = fork();
    if (pid < 0) {
	nexp_error(interp, "can't fork: %s", strerror(errno) );
	goto error;
    } else if (pid > 0) {
	/*
	 * Parent.
	 */

	/*
	 * Establish the child's process group here too (it is established
	 * for the child below).
	 */
	if (pgid == -1)
	    pgid = pid;

	setpgid(pid, pgid);

	/*
	 * Make the result of the "expect_network_background" command
	 * the PID of the child process.
	 */
	obj = Tcl_NewIntObj(pid);
	Tcl_SetObjResult(interp, obj);

	free_ecases(&eg);

	return TCL_OK;
    }

    /*
     * Child.
     */

    /*
     * Establish new process group for the child process since
     * we inherit the parent's process group after a fork(), which
     * is something we do not want.
     */
    if (pgid == -1)
	/*
	 * A process group ID does not exist so we create a new
	 * one. The ID will be this child process' PID.
	 */
	pgid = getpid();

    setpgid(0 /* use the child's PID */, pgid);

    retval = expect_network(interp, &eg);

    free_ecases(&eg);

    /*
     * This is the child so we do not return, we just finish execution of
     * the thread.
     */
    _exit(EXIT_SUCCESS);

error:
    free_ecases(&eg);

    /*
     * This is the child so we do not return, we just finish execution of
     * the thread.
     */
    _exit(EXIT_FAILURE);
}

static struct nexp_cmd_data cmd_data[] = {
    {
	"expect_network",
	NExp_ExpectNetworkObjCmd,
	NULL,
	0,
	0
    },
    {
	"expect_network_before",
	NExp_ExpectGlobalObjCmd,
	NULL,
	(ClientData) &nexp_cmd_before,
	0
    },
    {
	"expect_network_after",
	NExp_ExpectGlobalObjCmd,
	NULL,
	(ClientData) &nexp_cmd_after,
	0
    },
    {
	"expect_network_background",
	NExp_ExpectNetworkBackgroundObjCmd,
	NULL,
	NULL,
	0
    },

    {
	NULL,
	NULL,
	NULL,
	0,
	0
    }
};

/*
 * Function registered with atexit() and whose funcion is to kill all child
 * processes that are members of the "expect_network_background" process
 * group. We do this so we don't leave stray processes that have been
 * created by expect_network_background commands.
 */
static void
terminate_children(void)
{
    if (pgid != -1)
	kill(-pgid, SIGTERM);
}

/*
 * The signal handler function -- only gets called when a SIGCHLD is received,
 * i.e. when a child terminates. The only job of the signal handler is to reap
 * the children processes to prevent zombies.
 */
static void
sig_chld(int signo _U_)
{
    int status;

    if (pgid != -1)
	waitpid(-pgid, &status, WNOHANG);
}

void
nexp_init_expect_cmds(Tcl_Interp *interp)
{
    Tcl_Obj *obj;
    struct sigaction act;

    nexp_create_commands(interp, cmd_data);

    /* Initialize default expect_network timeout */
    obj = Tcl_NewDoubleObj(DEFAULT_TIMEOUT);
    Tcl_SetVar2Ex(interp, TIMEOUT_VARNAME, NULL, obj, 0);

    /*
     * Register an atexit() function whose job will be to make sure that we
     * don't leave stray processes when we exit.
     */
    if (atexit(terminate_children) != 0) {
	fprintf(stderr, "cannot set atexit() function terminate_children()\n");
	exit(EXIT_FAILURE);
    }

    act.sa_handler = sig_chld;
    sigemptyset(&act.sa_mask); /* We don't want to block any other signals */
    /*
     * We're only interested in children that have terminated, not ones
     * which have been stopped.
     */
    act.sa_flags = SA_NOCLDSTOP;

    /*
     * Make these values effective. If we were writing a real 
     * application, we would probably save the old value instead of 
     * passing NULL.
     */
    if (sigaction(SIGCHLD, &act, NULL) < 0) {
	fprintf(stderr, "sigaction failed\n");
	exit(EXIT_FAILURE);
    }
}
