/*
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Collections;
using Nemerle.Utility;

using Nemerle.Compiler;
using Nemerle.Compiler.Typedtree;
using Nemerle.Compiler.SolverMacros;

using PT = Nemerle.Compiler.Parsetree;

namespace Nemerle.Compiler
{
  public partial class Typer
  {
    public abstract class DelayedAction
    {
      public abstract Resolve (fail_loudly : bool) : option [TExpr];

      public override ToString () : string
      {
        "delayed macro"
      }
    }

    public class DelayedLambdaAction : DelayedAction
    {
      desc : string;
      fn : bool -> option [TExpr];

      public override ToString () : string
      { 
        desc
      }

      public override Resolve (fail_loudly : bool) : option [TExpr]
      {
        fn (fail_loudly)
      }
      
      public this (fn : bool -> option [TExpr], desc = "delayed typing action")
      {
        this.fn = fn;
        this.desc = desc;
      }
    }

    
    // TODO: object model seems to fit better here, upgrade most kinds to use DelayedAction
    public class DelayedTyping : Located, Nemerle.IComparable [DelayedTyping]
    {
      id : int;
      typer : Typer;
      local_context : LocalContext;
      expected : TyVar;
      mutable generic_specifier : list [TyVar];
      static mutable current_id : int;

      [Record]
      public variant Kind
      {
        | MemberAccess { expr : TExpr; name : PT.Name; }
        | Overloaded { overloads : list [OverloadPossibility]; }
        | Operator { t1 : TyVar; t2 : TyVar; name : string; env : GlobalEnv; }
        | Resolved { expr : TExpr; }
        | Macro { action : DelayedAction }
        | Error

        public mutable filtering_expression : TExpr.Call;
        public mutable need_ref : bool;
        public mutable need_write : bool;

        public this () {}
      }
      

      public CompareTo (o : DelayedTyping) : int
      {
        id - o.id
      }

      
      public DtKind : Kind
      {
        get {
          typer.solver.dt_store.Get (this)
        }
      }


      public ApplyGenericSpecifier (tyvars : list [TyVar]) : void
      {
        assert (generic_specifier == null);
        generic_specifier = tyvars;
        // apply it (possibly)
        SetKind (DtKind);
      }
      
 
      SetKind (k : Kind) : void
      {
        match (k) {
          | Kind.Resolved (expr) =>
            _ = typer.Expect (expected, expr.Type, "resolved overload");
          | _ => {}
        }

        def k =
          match (k) {
            | Kind.Resolved ((TExpr.StaticRef (_, meth is IMethod, _)) as expr) =>
              match (meth.BuiltinKind) {
                | BuiltinMethodKind.OpCode (ch, unch) =>
                  def opcode = if (local_context.IsChecked) ch else unch;
                  Kind.Resolved (TExpr.OpCode (expr.loc, expr.ty, opcode))
                | _ => k
              }
            | Kind.Overloaded (lst) =>
              Kind.Overloaded (OverloadPossibility.Unique (lst))
            | _ => k
          }

        when (generic_specifier != null)
          match (k) {
            | Kind.Overloaded (lst) =>
              foreach (o in lst)
                o.SetGenericSpecifier (generic_specifier);
              generic_specifier = null;
              
            | Kind.Resolved (expr) =>
              typer.ApplyGenericSpecifier (expr, generic_specifier);
              generic_specifier = null;
              
            | _ => {}
          }
          
        def solver = Passes.Solver;

        match (solver.dt_store.Find (this)) {
          | Some (k') =>
            when (k.filtering_expression == null)
              k.filtering_expression = k'.filtering_expression;
            k.need_ref = k'.need_ref;
            k.need_write = k'.need_write;
          | None => {}
        }

        solver.dt_store = solver.dt_store.Replace (this, k);

        match (k) {
          | Kind.Resolved (expr) =>
            when (k.need_write)
              typer.CheckLValue (expr, k.need_ref, 
                 if (k.need_ref) "ref/out parameter"
                 else "assignment target");
          | _ => {}
        }
      }
      
      
      public this (t : Typer, k : Kind, e : TyVar)
      {
        typer = t;
        local_context = t.local_context;
        current_id++;
        id = current_id;
        expected = e;
        SetKind (k)
      }


      public IsMutableIndexer : bool
      { 
        get {
          match (DtKind) {
            | Kind.Overloaded (lst) =>
              List.ForAll (lst, fun (op : OverloadPossibility) {
                match (op.Member) {
                  | p is IProperty => p.IsMutable && p.IsIndexer
                  | _ => false
                }
              })

            | Kind.Resolved (TExpr.PropertyMember (_, p))
            | Kind.Resolved (TExpr.StaticPropertyRef (_, p)) =>
              p.IsIndexer && p.IsMutable
            
            | _ => false
          }
        }
      }

      public IsFunctional : bool
      {
        get {
          match (DtKind) {
            | Kind.Overloaded (lst) =>
              lst.ForAll (fun (op) { op.Member is IMethod })

            | Kind.Resolved (TExpr.StaticRef (_, _ is IMethod, _)) => true
            
            | _ => false
          }
        }
      }
      
      public ExpectLValue (need_ref : bool) : bool
      {
        def kind = DtKind;
        kind.need_write = true;
        when (need_ref)
          kind.need_ref = true;
          
        match (kind) {
          | Kind.Overloaded (lst) =>
            !need_ref &&
            match (List.RevFilter (lst, fun (o : OverloadPossibility) {
              match (o.Member) {
                | p is IProperty => p.IsMutable && !need_ref
                | f is IField => f.IsMutable
                | _ => false
              }
            })) {
              | [] => false
              | newlst =>
                SetKind (Kind.Overloaded (newlst));
                true
            }
 
          | Kind.Resolved (r) => typer.IsLValue (r, need_ref)

          | Kind.MemberAccess
          | Kind.Macro
          | Kind.Error => true

          | Kind.Operator => false
        }
      }
      
      public IsResolved : bool
      {
        get {
          match (DtKind) {
            | Kind.Resolved
            | Kind.Error => true
            | _ => false
          }
        }
      }


      public ResolutionResult : TExpr
      {
        get {
          match (DtKind) {
            | Kind.Resolved (e) => e
            | Kind.Error => TExpr.Error ()
            | _ => assert (false)
          }
        }
      }


      public override ToString () : string
      {
        match (DtKind) {
          | Resolved (a) => "resolved expression " + a.ToString ()
          | Error => "an error"
          | Overloaded (overloads) =>
            "ambiguity between overloads:\n  " + overloads.ToString ("\n  ")
          | MemberAccess (expr, name) =>
            $ "accessing member `$(name)' in `$(expr)' : $(expr.Type)"
          | Operator (t1, t2, name, _) =>
            $ "operator `$(name)' on $(t1)"
            + (if (t2 == null) "" else $ " and $(t2)")
          | Macro (action) => action.ToString ()
        }
      }


      public GetDescriptionForError () : string
      {
        match (DtKind) {
          | Overloaded (overloads) =>
            def used (o) { o.UsedLastTime }
            "ambiguity between overloads:\n  " + 
            (if (overloads.Exists (used))
               overloads.Filter (used) 
             else 
               overloads).ToString ("\n  ")
          | Macro (action) =>
            // the Resolve function should dump the error message
            Util.locate (loc, {
              def res = action.Resolve (true);
              when (res.IsSome)
                Message.Error ($ "the macro resolution function unexpectedly succeeded "
                                 "when run for the second time, the result is "
                                 "$(Option.UnSome (res))");
            });
            ToString ()
            
          | _  => ToString ()
        }
      }


      public IsOverloaded : bool
      {
        get { DtKind is Kind.Overloaded }
      }


      public Context : LocalContext
      {
        get { local_context }
      }


      public CanSetCallExpr : bool
      {
        get {
          match (DtKind) {
            | MemberAccess
            | Overloaded
            | Error
            | Operator => true
            | Resolved
            | Macro => false
          }
        }
      }
      

      public SetCallExpr (e : TExpr.Call) : void
      {
        def kind = DtKind;
        Util.cassert (CanSetCallExpr);
        if (kind.filtering_expression == null)
          kind.filtering_expression = e;
        else
          Util.cassert (kind.filtering_expression : object == e : object)
      }


      /*
        A side, future implementation, note:
          [Resolve] should have [badness] parameter. It would tell [Resolve]
          what kind of bad things it can do, for example:
            0 -- nothing bad
            1 -- choose a better overload, even when it is not sure it will
                 be possible to enforce
            2 -- invoke macros taking typed arguments

          Now, after we are done with typing a function:

          start:
            foreach (e in yet untyped)
              e.Resolve(0)
            if (something changed) goto start
            foreach (e in yet untyped)
              if (e.Resolve(1) succeeded) goto start
            foreach (e in yet untyped)
              if (e.Resolve(2) succeeded) goto start
            if (yet untyped.IsEmpty) ok
            else error
      */


      ResolveOverload (overloads : list [OverloadPossibility], 
                       parms : list [Parm], expected : TyVar)
                       : list [OverloadPossibility]
      {
        def solver = typer.solver;
        def messenger = solver.CurrentMessenger;

        foreach (overload in overloads)
          overload.UsedLastTime = false;

        def try_type (overload : OverloadPossibility, final) {
          def parms =
            if (overload.ExtensionMethodObject == null) parms
            else Parm (overload.ExtensionMethodObject) :: parms;
          def ct = CallTyper (typer, overload.Compile (), parms, expected);
          ct.overload_possibility = overload;
          ct.is_final = final;
          ct.is_var_args = overload.VarArgs;
          overload.ResetOverloadSelectionStuff ();
          ct.Run ();
          if (ct.result == null || (overload.VarArgs && !ct.used_var_args))
            TExpr.Error ()
          else {
            unless (ct.result is TExpr.Error) {
              overload.PermutationArray = ct.GetPermutationArray ();
              overload.UsedDefaultParms = ct.used_default_parms;
            }
            ct.result
          }
        }

        def print_error_message (overload : OverloadPossibility) {
          solver.PushState ();
          messenger.NeedMessage = true;
          _ = try_type (overload, final = false);
          solver.PopState ();
        }

        match (OverloadPossibility.OnlyPossible (overloads, null)) {
          | [] => []

          | overloads =>
            mutable ok = [];

            // Message.Debug ($ "try type: $overloads");
            foreach (overload in overloads) {
              solver.PushState ();
              unless (IsError (try_type (overload, final = false)))
                ok = overload :: ok;
              solver.PopState ();
            }
            // Message.Debug ($ "res: $ok");

            match (typer.GetBestOverloads (ok)) {
              | [] =>
                when (messenger.NeedMessage) {
                  match (overloads) {
                    | [o] =>
                      print_error_message (o);
                    | _ =>
                      ReportError (messenger, "each overload has an error during call:");
                      mutable cnt = 1;
                      foreach (o in overloads) {
                        ReportError (messenger, $ "overload #$cnt, $o");
                        print_error_message (o);
                        ++cnt;
                      }
                  }
                }
                []

              | [one] =>
                match (try_type (one, final = true)) {
                  | TExpr.Call (_func, parms, _) =>
                    def expr = DtKind.filtering_expression;
                    // XXX this shouldn't be needed
                    // expr.func = _func;
                    expr.parms = parms;
                  | _ => assert (false)
                }
                one.UsedLastTime = true;
                [one]
              
              // This is important -- we only drop the clearly evil
              // solutions. We leave the less general ones (dropped by 
              // GetBestOverloads), as it is possible we will need them 
              // later.
              //
              // However we still mark only the best overloads in case we want
              // an error message later.
              | lst => 
                foreach (o in lst)
                  o.UsedLastTime = true;
                ok 
            }
        }
      }


      LookupOperatorIn (bound : MType, name : string) : list [OverloadPossibility]
      {
        mutable res = [];
        
        def loop (ti : TypeInfo) {
          foreach (meth is IMethod in ti.LookupMember (name))
            when (meth.IsStatic)
              res = typer.ConstructMethodOverload (meth) + res;
          match (ti.GetTydecl ()) {
            | TypeDeclaration.Interface =>
              loop (InternalType.Object_tc)
            | _ =>
              Option.Iter (ti.SuperClass (), loop)
          }
        }

        match (bound) {
          | MType.Class (ti, _) =>
            loop (ti);
            res
          | _ => assert (false)
        }
      }


      LookupOperator (t : TyVar, name : string) : option [list [OverloadPossibility]]
      {
        assert (name != null);
        if (t == null)
          None ()
        else {
          match (t.AnyHint) {
            | Some (MType.Class as bound) =>
              Some (LookupOperatorIn (bound, name))
              
            | Some (MType.Array) =>
              Some (LookupOperatorIn (InternalType.Array, name))

            | Some (MType.TyVarRef (a)) =>
              LookupOperator (a.LowerBound, name)

            | Some (MType.Intersection (lst)) =>
              mutable res = [];
              foreach (MType.Class as t in lst) {
                res = LookupOperatorIn (t, name) + res
              }
              Some (res)
              
            | Some (MType.Fun) =>
              Some (LookupOperatorIn (InternalType.Object, name))

            | _ => None ()
          }
        }
      }

      DoResolve () : void
      {
        def solver = typer.solver;
        def messenger = solver.CurrentMessenger;
        match (DtKind) {
          | Kind.MemberAccess (e, name) =>
            match (typer.TypeMember (e, name, expected)) {
              | Some ([]) =>
                ReportError (messenger,
                             $ "there is no member named `$(name)' "
                               "in $(TypeOf (e)) with type $expected");
                SetKind (Kind.Error ())
                
              | Some (lst) =>
                SetKind (Kind.Overloaded (lst));
                when (DtKind.need_write && !ExpectLValue (false))
                  ReportError (messenger, $"needed writable value, got $lst");
                
                // restart with more information
                Resolve ()
                
              | None => {}
            }
          
          | Kind.Overloaded (overloads) =>
            def expr = DtKind.filtering_expression;
            def o' =
              if (expr == null) {
                OverloadPossibility.OnlyPossible (overloads, expected)
              } else {
                if (typer.Expect (expected, expr.func.Type, 
                                  "overloaded function call"))
                  ResolveOverload (overloads, expr.parms, expr.Type)
                else
                  []
              }
            
            when (o'.Length != overloads.Length || o'.Length == 1)
              match (o') {
                | [] =>
                  SetKind (Kind.Error ())
                  
                | [one] =>
                  SetKind (Kind.Resolved (one.Compile ()))

                | lst =>
                  SetKind (Kind.Overloaded (lst))
              }
            
          | Kind.Operator (t1, t2, name, env) =>
            def operators =
              match ((LookupOperator (t1, name), LookupOperator (t2, name))) {
                | (Some (l1), Some (l2)) => Some (l1 + l2)
                | (Some as s, None)
                | (None, Some as s) => s
                | (None, None) => None ()
              }
            def operators = {
              def globals = env.LookupSymbol ([name], typer.current_type);
              mutable res = [];
              foreach (meth is IMethod in globals)
                when (meth.IsStatic)
                  res = typer.ConstructMethodOverload (meth) + res;

              match ((operators, res)) {
                | (x, []) => x
                | (Some (l1), l2) => Some (l1 + l2)
                | (None, l) => Some (l)
              }
            }
            match (operators) {
              | Some ([]) =>
                ReportError (messenger, "cannot find the " + ToString ());
                SetKind (Kind.Error ())
                
              | Some (lst) =>
                SetKind (Kind.Overloaded (lst));
                Resolve ()
                
              | None => {}
            }
          
          | Kind.Resolved (expr) =>
            unless (typer.Expect (expected, expr.Type, "already resolved overload"))
              SetKind (Kind.Error ())

          | Kind.Macro (action) =>
            Util.locate (loc,
              match (action.Resolve (false)) {
                | Some (expr) =>
                  if (typer.Expect (expected, expr.Type, $ "result of $action execution"))
                    SetKind (Kind.Resolved (expr))
                  else
                    SetKind (Kind.Error ())
                  
                | None => {}
              })
           
          | Kind.Error => {}
        }
      }

      public Resolve () : void
      {
        Util.locate (loc, {
          def tmp = typer.local_context;
          try {
            typer.local_context = local_context;
            DoResolve ();
          } finally {
            typer.local_context = tmp;
          }
        })
      }
    }
  }
}
