/*
 * Copyright (c) 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle;
using Nemerle.Collections;

namespace Nemerle.Compiler 
{
  /** Represent a substitution, from static type variables to types 
      (represented by plain type variables).  */
  public class Subst
  {
    mutable map : SystemMap [int, TyVar];
    mutable solver : Solver;
    mutable empty : bool;

    public this ()
    {
      this.solver = Passes.Solver;
      empty = true;
      map = SystemMap ();
    }


    public override ToString () : string
    {
      def sb = System.Text.StringBuilder ("\n");
      map.Iter (fun (k, v) {
        _ = sb.Append ($".$k -> $v\n");
      });
      _ = sb.Append ("\n");
      sb.ToString ()
    }
    

    public IsEmpty : bool
    {
      get { empty }
    }

    
    public Add (tv : StaticTyVar, ty : TyVar) : void
    {
      empty = false;
      map = map.Add (tv.id, ty);
    }
    

    public AddList (tv : list [StaticTyVar], ty : list [TyVar]) : void
    {
      List.Iter2 (tv, ty, Add)
    }
    

    public Copy () : Subst
    {
      def res = Subst ();
      res.map = map;
      res.solver = solver;
      res.empty = empty;
      res
    }

    public MonoApply (t : MType) : MType
    {
      if (empty) t else
      match (t) {
        | MType.Class (_, []) => t
          
        | MType.Class (tc, args) =>
          MType.Class (tc, List.Map (args, ApplyTv))
          
        | MType.Fun (f, t) =>
          MType.Fun (ApplyTv (f), ApplyTv (t))
          
        | MType.Tuple (lst) =>
          MType.Tuple (List.Map (lst, ApplyTv))
          
        | MType.Void => t

        | MType.Ref (t) =>
          MType.Ref (ApplyTv (t))
          
        | MType.Out (t) =>
          MType.Out (ApplyTv (t))
          
        | MType.Array (t, rank) =>
          MType.Array (ApplyTv (t), rank)
          
        | MType.Intersection (lst) =>
          MType.Intersection (List.Map (lst, MonoApply))

//        | MType.TyVarRef (tv) => assert (!Option.IsSome (map.Find (tv.id))); t
        | MType.TyVarRef => t
      }
    }


    ApplyTv (tv : TyVar) : TyVar
    {
      if (tv.NeedNoSubst) tv
      else
        Apply (tv.Fix ())
    }

    
    public Apply (t : MType) : TyVar
    { 
      // def from = $ "$t";
      // def res =
        if (empty) t else
        match (t) {
          | MType.TyVarRef (tv) when map.Member (tv.id) =>
            Option.UnSome (map.Find (tv.id))
          | _ =>
            MonoApply (t)
         }
      // Message.Debug ($ "apply($from) --> $res");
      // res
    }


    /** [other] has to be independent, that is [this] and [other] shall not
        share variables, or use variables from the other one in result */
    public AddSubst (other : Subst) : void
    {
      if (other.empty) {}
      else {
        empty = false;
        map =
          other.map.Fold (map, fun (f, t, map : SystemMap [int, _]) {
            map.Add (f, t)
          })
      }
    }

    /** Used for combining T3 substitutions together.  */
    internal Combine (other : Subst) : void
    {
      if (other.empty) {}
      else {
        
        empty = false;
        map =
          // we have f->t in this
          map.Fold (map, fun (f, t, map) {
            def replace_targets (from, to) {
              other.map.Fold (map, fun (_) {
                | (x, MType.TyVarRef (from'), map) when from'.Id == from.Id =>
                  map.Replace (x, to)
                | (_, _, map) => map
              })
            }

            match (other.map.Find (f)) {
              // there is f->of in other
              | Some (MType.TyVarRef (of)) =>
                replace_targets (of, t).Replace (of.Id, t)
              | None => map
              | x => Util.ice ($"$x")
            }
          })
      }
    }

    public static Fresh (vars : list [StaticTyVar]) : Subst * list [TyVar]
    {
      def res = Subst ();

      if (vars.IsEmpty) {
        (res, [])
      } else {
        // Message.Debug ($ "start copying, $(Passes.Solver.CurrentMessenger.LocalError)");
        def was_error = Passes.Solver.CurrentMessenger.LocalError;
        def vars' = 
          List.Map (vars, fun (v) { 
            def tv = Solver.FreshTyVar ();
            res.Add (v, tv);
            // Message.Debug ($ "copy $v, $(v.Constraints) $(v.LowerBound)");
            tv
          });

        List.Iter2 (vars, vars', fun (v : StaticTyVar, tv : TyVar) {
          when (!v.LowerBound.Equals (InternalType.Object)) {
            def ok = tv.Require (res.Apply (v.LowerBound));
            Util.cassert (was_error || ok, $ "req fail, $tv $(res.Apply (v.LowerBound))");
          }
        });
        (res, vars')
      }
    }
  }
}
