/** \file message_recorder.h
 * Class CMessageRecorder (allows to record/replay message input/output)
 *
 * $Id: message_recorder.h,v 1.6 2005-02-22 10:14:13 besson Exp $
 */

/* Copyright, 2001 Nevrax Ltd.
 *
 * This file is part of NEVRAX NEL.
 * NEVRAX NEL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.

 * NEVRAX NEL is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with NEVRAX NEL; see the file COPYING. If not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
 * MA 02111-1307, USA.
 */

#ifndef NL_MESSAGE_RECORDER_H
#define NL_MESSAGE_RECORDER_H

#include "nel/misc/types_nl.h"
#include "buf_net_base.h"
//#include "callback_net_base.h"
#include "message.h"
#include "nel/misc/time_nl.h"
#include "nel/misc/mem_stream.h"

#include <fstream>
#include <queue>
#include <string>

using namespace std;


namespace NLNET {


class CInetAddress;

/// Type of network events (if changed, don't forget to change EventToString() and StringToEvent()
enum TNetworkEvent { Sending, Receiving, Connecting, ConnFailing, Accepting, Disconnecting, Error };


/// TNetworkEvent -> string
string EventToString( TNetworkEvent e );

/// string -> TNetworkEvent
TNetworkEvent StringToEvent( string& s );


/*
 * TMessageRecord 
 */
struct TMessageRecord
{
	/// Default constructor
	TMessageRecord( bool input = false ) : UpdateCounter(0), SockId(InvalidSockId), Message( "", input ) {}

	/// Alt. constructor
	TMessageRecord( TNetworkEvent event, TSockId sockid, CMessage& msg, sint64 updatecounter ) :
		UpdateCounter(updatecounter), Event(event), SockId(sockid), Message(msg) {}

	/// Serial to string stream
	void serial( NLMISC::CMemStream& stream )
	{
		nlassert( stream.stringMode() );
		
		uint32 len;
		string s_event;
		stream.serial( UpdateCounter );
		if ( stream.isReading() )
		{
			stream.serial( s_event );
			Event = StringToEvent( s_event );
			stream.serialHex( (uint32&)SockId );
			stream.serial( len );
			stream.serialBuffer( Message.bufferToFill( len ), len );
		}
		else
		{
			s_event = EventToString( Event );
			stream.serial( s_event );
			stream.serialHex( (uint32&)SockId );
			len = Message.length();
			stream.serial( len );
			stream.serialBuffer( const_cast<uint8*>(Message.buffer()), len ); // assumes the message contains plain text
		}
	}

	//NLMISC::TTime		Time;
	sint64				UpdateCounter;
	TNetworkEvent		Event;
	TSockId				SockId;
	CMessage			Message;
};



/**
 * Message recorder.
 * The service performs sends normally. They are intercepted and the recorder
 * plays the receives back. No communication with other hosts.
 * Warning: it works only with messages as plain text.
 * \author Olivier Cado
 * \author Nevrax France
 * \date 2001
 */
class CMessageRecorder
{
public:

	/// Constructor
	CMessageRecorder();

	/// Destructor
	~CMessageRecorder();

	/// Start recording
	bool	startRecord( const std::string& filename, bool recordall=true );

	/// Add a record
	void	recordNext( sint64 updatecounter, TNetworkEvent event, TSockId sockid, CMessage& message );

	/// Stop recording
	void	stopRecord();

	/// Start replaying
	bool	startReplay( const std::string& filename );

	/// Push the received blocks for this counter into the receive queue
	void	replayNextDataAvailable( sint64 updatecounter );

	/**
	 * Returns the event type if the counter of the next event is updatecounter,
	 * and skip it; otherwise return Error.
	 */
	TNetworkEvent	checkNextOne( sint64 updatecounter );

	/// Get the first stored connection attempt corresponding to addr
	TNetworkEvent	replayConnectionAttempt( const CInetAddress& addr );

	/// Stop playback
	void	stopReplay();

	/// Receive queue (corresponding to one update count). Use empty(), front(), pop().
	std::queue<NLNET::TMessageRecord>	ReceivedMessages;

protected:

	/// Load the next record from the file (throws EStreamOverflow)
	bool	loadNext( TMessageRecord& record );

	/// Get the next record (from the preloaded records, or from the file)
	bool	getNext( TMessageRecord& record, sint64 updatecounter );

private:

	// Input/output file
	std::fstream								_File;

	// Filename
	std::string									_Filename;

	// Preloaded records
	std::deque<TMessageRecord>					_PreloadedRecords;

	// Connection attempts
	std::deque<TMessageRecord>					_ConnectionAttempts;

	// If true, record all events including sends
	bool										_RecordAll;
};


} // NLNET


#endif // NL_MESSAGE_RECORDER_H

/* End of message_recorder.h */
