--------------------------------------------------------------------------
--                                                                      --
--           Copyright: Copyright (C) 2000-2010 CNRS/IN2P3              --
--                                                                      --
-- Narval framework is free  software; you can redistribute  it and/or  --
-- modify  it   under  terms  of  the  GNU General  Public  License as  --
-- published  by  the  Free Software Foundation; either version  2, or  --
-- (at your option) any later version. Narval framework is distributed  --
-- in the hope  that  they will  be useful, but  WITHOUT ANY WARRANTY;  --
-- without even the implied warranty of  MERCHANTABILITY or FITNESS FOR --
-- A PARTICULAR PURPOSE. See the  GNU. General Public License for more  --
-- details. You should have received  a copy of the GNU General Public  --
-- License distributed with Narval; see file COPYING. If not, write to  --
-- the Free Software  Foundation,  Inc., 51 Franklin St,  Fifth Floor,  --
-- Boston, MA 02110-1301 USA.                                           --
--------------------------------------------------------------------------
package body Narval.State_Machine is

   Error_Available_Actions : constant Action_Array := (Full_Reset,
                                                     Partial_Reset);
   Initial_Available_Actions : constant Action_Array := (1 => Configure);
   Configured_Available_Actions : constant Action_Array := (Unconfigure, Load);
   Loaded_Available_Actions : constant Action_Array := (Unload, Initialise);
   Stopped_Available_Actions : constant Action_Array := (Reset_Com,
                                                       Start);
   Running_Available_Actions : constant Action_Array := (Stop, Pause);
   Paused_Available_Actions : constant Action_Array := (Stop, Resume);

   State_After_Action : constant array (Action) of Acquisition_State_Type :=
     (Full_Reset => Initial,
      Partial_Reset => Configured,
      Configure => Configured,
      Unconfigure => Initial,
      Load => Loaded,
      Unload => Configured,
      Configure_Load => Loaded,
      Unload_Unconfigure => Initial,
      Initialise => Stopped,
      Reset_Com => Loaded,
      Start => Running,
      Stop => Stopped,
      Pause => Paused,
      Resume => Running);

   -------------
   -- Machine --
   -------------

   protected body Machine is

      function Image return String is
      begin
         return Current_Acquisition_State'Img;
      end Image;

      ------------------
      -- Etat_Courant --
      ------------------

      function Current_State return Acquisition_State_Type is
      begin
         return Current_Acquisition_State;
      end Current_State;

      ------------------
      -- Changer_Etat --
      ------------------

      procedure Change_State (Order : Action) is
      begin
         if not Order_Coherency (Order) then
            raise Invalid_Order;
         end if;
         Current_Acquisition_State := State_After_Action (Order);
      end Change_State;

      --------------------
      -- Ordre_Coherent --
      --------------------

      function Order_Coherency (Order : Action) return Boolean is
         Vector : constant Action_Array := Available_Orders;
         Valid_Order : Boolean := False;
      begin
         for I in Vector'Range loop
            Valid_Order := Valid_Order or Order = Vector (I);
         end loop;
         return Valid_Order;
      end Order_Coherency;

      -----------------------
      -- Actions_Possibles --
      -----------------------

      function Available_Orders return Action_Array is
      begin
         case Current_Acquisition_State is
            when Error =>
               return Error_Available_Actions;
            when Initial =>
               return Initial_Available_Actions;
            when Configured =>
               return Configured_Available_Actions;
            when Loaded =>
               return Loaded_Available_Actions;
            when Running =>
               return Running_Available_Actions;
            when Paused =>
               return Paused_Available_Actions;
            when Stopped =>
               return Stopped_Available_Actions;
         end case;
      end Available_Orders;

      --------------------
      -- Mise_En_Erreur --
      --------------------

      procedure Put_In_Error is
      begin
         Memory_Of_State_Before_Error := Current_Acquisition_State;
         Current_Acquisition_State := Error;
      end Put_In_Error;

      -----------------------
      -- Etat_Avant_Erreur --
      -----------------------

      function State_Before_Error return Acquisition_State_Type is
      begin
         return Memory_Of_State_Before_Error;
      end State_Before_Error;

   end Machine;

end Narval.State_Machine;
