/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <string.h>
#include <lcms.h>

#include "MRI.h"

extern char *progname;

typedef struct {
	unsigned short row;
	unsigned short col;
} StuckPixel;

struct StuckPixelData {
	struct link *next;
	int	width;
	int	height;
	int	row;
	int	nextStuck;
	int	allocStuck;		/* Number allocated. */
	StuckPixel *stuckPixels;
};

static void
InitStuckPixels (struct StuckPixelData *wd)
{
	wd->nextStuck = 0;
	wd->allocStuck = 0;
}

static void
AddStuckPixel (struct StuckPixelData *wd, int row, int col)
{
	void *new;

	if (wd->nextStuck == wd->allocStuck) {
		wd->allocStuck += 10;
		if (wd->allocStuck == 10)
			new = malloc(sizeof(StuckPixel)*10);
		else
			new = realloc((void *)wd->stuckPixels, sizeof(StuckPixel)*wd->allocStuck);
		if (new == (void *)0)
			wd->allocStuck -= 10;
		else
			wd->stuckPixels = (StuckPixel *)new;
	}
	if (wd->nextStuck < wd->allocStuck) {
		wd->stuckPixels[wd->nextStuck].row = row;
		wd->stuckPixels[wd->nextStuck].col = col;
		wd->nextStuck++;
	}
}

static int
StuckPixelCompare (const void *a, const void *b)
{
	const StuckPixel *ap = a;
	const StuckPixel *bp = b;

	if (ap->row < bp->row) return -1;
	else if (ap->row > bp->row) return 1;
	else if (ap->col < bp->col) return -1;
	else if (ap->col > bp->col) return 1;
	else return 0;
}

static void
SortStuckPixels (struct StuckPixelData *wd)
{
	qsort (wd->stuckPixels, wd->nextStuck, sizeof(StuckPixel), StuckPixelCompare);
}

static void
StuckPixelStart (void *private, int width, int height, int freedata)
{
	struct StuckPixelData *wd = private;

	(*wd->next->start) (wd->next->private, width, height, freedata);
	wd->width = width;
	wd->height = height;
	wd->nextStuck = 0;
	wd->row = 0;
}

static void
StuckPixelRow (void *private, void *data)
{
	struct StuckPixelData *wd = private;
	struct MRI_ScanLine *sl = data;
	unsigned short *R, *G, *B;
	int col;

	if (sl->scanLineType != LINETYPE_SHORT) {
		fprintf (stderr, "StuckPixelRow: Unexpected linetype %d\n", sl->scanLineType);
		exit (1);
	}
	R = (unsigned short *)sl->R;
	G = (unsigned short *)sl->G;
	B = (unsigned short *)sl->B;
	while (wd->stuckPixels[wd->nextStuck].row == wd->row) {
		col = wd->stuckPixels[wd->nextStuck].col;
		if (col < 2) {
			R[col] = R[col+2];
			G[col] = G[col+2];
			B[col] = B[col+2];
		}
		else if (col >= wd->width-2) {
			R[col] = R[col-2];
			G[col] = G[col-2];
			B[col] = B[col-2];
		}
		else {
			R[col] = 0; /*(R[col-2] + R[col+2]) >> 1; */
			G[col] = 0; /*(G[col-2] + G[col+2]) >> 1; */
			B[col] = 0; /*(B[col-2] + B[col+2]) >> 1; */
		}
		wd->nextStuck++;
	}
	wd->row++;
	(*wd->next->row) (wd->next->private, sl);
}

static void
StuckPixelClose (void *private)
{
	struct StuckPixelData *wd = private;

	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

struct link *
MRI_GenStuckPixelFilter (const char *filename, int rotate, MRI *mp, struct link *next)
{
	FILE	*f;
	struct StuckPixelData *wd = malloc (sizeof (struct StuckPixelData));
	struct link *ep = malloc (sizeof (*ep));
	int byRow, i, j, di, dj;
	int row, col;
	char buffer[128];

	if (wd == (struct StuckPixelData *)0 ||
	    ep == (struct link *)0) {
		fprintf (stderr, "StuckPixelFilter: Error: unable to allocate memory!\n");
		return next;
	}
	switch (rotate) {
	case 0: byRow = TRUE; i = 0; di = 1; j = 0; dj = 1; break;
	case 90: byRow = FALSE; i = 0; di = 1; j = mp->height-1; dj = -1; break;
	case 180: byRow = TRUE; i = mp->width-1; di = -1; j = mp->height-1; dj = -1; break;
	case 270: byRow = FALSE; i = mp->height-1; di = -1; j = 0; dj = 1; break;
	default:
		fprintf (stderr, "%s: Error: rotate must be 0, 90, 180, or 270\n", progname);
		return next;
	}
	if ((f = fopen (filename, "r")) == NULL) {
		fprintf (stderr, "%s: Error: cannot open stuck pixel file %s\n", progname, filename);
		return next;
	}
	InitStuckPixels (wd);
	while (fgets (buffer, sizeof(buffer), f) != NULL) {
		if (sscanf (buffer, "%d %d", &col, &row) == 2) {
			if (byRow)
				AddStuckPixel (wd, i+di*row, j+dj*col);
			else
				AddStuckPixel (wd, i+di*col, j+dj*row);
		}
	}
	fclose (f);
	if (byRow)
		AddStuckPixel (wd, mp->height, mp->width);
	else
		AddStuckPixel (wd, mp->width, mp->height);
	SortStuckPixels (wd);

	for (i = 0; i < wd->nextStuck; i++)
		fprintf (stderr, "Stuck pixel at %d,%d\n", wd->stuckPixels[i].row, wd->stuckPixels[i].col);

	ep->start = StuckPixelStart;
	ep->row = StuckPixelRow;
	ep->close = StuckPixelClose;
	ep->private = wd;
	wd->next = next;

	return ep;
}
