/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include "MRI.h"

struct MRISelectData {
	MRI_Link *next;
	int	iwidth, owidth;
	int	iheight, oheight;
	int	nrows;
	int	freedata;
	int	x, y;
};

void
MRISelectStart (void *private, int width, int height, int freedata)
{
	struct MRISelectData *wd = private;

	wd->iwidth = width;
	wd->iheight = height;
	if ((wd->x + wd->owidth) > wd->iwidth) {
		fprintf (stderr, "MRISelectStart: output width (%d) + x offset (%d) greater than input width (%d)\n", wd->owidth, wd->x, wd->iwidth);
		exit (1);
	}
	if ((wd->y + wd->oheight) > wd->iheight) {
		fprintf (stderr, "MRISelectStart: output height (%d) + y offset (%d) greater than input height (%d)\n", wd->oheight, wd->y, wd->iheight);
		exit (1);
	}
	wd->nrows = 0;
	wd->freedata = freedata;
	(*wd->next->start) (wd->next->private, wd->owidth, wd->oheight, TRUE);
}

void
MRISelectRow (void *private, void *data)
{
	struct MRISelectData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	if (sl->scanLineType != LINETYPE_SHORT) {
		fprintf (stderr, "MRISelectRow: unexpected line type %d\n", sl->scanLineType);
		exit (1);
	}
	if (wd->nrows >= wd->y && (wd->nrows-wd->y) < wd->oheight) {
		struct MRI_ScanLine *ol = MRI_NewScanLine (LINETYPE_SHORT, wd->owidth);
		for (x = 0; x < wd->owidth; x++) {
			((unsigned short *)(ol->R))[x] = ((unsigned short *)(sl->R))[x+wd->x];
			((unsigned short *)(ol->G))[x] = ((unsigned short *)(sl->G))[x+wd->x];
			((unsigned short *)(ol->B))[x] = ((unsigned short *)(sl->B))[x+wd->x];
		}
		(*wd->next->row) (wd->next->private, ol);
	}
	if (wd->freedata) MRI_FreeScanLine (sl);
	wd->nrows++;
}

void
MRISelectClose (void *private)
{
	struct MRISelectData *wd = private;

	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

MRI_Link *
MRI_GenSelector (int x, int y, int width, int height, MRI_Link *next)
{
	struct MRISelectData *wd = malloc (sizeof (struct MRISelectData));
	MRI_Link *ep = malloc (sizeof (*ep));

	if ((x & 1) || (y & 1)  || (width & 1) || (height & 1)) {
		fprintf (stderr, "MRI_Select: x, y, width and height must be even\n");
		exit (1);
	}
	if (wd == (struct MRISelectData *)0 || ep == (MRI_Link *)0) {
		fprintf (stderr, "MRI_Select: unable to allocate memory\n");
		exit (1);
	}
	ep->start = MRISelectStart;
	ep->row = MRISelectRow;
	ep->close = MRISelectClose;
	ep->private = wd;
	wd->next = next;
	wd->x = x;
	wd->y = y;
	wd->owidth = width;
	wd->oheight = height;
	return ep;
}
