
/*
 * $Header: /cvsroot/mpdist/mpdist/mp/print.c,v 1.3 2002/10/15 19:09:32 richbastard Exp $
 * 
 * Copyright (c) 1987-2002 Rich Burridge, Sun Microsystems Inc.
 * All rights reserved.
 * 
 * This software is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Library General Public
 * License for more details.
 * 
 * You should have received a copy of the GNU Library General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include "option.h"
#include "header.h"
#include "input.h"
#include "print.h"

static char curfname[MAXPATHLEN];    /* Current file being printed. */
static char *message_for;            /* "[Mail,News,Listing] for " line. */

static int colct;          /* Column count on current page. */
static int linect;         /* Line count on current page. */
static int pageno;         /* Page number within message. */
static int tpn;            /* Total number of pages printed. */

static FILE *ofp;          /* Output file pointer (stdout or print cmd). */

static void do_date  (void);
static void expand   (unsigned char *);
static void psdef    (char *, char *);
static void psdict   (char *, char *);
static void set_defs (void);
static void useline  (void);


char *
current_filename(void)
{
    return(curfname);
}


static void
do_date(void)     /* Output Postscript definition for the date and time. */
{
    char        *ptr;      /* Pointer to current time or date string. */
    int         len;
    time_t      clock;     /* Used by the localtime function call. */
    struct stat status;    /* Stat(2) info from current file (-modtime). */
    struct tm   *tm;       /* Used by the localtime and asctime calls. */
    char        *timenow;  /* Used to set TimeNow field with users name. */

    if (int_opt_val(O_MODTIME) &&
        int_opt_val(O_NUMNAMES)) {    /* -modtime given on command line? */
        if (!stat(curfname, &status)) {
            tm = localtime(&status.st_mtime);
            set_date_hdr(asctime(tm));
        }
    }
    if (get_date_hdr() == NULL) {
        clock = time((time_t *) 0);
        tm = localtime(&clock);
        ptr = asctime(tm);
        ptr[24] = '\0';
    } else {
        ptr = get_date_hdr();
    }

    if (!int_opt_val(O_ARTICLE)) {
        psdef("TimeNow", ptr);
    } else {
        len = strlen(ptr);
        timenow = (char *)
            malloc((unsigned) (len + 6 + strlen(who_am_i()) + 1));
        SPRINTF(timenow, "%s  - (%s)", ptr, who_am_i());
        psdef("TimeNow", timenow);
    }
}


void
End(enum action_type op)
{
    int cval, pval;

    switch (op) {
        case IS_COL:
            write_output("sf ");
            linect = 1;
            ++colct;
            if (int_opt_val(O_LANDSCAPE) &&
                !int_opt_val(O_FORCEPAGE) &&
                (int_opt_val(O_NUMCOLS) > 1)) {
                pval = (2 * pageno) + colct - 2;
            } else {
                pval = pageno;
            }
            if (int_opt_val(O_FORCEPAGE)) {
                cval = 1;
            } else {
                cval = colct;
            }
            FPRINTF(ofp, "(%1d) %1d endcol\n", pval, cval);

/* Needed for correct heading in multiple columns. */

            if (int_opt_val(O_NUMCOLS) > 1) {
                set_defs();
            }
            if (int_opt_val(O_LANDSCAPE) &&
                !int_opt_val(O_FORCEPAGE) &&
                colct != int_opt_val(O_NUMCOLS)) {
                return;
            }
            pageno++;
            break;

        case IS_FILE:
            linect = 0;
            if (int_opt_val(O_NEWPAGE) && tpn % 2) {
                Start(IS_PAGE);
                End(IS_PAGE);
            }
            break;

        case IS_LINE:
            write_output(") showline\n");
            break;

        case IS_PAGE:
            linect = 0;
            FPRINTF(ofp, "(%1d) endpage\n", pageno - 1);
    }
}


/* Display a string with PostScript-sensitive characters escaped. */

static void
expand(unsigned char *s)
{
    for (; s && *s; s++) {
        switch (*s) {
            case '\\':
                write_output("\\\\");
                break;

            case '(':
                write_output("\\(");
                break;

            case ')':
                write_output("\\)");
                break;

            default:
                if ((*s < 0x20) || (*s > 0x7e)) {
                    FPRINTF(ofp, "\\%03o", *s);
                } else if (isprint(*s)) {
                    PUTC(*s, ofp);
                }
        }
    }
}


FILE *
makecmd(char *fname)
{
    FILE *fp;              /* Output stream pointer from popen(). */
    char cmd[MAXLINE];     /* The print command to execute. */
    int bsdprint = int_opt_val(O_BSDPRINT);

    if (ofp && int_opt_val(O_TOPRINT)) {
        PCLOSE(ofp);
    }
    STRCPY(cmd, (bsdprint ? "lpr -J " : "lp -t "));
    STRCAT(cmd, fname);
    STRCAT(cmd, " ");
    if (!int_opt_val(O_BURSTPAGE)) {
        STRCAT(cmd, bsdprint ? "-h " : "-o nobanner ");
    }

/*
 * If no printer name given, try to get the default from the
 * environment.
 */

    if (!str_opt_val(O_PR_NAME)) {
        if (bsdprint) {
            set_str_opt_val(O_PR_NAME, getenv("PRINTER"));
        } else {
            set_str_opt_val(O_PR_NAME, getenv("LPDEST"));
        }
    }
    if (str_opt_val(O_PR_NAME)) {
        STRCAT(cmd, (bsdprint ? "-P " : "-d "));
        STRCAT(cmd, str_opt_val(O_PR_NAME));
    }
    if ((fp = popen(cmd, "w")) == NULL) {
        FPRINTF(stderr, "Invalid print command: %s\n", cmd);
        FPRINTF(stderr, "Defaulting to standard output.\n");
        fp = stdout;
    }
    return(fp);
}


void
Print(void)
{
    colct = 0;         /* Column count on current page. */
    linect = 0;        /* Line count on current page. */
    ofp = NULL;        /* No output stream open initially. */
    pageno = 1;        /* Page number within message. */
    tpn = 0;           /* Total number of pages printed. */
}


void
print_extra(void)
{
    char *current;
    int i;

    for (i = 0; i < int_opt_val(O_ADDHDRN); i++) {
        if ((current = get_header_n(i)) != NULL) {
            show_text(T_MIXED, get_hdr(H_ADD, i), current);
        }
    }
}


static void
psdef(char *name, char *def)     /* Do a PostScript define. */
{
    FPRINTF(ofp, "/%s (", name);
    expand((unsigned char *) def);
    write_output(") def\n");
}


static void
psdict(char *name, char *state)     /* Write dictionary definition. */
{
    FPRINTF(ofp, "statusdict /%s known { statusdict begin %s %s end } if\n",
            name, state, name);
}


void
set_column_count(int val)           /* Set column count for current page. */
{
    colct = val;
}


void
set_current_filename(char *value)
{
    STRCPY(curfname, value);
}


static void
set_defs(void)                      /* Setup PostScript definitions. */
{
    char *newsgroups;
    int i;

    if (int_opt_val(O_ARTICLE)) {
        message_for = "Article from";
    } else if (int_opt_val(O_PR_ORIG) && get_from_hdr()) {
        message_for = "From";
    }
    psdef("MailFor", message_for);

    newsgroups = get_newsgroups_hdr();
    if (int_opt_val(O_ARTICLE) && newsgroups) {    /* User. */
        for (i = 0; i < (int) strlen(newsgroups); i++) {
            if (newsgroups[i] == ',' || newsgroups[i] == '\0') {
                break;
            }
        }
        owner = (char *) realloc(owner, (unsigned int) i + 1);
        STRNCPY(owner, newsgroups, i);
        owner[i] = '\0';
    } else if (int_opt_val(O_PR_ORIG) && get_from_hdr()) {
        i = strlen(get_from_hdr());
        owner = (char *) realloc(owner, (unsigned int) i + 1);
        STRNCPY(owner, get_from_hdr(), i);
        owner[i] = '\0';
    }
    psdef("User", owner);

    do_date();                   /* TimeNow. */

    if (int_opt_val(O_TEXTDOC) && int_opt_val(O_NUMNAMES)) {
        set_subject_hdr(curfname);
    }
    psdef("Subject", (str_opt_val(O_GSUBJECT) != NULL)
          ? str_opt_val(O_GSUBJECT) : get_subject_hdr());
}


void
set_line_count(int val)          /* Set line count for current page. */
{
    linect = val;
}


void
set_message_for(char *value)
{
    message_for = value;
}


void
set_output(FILE *fp)
{
    ofp = fp;
}


void
set_page_number(int val)          /* Set page number for this message. */
{
    pageno = val;
}


/* Display the PostScript prologue file for mp. */

void
show_prologue(char *pro)
{
    FILE *cf, *pf;
    char buf[MAXLINE], tmpstr[MAXLINE];
    char cpro[MAXPATHLEN];    /* Full pathname of the common prologue file. */
    int t2;                   /* Possible extract page or line length. */

    if ((pf = fopen(pro, "r")) == NULL) {
        SPRINTF(tmpstr, "%s/%s", prologue, pro);
        if ((pf = fopen(tmpstr, "r")) == NULL) {
            FPRINTF(stderr, "%s: Prologue file %s not found.\n",
                    str_opt_val(O_APPNAME), pro);
            exit(1);
        }
    }
    while (fgets(buf, MAXLINE, pf) != NULL) {

/*
 * If "-columns #", "-linelength #" or "-pagelength #" was
 * given on the command line, then use that value, otherwise
 * get the appropriate value from the prologue file.
 */

        if (is_opt_given(O_PLEN) && EQUALN(buf, "%%PageLength")) {
            SPRINTF(buf, "%%%%PageLength %1d\n", int_opt_val(O_PLEN));
        } else if (is_opt_given(O_LLEN) && EQUALN(buf, "%%LineLength")) {
            SPRINTF(buf, "%%%%LineLength %1d\n", int_opt_val(O_LLEN));
        } else if (is_opt_given(O_NUMCOLS) && EQUALN(buf, "%%NumCols")) {
            SPRINTF(buf, "%%%%NumCols %1d\n", int_opt_val(O_NUMCOLS));
        }
        write_output(buf);

/*
 * If appropriate, extract out the pagelength, linelength or
 * the number of columns. Note that they might have just been
 * set above. It was simpler to do it this way.
 */

        tmpstr[0] = '\0';
        SSCANF(buf, "%s %d", tmpstr, &t2);
        if (EQUAL(tmpstr, "%%PageLength")) {
            set_int_opt_val(O_PLEN, t2);     /* Change the page length. */
        } else if (EQUAL(tmpstr, "%%LineLength")) {
            set_int_opt_val(O_LLEN, t2);     /* Change the line length. */
        } else if (EQUAL(tmpstr, "%%NumCols")) {
            set_int_opt_val(O_NUMCOLS, t2);  /* Change the number of columns. */
        } else if (EQUAL(tmpstr, "%%EndComments")) {

/*
 * If this is the %%EndComments line from the prologue file, then we need 
 * to read (and output to stdout), the contents of the common prologue file,
 * mp.common.ps
 */

            SPRINTF(cpro, "%s/mp.common.ps", prologue);
            if ((cf = fopen(cpro, "r")) == NULL) {
                FPRINTF(stderr, "%s: Common prologue file %s\n",
                        str_opt_val(O_APPNAME), cpro);
                exit(1);
            }
            while (fgets(buf, MAXLINE, cf) != NULL) {
                write_output(buf);
            }
            FCLOSE(cf);

/* Output the sizes of the four paper margins. */

            SPRINTF(buf, "/bottomOffset %1d def\n", get_margin(M_BOTTOM));
            write_output(buf);
            SPRINTF(buf, "/leftOffset %1d def\n", get_margin(M_LEFT));
            write_output(buf);
            SPRINTF(buf, "/rightOffset %1d def\n", get_margin(M_RIGHT));
            write_output(buf);
            SPRINTF(buf, "/topOffset %1d def\n", get_margin(M_TOP));
            write_output(buf);
        }
    }
    FCLOSE(pf);

#ifndef NO_DUPLEX
    psdict("setduplexmode", (int_opt_val(O_DUPLEX)) ? "true" : "false");
#endif /* !NO_DUPLEX */

    if (int_opt_val(O_TUMBLE)) {
        psdict("settumble", "true");
    }
    if (int_opt_val(O_FLIP)) {
        write_output("switchtopbottom\n");
    }

    if (!int_opt_val(O_BANNERS)) {
        write_output("/dobanners where { /dobanners { } put } if\n");
        write_output("/boxHeight where { /boxHeight 0 put } if\n");
    }
    write_output("%%EndProlog\n");
}


void
show_text(enum text_type op, char *hdr, char *str)
{
    static char    *fontstr[MAXTTYPES] = {
        "BoldFont ",    /* T_BOLD. */
        "BoldFont ",    /* T_MIXED. */
        NULL,           /* T_PLAIN. */
        "pf "           /* T_ROMAN. */
    };

    useline();
    if (fontstr[(int) op]) {
        write_output(fontstr[(int) op]);
    }

    if (end_of_page() && strlen(str) == 0) {
        return;
    }

    Start(IS_LINE);
    switch (op) {
        case T_BOLD:
            expand((unsigned char *) hdr);
            expand((unsigned char *) str);
            break;

        case T_MIXED:
            expand((unsigned char *) hdr);
            write_output(") show pf (");
            expand((unsigned char *) str);
            break;

        case T_PLAIN:
            expand((unsigned char *) str);
            break;

        case T_ROMAN:
            expand((unsigned char *) str);
    }
    End(IS_LINE);
}


void
show_trailer(void)
{
    write_output("%%Trailer\n");
    FPRINTF(ofp, "%%%%Pages: %1d\n", tpn);
}


void
Start(enum action_type op)
{
    switch (op) {
        case IS_COL:
            break;

        case IS_FILE:
            break;

        case IS_LINE:
            PUTC('(', ofp);
            break;

        case IS_PAGE:
            FPRINTF(ofp, "%%%%Page: ? %1d\n", ++tpn);
            FPRINTF(ofp, "%1d newpage\n", pageno);
            set_defs();
            write_output("sf ");
            break;
    }
}


static void
useline(void)        /* Called in order to ready a line for printing. */
{
    if (++linect > int_opt_val(O_PLEN) || end_of_page()) {
        End(IS_COL);
        if (end_of_page() && int_opt_val(O_FORCEPAGE)) {
            colct = int_opt_val(O_NUMCOLS);
        }
        if (colct < int_opt_val(O_NUMCOLS)) {
            return;
        }
        colct = 0;
        End(IS_PAGE);
        linect = 1;
        Start(IS_PAGE);
    }
}


void
write_output(char *line)
{
    FPUTS(line, ofp);
}
