//
// Gendarme.Rules.Design.ConsiderConvertingMethodToPropertyRule
//
// Authors:
//	Adrian Tsai <adrian_tsai@hotmail.com>
//
// Copyright (c) 2007 Adrian Tsai
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

using System;

using Mono.Cecil;

using Gendarme.Framework;
using Gendarme.Framework.Rocks;

namespace Gendarme.Rules.Design {

	public class ConsiderConvertingMethodToPropertyRule : IMethodRule {

		static readonly string [] whitelist = {	"GetEnumerator",
							"GetHashCode",
							"GetType",
							"GetTypeCode",
							"GetValue"};

		static bool StartsWith (string start, string name)
		{
			return name.StartsWith (start, true, null);
		}

		public MessageCollection CheckMethod (MethodDefinition method, Runner runner)
		{
			// rules do not apply to constructors, methods returning an array, properties
			if (method.IsConstructor || method.IsSpecialName ||
				method.ReturnType.ReturnType.IsArray () || method.IsProperty ())
				return runner.RuleSuccess;

			// rules do not apply to code generated by the compiler (e.g. anonymous methods)
			// or generated by a tool (e.g. web services)
			if (method.IsGeneratedCode ())
				return runner.RuleSuccess;

			// ignore the some common Get* method names used in the framework
			foreach (string s in whitelist) {
				if (method.Name == s)
					return runner.RuleSuccess;
			}

			// rule applies

			// If it starts with "get" or "is", has no parameters and returns something
			// -or-
			// If it starts with "set" and has 1 parameter
			if ((StartsWith ("get", method.Name) || StartsWith ("is", method.Name)) && method.Parameters.Count == 0 && method.ReturnType.ReturnType.Name != "void"
				|| StartsWith ("set", method.Name) && method.Parameters.Count == 1) {
				Location loc = new Location (method);
				Message msg = new Message ("This property-like method should probably be a property.", loc, MessageType.Warning);
				return new MessageCollection (msg);
			}

			return runner.RuleSuccess;
		}
	}
}
