/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file Curve.cc
    \brief Implementation of the Template class Curve.
    
    Magics Team - ECMWF 2004
    
    Started: Wed 5-May-2004
    
    Changes:
    
*/



#include "Curve.h"
#include "PointsHandler.h"
#include "LegendVisitor.h"


using namespace magics;

Curve::Curve() 
{}


Curve::~Curve() 
{}

/*!
 Class information are given to the output-stream.
*/		
void Curve::print(ostream& out)  const
{
	out << "Curve[";
	out << "]";
}
void Curve::shade() 
{
	setShade(new NoGraphShade());
}
void CurveArea::shade() 
{
	
}
Polyline* Curve::newCurve(BasicGraphicsObjectContainer& task) const
{

	Polyline* curve  = new Polyline();
	(*curve).setColour(*colour_);
	(*curve).setLineStyle(style_);
	(*curve).setThickness(thickness_);

	return curve;
	
}
	
bool  Curve::missing(const UserPoint& point) const
{
	if ( point.missing() ) return true;
	double x = point.x_;

	if ( x < this->x_below_ ) return true;
	if ( x > this->x_above_ ) return true;
	if ( same(x, this->x_below_) ) return true;
	if ( same(x, this->x_above_) ) return true;

	double y = point.y_;

	if ( y < this->y_below_ ) return true;
	if ( y > this->y_above_ ) return true;
	if ( same(y, this->y_below_) ) return true;
	if ( same(y, this->y_above_) ) return true;
	return false;
}



void Curve::operator()(Data<UserPoint>& data, BasicGraphicsObjectContainer& task)
{
	
	vector<double> xpos;
	vector<double> ypos;
    const Transformation& transformation = task.transformation();
    vector<UserPoint> points;

    if ( shade_->needCustomised() ) {
    	CustomisedPointsList list;
    	std::set<string> def;
    	data.customisedPoints(transformation, def, list);

    	shade();
    	(*shade_)(list, points);

    }
    else
    	data.points(points);
	if ( legend_text_ == "?" ) legend_text_ = data.legend(); 
	
	
	Polyline* curve_ = newCurve(task);


	
	
	bool last_out = false;	
	
	
	PaperPoint last, current;
	
    vector<PaperPoint> missing;
    
    for ( vector<UserPoint>::iterator point = points.begin(); point != points.end(); ++point)  	
	{
		current = transformation(*point);
		
		if (!this->missing(*point) ) {
			if ( last_out ) {
				if ( (*missing_mode_)(last, current, missing, task) ) {
					if ( line_ )
						transformation(*curve_, task);
					curve_ = newCurve(task);
				}
				missing.clear();
			}
			last = current;
			last_out = false;
			curve_->push_back(current);
			  
		} 
		else {
			missing.push_back( current );
			last_out = true;
		} 
		
	}
    if ( missing.empty() == false ) {
    	 (*missing_mode_)(last, missing.front(), missing, task);
    }

	
	
	// apply the symbol
	(*symbol_)(points, task);
	(*shade_)(*curve_);
	if ( line_ ) transformation(*curve_, task);
	
}

struct DateToFloat {
	DateToFloat() {}
	~DateToFloat() {}
	double operator()(const string& date) {
		DateTime val(date);
		return double(val);
	}
		 
};


class CurveEntry : public LegendEntry
{
public:
	CurveEntry(const string& legend, Curve& curve) : 
		LegendEntry(legend), curve_(curve) {}
	void set(const PaperPoint& point, BasicGraphicsObjectContainer& legend) 
		{ curve_.set(point, legend, *this); }
protected:
	Curve& curve_;
};

void Curve::visit(LegendVisitor& legend)
{
	if ( !legend_) return;
	CurveEntry* entry = new CurveEntry(legend_text_, *this);
	entry->userText(legend_text_);
	legend.add(entry);
}

void  Curve::set(const PaperPoint& point, BasicGraphicsObjectContainer& legend, LegendEntry& entry)
{
	Polyline* curve  = new Polyline();
	curve->setColour(*colour_);
	curve->setLineStyle(style_);
	curve->setThickness(thickness_);
	
	double width = entry.computeWidth(0.8)/2;

	PaperPoint p = entry.centreSymbolBox(point);

	
	double x = p.x();
	double y = point.y();

	curve->push_back(PaperPoint(x-width, y));
	curve->push_back(PaperPoint(x+width,  y));
	
	symbol_->legend(p, legend);
	shade_->legend(*curve);
	
	    
	legend.push_back(curve);
	
}


