

/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 
 ******************************** LICENSE ********************************/

/*! \file Matrix.h
    \brief Implementation of the Template class Matrix.
    
    Magics Team - ECMWF 2004
    
    Started: Wed 18-Feb-2004
    
    Changes:
    
*/

#include "Matrix.h"
#include "Timer.h"

using namespace magics;


bool pointInPoly(vector<double>& x, vector<double>& y, double px, double py)
{
  double x1, x2;

  /* How many times the ray crosses a line-segment */
  int crossings = 0;
  unsigned int nb = x.size();

  /* Iterate through each line */

  for ( unsigned int i = 0; i < nb; i++ ){
	
	/* This is done to ensure that we get the same result when
	   the line goes from left to right and right to left */
	if ( x[i] < x[ (i+1)%nb ] ){
		x1 = x[i];
		x2 = x[(i+1)%nb];
	} else {
		x1 = x[(i+1)%nb];
		x2 = x[i];
	}
	
	/* First check if the ray is possible to cross the line */
	if ( px > x1 && px <= x2 && ( py < y[i] || py <= y[(i+1)%nb] ) ) {
		static const float eps = 0.000001;

		/* Calculate the equation of the line */
		float dx = x[(i+1)%nb] - x[i];
		float dy = y[(i+1)%nb] - y[i];
		float k;

		if ( fabs(dx) < eps ){
			k = INFINITY;	// math.h
		} else {
			k = dy/dx;
		}

		float m = y[i] - k * x[i];
		
		/* Find if the ray crosses the line */
		float y2 = k * px + m;
		if ( py <= y2 ){
			crossings++;
	}
   }
  }
  return  ( crossings % 2 == 1 );
}

void ProjectedMatrix::build()
{
	vector<pair<int, int>  > coords(4);
	for (int r = 0; r < rows_; r++)
		for (int c = 0; c < columns_; c++)
			push_back(missing_);
	
	coords[0] = std::make_pair(0,0);
	coords[1] = std::make_pair(0,1);
	coords[2] = std::make_pair(1,1);
	coords[3] = std::make_pair(1,0);
	
	Timer  t("GridHelper::build", "GridHelper");
	for ( int row = 0 ; row < origRows_-1; row++)
		for ( int column = 0 ; column < origColumns_-1; column++)
		{
			double xmin = origColumns_;
			double xmax = 0;
			double ymin = rows_;
			double ymax = 0;
			
			vector<double> x, y, v;
			vector<double> xs, ys, vs;
			
			bool stop = true;
			
			for (vector<pair<int, int>  >::iterator p = coords.begin(); p != coords.end(); ++p) {
				int r = row+ p->first;
				int c = column+ p->second;
				double rr = rowsArray_[index(r, c)];
				if ( rr >= miny_ -  stepy_ && rr <= maxy_ + stepy_)
					stop = false;
				double cc = columnsArray_[index(r, c)];
				if ( cc >= minx_ -  stepx_ && cc <= maxx_ + stepx_ )
					stop = false;
				
			}
			if ( stop )
				// Go to next cell...
				continue;
			for (vector<pair<int, int>  >::iterator p = coords.begin(); p != coords.end(); ++p) {
				
				int r = row+ p->first;
				int c = column+ p->second;
				double  rr = rowsArray_[index(r, c)];
				double cc = columnsArray_[index(r, c)];
				
				
				 x.push_back( (cc - minx_ ) / stepx_ );
				 y.push_back( (rr - miny_ ) / stepy_);			
				 v.push_back( values_[index(r, c)]);  						
			}
			xmin = *std::min_element(x.begin(), x.end());
			xmax = *std::max_element(x.begin(), x.end());
			ymin = *std::min_element(y.begin(), y.end());
			ymax = *std::max_element(y.begin(), y.end());

			int ii = (xmin < 0) ? 0 : xmin;

			while (ii <= xmax && ii < columns_ )
			{
				int jj = (ymin < 0) ? 0 : ymin;
				while ( jj <= ymax && jj < rows_ ) { 
			
				
					// is the point inside or oiutside .. if inside use it! 
					if ( pointInPoly(x, y, ii, jj) ) 
					 {
						//feed the matrix...
						vector<double> w;
						double total = 0.;
						for (unsigned int i = 0; i < x.size(); i++) {
							double val = sqrt( ((ii - x[i])*(ii-x[i]))  + ( (jj-y[i])*(jj-y[i]) ) );
						
							w.push_back(1/ (val*val));
							if (v[i] != missing_) 
							    total += w[i];
						}
						
						double val = 0.;
						for (unsigned int i = 0; i < w.size(); i++) {
							if (v[i] != missing_) 
								val += (v[i]*w[i]/total);
							}

				     (*this)[(jj*columns_) + ii] = val;
					}
					jj++;
				}
				ii++;
			}
		}
}

void ProjectedMatrix::getReady()
{
	assert(!values_.empty());
	assert(!rowsArray_.empty());
	assert(!columnsArray_.empty());
	
	// find the bounding box! 
	
	minx_ = *std::min_element(columnsArray_.begin(), columnsArray_.end());
	maxx_ = *std::max_element(columnsArray_.begin(), columnsArray_.end());
	miny_ = *std::min_element(rowsArray_.begin(), rowsArray_.end());
	maxy_ = *std::max_element(rowsArray_.begin(), rowsArray_.end());
	
	stepx_ = (maxx_ - minx_)/(columns_-1);
    stepy_ = (maxy_ - miny_)/(rows_-1);
	double x = minx_;
	double y = miny_;
	// Create the Axis for Regular Matrix..
	for ( int i = 0; i < columns_; i++) {
		columnsAxis_.push_back(x); 
		x += stepx_;
	}
	for ( int j = 0; j < rows_; j++) {
			rowsAxis_.push_back(y); 
			y +=  stepy_;
	}
	setMapsAxis();
	
	// Now preapre the matrix! 
	build();
}


ProjectedMatrix::ProjectedMatrix(int rows, int columns): Matrix(rows, columns)
{
	       origColumns_ = columns;
	       origRows_ = rows;
	       rowsArray_.reserve(rows*columns);  
	       columnsArray_.reserve(rows*columns);
}



double Matrix::min() const {
  	 if ( min_ < DBL_MAX) 
  		 return min_;
  	
  	 for ( const_iterator val = begin(); val != end(); ++val) {
  		 if ( *val == missing_ ) continue; 
  		 if ( *val < min_ ) min_ = *val;
  		 if ( *val > max_ ) max_ = *val;
  	 }
  	 return min_;
 }


double Matrix::max() const
{
  	 if ( max_ > -DBL_MAX) 
  		 return max_; 
  	 
  	  for ( const_iterator val = begin(); val != end(); ++val) {
  	  		 if ( *val == missing_ ) continue; 
  	  		 if ( *val < min_ ) min_ = *val;
  	  		 if ( *val > max_ ) max_ = *val;
  	  }
  	 return max_;
}
   
double Matrix::interpolate(double i, double j) const    
{  
	   double xleft = std::min( left(), right());
	   double xright = std::max( left(), right());
	   double ybottom = std::min( bottom(), top());
	   double ytop = std::max( bottom(), top());
	   
	   if ( columns() == 0 || j < xleft || j > xright ) 
		   return missing_;
	   if ( i < ybottom || i > ytop ) 
   			return missing_;
		int ii = rowIndex(i);
	    	if (ii == -1) {
	    		// interpolate between 2 rows.
	    		double v1, v2;
	    		int i1, i2;
	    		boundRow(i, v1, i1, v2, i2);
	    		
	    		if (i1 == -1) return missing(); 
	    		
	    	    double a = (*this).interpolate(v1, j);
	            double b = (*this).interpolate(v2, j);
	          
	            if ( same(a, missing()) || same(b, missing()) ) return missing();
	            
	            double da = (v2-i)/(v2-v1);
	            double db = (i-v1)/(v2-v1);
	            double val = (a*da) + (b*db);
	            return val;
	        }
	        int jj = columnIndex(j);
	        if (jj == -1) {
	        	double v1, v2;
	    		int i1, i2;
	    		boundColumn(j, v1, i1, v2, i2);
	    		if (i1 == -1) return missing(); 
	    		
	        	
	    		double a = (*this)(ii, i1);
	            double b = (*this)(ii, i2);
	            
	            if ( same(a, missing()) || same(b, missing()) ) return missing();
	            
	            double da = (v2-j)/(v2-v1);
	            double db = (j-v1)/(v2-v1);
	            double val = (a*da) + (b*db);
	            return val;         
	   }
	   return (*this)(ii, jj);
}

double Matrix::nearest(double row, double col,double &rowOut, double &colOut) const    
{  
	double xleft = std::min( left(), right());
	double xright = std::max( left(), right());
	double ybottom = std::min( bottom(), top());
	double ytop = std::max( bottom(), top());
	 
	
	if ( columns() == 0 || col < xleft || col > xright ) 
		return missing_;
	if ( columns() == 0 ||  row < ybottom || row > ytop ) 
  		return missing_;
	
	double row1, row2, col1, col2;
	int rowIdx1, rowIdx2, colIdx1, colIdx2;
	
	boundRow(row, row1, rowIdx1, row2, rowIdx2);   
	if(rowIdx1 == -1 || rowIdx2 == -1)
		return missing_;
	
	boundColumn(col, col1, colIdx1, col2, colIdx2);   
	if(colIdx1 == -1 || colIdx2 == -1)
		return missing_;
	  
	int rowIdx, colIdx;
	if(fabs(row1-row) < fabs(row2-row))
	{
		rowOut= row1;
		rowIdx=rowIdx1;
	}
	else
	{
	  	rowOut= row2;
		rowIdx=rowIdx2;
	}
	
	if(fabs(col1-col) < fabs(col2-col))
	{
		colOut= col1;
		colIdx=colIdx1;
	}
	else
	{
	  	colOut= col2;
		colIdx=colIdx2;
	}
		
	return (*this)(rowIdx, colIdx);
	
}	
	