/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file AxisMethod.h
    \brief Definition of the Template class AxisMethod.
    
    Magics Team - ECMWF 2005
    
    Started: Fri 7-Oct-2005
    
    Changes:
    
*/

#ifndef AxisMethod_H
#define AxisMethod_H

#include "magics.h"
#include "AxisMethodAttributes.h"
#include "Translator.h"
#include "Factory.h"
#include "AxisItem.h"
#include "Transformation.h"
#include "PositionListAxisMethodAttributes.h"


namespace magics {

class Box;

class AxisMethod : public AxisMethodAttributes {

public:
	AxisMethod();
	virtual ~AxisMethod();
    
    virtual void set(const XmlNode& node) {
       AxisMethodAttributes::set(node);
    }
    virtual void set(const map<string, string>& map) {
    	AxisMethodAttributes::set(map);
    }
    virtual AxisMethod* clone() const {
        Log::dev() << "(const map<string, string&)---> to be checked!...\n";
        return new AxisMethod();
    }
    
    virtual void updateX(const Transformation&);
    virtual void updateY(const Transformation&);
    
    virtual double getMin() const { return AxisMethodAttributes::getMin(); }
    virtual double getMax() const { return AxisMethodAttributes::getMax(); }
    
    
    virtual void     prepare(double, double, AxisItems&);
    
    virtual void addItem(AxisItems& items, double val) const 
    	{ items.push_back(new AxisItem(val));  }
protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
	 virtual void print(ostream&) const; 
	 virtual double value(double val) const { return val; }

private:
    //! Copy constructor - No copy allowed
	AxisMethod(const AxisMethod&);
    //! Overloaded << operator to copy - No copy allowed
	AxisMethod& operator=(const AxisMethod&);

// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const AxisMethod& p)
		{ p.print(s); return s; }

};

class LogarithmicAxisMethod : public AxisMethod
{
public:
	LogarithmicAxisMethod() {}
	virtual ~LogarithmicAxisMethod() {}
	void     prepare(double, double, AxisItems&);
	
	double value(double val) const { return exp10(val); }
	
	  virtual AxisMethod* clone() const {
        Log::dev() << "LogarithmicAxisMethod::clone()---> to be checked!...\n";
        return new LogarithmicAxisMethod();
    }
  
}; 
class HyperAxisMethod : public AxisMethod
{
public:
	HyperAxisMethod() {}
	virtual ~HyperAxisMethod() {}
	void     prepare(double, double, AxisItems&);
	
	
	
	  virtual AxisMethod* clone() const {
        Log::dev() << "HyperAxisMethod::clone()---> to be checked!...\n";
        return new HyperAxisMethod();
    }
	 void updateX(const Transformation&);
	 void updateY(const Transformation&);
	 
protected:
	vector<double> hyperMin_;
	vector<double> hyperMax_;
	
		
}; 
class PositionListAxisMethod : public AxisMethod, public PositionListAxisMethodAttributes
{
public:
	PositionListAxisMethod() {}
	virtual ~PositionListAxisMethod() {}
	virtual void set(const XmlNode& node) {
	       AxisMethodAttributes::set(node);
	       PositionListAxisMethodAttributes::set(node);
	    }
	    virtual void set(const map<string, string>& map) {
	    	AxisMethodAttributes::set(map);
	    	PositionListAxisMethodAttributes::set(map);
	    }
	  virtual AxisMethod* clone() const {
    
        return new PositionListAxisMethod();
    }
	 void     prepare(double, double, AxisItems&);
}; 


class LatitudeAxisMethod : public AxisMethod
{
public:
	LatitudeAxisMethod() {}
	virtual ~LatitudeAxisMethod() {}
	
	  virtual AxisMethod* clone() const {
        Log::dev() << "LatitudeAxisMethod::clone()---> to be checked!...\n";
        return new LatitudeAxisMethod();
    }
    void addItem(AxisItems& items, double val) const 
    	{ items.push_back(new AxisLatitudeItem(val));  }	
    
}; 

class LongitudeAxisMethod : public AxisMethod
{
public:
	LongitudeAxisMethod() {}
	virtual ~LongitudeAxisMethod() {}
	
	  virtual AxisMethod* clone() const {
        Log::dev() << "LatitudeAxisMethod::clone()---> to be checked!...\n";
        return new LongitudeAxisMethod();
    }
    void addItem(AxisItems& items, double val) const 
    	{ items.push_back(new AxisLongitudeItem(val));  }	
}; 

template <>
class Translator<string, AxisMethod> { 
public:
	AxisMethod* operator()(const string& val )
	{
		return SimpleObjectMaker<AxisMethod>::create(val);
	}     

	AxisMethod* magics(const string& param)
	{
		AxisMethod* object;
		ParameterManager::update(param, object);
		return object;
	}
};

} // namespace magics
#endif
