/* 
-------------------------------------------------------------- 
Copyright 2007-2008 Max Cavallo ixamit@gmail.com - All Rights Reserved

This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, Inc., 675 Mass Ave, Cambridge MA 02139,USA; either version 2 of the License, or (at your option) any later version; incorporated herein by reference.
-------------------------------------------------------------- 
*/

#include "finestre.h"

/*
Gray Scaling Images
Reducing images to a gray scale is a common image processing task which is also very simple to implement. The simplest method to do this is simply to take the green value and use apply the same value to Red and Blue pixels. This works well because the eye responds to green light in RGB colours much more strongly that the Red and Blue values. Obviously completely saturated Reds and Blues will not be represented this way. A more accurate conversion can be performed using the ITU standard, which is:
    (222 * Red + 707 * Green + 71 * Blue) / 1000 

Using this calculation results in a gray value between 0 and 255 which should be applied to each of the Red, Green and Blue pixels.


  grayv = (r * 30 + 59 * g + 11 *b)/100;
  grayv = grayv << 8;

*/

void WX11_ColorGradiant (Pixmap pixmap, int width, int hight, int from_color, int to_color)
{

    unsigned int new_color=from_color;
    int new_R=0;
    int new_G=0;
    int new_B=0;

    int from_R=(from_color & 0xFF0000)>>16;
    int from_G=(from_color & 0x00FF00)>>8;
    int from_B=(from_color & 0x0000FF)>>0;
    int to_R=(to_color & 0xFF0000)>>16;
    int to_G=(to_color & 0x00FF00)>>8;
    int to_B=(to_color & 0x0000FF)>>0;

    int delta_R=to_R-from_R;
    int delta_G=to_G-from_G;
    int delta_B=to_B-from_B;

    Display *dpy=WX11_GetDisplay();
    GC gc=WX11_GCGet();

    int k;
    for (k=0;k<hight;k++)
    {
        new_R=(from_R+(delta_R * k) / hight);
        new_G=(from_G+(delta_G * k) / hight);
        new_B=(from_B+(delta_B * k) / hight);

        new_color=(new_R<<16)+(new_G<<8)+new_B;
        XSetForeground(dpy, gc, WX11_RGB(new_color));

        XDrawLine(dpy, pixmap, gc, 0, k, width,k );

    }

}

void WX11_ColorBackgroundSet (CmdInfo *cmd, unsigned int gradient_from, unsigned int gradient_to)
{

	Display *dpy=WX11_GetDisplay();
	Pixmap pixmap;
	if (gradient_from == gradient_to)
	{
		XSetWindowBackground(dpy, cmd->win, WX11_RGB(gradient_from));
	}
	else
	{
        	pixmap=XCreatePixmap(dpy, cmd->win, cmd->Width, cmd->Height, WX11_GetDepth ());

        	WX11_ColorGradiant (pixmap, cmd->Width, cmd->Height, gradient_from,gradient_to);

        	XSetWindowBackgroundPixmap (dpy,cmd->win,pixmap);

        	XFreePixmap(dpy, pixmap);
	}

        if (cmd->Bordo)
            WX11_BorderRelief (cmd);

	XClearWindow (WX11_GetDisplay(),cmd->win);
}


unsigned int WX11_RGB (unsigned int colore)
{
	// Input 24 Bit Color
	int R=(colore & 0xFF0000)>>16;
	int G=(colore & 0x00FF00)>>8;
	int B=(colore & 0x0000FF)>>0;
	float PR=0;
	float PG=0;
	float PB=0;

	//return;
	if (WX11_GetDepth ()==24) 
	{
		return (colore) ;
	}

	// RGB percent
	if (R) PR=(float) R*100/255;
	if (G) PG=(float) G*100/255;
	if (B) PB=(float) B*100/255;

	
	if (WX11_GetDepth ()==16)
	{
		// RRRRRGGGGGGBBBBB (565)
		R=0x1F*PR/100;
		G=0x3F*PG/100;
		B=0x1F*PB/100;
		//
		return ((R<<11)+(G<<5)+B);
	}
	if (WX11_GetDepth ()==15)
	{
		// RRRRRGGGGGBBBBB (555)
		R=0x1F*PR/100;
		G=0x1F*PG/100;
		B=0x1F*PB/100;
		//
		return ((R<<10)+(G<<5)+B);
	}

	printf ("Color %dbits not supported\n",WX11_GetDepth ());
	return (colore) ;
}

unsigned int WX11_ColorSet (int Where,unsigned int colore)
{
	//ConversioneColore (&colore);

	switch (Where)
	{
		case WINBG:
			ColorInfo.WinBg=colore;
			break;
		case WINFG:
			ColorInfo.WinFg=colore;
			break;
		case BUTTONBG:
			ColorInfo.ButtonBg=colore;
			break;
		case BUTTONFG:
			ColorInfo.ButtonFg=colore;
			break;
		case INPUTBG:
			ColorInfo.InputBg=colore;
			break;
		case INPUTFG:
			ColorInfo.InputFg=colore;
			break;
		case MENUBG:
			ColorInfo.MenuBg=colore;
			break;
		case MENUFG:
			ColorInfo.MenuFg=colore;
			break;
		case LISTBOXBG:
			ColorInfo.ListCmdBg=colore;
			break;
		case LISTBOXFG:
			ColorInfo.ListCmdFg=colore;
			break;
	default:
			WX11_FatalError ("Errore Funzione WX11_ColorSet");
			break;
	}
	return (colore);	
}

unsigned int WX11_ColorGet (int Where)
{
	unsigned int colore=0;

	switch (Where)
	{
		case WINBG:
			colore=ColorInfo.WinBg;
			break;
		case WINFG:
			colore=ColorInfo.WinFg;
			break;
		case BUTTONBG:
			colore=ColorInfo.ButtonBg;
			break;
		case BUTTONFG:
			colore=ColorInfo.ButtonFg;
			break;
		case INPUTBG:
			colore=ColorInfo.InputBg;
			break;
		case INPUTFG:
			colore=ColorInfo.InputFg;
			break;
		case MENUBG:
			colore=ColorInfo.MenuBg;
			break;
		case MENUFG:
			colore=ColorInfo.MenuFg;
			break;
		case LISTBOXBG:
			colore=ColorInfo.ListCmdBg;
			break;
		case LISTBOXFG:
			colore=ColorInfo.ListCmdFg;
			break;
		default:
			WX11_FatalError ("Errore Funzione WX11_ColorGet");
			break;
	}

	//ConversioneColore (&colore);

	return (colore);	
}

void WX11_ColorChange (CmdInfo *cmd)
{
	Display *dp=WX11_GetDisplay();
	unsigned int bg=NERO;
	unsigned int fg=BIANCO;
	CmdInfo *comando=cmd;

	if (!cmd) comando=PrimoComando(0,NULL);
	while (comando)
	{
		if (comando->Tipo==MAIN)
		{	bg=WX11_ColorGet(WINBG);fg=WX11_ColorGet(WINFG); }
		if (comando->Tipo==LABEL)
		{	bg=WX11_ColorGet(WINBG);fg=WX11_ColorGet(WINFG); }
		if (comando->Tipo==BUTTON)
		{	
			bg=WX11_ColorGet(BUTTONBG);fg=WX11_ColorGet(BUTTONFG); 
			if (comando->BUpDn==0) bg=WX11_ColorLight(bg);
		}
		if (comando->Tipo==INPUT)
		{	bg=WX11_ColorGet(INPUTBG);fg=WX11_ColorGet(INPUTFG); }
		if (comando->Tipo==MENU)
		{	bg=WX11_ColorGet(MENUBG);fg=WX11_ColorGet(MENUFG); }
		if (comando->Tipo==LIST_BOX || comando->Tipo==COMBO_BOX)
		{	bg=WX11_ColorGet(LISTBOXBG);fg=WX11_ColorGet(LISTBOXFG); }
		if (comando->Tipo==SCROLL)
		{	bg=WX11_ColorLight(WX11_ColorGet(WINBG));
			fg=WX11_ColorLight(WX11_ColorGet(WINFG)); 
		}
		if (comando->Tipo==SCROLL_UP || comando->Tipo==SCROLL_DN)
		{	bg=WX11_ColorGet(WINBG);
			fg=WX11_ColorGet(WINFG); 
		}
		if (comando->Tipo==SCROLL_PGUP || comando->Tipo==SCROLL_PGSX)
		{	bg= WX11_ColorLightSmall(WX11_ColorGet(WINBG));
			fg= WX11_ColorLightSmall(WX11_ColorGet(WINFG)); 
		}
		if (comando->Tipo==SCROLL_PGDN || comando->Tipo==SCROLL_PGDX)
		{	bg= WX11_ColorLightSmall(WX11_ColorGet(WINBG));
			fg= WX11_ColorLightSmall(WX11_ColorGet(WINFG)); 
		}
		if (comando->Tipo==SCROLL_VCUR || comando->Tipo==SCROLL_HCUR)
		{	bg=WX11_ColorGet(WINBG);fg=WX11_ColorGet(WINFG); }

	
                if (comando->Tipo!=BORDO)
                {
			if (comando->Gradiant==1)
			{
				Pixmap pixmap;
			        unsigned int gradient_from= WX11_ColorLightSmall(bg);
        			unsigned int gradient_to  =WX11_ColorDarkSmall(bg);

				pixmap=XCreatePixmap(dp, comando->win, comando->Width, comando->Height, WX11_GetDepth ());

				WX11_ColorGradiant (pixmap, comando->Width, comando->Height, gradient_from,gradient_to);
				XSetWindowBackgroundPixmap (dp,comando->win,pixmap);

        			XFreePixmap(dp, pixmap);
			}
			else
			{
		        	XSetWindowBackground(dp, comando->win, WX11_RGB(bg));
			}
		        comando->Bg=bg;
		        comando->Fg=fg;
		        XClearWindow(dp,comando->win);
                        if (comando->Bordo)
                        {
               		        XSetWindowBackground(dp, comando->Prev->win, WX11_RGB(bg));
        		        comando->Prev->Bg=bg;
	           	        comando->Prev->Fg=fg;
		                XClearWindow(dp,comando->Prev->win);
                        }
                }

		if (!cmd) 
			comando=comando->Next;
		else
			comando=NULL;
	}
	WX11_PrintFlush(NULL);
}

unsigned int WX11_ColorLight(unsigned int colore)
{
	unsigned int light;
		
	int Rosso=(colore & 0xFF0000)>>16;
	int Verde=(colore & 0x00FF00)>>8;
	int Blu  =(colore & 0x0000FF)>>0;

	Rosso+=0x20;	Verde+=0x20;	Blu+=0x20;
	if (Rosso > 0xFF)	Rosso=0xFF;
	if (Verde > 0xFF)	Verde=0xFF;
	if (Blu   > 0xFF)     	Blu  =0xFF;

	light=(Rosso<<16)+(Verde<<8)+Blu;

	return (light);
}

unsigned int  WX11_ColorLightSmall(unsigned int colore)
{
	unsigned int light;
		
	int Rosso=(colore & 0xFF0000)>>16;
	int Verde=(colore & 0x00FF00)>>8;
	int Blu  =(colore & 0x0000FF)>>0;

	Rosso+=0x10;	Verde+=0x10;	Blu+=0x10;
	if (Rosso > 0xFF)	Rosso=0xFF;
	if (Verde > 0xFF)	Verde=0xFF;
	if (Blu   > 0xFF)     	Blu  =0xFF;

	light=(Rosso<<16)+(Verde<<8)+Blu;

	return (light);
}

unsigned int WX11_ColorDarkSmall(unsigned int colore)
{
	unsigned int light;
	
	int Rosso=(colore & 0xFF0000)>>16;
	int Verde=(colore & 0x00FF00)>>8;
	int Blu  =(colore & 0x0000FF)>>0;

	Rosso-=0x06;	Verde-=0x06;	Blu-=0x06;
	if (Rosso < 0x0)	Rosso=0x0;
	if (Verde < 0x0)	Verde=0x0;
	if (Blu   < 0x0)     	Blu  =0x0;

	light=(Rosso<<16)+(Verde<<8)+Blu;

	return (light);
}

unsigned int WX11_ColorDark(unsigned int colore)
{
	unsigned int light;
	
	int Rosso=(colore & 0xFF0000)>>16;
	int Verde=(colore & 0x00FF00)>>8;
	int Blu  =(colore & 0x0000FF)>>0;

	Rosso-=0x30;	Verde-=0x30;	Blu-=0x30;
	if (Rosso < 0x0)	Rosso=0x0;
	if (Verde < 0x0)	Verde=0x0;
	if (Blu   < 0x0)     	Blu  =0x0;

	light=(Rosso<<16)+(Verde<<8)+Blu;

	return (light);
}

