////////////////////////////////////////////////////////////////////////////////
//
//	Pico Technology USB TC08 Driver
//
///	\file     TC08DeviceInternalData.h
///	\brief    Internals of the TC08 device
//
//	Copyright (c) 2007, Pico Technology.
//	All rights reserved.
//   
//	Redistribution and use in source and binary forms, with or without
//	modification, are permitted provided that the following conditions are met:
//		* Redistributions of source code must retain the above copyright
//		  notice, this list of conditions and the following disclaimer.
//		* Redistributions in binary form must reproduce the above copyright
//		  notice, this list of conditions and the following disclaimer in the
//		  documentation and/or other materials provided with the distribution.
//		* The name of Pico Technology may not be used to endorse or promote
//		  products derived from this software without specific prior written
//		  permission.
//
//	THIS SOFTWARE IS PROVIDED BY PICO TECHNOLOGY ``AS IS'' AND ANY
//	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL PICO TECHNOLOGY BE LIABLE FOR ANY
//	DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
//	ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
//	THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//	Version $Id: TC08DeviceInternalData.h,v 1.8 2008/02/19 20:12:58 douglas Exp $
//
////////////////////////////////////////////////////////////////////////////////

#ifndef TC08DEVICEINTERNALDATA_H
#define TC08DEVICEINTERNALDATA_H



////////////////////////////////////////////////////////////////////////////////
//	INCLUDES
////////////////////////////////////////////////////////////////////////////////

#include <assert.h>
#include <stdio.h>
#include <ctype.h>
#include <limits>
#include <math.h>
#include <pthread.h>
#include <unistd.h>
#include <stdint.h>

#include <sys/times.h>

#include "../shared/PicoStatus.h"
#include "../shared/PicoPortability.h"
#include "../shared/PicoUsbID.h"
#include "../shared/PicoUsbDevice.h"
#include "TC08Device.h"
#include "readingbuff.h"
#include "tctables.h"



////////////////////////////////////////////////////////////////////////////////
// DEFINES
////////////////////////////////////////////////////////////////////////////////

// #define DEBUG 1

/// Bulk out pipe for data transfer from PC to device
#define USBTC08_BULKOUT_PIPE_ID       0
/// Bulk out pipe for data transfer from PC to device
#define USBTC08_BULKIN_PIPE_ID        1
#define USBTC08_NUMBER_OF_PIPES       2

// Command buffer size
#define USBTC08_CMD_BUF_SIZE         64
#define USBTC08_STREAM_PKT_SIZE      36

////////////////////////////////////////////////////////////////////////////////
/// \name USB Commands
////////////////////////////////////////////////////////////////////////////////
/// Each USB command consists of a single-byte instruction sent to the unit 
/// over the bulk-out endpoint, followed in some cases by futher bytes
/// constituting the "parameters" for the command. Some commands will cause the
/// unit to respond by transferring back bytes over the bulk-in endpoint. 
//@{

/// USB command to initialise the unit. Must be sent before any other commands 
/// will be recognized. Takes no parameters, returns no bytes. Any other
/// commands sent to the unit before INIT will be silently ignored.
#define USBTC08_CMD_INIT					0x80

/// USB command to set the enabled channels.
#define USBTC08_CMD_SETUP_CHANNELS			0x81

/// USB command to acquire a single reading on all enabled channels.
#define USBTC08_CMD_READ_CHANNELS			0x82

/// USB Command to set the LED on the unit. Takes one byte, 0x00=Off, 0x01=Red,
/// 0x02=Green. Returns no bytes. The LED remains set until another operation
/// (such as acquiring a reading) resets it.f
#define USBTC08_CMD_SET_LED					0x83

/// USB command to read the unit information from onboard EEPROM. 
#define USBTC08_CMD_READ_UNIT_INFO			0x84

/// USB command to set the mains frequency for noise rejection.
#define USBTC08_CMD_SET_MAINS				0x85

/// USB command to set the unit running in streaming mode.
#define USBTC08_CMD_RUN_STREAMING			0x86

/// USB command to terminate streaming mode.
#define USBTC08_CMD_STOP_STREAMING			0x87

/// USB Command to get a single byte back from the unit. Takes no bytes, 
/// returns one byte of 0x01. Useful to check that the unit is present
/// and responding correctly
#define USBTC08_CMD_ACKNOWLEDGE				0x88

/// USB Command to get the firmware version. 
#define USBTC08_CMD_GET_FIRMWARE_VERSION	0x89

//@} 

/// Allowable thermocouple types
#define USBTC08_DEF_TCTYPES "BEJKNRST"
#define USBTC08_ALL_TCTYPES USBTC08_DEF_TCTYPES " X"
#define USBTC08_TCTYPE_THERMISTOR 'C'
#define USBTC08_TCTYPE_MV         'X'
#define USBTC08_FIXED_AMP_GAIN        16.0 
#define USBTC08_REFERENCE_MILLIVOLTS  5000.0
#define USBTC08_ADC_COUNTS            16777216L // 2^24

//  Use this to limit the input range of the 
//  thermocouple channels
#define THERMOCOUPLE_MV_LIMIT 70.0f 
#define USBTC08_CONVERSION_TIME 68L  // Channel conversion time with 50Hz Filter (*6/5 for 60Hz filter)
#define USBTC08_INTERRUPT_MS    1L
#define USBTC08_MIN_INTERVAL_MS 100L
#define USBTC08_MAX_INTERVAL_MS 65535L

#define USBTC08_UNIT_INFO_LENGTH  30
#define USBTC08_FIRMWARE_VERSION_LENGTH  3

/* ***************************************************************************
 * MACROS
 *************************************************************************** */

/// Convert a three-byte ADC reading from the unit to a long integer
#define ConstructReading(a, b, c) (long)(((a << 24) + (b << 16) + (c << 8)) >> 8)

#define Truncate20Bit(reading) (reading &= -16L/*0xFFFFFFFFFFFFFFF0L*/)
//#define Round(x) floor(x + 0.5f)
#define __min(a,b) (((a) < (b)) ? (a) : (b))
#define __max(a,b) (((a) > (b)) ? (a) : (b))

/// Convert an ADC reading to its voltage value
#define AdcToMilliVolts(gain,adcCounts) (adcCounts * (((float)(USBTC08_REFERENCE_MILLIVOLTS / gain)) / USBTC08_ADC_COUNTS))

/// Convert a Celsius temperature to Farenheit
#define ConvertToFahrenheit(centigrade)			((((centigrade)*9.0f)/5.0f)+32.0f)

/// Convert a Celsius temperature to Kelvin
#define ConvertToKelvin(centigrade)				((centigrade)+273.0f)

/// Convert a Celsius temperature to Rankine 
/// Rankine units are same interval as Fahrenheit,
/// but with a -459.27 degree offset
//																	427.67 = 459.67 - 32
#define ConvertToRankine(centigrade)			((((centigrade)*9.0f)/5.0f)-427.67f)

////////////////////////////////////////////////////////////////////////////////
// ENUMS
////////////////////////////////////////////////////////////////////////////////

/// The USB TC08 applies different voltage gain to the thermocouple channels to
/// the CJC channel. 
enum USBTC08_VOLTAGE_GAINS {
  USBTC08_GAIN_1V25         = 2, ///< CJC thermistor reads in the range [0V,1.25V]
  USBTC08_GAIN_THERMOCOUPLE = 32 ///< Thermocouples read in the range [-70mV,70mV]
};


////////////////////////////////////////////////////////////////////////////////
// STRUCTS
////////////////////////////////////////////////////////////////////////////////

typedef struct 
{
	char  TCType;
	float MinTemp;
	float MaxTemp;
	float Interval;
} USBTC08_TCTABLE;

/// Channel setup (for internal use)
typedef struct {
	bool          fEnabled;
	char          bTCType; 
	ReadingBuffer SampleBuffer;
} USBTC08_CHANNEL;


////////////////////////////////////////////////////////////////////////////////
// TC08DeviceInternalData class
////////////////////////////////////////////////////////////////////////////////

/// Contains the internal methods and data members of TC08Device.
//
class TC08DeviceInternalData
{
	// Static fields
	private:
		friend class TC08Device;
		static int tablesInitialized;
	private:
		/// Lookup tables for converting voltages to temperature readings
		static ThermocoupleTables TCTables;
		
	// Static functions
	private:
		/// Initialize the thermocouple lookup tables 
		static void InitializeTables(void);
			
	private:
		/// Pointer to the underlying USB device
		PicoUsbDevice *usbDevice;
		/// The curent state of the unit (e.g. idle, streaming)
		USBTC08_STATES state;
		/// Setup information for each channel
		USBTC08_CHANNEL Channels[USBTC08_MAX_CHANNELS + 1];
		USBTC08_MAINS_FREQUENCY freq;
		unsigned int interval;
		pthread_t *thread;
		float LastGoodValues[9];
		USBTC08_UNIT_INFO *info;
	  pthread_mutex_t threadLock;
		bool threadStop;
	
	
	private:
		TC08DeviceInternalData(void);	
		void	ResetBuffers(void);

		PICO_RETURNS		ReadPipe(unsigned char *buf,unsigned int *size);
		PICO_RETURNS		WritePipe(unsigned char *buf,unsigned int size);
		PICO_RETURNS		ResetPipes(void);
		
		void	Init(void);
		bool	SetAllChannels(void);
		bool	LimitReading(float *reading, float limit);
		bool	CJCUsed(void);
		
		static void	*threadProc(void *args);
		void PollUnitThreaded(void);

};
	
#endif // not defined TC08DEVICEINTERNALDATA_H

