/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDIMATRIX_HPP
  #define TEPDIMATRIX_HPP
  
  #include <TeAgnostic.h>
  #include <TeSharedPtr.h>
  #include <TeTempFilesRemover.h>
  #include <TeUtils.h>
  
  #include <vector>
  #include <string>
  

  /**
   * @brief This is the template class to deal with a generic matrix.
   * @author Emiliano F. Castejon <castejon@dpi.inpe.br>
   * @ingroup PDIAux
   */
  template< class T >
  class TePDIMatrix {
    public :
      /** @typedef TeSharedPtr< TePDIMatrix< T > > pointer 
          Type definition for a instance pointer */
      typedef TeSharedPtr< TePDIMatrix< T > > pointer;
      
      /**
       * @brief Memory polycy.
       */
      enum MemoryPolicy {
        /**
         * Automatic memory policy ( Try to use RAM or DISK, if there is no 
         * avaliable RAM ) -
         * DO NOT USE AutoMemPol FOR COMPLEX DATA TYPES !
         * 
         */
        AutoMemPol,
        /**
         * RAM memory policy.
         */
        RAMMemPol,
        /**
         * Disk memory policy ( virtual mapped memory ) -
         *  DO NOT USE DiskMemPol FOR COMPLEX DATA TYPES !
         */
        DiskMemPol
      };
      
      /**
       * @brief Default Constructor.
       * @note The default mamory policy is RAMMemPol.
       */
      TePDIMatrix();      
      
      /**
       * @brief Alternative Constructor.
       *
       * @param lines Number of lines.
       * @param columns Number of columns.
       * @param mp Memory policy.
       */
      TePDIMatrix( unsigned int lines, unsigned int columns, 
        MemoryPolicy mp );
      
      /**
       * @brief Alternative Constructor.
       *
       * @param mp Memory policy.
       */
      TePDIMatrix( MemoryPolicy mp );      
        
      /**
       * @brief Alternative Constructor.
       *
       * @param lines Number of lines.
       * @param columns Number of columns.
       * @note The default mamory policy is RAMMemPol.
       */
      TePDIMatrix( unsigned int lines, unsigned int columns );        

      /**
       * @brief Alternative Constructor.
       *
       * @param external External object reference.
       * @param mp Memory policy.
       */
      TePDIMatrix( const TePDIMatrix< T >& external );

      /**
       * @brief Default Destructor
       */
      ~TePDIMatrix();
      
      /**
       * @brief Reset the active instance the the new parameters.
       *
       * @param lines Number of lines.
       * @param columns Number of columns.
       * @return true if OK, false on error.
       */
      bool Reset( unsigned int lines = 0, 
        unsigned int columns = 0 );      

      /**
       * @brief Reset the active instance the the new parameters.
       *
       * @param lines Number of lines.
       * @param columns Number of columns.
       * @param mp Memory policy.
       * @return true if OK, false on error.
       */
      bool Reset( unsigned int lines, unsigned int columns,
        MemoryPolicy mp );
      
      /**
       * @brief The number of current matrix lines.
       *
       * @return The number of current matrix lines.
       */
      unsigned int GetLines() const;
      
      /**
       * @brief The number of current matrix columns.
       *
       * @return The number of current matrix columns
       */
      unsigned int GetColumns() const;
      
      /**
       * @brief Empty Matrix verification.
       *
       * @return true if the matrix is empty.
       */
      bool IsEmpty() const;

      /**
       * @brief Operator = overload.
       *
       * @note The external memory policy will be used as reference.
       *
       * @param external External instance reference.
       * @return A reference to the current matrix.
       */
      const TePDIMatrix< T >& operator=( 
        const TePDIMatrix< T >& external );

      /**
       * @brief Operator () overload.
       *
       * @param line Line number.
       * @param column Column number.
       * @return A reference to the required element.
       */
      inline T& operator()( const unsigned int& line, 
        const unsigned int& column )
      {
        TEAGN_DEBUG_CONDITION( ( line < totalLines_ ),
          "Invalid line" )
        TEAGN_DEBUG_CONDITION( ( column < totalColumns_ ),
          "Invalid columns" )
            
        return scanLine( line )[ column ];      
      };
      
      /**
       * @brief Operator () overload.
       *
       * @param line Line number.
       * @param column Column number.
       * @return A const reference to the required element.
       */
      inline const T& operator()( const unsigned int& line, 
        const unsigned int& column ) const
      {
        TEAGN_DEBUG_CONDITION( ( line < totalLines_ ),
          "Invalid line" )
        TEAGN_DEBUG_CONDITION( ( column < totalColumns_ ),
          "Invalid columns" )
          
        return scanLine( line )[ column ];      
      }; 
      
      /**
       * @brief Operator [] overload.
       *
       * @param line Line number.
       * @param column Column number.
       * @return A pointer to the required line.
       * @note The returned pointer is garanteed to 
       * be valid until an acess to another line occurs.
       */
      inline T* operator[]( const unsigned int& line )
      {
        TEAGN_DEBUG_CONDITION( ( line < totalLines_ ),
          "Invalid line" )
      
        return scanLine( line );      
      };
      
      /**
       * @brief Operator [] overload.
       *
       * @param line Line number.
       * @param column Column number.
       * @return A pointer to the required line.
       * @note The returned pointer is garanteed to 
       * be valid until an acess to another line occurs.
       */
      inline T const* operator[]( const unsigned int& line ) const
      {
        TEAGN_DEBUG_CONDITION( ( line < totalLines_ ),
          "Invalid line" )
      
        return scanLine( line );      
      };      
            
    protected :
    
      /**
       * @brief File tiles data node.
       */         
      class DiskLinesVecNodeT
      {
        public :
          
          FILE* filePtr_;
          unsigned int fileOff_;
          std::string fullFilePath_;
          
          DiskLinesVecNodeT()
          {
            filePtr_ = 0;
            fileOff_ = 0;
          }
      };
      
      typedef std::vector< DiskLinesVecNodeT > DiskLinesVecT;
      
      /**
       * @brief Max bytes per temp file (for swapped tiles only).
       */
      unsigned long int maxTmpFileSize_;      
      
      /**
       * @brief The total lines number.
       */
      unsigned int totalLines_;
      
      /**
       * @brief The total columns number.
       */
      unsigned int totalColumns_;     
      
      /**
       * @brief The current used memory policy.
       */
      MemoryPolicy currentMemPolicy_;
      
      /**
       * @brief The index of the current active disk line copy in RAM
       * ( default -1 , no disk line active ).
       */     
      mutable long int curDiskLineIdx_;    
        
      /**
       * @brief A pointer to the RAM memory used for disk lines swap..
       */     
      T* currDiskLinePtr_;
      
      /**
       * @brief The lines size (bytes).
       */     
      unsigned int linesSize_;      

      /**
       * @brief The lines pointers vector (RAM + disk lines).
       */
      mutable std::vector< T* > linesPtrsVec_;     
       
      /**
       * @brief The disk lines data vector.
       */      
      DiskLinesVecT diskLinesVec_;    
     
      /**
       * @brief Reset the internal variables to the initial state.
       */      
      void init();
      
      /**
       * @brief Clear all allocated resources and go back to the initial
       * state.
       */      
      void clear();
      
      /**
       * @brief Allocate disk lines.
       * @param startingLineIdx Starting line index.
       * @return true if OK, false on errors.
       */    
      bool allocateDiskLines( 
        unsigned int startingLineIdx );      
      
      /**
       * @brief Returns a pointer to the required line.
       *
       * @param line Line number.
       * @param column Column number.
       * @return A pointer to the required line.
       * @note The returned pointer is garanteed to 
       * be valid until an acess to another line occurs
       */
      T* scanLine( const unsigned int& line ) const;
        
      /**
       * @brief Create a new disk file.
       * @param filename The file name.
       * @param size The file size.
       * @param fileptr The file pointer.
       * @return true if OK. false on errors.
       */
      bool createNewDiskFile( unsigned long int size,
        const std::string& filename, FILE** fileptr ) const;      
  };

  template< class T >
  void TePDIMatrix< T >::init()
  {
    maxTmpFileSize_ = ( 1024 * 1024 * 100 );
    totalLines_ = 0;
    totalColumns_ = 0;  
    currentMemPolicy_ = RAMMemPol;
    curDiskLineIdx_ = -1;
    currDiskLinePtr_ = 0;
    linesSize_ = 0;
  }
  

  template< class T >
  void TePDIMatrix< T >::clear()
  {
    if( curDiskLineIdx_ > -1 )
    {
      delete[] currDiskLinePtr_;
      linesPtrsVec_[ curDiskLineIdx_ ] = 0;
    }
    
    const unsigned int linesPtrsVecSize = linesPtrsVec_.size();
    for( unsigned int linesPtrsVecIdx = 0 ; linesPtrsVecIdx < linesPtrsVecSize ;
      ++linesPtrsVecIdx ) {
     
      if( linesPtrsVec_[ linesPtrsVecIdx ] ) {
        delete[] linesPtrsVec_[ linesPtrsVecIdx ];
      }
    }  
      
    linesPtrsVec_.clear();
    
    const unsigned int diskLinesVecSize = diskLinesVec_.size();
    for( unsigned int diskLinesVecIdx = 0 ; diskLinesVecIdx <
      diskLinesVecSize ; ++diskLinesVecIdx )
    {
      TeTempFilesRemover::instance().removeFile( diskLinesVec_[ 
        diskLinesVecIdx ].fullFilePath_ ); 
    }
    
    diskLinesVec_.clear();
  
    init();
  }  
  

  template< class T >
  TePDIMatrix< T >::TePDIMatrix()
  {
    init();
  }    
  
  
  template< class T >
  TePDIMatrix< T >::TePDIMatrix( unsigned int lines, 
    unsigned int columns, MemoryPolicy mp )
  {
    init();
    
    TEAGN_TRUE_OR_THROW( Reset( lines, columns, mp ),
      "Unable to initiate the matrix object" );
  }  
  
  template< class T >
  TePDIMatrix< T >::TePDIMatrix( MemoryPolicy mp )
  {
    init();
    
    currentMemPolicy_ = mp;
  }    
  
  template< class T >
  TePDIMatrix< T >::TePDIMatrix( unsigned int lines, 
    unsigned int columns )
  {
    init();
    
    TEAGN_TRUE_OR_THROW( Reset( lines, columns, RAMMemPol ),
      "Unable to initiate the matrix object" );
  }   
  
  
  template< class T >
  TePDIMatrix< T >::TePDIMatrix( const TePDIMatrix< T >& external )
  {
    init();
    
    operator=( external );
  }


  template< class T >
    TePDIMatrix< T >::~TePDIMatrix()
  {
    clear();
  }

  
  template< class T >
  bool TePDIMatrix< T >::Reset( unsigned int lines, 
    unsigned int columns )
  {
    return Reset( lines, columns, currentMemPolicy_ );
  }  

  template< class T >
  bool TePDIMatrix< T >::Reset( unsigned int lines, 
    unsigned int columns,
    MemoryPolicy mp )
  {
    /* Update the old buffer if necessary */
    
    if( ( lines != totalLines_ ) || ( columns != totalColumns_ ) ||
        ( currentMemPolicy_ != mp ) ) {
    
      /* free the old resources */
      
      clear();
    
      /* Allocate the new resources */
      
      totalLines_ = lines;
      totalColumns_ = columns;       
      linesSize_ = sizeof( T ) * totalColumns_;
      
      currentMemPolicy_ = mp;
      
      if( ( lines != 0 ) && ( columns != 0 ) ) {
        /* Guessing the memory source, if in automatic mode */
     
        unsigned int line = 0;
        
        /* Allocating the main lines pointers vector */
        
        try
        {
          linesPtrsVec_.resize( totalLines_, 0 );
        }
        catch(...)
        {
          clear();
          TEAGN_LOG_AND_RETURN( "Memory allocation error" );
        }
        
        if( linesPtrsVec_.size() != totalLines_ ) {
          clear();
          TEAGN_LOG_AND_RETURN( "Memory allocation error" );
        }
        
        /* Allocating the disk lines data vector */
        
        try
        {
          diskLinesVec_.resize( totalLines_ );
        }
        catch(...)
        {
          clear();
          TEAGN_LOG_AND_RETURN( "Memory allocation error" );
        }
        
        if( diskLinesVec_.size() != totalLines_ ) {
          clear();
          TEAGN_LOG_AND_RETURN( "Memory allocation error" );
        }        
        
        /* Allocating the RAM file swap area */
        
        try
        {
          currDiskLinePtr_ = new T[ totalColumns_ ];
        }
        catch(...)
        {
          clear();
          TEAGN_LOG_AND_RETURN( "Memory allocation error" );
        }
        
        if( currDiskLinePtr_ == 0 ) {
          clear();
          TEAGN_LOG_AND_RETURN( "Memory allocation error" );
        }    

        /* Allocating lines */

        switch( currentMemPolicy_ ) {
          case RAMMemPol :
          {
            T* new_line_ptr = 0;
        
            for( line = 0 ; line < totalLines_ ; ++line ) {
              try {
                new_line_ptr = new T[ totalColumns_ ];
              }
              catch(...) {
                clear();
                
                TEAGN_LOG_AND_RETURN( "Memory allocation error" );               
              }
              
              if( new_line_ptr == 0 ) {
                clear();
                
                TEAGN_LOG_AND_RETURN( "Memory allocation error" );              
              } 
              
              linesPtrsVec_[ line ] = new_line_ptr;
            }
            
            break;
          }
          case DiskMemPol :
          {
            if( ! allocateDiskLines( 0 ) ) {
              clear();
              
              TEAGN_LOG_AND_RETURN( 
                "Error allocating mapped memory lines" )
            }
              
            break;
          }       
          case AutoMemPol :
          {
            const unsigned long int freeVm = TeGetFreeVirtualMemory();
            unsigned long int lineBytes = sizeof( T ) * totalColumns_;
            const unsigned long int maxRam  = (unsigned long int)
              ( 0.90 * ( (double)( freeVm ) ) );    
            unsigned long int maxRamLines = (unsigned long int)
              ( ( (double)maxRam ) / ( (double) lineBytes ) ); 
              
            T* new_line_ptr = 0;
        
            for( unsigned int line = 0 ; line < totalLines_ ; ++line ) 
            {
              if( line < maxRamLines ) 
              {
                try {
                  new_line_ptr = new T[ totalColumns_ ];
                }
                catch(...) 
                {
                  if( allocateDiskLines( line ) )
                  {
                    return true;
                  }
                  else 
                  {
                    clear();
                    
                    TEAGN_LOG_AND_RETURN( 
                      "Error allocating mapped memory lines" )
                  }
                }
                
                if( new_line_ptr == 0 ) {
                  if( allocateDiskLines( line ) )
                  {
                    return true;
                  }
                  else 
                  {
                    clear();
                    
                    TEAGN_LOG_AND_RETURN( 
                      "Error allocating mapped memory lines" )
                  }         
                } 
                
                linesPtrsVec_[ line ] = new_line_ptr;
              } else { // ( line >= maxRamLines )
                if( allocateDiskLines( line ) ) 
                {
                  return true;
                } 
                else 
                {
                  clear();
                  
                  TEAGN_LOG_AND_RETURN( 
                    "Error allocating mapped memory lines" )
                }              
              }
            }
            
            break;
          }                 
          default :
          {
            TEAGN_LOG_AND_THROW( "Invalid memory policy" );
            break;
          }
        }
      }
    }
    
    return true;
  }
  
  
  template< class T >
  unsigned int TePDIMatrix< T >::GetLines() const
  {
    return totalLines_;
  }

  
  template< class T >
  unsigned int TePDIMatrix< T >::GetColumns() const
  {
    return totalColumns_;
  }
  
  
  template< class T >
  bool TePDIMatrix< T >::IsEmpty() const
  {
    return ( totalLines_ == 0 ) ? true : false;
  }
  

  template< class T >
  const TePDIMatrix< T >& TePDIMatrix< T >::operator=(
    const TePDIMatrix< T >& external )
  {
    TEAGN_TRUE_OR_THROW( 
      Reset( external.totalLines_, external.totalColumns_,
      external.currentMemPolicy_ ),
      "Unable to initiate the matrix object" );
    
    unsigned int line;
    unsigned int column;
    T const* inLinePtr = 0;
    T* outLinePtr = 0;
    
    for( line = 0 ; line < totalLines_ ; ++line ) 
    {
      inLinePtr = external.scanLine( line );
      outLinePtr = scanLine( line );
      
      for( column = 0 ; column < totalColumns_ ; ++column ) {
        outLinePtr[ column ] = inLinePtr[ column ];
      }
    }

    return *this;
  }
  
  template< class T >
  bool TePDIMatrix< T >::allocateDiskLines( unsigned int startingLineIdx )
  {
    TEAGN_DEBUG_CONDITION( diskLinesVec_.size() == totalLines_,
      "Invalid disk lines vector size" );
    
    const unsigned int lineBytes = sizeof( T ) * totalColumns_;
    const unsigned int linesPerfile = ( unsigned long int )
      floor( ( (double)maxTmpFileSize_ ) / ( (double) lineBytes ) );    
    const unsigned long int fileSize = (unsigned long int)
      ( linesPerfile * lineBytes );
    
    unsigned int currFileLines = linesPerfile + 1;
    DiskLinesVecNodeT dummyNode;
     
    for( unsigned long int lineIdx = startingLineIdx ; lineIdx < 
      totalLines_ ; ++lineIdx )
    {
      if( currFileLines >= linesPerfile )
      {
        TEAGN_TRUE_OR_RETURN( TeGetTempFileName( dummyNode.fullFilePath_ ),
          "Unable to create temporary file name" );
        TEAGN_TRUE_OR_RETURN( createNewDiskFile( fileSize, 
          dummyNode.fullFilePath_, &(dummyNode.filePtr_) ), 
          "Error creating temporary file" );
        
        TeTempFilesRemover::instance().addFile( dummyNode.fullFilePath_,
          dummyNode.filePtr_ );        
        
        currFileLines = 0;
      }
      
      dummyNode.fileOff_ = currFileLines * lineBytes;
      diskLinesVec_[ lineIdx ] = dummyNode;
      
      ++currFileLines;
    } 
    
    return true;
  }
  
  template< class T >
  T* TePDIMatrix< T >::scanLine( const unsigned int& line ) const
  {
    TEAGN_DEBUG_CONDITION( line < linesPtrsVec_.size(),
      "Invalid tile index" );
      
    T* line_ptr = linesPtrsVec_[ line ];
      
    if( line_ptr ) {
      return line_ptr;
    } else {
      /* flush the current disk line from RAM */
      
      if( curDiskLineIdx_ > -1 )
      {
        TEAGN_DEBUG_CONDITION( curDiskLineIdx_ < (long int)diskLinesVec_.size(),
          "Invalid disk tile index" );
        TEAGN_DEBUG_CONDITION( curDiskLineIdx_ < (long int)linesPtrsVec_.size(),
          "Invalid disk tile index" );
        TEAGN_DEBUG_CONDITION( currDiskLinePtr_,
          "Invalid swap line pointer" );
        
        const DiskLinesVecNodeT& node = diskLinesVec_[ curDiskLineIdx_ ];
        
        TEAGN_TRUE_OR_THROW( 0 == fseek( node.filePtr_, 
          (long)( node.fileOff_ ), SEEK_SET ),
          "File seek error" );        
        
        TEAGN_TRUE_OR_THROW( 1 == fwrite( (void*)currDiskLinePtr_, 
          (size_t)( linesSize_ ), 1, node.filePtr_ ),
          "File write error" )
            
        linesPtrsVec_[ curDiskLineIdx_ ] = 0;
      }
      
      // Read the required line into RAM
      
      TEAGN_DEBUG_CONDITION( line < (unsigned int)diskLinesVec_.size(),
        "Invalid disk tile index" );      
      const DiskLinesVecNodeT& node = diskLinesVec_[ line ];
      
      TEAGN_TRUE_OR_THROW( 0 == fseek( node.filePtr_, 
        (long)( node.fileOff_ ), SEEK_SET ),
        "File seek error" );
      
      TEAGN_TRUE_OR_THROW( 1 == fread( (void*)currDiskLinePtr_, 
        (size_t)( linesSize_ ), 1, node.filePtr_ ),
        "File read error" )      
          
      curDiskLineIdx_ =  line;   
      linesPtrsVec_[ curDiskLineIdx_ ] = currDiskLinePtr_;
      
      return currDiskLinePtr_;
    }
  } 

  template< class T >
  bool TePDIMatrix< T >::createNewDiskFile( unsigned long int size,
    const std::string& filename, FILE** fileptr ) const
  {
    TEAGN_TRUE_OR_RETURN( ! filename.empty(),
      "Invalid file name" );
      
    (*fileptr) = fopen( filename.c_str(), "wb+" );
    TEAGN_TRUE_OR_RETURN( (*fileptr) != 0, "Invalid file pointer" )
    
    long seekoff = (long)( size - 1 );
    
    if( 0 != fseek( (*fileptr), seekoff, SEEK_SET ) )
    {
      fclose( (*fileptr) );
      TEAGN_LOGERR( "File seek error" );
      return false;
    }
  
    unsigned char c = '\0';
    if( 1 != fwrite( &c, 1, 1, (*fileptr) ) )
    {
      fclose( (*fileptr) );
      TEAGN_LOGERR( "File write error" );
      return false;
    }
      
    return true;
  }  
  
/** @example TePDIMatrix_test.cpp
 *    Shows how to use this class.
 */    
  
#endif //TEPDIMATRIX_HPP

