#include "TePDIMMIOMatching.hpp"

#include "TePDIInterpolator.hpp"
#include "TePDIUtils.hpp"
#include "TePDITypes.hpp"

#include <TeThreadJobsManager.h>
#include <TeFunctionCallThreadJob.h>
#include <TeCoord2D.h>
#include <TeGTFactory.h>
#include <TeDefines.h>
#include <TeMutex.h>
#include <TeUtils.h>

#include <math.h>
#include <limits.h>
#include <TeRTree.h>

#include <map>

#define BICUBIC_MODULE( x ) ( ( x < 0 ) ? ( -1 * x ) : x )
#define BICUBIC_K1( x , a ) ( ( ( a + 2 ) * x * x * x ) - \
  ( ( a + 3 ) * x * x ) + 1 )
#define BICUBIC_K2( x , a ) ( ( a * x * x * x ) - ( 5 * a * x * x ) + \
  ( 8 * a * x ) - ( 4 * a ) )
#define BICUBIC_RANGES(x,a) \
  ( ( ( 0 <= x ) && ( x <= 1 ) ) ? \
    BICUBIC_K1(x,a) \
  : ( ( ( 1 < x ) && ( x <= 2 ) ) ? \
      BICUBIC_K2(x,a) \
    : 0 ) )
#define BICUBIC_KERNEL( x , a ) BICUBIC_RANGES( BICUBIC_MODULE(x) , a )

TePDIMMIOMatching::TePDIMMIOMatching()
{
  input_channel1_ = 0;
  input_channel2_ = 0;
  matching_method_ = NormCrossCorrMethod;
}


TePDIMMIOMatching::~TePDIMMIOMatching()
{
}

bool TePDIMMIOMatching::CheckParameters( const TePDIParameters& parameters ) const
{
  /* Checking input_image1_ptr */
    
  TePDITypes::TePDIRasterPtrType input_image1_ptr;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( 
    "input_image1_ptr", input_image1_ptr ),
    "Missing parameter: input_image1_ptr" );
  TEAGN_TRUE_OR_RETURN( input_image1_ptr.isActive(),
    "Invalid parameter: input_image1_ptr inactive" );
  TEAGN_TRUE_OR_RETURN( input_image1_ptr->params().status_ != 
    TeRasterParams::TeNotReady, 
    "Invalid parameter: input_image1_ptr not ready" );
  
  /* Checking matching method */
  
  FeatMatchingMethod matching_method = TePDIMMIOMatching::NormCrossCorrMethod;
  if( parameters.CheckParameter< FeatMatchingMethod >( 
    "matching_method" ) ) 
  {
    parameters.GetParameter( "matching_method", matching_method );
    
    TEAGN_TRUE_OR_RETURN( 
      ( ( matching_method == TePDIMMIOMatching::EuclidianDistMethod ) ||
      ( matching_method == TePDIMMIOMatching::NormCrossCorrMethod ) ),
      "Invalid parameter : matching_method" );
  }     
    
  /* Checking input_channel1 */
    
  unsigned int input_channel1 = 0;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( 
    "input_channel1", input_channel1 ),
    "Missing parameter: input_channel1" );
  TEAGN_TRUE_OR_RETURN( 
    ( ( (int)input_channel1 ) < 
    input_image1_ptr->params().nBands() ),
    "Invalid parameter: input_channel1" );
  
    
  /* Checking input_image1_ptr */
    
  TePDITypes::TePDIRasterPtrType input_image2_ptr;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( 
    "input_image2_ptr", input_image2_ptr ),
    "Missing parameter: input_image2_ptr" );
  TEAGN_TRUE_OR_RETURN( input_image2_ptr.isActive(),
    "Invalid parameter: input_image2_ptr inactive" );
  TEAGN_TRUE_OR_RETURN( input_image2_ptr->params().status_ != 
    TeRasterParams::TeNotReady, 
    "Invalid parameter: input_image2_ptr not ready" );    
    
  /* Checking input_channel2 */
    
  unsigned int input_channel2 = 0;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( 
    "input_channel2", input_channel2 ),
    "Missing parameter: input_channel2" );
  TEAGN_TRUE_OR_RETURN( 
    ( ( (int)input_channel2 ) < 
    input_image2_ptr->params().nBands() ),
    "Invalid parameter: input_channel2" );
    
  /* Checking out_tie_points_ptr */
    
  TeSharedPtr< TeCoordPairVect > out_tie_points_ptr;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( 
    "out_tie_points_ptr", out_tie_points_ptr ),
    "Missing parameter: out_tie_points_ptr" );
  TEAGN_TRUE_OR_RETURN( out_tie_points_ptr.isActive(),
    "Invalid parameter: out_tie_points_ptr inactive" );
    
  /* Checking gt_params */
    
  TeGTParams gt_params;
  if( parameters.CheckParameter< TeGTParams >( "gt_params" ) ) {
    parameters.GetParameter( "gt_params", gt_params );
    
    TEAGN_TRUE_OR_RETURN( ( TeGTFactory::instance().find( 
      gt_params.transformation_name_ ) != 
      TeGTFactory::instance().end() ),
      "Invalid parameter : gt_params" )
  }   
  
  /* Checking out_gt_params */
  
  TeGTParams::pointer out_gt_params_ptr;
  if( parameters.CheckParameter< TeGTParams::pointer >( 
    "out_gt_params_ptr" ) ) 
  {
    parameters.GetParameter( "out_gt_params_ptr", out_gt_params_ptr );
    
    TEAGN_TRUE_OR_RETURN( ( out_gt_params_ptr.isActive() ),
      "Invalid parameter : out_gt_params_ptr" )
  }      
    
  /* Checking input_box1 */
    
  TeBox input_box1;
  if( parameters.CheckParameter< TeBox >( "input_box1" ) ) {
    parameters.GetParameter( "input_box1", input_box1 );
    
    TEAGN_TRUE_OR_RETURN( ( input_box1.x1() >= 0 ),
      "Invalid parameter : input_box1" )
    TEAGN_TRUE_OR_RETURN( ( input_box1.x2() >= 0 ),
      "Invalid parameter : input_box1" )      
    TEAGN_TRUE_OR_RETURN( ( input_box1.y1() >= 0 ),
      "Invalid parameter : input_box1" )
    TEAGN_TRUE_OR_RETURN( ( input_box1.y2() >= 0 ),
      "Invalid parameter : input_box1" )         
      
    TEAGN_TRUE_OR_RETURN( 
      ( input_box1.x1() <= 
        ( input_image1_ptr->params().ncols_ - 1 ) ),
      "Invalid parameter : input_box1" )
    TEAGN_TRUE_OR_RETURN( 
      ( input_box1.x2() <= 
        ( input_image1_ptr->params().ncols_ - 1 ) ),
      "Invalid parameter : input_box1" )       
      
    TEAGN_TRUE_OR_RETURN( 
      ( input_box1.y1() <= 
        ( input_image1_ptr->params().nlines_ - 1 ) ),
      "Invalid parameter : input_box1" )
    TEAGN_TRUE_OR_RETURN( 
      ( input_box1.y2() <= 
        ( input_image1_ptr->params().nlines_ - 1 ) ),
      "Invalid parameter : input_box1" )        
  }
    
  /* Checking input_box2 */
    
  TeBox input_box2;
  if( parameters.CheckParameter< TeBox >( "input_box2" ) ) {
    parameters.GetParameter( "input_box2", input_box2 );
    
    TEAGN_TRUE_OR_RETURN( ( input_box2.x1() >= 0 ),
      "Invalid parameter : input_box2" )
    TEAGN_TRUE_OR_RETURN( ( input_box2.x2() >= 0 ),
      "Invalid parameter : input_box2" )      
    TEAGN_TRUE_OR_RETURN( ( input_box2.y1() >= 0 ),
      "Invalid parameter : input_box2" )
    TEAGN_TRUE_OR_RETURN( ( input_box2.y2() >= 0 ),
      "Invalid parameter : input_box2" )         
      
    TEAGN_TRUE_OR_RETURN( 
      ( input_box2.x1() <= 
        ( input_image2_ptr->params().ncols_ - 1 ) ),
      "Invalid parameter : input_box2" )
    TEAGN_TRUE_OR_RETURN( 
      ( input_box2.x2() <= 
        ( input_image2_ptr->params().ncols_ - 1 ) ),
      "Invalid parameter : input_box2" )       
      
    TEAGN_TRUE_OR_RETURN( 
      ( input_box2.y1() <= 
        ( input_image2_ptr->params().nlines_ - 1 ) ),
      "Invalid parameter : input_box2" )
    TEAGN_TRUE_OR_RETURN( 
      ( input_box2.y2() <= 
        ( input_image2_ptr->params().nlines_ - 1 ) ),
      "Invalid parameter : input_box2" )        
  }    
    
  /* Checking max_tie_points */
  
  unsigned int max_tie_points = 0;
  if( parameters.CheckParameter< unsigned int >( "max_tie_points" ) ) {
    parameters.GetParameter( "max_tie_points", max_tie_points );
    
    TEAGN_TRUE_OR_RETURN( ( max_tie_points > 0 ),
      "Invalid parameter : max_tie_points" )    
  }
  
  /* Checking corr_window_width */
  
  unsigned int corr_window_width = 0;
  if( parameters.CheckParameter< unsigned int >( 
    "corr_window_width" ) ) {
    
    parameters.GetParameter( "corr_window_width", 
      corr_window_width );
      
    TEAGN_TRUE_OR_RETURN( ( ( corr_window_width % 2 ) > 0 ),
      "Invalid parameter : corr_window_width" )      
    
    TEAGN_TRUE_OR_RETURN( ( corr_window_width >= 13 ),
      "Invalid parameter : corr_window_width" )
  }    
  
  /* Checking moravec_window_width */
  
  unsigned int moravec_window_width = corr_window_width / 2;
  if( parameters.CheckParameter< unsigned int >( 
    "moravec_window_width" ) ) {
    
    parameters.GetParameter( "moravec_window_width", 
      moravec_window_width );
      
    TEAGN_TRUE_OR_RETURN( ( ( moravec_window_width % 2 ) > 0 ),
      "Invalid parameter : moravec_window_width" )      
    
    TEAGN_TRUE_OR_RETURN( ( moravec_window_width >= 3 ),
      "Invalid parameter : corr_window_width" )
  }   
  
  /* Checking pixel_x_relation */
  
  double pixel_x_relation = 1.0;
  if( parameters.CheckParameter< double >( "pixel_x_relation" ) ) {
    parameters.GetParameter( "pixel_x_relation", pixel_x_relation );
    
    TEAGN_TRUE_OR_RETURN( ( pixel_x_relation != 0.0 ),
      "Invalid parameter : pixel_x_relation" )
  }
  
  /* Checking pixel_y_relation */
  
  double pixel_y_relation = 1.0;
  if( parameters.CheckParameter< double >( "pixel_y_relation" ) ) {
    parameters.GetParameter( "pixel_y_relation", pixel_y_relation );
    
    TEAGN_TRUE_OR_RETURN( ( pixel_y_relation != 0.0 ),
      "Invalid parameter : pixel_y_relation" )
  }
  
  /* Checking variance_min_thrs */
  
  double variance_min_thrs = 0;
  if( parameters.CheckParameter< double >( "variance_min_thrs" ) ) {
    parameters.GetParameter( "variance_min_thrs", variance_min_thrs );
    
    TEAGN_TRUE_OR_RETURN( ( ( variance_min_thrs >= 0.0 ) &&
      ( variance_min_thrs <= 3.0 ) ),
      "Invalid parameter : variance_min_thrs" )
  }  
  
  return true;
}

bool TePDIMMIOMatching::RunImplementation()
{
  /* Retriving Parameters */

  TeSharedPtr< TeCoordPairVect > out_tie_points_ptr;
  params_.GetParameter( "out_tie_points_ptr", out_tie_points_ptr );
  
  double pixel_x_relation = 1.0;
  if( params_.CheckParameter< double >( "pixel_x_relation" ) ) {
    params_.GetParameter( "pixel_x_relation", pixel_x_relation );
  }
  
  double pixel_y_relation = 1.0;
  if( params_.CheckParameter< double >( "pixel_y_relation" ) ) {
    params_.GetParameter( "pixel_y_relation", pixel_y_relation );
  }    
  
  TeGTParams gt_params;
  gt_params.out_rem_strat_ = TeGTParams::LWAOutRemotion;
  gt_params.max_dmap_error_ = 2.0;
  gt_params.max_imap_error_ = 2.0;
  gt_params.transformation_name_ = "affine";
  if( params_.CheckParameter< TeGTParams >( "gt_params" ) ) {
    params_.GetParameter( "gt_params", gt_params );
  }
  
  bool enable_multi_thread = false;
  if( params_.CheckParameter< int >( "enable_multi_thread" ) ) {
    enable_multi_thread = true;
  }
  
  bool skip_geom_filter = false;
  if( params_.CheckParameter< int >( "skip_geom_filter" ) ) {
    skip_geom_filter = true;
  }  
  
  bool enable_upsampled_filter = false;
  if( params_.CheckParameter< int >( "enable_upsampled_filter" ) ) {
    enable_upsampled_filter = true;
  }    
  
  unsigned int max_tie_points = 1000;
  if( params_.CheckParameter< unsigned int >( "max_tie_points" ) ) {
    params_.GetParameter( "max_tie_points", max_tie_points );
  }    
  
  unsigned int corr_window_width = 13;
  if( params_.CheckParameter< unsigned int >( 
    "corr_window_width" ) ) {
    
    params_.GetParameter( "corr_window_width", corr_window_width );
  }    
  
  unsigned int moravec_window_width = 11;
  if( params_.CheckParameter< unsigned int >( "moravec_window_width" ) ) 
  {
    params_.GetParameter( "moravec_window_width", 
      moravec_window_width );
  }     
  
  double variance_min_thrs = 1.0;
  if( params_.CheckParameter< double >( "variance_min_thrs" ) ) {
    params_.GetParameter( "variance_min_thrs", variance_min_thrs );
  }    
  
  // Do wee need to process what images ??
  
  const bool processImg1 = img1featMtx_.IsEmpty();
  const bool processImg2 = img2featMtx_.IsEmpty();
  
  // The original boxes postion over the original images
  
  const unsigned int orig_box1_x_off = (unsigned int)
    input_box1_.lowerLeft().x();
  const unsigned int orig_box1_y_off = (unsigned int)
    input_box1_.upperRight().y();
  const unsigned int orig_box1_nlines = 1 + (unsigned int)
    ABS( input_box1_.height() );
  const unsigned int orig_box1_ncols = 1 + (unsigned int)
    ABS( input_box1_.width() );
  
  const unsigned int orig_box2_x_off = (unsigned int)
    input_box2_.lowerLeft().x();
  const unsigned int orig_box2_y_off = (unsigned int)
    input_box2_.upperRight().y();
  const unsigned int orig_box2_nlines = 1 + (unsigned int)
    ABS( input_box2_.height() );
  const unsigned int orig_box2_ncols = 1 + (unsigned int)
    ABS( input_box2_.width() );  
  
  /* Calculating the rescale factors 
     factor = rescaled_orignal_image / original_image */
  
  double i1XRescFact = 1.0;
  double i1YRescFact = 1.0;
  double i2XRescFact = 1.0;
  double i2YRescFact = 1.0;
  {
    double mean_pixel_relation = ( pixel_x_relation + pixel_y_relation ) /
      2.0;
      
    if( mean_pixel_relation > 1.0 ) {
      /* The image 1 has poor resolution - bigger pixel resolution values -
         and image 2 needs to be rescaled down */
      
      i2XRescFact = 1.0 / pixel_x_relation;
      i2YRescFact = 1.0 / pixel_y_relation;
    } else if( mean_pixel_relation < 1.0 ) {
      /* The image 2 has poor resolution - bigger pixel resolution values
        and image 1 needs to be rescaled down */
      
      i1XRescFact = pixel_x_relation;
      i1YRescFact = pixel_y_relation;
    }
  }   
  
  /* Calculating the max tie-points to be generated for each image
     trying to keep the same tie-point density for both images */
     
  unsigned int img1MaxTps = max_tie_points;
  unsigned int img2MaxTps = max_tie_points;
  
  {
    double resc_box1_nlines =  ( (double)orig_box1_nlines  ) * 
      i1YRescFact;
    double resc_box1_ncols =  ( (double)orig_box1_ncols ) * 
      i1XRescFact;
    double resc_box2_nlines =  ( (double)orig_box2_nlines  ) * 
      i2YRescFact;
    double resc_box2_ncols =  ( (double)orig_box2_ncols ) * 
      i2XRescFact;
      
    double resc_box1_area = resc_box1_nlines * resc_box1_ncols;
    double resc_box2_area = resc_box2_nlines * resc_box2_ncols;
      
    if( resc_box1_area > resc_box2_area )
    {
      img1MaxTps = (unsigned int)( 
        resc_box1_area / 
        ( resc_box2_area / ( (double)max_tie_points ) ) );
    }
    else if( resc_box1_area < resc_box2_area )
    {
      img2MaxTps = (unsigned int)( 
        resc_box2_area / 
        ( resc_box1_area / ( (double)max_tie_points ) ) ); 
    }
  }
  
  // Moravec specific variables
  
  // factor = original_image / downsampled_image
  const double moravecDownSampleFactor = 1.5;
  // factor = original_image / upsampled_image
  const double moravecUpSampleFactor = 0.5;
  
  const double moravecRepeatabilityMinTolerance = 3.0;

  unsigned int moravecDSWindowSize = (unsigned int)( 
    moravec_window_width / moravecDownSampleFactor );
  if( ( moravecDSWindowSize % 2 ) == 0 ) ++moravecDSWindowSize;
  TEAGN_DEBUG_CONDITION( moravecDSWindowSize > 2, 
    "Invalid moravecDSWindowSize ")

  unsigned int moravecUSWindowSize = (unsigned int)( 
    moravec_window_width / moravecUpSampleFactor );
  if( ( moravecUSWindowSize % 2 ) == 0 ) ++moravecUSWindowSize;
  TEAGN_DEBUG_CONDITION( moravecUSWindowSize > 2, 
    "Invalid moravecUSWindowSize ")
       
  /* Calculating the number of allowed threads - this process
     not included into the count */
  
  if( enable_multi_thread )
  {
    jobsMan_.setAutoMaxSimulJobs();
  }
  else
  {
    jobsMan_.setMaxSimulJobs( 0 );
  }
  
 /* Loading images */
  
  TeSharedPtr< TePDIMtxDoubleAdptInt > img1_matrix_ptr;
  TeSharedPtr< TePDIMtxDoubleAdptInt > img1_matrix_ds_ptr; // Downsampled version
  TeSharedPtr< TePDIMtxDoubleAdptInt > img1_matrix_us_ptr; // Upsampled version  

  if( processImg1 )
  {
    createImgMatrix( input_image1_ptr_->params().dataType_[input_channel1_],
      img1_matrix_ptr );
    createImgMatrix( input_image1_ptr_->params().dataType_[input_channel1_],
      img1_matrix_ds_ptr );
    if( enable_upsampled_filter )
    {
      createImgMatrix( input_image1_ptr_->params().dataType_[input_channel1_],
        img1_matrix_us_ptr );
    }
  }
  
  TeSharedPtr< TePDIMtxDoubleAdptInt > img2_matrix_ptr;
  TeSharedPtr< TePDIMtxDoubleAdptInt > img2_matrix_ds_ptr; // Downsampled version
  TeSharedPtr< TePDIMtxDoubleAdptInt > img2_matrix_us_ptr; // Upsampled version

  if( processImg2 )
  {
    createImgMatrix( input_image2_ptr_->params().dataType_[input_channel2_],
      img2_matrix_ptr );
    createImgMatrix( input_image2_ptr_->params().dataType_[input_channel2_],
      img2_matrix_ds_ptr );
    if( enable_upsampled_filter )
    {
      createImgMatrix( input_image2_ptr_->params().dataType_[input_channel2_],
        img2_matrix_us_ptr );  
    }
  }
 
  {
    LoadImageParams img1pars;
    img1pars.input_image_ptr_ = input_image1_ptr_;
    img1pars.input_image_channel_ = input_channel1_;
    img1pars.img_matrix_ptr_ = img1_matrix_ptr.nakedPointer();
    img1pars.img_matrix_ds_ptr_ = img1_matrix_ds_ptr.nakedPointer();
    img1pars.img_matrix_us_ptr_ = img1_matrix_us_ptr.nakedPointer();
    img1pars.in_box_x_off_ = orig_box1_x_off;
    img1pars.in_box_y_off_ = orig_box1_y_off;
    img1pars.in_box_nlines_ = orig_box1_nlines;
    img1pars.in_box_ncols_ = orig_box1_ncols;
    img1pars.progress_enabled_ = progress_enabled_;
    img1pars.glb_mem_lock_ptr_ = &globalMutex_;
    img1pars.origImgXRescFact_ = i1XRescFact;
    img1pars.origImgYRescFact_ = i1YRescFact;
    img1pars.moravecDownSampleFactor_ = moravecDownSampleFactor;
    img1pars.moravecUpSampleFactor_ = moravecUpSampleFactor;
    img1pars.returnValue_ = true;
    img1pars.generateUpsampledImage_ = enable_upsampled_filter;
    
    LoadImageParams img2pars = img1pars;
    img2pars.input_image_ptr_ = input_image2_ptr_;
    img1pars.input_image_channel_ = input_channel2_;
    img2pars.img_matrix_ptr_ = img2_matrix_ptr.nakedPointer();
    img2pars.img_matrix_ds_ptr_ = img2_matrix_ds_ptr.nakedPointer();
    img2pars.img_matrix_us_ptr_ = img2_matrix_us_ptr.nakedPointer();
    img2pars.in_box_x_off_ = orig_box2_x_off;
    img2pars.in_box_y_off_ = orig_box2_y_off;
    img2pars.in_box_nlines_ = orig_box2_nlines;
    img2pars.in_box_ncols_ = orig_box2_ncols;    
    img2pars.origImgXRescFact_ = i2XRescFact;
    img2pars.origImgYRescFact_ = i2YRescFact;
    
    if( processImg1 )
    {    
      TeFunctionCallThreadJob job1;
      job1.jobFunc_ = loadImage;
      job1.jobFuncParsPtr_ = (void*)&img1pars;
      jobsMan_.executeJob( job1 );
    }

    if( processImg2 )
    {    
      TeFunctionCallThreadJob job2;
      job2.jobFunc_ = loadImage;
      job2.jobFuncParsPtr_ = (void*)&img2pars;
      jobsMan_.executeJob( job2 );
    }
    
    jobsMan_.waitAllToFinish();
    
    TEAGN_TRUE_OR_RETURN( img1pars.returnValue_, "Error loading image" )
    TEAGN_TRUE_OR_RETURN( img2pars.returnValue_, "Error loading image" )
  }  

/* 
matrix2Tiff( *img1_matrix_ptr, "img1_matrix_ptr_loaded.tif" );
matrix2Tiff( *img1_matrix_ds_ptr, "img1_matrix_ds_ptr_loaded.tif" );
matrix2Tiff( *img1_matrix_us_ptr, "img1_matrix_us_ptr_loaded.tif" );
matrix2Tiff( *img2_matrix_ptr, "img2_matrix_ptr_loaded.tif" );
matrix2Tiff( *img2_matrix_ds_ptr, "img2_matrix_ds_ptr_loaded.tif" );
matrix2Tiff( *img2_matrix_us_ptr, "img2_matrix_us_ptr_loaded.tif" );
*/
  
  // Extracting maximas from all images
  
  MaximasListT img1ds_maximas_list;
  MaximasListT img1us_maximas_list;
  MaximasListT img2ds_maximas_list;
  MaximasListT img2us_maximas_list;
  
  {
    // Creating job execution parameters 
    
    TeFunctionCallThreadJob dummyJob;
    dummyJob.jobFunc_ = extractLocalMaximas;
    
    ExtractLocalMaximasParams img1pars;
    img1pars.imgMatrixPtr_ = img1_matrix_ptr.nakedPointer();
    img1pars.outMaximasListPtr_ = &img1_maximas_list_;
    img1pars.moravecWindowSize_ = moravec_window_width;
    img1pars.maximasNumber_ = img1MaxTps;
    img1pars.moravecMinVarianceThreshold_ = variance_min_thrs;
    img1pars.seachSubImagesNmb_ = 9;
    img1pars.returnValue_ = true;
    img1pars.progressEnabled_ = progress_enabled_;
    img1pars.glbMutexptr_ = &globalMutex_;
    
    ExtractLocalMaximasParams img1dspars = img1pars;
    img1dspars.imgMatrixPtr_ = img1_matrix_ds_ptr.nakedPointer();
    img1dspars.outMaximasListPtr_ = &img1ds_maximas_list;
    img1dspars.moravecWindowSize_ = moravecDSWindowSize;

    ExtractLocalMaximasParams img1uspars = img1pars;
    img1uspars.imgMatrixPtr_ = img1_matrix_us_ptr.nakedPointer();
    img1uspars.outMaximasListPtr_ = &img1us_maximas_list;
    img1uspars.moravecWindowSize_ = moravecUSWindowSize;
    
    ExtractLocalMaximasParams img2pars = img1pars;;
    img2pars.imgMatrixPtr_ = img2_matrix_ptr.nakedPointer();
    img2pars.outMaximasListPtr_ = &img2_maximas_list_;
    img2pars.maximasNumber_ = img2MaxTps;
    
    ExtractLocalMaximasParams img2dspars = img2pars;
    img2dspars.imgMatrixPtr_ = img2_matrix_ds_ptr.nakedPointer();
    img2dspars.outMaximasListPtr_ = &img2ds_maximas_list;
    img2dspars.moravecWindowSize_ = moravecDSWindowSize;

    ExtractLocalMaximasParams img2uspars = img2pars;
    img2uspars.imgMatrixPtr_ = img2_matrix_us_ptr.nakedPointer();
    img2uspars.outMaximasListPtr_ = &img2us_maximas_list;
    img2uspars.moravecWindowSize_ = moravecUSWindowSize;
    
    // Starting jobs
    
    if( processImg1 )
    {
      dummyJob.jobFuncParsPtr_ = (void*)&img1pars;
      jobsMan_.executeJob( dummyJob );     
      
      dummyJob.jobFuncParsPtr_ = (void*)&img1dspars;
      jobsMan_.executeJob( dummyJob ); 
      
      if( enable_upsampled_filter )
      {
        dummyJob.jobFuncParsPtr_ = (void*)&img1uspars;
        jobsMan_.executeJob( dummyJob ); 
      }
    }
    
    if( processImg2 )
    {
      dummyJob.jobFuncParsPtr_ = (void*)&img2pars;
      jobsMan_.executeJob( dummyJob );     
      
      dummyJob.jobFuncParsPtr_ = (void*)&img2dspars;
      jobsMan_.executeJob( dummyJob );   
      
      if( enable_upsampled_filter )
      {
        dummyJob.jobFuncParsPtr_ = (void*)&img2uspars;
        jobsMan_.executeJob( dummyJob );   
      }
    }
    
    // Waiting all jobs to finish
    
    jobsMan_.waitAllToFinish();
    
    TEAGN_TRUE_OR_RETURN( img1pars.returnValue_, 
      "Error loocking for maximas" )
    TEAGN_TRUE_OR_RETURN( img1dspars.returnValue_,
      "Error loocking for maximas" )
    TEAGN_TRUE_OR_RETURN( img1uspars.returnValue_, 
      "Error loocking for maximas" )
    TEAGN_TRUE_OR_RETURN( img2pars.returnValue_, 
      "Error loocking for maximas" )
    TEAGN_TRUE_OR_RETURN( img2dspars.returnValue_, 
      "Error loocking for maximas" )
    TEAGN_TRUE_OR_RETURN( img2uspars.returnValue_, 
      "Error loocking for maximas" )
  }
  
/*
matrix2Tiff( *img1_matrix_ptr, "img1_matrix_ptr_max.tif",
  img1_maximas_list_ );
matrix2Tiff( *img1_matrix_ds_ptr, "img1_matrix_ds_ptr_max.tif",
  img1ds_maximas_list );
matrix2Tiff( *img1_matrix_us_ptr, "img1_matrix_us_ptr_max.tif",
  img1us_maximas_list );
matrix2Tiff( *img2_matrix_ptr, "img2_matrix_ptr_max.tif",
  img2_maximas_list_ );
matrix2Tiff( *img2_matrix_ds_ptr, "img2_matrix_ds_ptr_max.tif",
  img2ds_maximas_list );
matrix2Tiff( *img2_matrix_us_ptr, "img2_matrix_us_ptr_max.tif",
  img2us_maximas_list );  
*/

  // Applying moravec repeatability filter 
  
  if( processImg1 )
  {
    MaximasListT auxList;
    
    moravecRepeatability( img1_maximas_list_, img1ds_maximas_list,
      img1_matrix_ds_ptr->getNCols(),
      img1_matrix_ds_ptr->getNLines(), 
      moravecRepeatabilityMinTolerance / moravecDownSampleFactor, 
      moravecDownSampleFactor, auxList );
    
    if( enable_upsampled_filter )
    {
      moravecRepeatability( auxList, img1us_maximas_list,
        img1_matrix_us_ptr->getNCols(),
        img1_matrix_us_ptr->getNLines(), 
        moravecRepeatabilityMinTolerance / moravecUpSampleFactor, 
        moravecUpSampleFactor, img1_maximas_list_ );    
    }
    else
    {
      img1_maximas_list_ = auxList;
    }
  }
  
  if( processImg2 )
  {
    MaximasListT auxList;
    
    moravecRepeatability( img2_maximas_list_, img2ds_maximas_list,
      img2_matrix_ds_ptr->getNCols(),
      img2_matrix_ds_ptr->getNLines(), 
      moravecRepeatabilityMinTolerance / moravecDownSampleFactor, 
      moravecDownSampleFactor, auxList );
    
    if( enable_upsampled_filter )
    {
      moravecRepeatability( auxList, img2us_maximas_list,
        img2_matrix_us_ptr->getNCols(),
        img2_matrix_us_ptr->getNLines(), 
        moravecRepeatabilityMinTolerance / moravecUpSampleFactor, 
        moravecUpSampleFactor, img2_maximas_list_ );
    }
    else
    {
      img2_maximas_list_ = auxList;
    }
  }
/*  
matrix2Tiff( *img1_matrix_ptr, "img1_matrix_ptr_max_repeat.tif",
  img1_maximas_list_ );
matrix2Tiff( *img2_matrix_ptr, "img2_matrix_ptr_max_repeat.tif",
  img2_maximas_list_ );
*/
  
  img1us_maximas_list.clear();
  img1ds_maximas_list.clear();  
  img2us_maximas_list.clear();
  img2ds_maximas_list.clear();
  
  img1_matrix_ds_ptr.reset();
  img1_matrix_us_ptr.reset();
  img2_matrix_ds_ptr.reset();
  img2_matrix_us_ptr.reset();  
  
  // filtering out points closer to images borders to avoid
  // correlation windows generation errors 

  
  {
    /* The radius of a feature window rotated by 90 degrees. 
     * over the input image */
    const unsigned int winRadius = (unsigned int)
      (
        ceil( 
          sqrt( 
            2 
            * 
            ( 
              ( (double)corr_window_width ) 
              *
              ( (double)corr_window_width )
            )
          ) / 2.0
        )
      );
    
    unsigned int img1FistValidLine = winRadius;
    unsigned int img1FistValidCol = winRadius;
    unsigned int img2FistValidLine = winRadius;
    unsigned int img2FistValidCol = winRadius;
    
    // Removing maximas outside the image 1 allowed bondaries
    
    if( processImg1 )
    {
      unsigned int img1LinesBound = img1_matrix_ptr->getNLines() - winRadius;
      unsigned int img1ColsBound = img1_matrix_ptr->getNCols() - winRadius;
    
      MaximasListT::iterator maxIt = img1_maximas_list_.begin();
      MaximasListT::iterator maxItEnd = img1_maximas_list_.end();
      MaximasListT::iterator maxAuxIt;
      
      while( maxIt != maxItEnd )
      {
        if( ( maxIt->first < img1FistValidCol ) ||
          ( maxIt->first >= img1ColsBound ) || 
          ( maxIt->second < img1FistValidLine ) ||
          ( maxIt->second >= img1LinesBound ) )
        {
          maxAuxIt = maxIt;
          ++maxIt;
          img1_maximas_list_.erase( maxAuxIt );
        }
        else
        {
          ++maxIt;
        }
      }
    }
    
    // Removing maximas outside the image 2 allowed bondaries
    
    if( processImg2 )
    {
      unsigned int img2LinesBound = img2_matrix_ptr->getNLines() - winRadius;
      unsigned int img2ColsBound = img2_matrix_ptr->getNCols() - winRadius;
    
      MaximasListT::iterator maxIt = img2_maximas_list_.begin();
      MaximasListT::iterator maxItEnd = img2_maximas_list_.end();
      MaximasListT::iterator maxAuxIt;
      
      while( maxIt != maxItEnd )
      {
        if( ( maxIt->first < img2FistValidCol ) ||
          ( maxIt->first >= img2ColsBound ) || 
          ( maxIt->second < img2FistValidLine ) ||
          ( maxIt->second >= img2LinesBound ) )
        {
          maxAuxIt = maxIt;
          ++maxIt;
          img2_maximas_list_.erase( maxAuxIt );
        }
        else
        {
          ++maxIt;
        }
      }
    }
  }
  
/*
matrix2Tiff( *img1_matrix_ptr, "img1_matrix_ptr_max_outremoved.tif",
  img1_maximas_list_ );
matrix2Tiff( *img2_matrix_ptr, "img2_matrix_ptr_max_outremoved.tif",
  img2_maximas_list_ );  
*/
  
  /* Generating features matrix */
  
  {
    GenerateCorrelationFeaturesParams params1;
    
    switch( matching_method_ )
    {
      case EuclidianDistMethod :
      {
        params1.normalizeWindows_ = true;
        break;
      }      
      case NormCrossCorrMethod :
      {
        params1.normalizeWindows_ = true;
        break;
      }
      default :
      {
        TEAGN_LOG_AND_THROW( "Invalid correlation method" );
        break;
      }
    }
    
    params1.imgMatrixPtr_ = img1_matrix_ptr.nakedPointer();
    params1.inMaximasListPtr_ = &img1_maximas_list_;
    params1.featMtxPtr_ = &img1featMtx_;
    params1.returnValue_ = true;
    params1.glbMemLockPtr_ = &globalMutex_;
    params1.corrWindowsWidth_ = corr_window_width;
    params1.progressEnabled_ = progress_enabled_;
    
    GenerateCorrelationFeaturesParams params2 = params1;
    params2.imgMatrixPtr_ = img2_matrix_ptr.nakedPointer();
    params2.inMaximasListPtr_ = &img2_maximas_list_;
    params2.featMtxPtr_ = &img2featMtx_;
    
    // Starting jobs
    
    TeFunctionCallThreadJob dummyJob;
    dummyJob.jobFunc_ = generateCorrelationFeatures;
   
    // Starting jobs
    
    if( processImg1 )
    {
      dummyJob.jobFuncParsPtr_ = (void*)&params1;
      jobsMan_.executeJob( dummyJob );     
    }
    
    if( processImg2 )
    {
      dummyJob.jobFuncParsPtr_ = (void*)&params2;
      jobsMan_.executeJob( dummyJob );       
    }
    
    // Waiting all jobs to finish
    
    jobsMan_.waitAllToFinish();
    
    TEAGN_TRUE_OR_RETURN( params1.returnValue_, 
      "Error generating correlation features" )
    TEAGN_TRUE_OR_RETURN( params2.returnValue_,
      "Error generating correlation features" )    
  }
  
/*  
features2Tiff( corr_window_width, img1_maximas_list_, img1featMtx, 
  "img1feat" );
features2Tiff( corr_window_width, img2_maximas_list_, img2featMtx, 
  "img2feat" ); 
*/
  
  // Generating tie-points (images boxes reference)
  
  TEAGN_TRUE_OR_RETURN( matchFeatures( matching_method_, img1featMtx_,
    img1_maximas_list_, img2featMtx_, img2_maximas_list_, *out_tie_points_ptr
    ), "Error generating tie-points" );
  
/*  
matrix2Tiff( *img1_matrix_ptr, "img1_matrix_ptr_tps.tif",
  *out_tie_points_ptr, true );
matrix2Tiff( *img2_matrix_ptr, "img2_matrix_ptr_tps.tif",
  *out_tie_points_ptr, false );      
*/  
  
  img1_matrix_ds_ptr.reset();
  img1_matrix_us_ptr.reset();  
    
  // Bring tie-points to original images reference
    
  {
    for( unsigned int tpsVecIdx = 0 ; tpsVecIdx < 
      out_tie_points_ptr->size() ; ++tpsVecIdx )
    {
      TeCoordPair& coordPair = out_tie_points_ptr->operator[]( tpsVecIdx );
      
      coordPair.pt1.x( ( coordPair.pt1.x() / i1XRescFact ) + (double)orig_box1_x_off );
      coordPair.pt1.y( ( coordPair.pt1.y() / i1YRescFact ) + (double)orig_box1_y_off );
      
      coordPair.pt2.x( ( coordPair.pt2.x() / i2XRescFact ) + (double)orig_box2_x_off );
      coordPair.pt2.y( ( coordPair.pt2.y() / i2YRescFact ) + (double)orig_box2_y_off );
    }
  }
  
  /* Doing geometrical filtering using user supplied 
     geometrical transformation parameters */
     
  if( ! skip_geom_filter ) {
    gt_params.tiepoints_ = (*out_tie_points_ptr);
       
    TeGeometricTransformation::pointer trans_ptr( 
      TeGTFactory::make( gt_params ) );
    if( trans_ptr->reset( gt_params ) ) {
      trans_ptr->getParameters( gt_params );
      
      /* updating the output tie points */
      
      (*out_tie_points_ptr) = gt_params.tiepoints_;
      
      /* updating the output gt_params, if available */
      
      TeGTParams::pointer out_gt_params_ptr;
      if( params_.CheckParameter< TeGTParams::pointer >( 
        "out_gt_params_ptr" ) ) 
      {
        params_.GetParameter( "out_gt_params_ptr", out_gt_params_ptr );
        
        *out_gt_params_ptr = gt_params;
      }       
    } else {
      out_tie_points_ptr->clear();
    }
  }
  
  // So far.... so good !!
  
  return true;
}


void TePDIMMIOMatching::ResetState( const TePDIParameters& params )
{
  // Do we need to re-process input image 1 ??
  
  FeatMatchingMethod matching_method = NormCrossCorrMethod;
  if( params.CheckParameter< FeatMatchingMethod >( 
    "matching_method" ) ) 
  {
    params.GetParameter( "matching_method", matching_method );
  }   
  
  TePDITypes::TePDIRasterPtrType input_image1_ptr;
  params.GetParameter( "input_image1_ptr", input_image1_ptr ); 
  
  unsigned int input_channel1 = 0;
  params.GetParameter( "input_channel1", input_channel1 );
  
  TeBox input_box1;
  if( params.CheckParameter< TeBox >( "input_box1" ) ) {
    params.GetParameter( "input_box1", input_box1 );
  } else {
    TeBox input_box1_proj = input_image1_ptr->params().box();
    
    TePDIUtils::MapCoords2RasterIndexes( input_box1_proj, input_image1_ptr,
      input_box1 );
  }   
  
  if( ( matching_method_ != matching_method ) || ( input_image1_ptr != 
    input_image1_ptr_ ) || ( input_channel1 != 
    input_channel1_ ) || ( input_box1 != input_box1_ ) )
  {
    img1featMtx_.Reset();
    img1_maximas_list_.clear();
  }
  
  // Do we need to re-process input image 2 ??
  
  TePDITypes::TePDIRasterPtrType input_image2_ptr;
  params.GetParameter( "input_image2_ptr", input_image2_ptr ); 
  
  unsigned int input_channel2 = 0;
  params.GetParameter( "input_channel2", input_channel2 );
  
  TeBox input_box2;
  if( params.CheckParameter< TeBox >( "input_box2" ) ) {
    params.GetParameter( "input_box2", input_box2 );
  } else {
    TeBox input_box2_proj = input_image2_ptr->params().box();
    
    TePDIUtils::MapCoords2RasterIndexes( input_box2_proj, input_image2_ptr,
      input_box2 );
  }    
  
  if( ( matching_method_ != matching_method ) || ( input_image2_ptr != 
    input_image2_ptr_ ) || ( input_channel2 != 
    input_channel2_ ) || ( input_box2 != input_box2_ ) )
  {
    img2featMtx_.Reset();
    img2_maximas_list_.clear();
  }
  
  matching_method_ = matching_method;
  input_image1_ptr_ = input_image1_ptr;
  input_channel1_ = input_channel1;
  input_box1_ = input_box1;  
  input_image2_ptr_ = input_image2_ptr;
  input_channel2_ = input_channel2; 
  input_box2_ = input_box2;
}


void TePDIMMIOMatching::loadImage( void * paramsPtr )
{
 /* Retriving parameters */
  
  LoadImageParams& params = *( (LoadImageParams*) paramsPtr );
  
  params.returnValue_ = false;
  
  // Loading original image box
  
  const unsigned int origBoxRescNLines = (unsigned int)(
    params.in_box_nlines_ * params.origImgYRescFact_ );
  const unsigned int origBoxRescNCols = (unsigned int)(
    params.in_box_ncols_ * params.origImgXRescFact_ );
  
  {
    // Allocating resources 
    
    params.glb_mem_lock_ptr_->lock();
    
    if( ! params.img_matrix_ptr_->reset( origBoxRescNLines, 
      origBoxRescNCols ) )
    {
      TEAGN_LOGERR( "Unable to allocate space for the loaded image" )
      params.glb_mem_lock_ptr_->unLock();
      return;
    }
      
    params.glb_mem_lock_ptr_->unLock();
    
    // Interpolator - if needed
    
    TePDIInterpolator::InterpMethod intMethod = 
      TePDIInterpolator::NNMethod;
    if( ( params.origImgXRescFact_ != 1.0 ) || 
      ( params.origImgYRescFact_ != 1.0 ) )
    {
      intMethod = TePDIInterpolator::BicubicMethod;
    }
    
    TePDIInterpolator interp;
    TEAGN_TRUE_OR_THROW( interp.reset( params.input_image_ptr_,
      intMethod,
      params.input_image_ptr_->params().useDummy_ ?
      params.input_image_ptr_->params().dummy_[ params.input_image_channel_ ] : 
      0 ), "Interpolator error" );    
      
    unsigned int curr_out_line = 0;
    unsigned int curr_out_col = 0;
    unsigned int curr_input_line = 0;
    unsigned int curr_input_col = 0;
    double value = 0;
    TeRaster& inputRasterRef = (*params.input_image_ptr_);
    
    TePDIPIManager progress( "Loading image data", params.in_box_nlines_, 
       params.progress_enabled_ );
    
    for( curr_out_line = 0 ; curr_out_line < origBoxRescNLines ; 
      ++curr_out_line ) {
      
      curr_input_line = (unsigned int)( ( ( (double)curr_out_line ) / 
        params.origImgYRescFact_ ) + ( (double) params.in_box_y_off_ ) );      
      
      for( curr_out_col = 0 ; curr_out_col < origBoxRescNCols ; 
        ++curr_out_col ) {
        
        curr_input_col = (unsigned int)( ( ( (double)curr_out_col ) / 
            params.origImgXRescFact_ ) + ( (double) params.in_box_x_off_ ) );        
        
        inputRasterRef.getElement( curr_input_col, curr_input_line,
          value, params.input_image_channel_ );
        
        params.img_matrix_ptr_->setValue( curr_out_line, curr_out_col, 
          value );
      }
      
      if( progress.Increment() )
      {
        TEAGN_LOGMSG( "Canceled by the user" );
        return;
      }
    }
  }
  
  // Retriving raster channel max and min channel allowed values
  
  double rasterChnMin = 0;
  double rasterChnMax = 0;
  
  if( ! TePDIUtils::TeGetRasterMinMaxBounds( 
    params.input_image_ptr_,
    params.input_image_channel_, rasterChnMin, rasterChnMax ) )
  {
     TEAGN_LOGERR( "Error getting channels bounds" );  
     return;
  }
  
  // Loading downsampled image box
  
  {
    const unsigned int out_box_nlines = (unsigned int)
      ( ((double)origBoxRescNLines) / params.moravecDownSampleFactor_ );
    const unsigned int out_box_ncols = (unsigned int)
      ( ((double)origBoxRescNCols) / params.moravecDownSampleFactor_ );
    
    if( ! bicubicResampleMatrix( *( params.img_matrix_ptr_ ), out_box_nlines,
      out_box_ncols, params.progress_enabled_, *(params.glb_mem_lock_ptr_),
      rasterChnMin, rasterChnMax, *( params.img_matrix_ds_ptr_ ) ) )
    {
      TEAGN_LOGERR( "Error generating downsampled image" );
      return;
    }
  }
  
  // Loading upsampled image box
  
  if( params.generateUpsampledImage_ )
  {
    // Calculating sizes for resampled boxes
    
    const unsigned int out_box_nlines = (unsigned int)
      ( ((double)origBoxRescNLines) / params.moravecUpSampleFactor_ );
    const unsigned int out_box_ncols = (unsigned int)
      ( ((double)origBoxRescNCols) / params.moravecUpSampleFactor_ );
    
    if( ! bicubicResampleMatrix( *( params.img_matrix_ptr_ ), out_box_nlines,
      out_box_ncols, params.progress_enabled_, *(params.glb_mem_lock_ptr_),
      rasterChnMin, rasterChnMax, *( params.img_matrix_us_ptr_ ) ) )
    {
      TEAGN_LOGERR( "Error generating downsampled image" );
      return;
    }
  }    
  
  params.returnValue_ = true;
}

void TePDIMMIOMatching::createImgMatrix( TeDataType dataType, 
  TeSharedPtr< TePDIMtxDoubleAdptInt >& matrixPtr )
{
  switch( dataType )
  {
    case TeDOUBLE :
    {
      matrixPtr.reset( new TePDIMtxDoubleAdpt< double >( 
        TePDIMtxDoubleAdpt< double >::AutoMemPol ) );
      break;
    }
    case TeFLOAT :
    {
      matrixPtr.reset( new TePDIMtxDoubleAdpt< float >( 
        TePDIMtxDoubleAdpt< float >::AutoMemPol ) );
      break;
    }
    case TeLONG :
    {
      matrixPtr.reset( new TePDIMtxDoubleAdpt< long int >( 
        TePDIMtxDoubleAdpt< long int >::AutoMemPol ) );
      break;
    }
    case TeUNSIGNEDLONG :
    {
      matrixPtr.reset( new TePDIMtxDoubleAdpt< unsigned long int >( 
        TePDIMtxDoubleAdpt< unsigned long int >::AutoMemPol ) );
      break;
    }
    case TeINTEGER :
    {
      matrixPtr.reset( new TePDIMtxDoubleAdpt< int >( 
        TePDIMtxDoubleAdpt< int >::AutoMemPol ) );
      break;
    }
    case TeSHORT :
    {
      matrixPtr.reset( new TePDIMtxDoubleAdpt< short int >( 
        TePDIMtxDoubleAdpt< short int >::AutoMemPol ) );
      break;
    }
    case TeUNSIGNEDSHORT :
    {
      matrixPtr.reset( new TePDIMtxDoubleAdpt< unsigned short int >( 
        TePDIMtxDoubleAdpt< unsigned short int >::AutoMemPol ) );
      break;
    }
    case TeCHAR :
    {
      matrixPtr.reset( new TePDIMtxDoubleAdpt< char >( 
        TePDIMtxDoubleAdpt< char >::AutoMemPol ) );
      break;
    }
    case TeUNSIGNEDCHAR :
    {
      matrixPtr.reset( new TePDIMtxDoubleAdpt< unsigned char >( 
        TePDIMtxDoubleAdpt< unsigned char >::AutoMemPol ) );
      break;
    }
    case TeBIT :
    {
      matrixPtr.reset( new TePDIMtxDoubleAdpt< unsigned char >( 
        TePDIMtxDoubleAdpt< unsigned char >::AutoMemPol ) );
      break;
    }
    default :
    {
      TEAGN_LOG_AND_THROW( "Invalid data type ");
      break;
    }
  }
}

void TePDIMMIOMatching::extractLocalMaximas( void * paramsPtr )
{
 /* Retriving parameters */
  
  ExtractLocalMaximasParams params = *( (ExtractLocalMaximasParams*) 
    paramsPtr );
    
  TEAGN_DEBUG_CONDITION( params.seachSubImagesNmb_ > 0,
    "Invalid params.seachSubImagesNmb_" );
    
  ( (ExtractLocalMaximasParams*) paramsPtr )->returnValue_ = false;
  
  /* Fixing parameters */
   
  unsigned int sqrtSI = (unsigned int)ceil( sqrt( 
    (double)params.seachSubImagesNmb_ ) );
  if( sqrtSI == 0 ) sqrtSI = 1;
  
  params.seachSubImagesNmb_ = sqrtSI * sqrtSI;
  
  params.maximasNumber_ = MAX( params.maximasNumber_, 
    params.seachSubImagesNmb_ );
  params.maximasNumber_ = params.maximasNumber_ / 
    params.seachSubImagesNmb_;  
    
  if( params.moravecWindowSize_ % 2 == 0 ) ++params.moravecWindowSize_;
  
  /* Calling the maximas extraction for each sub-image */
  
  const unsigned int subWindowMaxWidth = params.imgMatrixPtr_->getNCols() / 
    sqrtSI;
  const unsigned int subWindowMaxHeight = params.imgMatrixPtr_->getNLines() / 
    sqrtSI;
    
  if( ( subWindowMaxWidth > 0 ) && ( subWindowMaxHeight > 0 ) )
  {
    unsigned int windColBound = 0;
    unsigned int windLinBound = 0;
    unsigned int windWidth = 0;
    unsigned int windHeight = 0;
    
    MaximasMapT maximasMap;
      
    TePDIPIManager progress( "Finding local maximas",
      ( ( params.imgMatrixPtr_->getNLines() - ( sqrtSI * 
      params.moravecWindowSize_ ) ) / params.moravecWindowSize_ ) * sqrtSI, 
      params.progressEnabled_ );
      
    for( unsigned int line = 0 ; line < params.imgMatrixPtr_->getNLines() ;
      line += subWindowMaxHeight )
    {
      windLinBound = MIN( line + subWindowMaxHeight, 
        params.imgMatrixPtr_->getNLines() );
      windHeight = windLinBound - line;
      
      if( windHeight > params.moravecWindowSize_ )
      {
        for( unsigned int col = 0 ; col < params.imgMatrixPtr_->getNCols() ; 
          col += subWindowMaxWidth ) 
        {
          windColBound = MIN( col + subWindowMaxWidth, 
            params.imgMatrixPtr_->getNCols() );
          windWidth = windColBound - col;
          
          if( windWidth > params.moravecWindowSize_ )
          {
            if( ! extractLocalMaximas( params, col, line, windWidth, 
              windHeight, maximasMap, progress ) )
            {
              return;
            }
          }
        }
      }
    }
    
    // Filtering out maximis below the variance threshold
    
    if( maximasMap.size() > 0 )
    {
      if( params.moravecMinVarianceThreshold_ > 0.0 )
      {
        double mean = 0;
        MaximasMapT::iterator it = maximasMap.begin();
        const MaximasMapT::iterator itEnd = maximasMap.end();  
        while( it != itEnd ) 
        {
          mean += it->first ;
          ++it;
        }    
        mean /= (double)( maximasMap.size() );
        
        double stddev = 0;
        double diff = 0;
        it = maximasMap.begin();
        while( it != itEnd ) 
        {
          diff = it->first - mean;
          stddev += ( diff * diff );
          
          ++it;
        }
        stddev /= (double)( maximasMap.size() );
        stddev = sqrt( stddev );
        
        const double minAllowedVariance = mean - ( ( stddev / 2.0 ) *   
          params.moravecMinVarianceThreshold_ );
        
        it = maximasMap.begin();
        
        while( it != itEnd ) 
        {
          if( it->first >= minAllowedVariance )
            params.outMaximasListPtr_->push_back( it->second );
          
          ++it;
        }
      }
      else
      {
        MaximasMapT::iterator it = maximasMap.begin();
        MaximasMapT::iterator itEnd = maximasMap.end();  
        
        while( it != itEnd ) 
        {
          params.outMaximasListPtr_->push_back( it->second );
          
          ++it;
        }
      }
    }
  }
    
  // So far... so good !
  
  ( (ExtractLocalMaximasParams*) paramsPtr )->returnValue_ = true;
}

bool TePDIMMIOMatching::extractLocalMaximas( 
  ExtractLocalMaximasParams& params,
  unsigned int xStart, unsigned int yStart, unsigned int width,
  unsigned int height, MaximasMapT& outMaximasMap,
  TePDIPIManager& progress )
{
  TEAGN_DEBUG_CONDITION( width > params.moravecWindowSize_, 
    "invalid block size" )
  TEAGN_DEBUG_CONDITION( height > params.moravecWindowSize_, 
    "invalid block size" )
  TEAGN_DEBUG_CONDITION( ( params.moravecWindowSize_ % 2 ), 
    "invalid moravec window size" )
    
    
  // Allocating the variance values buffer
  
  const int windRad = (unsigned int)( params.moravecWindowSize_ / 
    2 );    
    
  const unsigned int bufferLines = params.moravecWindowSize_;
  const unsigned int bufferCols = width - ( 2 * windRad );
  
  TePDIMatrix< double > buffer( TePDIMatrix< double >::RAMMemPol );
  if( ! buffer.Reset( bufferLines, bufferCols ) )
  {
    TEAGN_LOGERR( "Error allocating internal buffer" )
    return false;
  }

  // Iterating over image (reference: window center) looking for maximas
  
  // The found maximas coords ordered by diretional variance
  MaximasMapT maximasData;
  
  // Buffer over input 
  const unsigned int imgBufStartLinesBound = yStart + height - windRad -  
    params.moravecWindowSize_  + 1;
  const unsigned int imgBufStartCol = xStart + windRad;
  
  // Internal buffer block variables
  const unsigned int bufBlockStartLinesBound = bufferCols - ( bufferCols % 
    params.moravecWindowSize_ ) - params.moravecWindowSize_  + 1;
  unsigned int bufBlockStartCol = 0;
  unsigned int bufBlockCol = 0;
  unsigned int bufBlockLin = 0;

  int offset = 0;
  double hor_var = 0;
  double ver_var = 0;
  double diag_var = 0;
  double adiag_var = 0;
  double min_var = 0;
  double centerValue = 0;
  double neightValue = 0;
  double diffValue = 0;  
  
  TePDIMtxDoubleAdptInt& imgMatrix = *(params.imgMatrixPtr_);
  
  unsigned int bufLine = 0;
  unsigned int bufCol = 0;
  unsigned int im_line = 0;
  unsigned int im_col = 0;

  std::pair< double, std::pair< unsigned int, unsigned int > > 
    maxVarDataPair;
  
  // Starting from the first valid pixel where the variances can
  // be calculated
  for( unsigned int imgBufStartLine = windRad + yStart ; 
    imgBufStartLine < imgBufStartLinesBound ; 
    imgBufStartLine += params.moravecWindowSize_ )
  {
    // Generating variance information for the current buffer
    // position over input image
    
    for( bufLine = 0 ; bufLine < bufferLines ; ++bufLine )
    {
      im_line = bufLine + imgBufStartLine;
        
      for( bufCol = 0 ; bufCol < bufferCols ; ++bufCol )
      {        
        im_col = bufCol + imgBufStartCol;
        
        imgMatrix.getValue( im_line, im_col, centerValue );
        
        hor_var = 0;
        ver_var = 0;
        diag_var = 0;
        adiag_var = 0;
    
        for( offset = -windRad ; offset <= windRad ; ++offset )
        {
          imgMatrix.getValue( im_line, im_col + offset, neightValue );
          diffValue = neightValue - centerValue;
          hor_var += ( diffValue * diffValue );
          
          imgMatrix.getValue( im_line + offset, im_col, neightValue );
          diffValue = neightValue - centerValue;
          ver_var += ( diffValue * diffValue );

          imgMatrix.getValue( im_line + offset, im_col + offset, neightValue );
          diffValue = neightValue - centerValue;
          diag_var += ( diffValue * diffValue );
        
          imgMatrix.getValue( im_line - offset, im_col + offset, neightValue );
          diffValue = neightValue - centerValue;
          adiag_var += ( diffValue * diffValue );
        }
        
        min_var = MIN( hor_var, MIN( ver_var, MIN( diag_var, adiag_var ) ) );
        
        buffer[ bufLine ][ bufCol ] = min_var;
      }
    }
    
    // Selecting the higher variance levels for each window of 
    // moravecWindowSize_ X moravecWindowSize_ from inside the 
    // current buffer
    
    for( bufBlockStartCol = 0 ; bufBlockStartCol < 
      bufBlockStartLinesBound ; bufBlockStartCol += params.moravecWindowSize_ )
    {
      maxVarDataPair.first = DBL_MAX * (-1.0);
        
      for( bufBlockLin = 0 ; bufBlockLin < params.moravecWindowSize_ ; 
        ++bufBlockLin )
      {
        im_line = bufBlockLin + imgBufStartLine;
        
        for( bufBlockCol = 0 ; bufBlockCol < params.moravecWindowSize_ ; 
          ++bufBlockCol )
        {  
          const double& bufElement = buffer[ bufBlockLin ][ bufBlockCol + 
            bufBlockStartCol ];
          
          if( bufElement > maxVarDataPair.first )
          {
            im_col = bufBlockCol + bufBlockStartCol + imgBufStartCol;
            
            maxVarDataPair.first = bufElement;
            maxVarDataPair.second.second = im_line;
            maxVarDataPair.second.first = im_col;
          }
        }
      }
      
      // Adding maxima point to the ordered maximas list

      maximasData.insert( maxVarDataPair );
    }
    
    if( progress.Increment() )
    {
      TEAGN_LOGERR( "Canceled by the user");
      return false;
    }
  }  
  
  // Inserting params.maximasNumber_ maximas with higher directional 
  // variance into output map
  
  unsigned int addedMaximas = 0;
  
  MaximasMapT::reverse_iterator it = maximasData.rbegin();  
  MaximasMapT::reverse_iterator itEnd = maximasData.rend();  
    
  while( ( it != itEnd ) && ( addedMaximas < params.maximasNumber_ ) )
  {
    outMaximasMap.insert( *it );
    
    ++addedMaximas;
    ++it;
  }
  
  return true;
}

void TePDIMMIOMatching::moravecRepeatability( const MaximasListT& inputList,
  const MaximasListT& constraintList, 
  unsigned int constraintImageWidth,
  unsigned int constraintImageHeight,  
  double moravecRepeatabilityMinTolerance, 
  double moravecReSampleFactor, 
  MaximasListT& outputList )
{
  TEAGN_DEBUG_CONDITION( moravecRepeatabilityMinTolerance > 0,
    "Invalid moravecRepeatabilityMinTolerance ")
  TEAGN_DEBUG_CONDITION( moravecReSampleFactor > 0,
    "Invalid moravecReSampleFactor ")
      
  outputList.clear();
  
  // Criating an RTree for seach optmization
  
  const TeBox myTreeBBox( -1.0, -1.0, (double)(constraintImageWidth),
    (double)(constraintImageHeight) );  
  
  TeSAM::TeRTree<MaximasListNodeT const*, 8, 4> myTree( myTreeBBox );
  
  {
    MaximasListT::const_iterator cListIt = constraintList.begin();
    MaximasListT::const_iterator cListItEnd = constraintList.end();
    TeBox auxBox;
    
    while( cListIt != cListItEnd )
    {
      auxBox.x1_ = (double)(cListIt->first) - moravecRepeatabilityMinTolerance;
      auxBox.y1_ = (double)(cListIt->second) - moravecRepeatabilityMinTolerance;
      
      auxBox.x2_ = (double)(cListIt->first) + moravecRepeatabilityMinTolerance;
      auxBox.y2_ = (double)(cListIt->second) + moravecRepeatabilityMinTolerance;
      
      myTree.insert( auxBox, &(*cListIt) );
          
      ++cListIt;
    }
  }
  
  // Iterating over input list
  
  MaximasListT::const_iterator iListIt = inputList.begin();
  MaximasListT::const_iterator iListItEnd = inputList.end();
  std::vector< MaximasListNodeT const* > resultVec;
  TeBox auxBox;
    
  while( iListIt != iListItEnd )
  {
    const MaximasListNodeT& iListEle = *iListIt;
    
    auxBox.x1_ = auxBox.x2_ = iListEle.first / moravecReSampleFactor;
    auxBox.y1_ = auxBox.y2_ = iListEle.second / moravecReSampleFactor;
    
    if( myTree.search( auxBox, resultVec ) )
    {
      outputList.push_back( iListEle );  
    }
    
    ++iListIt;  
  }
  
  return;
}

void TePDIMMIOMatching::generateCorrelationFeatures( void * paramsPtr )
{
  /* Retriving parameters */
  
  GenerateCorrelationFeaturesParams& params = 
    *( (GenerateCorrelationFeaturesParams*) paramsPtr );
  
  const TePDIMtxDoubleAdptInt& img_matrix = *(params.imgMatrixPtr_);
  
  TePDIMatrix< double >& img_features_matrix = *( params.featMtxPtr_);
  
  params.returnValue_ = false;
    
  if( params.inMaximasListPtr_->size() > 0 ) {
    /* Allocating output matrix */
       
    params.glbMemLockPtr_->lock();
    
    if( ! params.featMtxPtr_->Reset( params.inMaximasListPtr_->size(),
      params.corrWindowsWidth_ * params.corrWindowsWidth_,
      TePDIMatrix< double >::RAMMemPol ) )
    {
      TEAGN_LOGERR( "Error allocating image features matrix" )
      params.glbMemLockPtr_->unLock();
      return;
    }
      
    params.glbMemLockPtr_->unLock();
      
    /* variables related to the current window over the hole image */
    unsigned int curr_window_x_start = 0;
    unsigned int curr_window_y_start = 0;
    unsigned int curr_window_x_center = 0;
    unsigned int curr_window_y_center = 0;
    unsigned int curr_window_x_bound = 0;
    unsigned int curr_window_y_bound = 0;
    
    /*used on the rotation calcule */

    const unsigned int wind_radius = params.corrWindowsWidth_ / 2;
    // output window radius
    const double wind_radius_double = (double)wind_radius;

    const unsigned int img_features_matrix_cols = 
      params.featMtxPtr_->GetColumns();
    unsigned int curr_x = 0;
    unsigned int curr_y = 0;
    double curr_x_minus_radius = 0;
    double curr_y_minus_radius = 0;
    unsigned int curr_offset = 0;
    double int_x_dir = 0;
    double int_y_dir = 0;
    double int_norm = 0;
    double rotated_curr_x = 0;/* rotaded coord - window ref */
    double rotated_curr_y = 0;/* rotaded coord - window ref */
    
     /* the found rotation parameters */
    double rot_cos = 0;
    double rot_sin = 0;
    
    /* the coords rotated but in the hole image reference */
    int rotated_curr_x_img = 0;
    int rotated_curr_y_img = 0;
    
    /* current window mean and standart deviation */
    double curr_wind_mean = 0.0;
    double curr_wind_stddev = 0.0;
    double curr_wind_stddev_aux = 0.0;
    
    // used on intensity vector calcule
    double imgMatrixValue1 = 0;
    double imgMatrixValue2 = 0;
    
    TePDIPIManager progress( "Generating correlation windows",
      params.inMaximasListPtr_->size(), params.progressEnabled_ );
    
    MaximasListT::const_iterator maximIt = params.inMaximasListPtr_->begin();
    MaximasListT::const_iterator maximItEnd = params.inMaximasListPtr_->end();
    unsigned int maximasListIndex = 0;
      
    while( maximIt != maximItEnd ) {
      
      /* Calculating the current window position */
     
      curr_window_x_center = maximIt->first;
      curr_window_y_center = maximIt->second;
      curr_window_x_start = curr_window_x_center - wind_radius;
      curr_window_y_start = curr_window_y_center - wind_radius;
      curr_window_x_bound = curr_window_x_start + params.corrWindowsWidth_;
      curr_window_y_bound = curr_window_y_start + params.corrWindowsWidth_;
        
      /* Estimating the intensity vector X direction */
      
      for( curr_offset = 0 ; curr_offset < wind_radius ;
        ++curr_offset ) {      

        for( curr_y = curr_window_y_start ; curr_y < curr_window_y_bound ; 
          ++curr_y ) 
        {
          img_matrix.getValue( curr_y, curr_window_x_bound - 1 - curr_offset, 
             imgMatrixValue1 );
          img_matrix.getValue( curr_y, curr_window_x_start + curr_offset, 
             imgMatrixValue2 );
        
          int_x_dir += imgMatrixValue1 - imgMatrixValue2;
        }
      }
      
      int_x_dir /= ( 2.0 * ( (double) wind_radius ) );
      
      /* Estimating the intensity vector y direction */
      
      for( curr_offset = 0 ; curr_offset < wind_radius ;
        ++curr_offset ) {      

        for( curr_x = curr_window_x_start ; 
          curr_x < curr_window_x_bound ;
          ++curr_x ) 
        {
          img_matrix.getValue( curr_window_y_start + curr_offset, curr_x , 
            imgMatrixValue1 );
          img_matrix.getValue( curr_window_y_bound - 1 - curr_offset, 
            curr_x, imgMatrixValue2 );          
        
          int_y_dir += imgMatrixValue1 - imgMatrixValue2;
        }
      }      
      
      int_y_dir /= ( 2.0 * ( (double) wind_radius ) );
      
      /* Calculating the rotation parameters - 
         counterclockwise rotation 
         
         | u |    |cos  -sin|   |X|
         | v | == |sin   cos| x |Y|
      */
      int_norm = sqrt( ( int_x_dir * int_x_dir ) + 
        ( int_y_dir * int_y_dir ) );
      
      if( int_norm != 0.0 ) {
        rot_cos = int_x_dir / int_norm;
        rot_sin = int_y_dir / int_norm;
      } else {
        /* No rotation */
        rot_cos = 1.0;
        rot_sin = 0.0;
      }
      
      /* Generating the rotated window data and inserting it into 
         the img_features_matrix by setting the intensity vector
         to the direction (1,0) by a clockwise rotation
         using the inverse matrix 
      
         | u |    |cos   sin|   |X|
         | v | == |-sin  cos| x |Y|
      */
         
      for( curr_y = 0 ; curr_y < params.corrWindowsWidth_ ; ++curr_y ) 
      {
        for( curr_x = 0 ; curr_x < params.corrWindowsWidth_ ; ++curr_x ) 
        {
          /* briging the window to the coord system center */
          
          curr_x_minus_radius = ((double)curr_x) - 
            wind_radius_double;
          curr_y_minus_radius = ((double)curr_y) - 
            wind_radius_double;
          
          /* rotating the centered window */
          
          rotated_curr_x = 
            ( ( rot_cos * curr_x_minus_radius ) + 
            ( rot_sin * curr_y_minus_radius ) );
          
          rotated_curr_y =
            ( ( -1.0 * rot_sin * curr_x_minus_radius ) + 
            ( rot_cos * curr_y_minus_radius ) );
            
          /* bringing the window back to its original
             location with the correct new scale */ 
            
          rotated_curr_x += wind_radius_double;
          rotated_curr_y += wind_radius_double;
          
          /* copy the new rotated window to the output vector */
            
          rotated_curr_x_img = curr_window_x_start +
            TeRound( rotated_curr_x );
          rotated_curr_y_img = curr_window_y_start +
            TeRound( rotated_curr_y );                        
           
          if( ( rotated_curr_x_img > 0 ) &&  
            ( rotated_curr_x_img < (int)img_matrix.getNCols() ) &&
            ( rotated_curr_y_img > 0 ) &&
            ( rotated_curr_y_img < (int)img_matrix.getNLines() ) )
          {
            img_matrix.getValue( rotated_curr_y_img, rotated_curr_x_img, 
              imgMatrixValue1 ); 
          }
          else
          {
            TEAGN_LOGERR( "Invalid matrix position -> rotated_curr_y_img=" +
              Te2String( rotated_curr_y_img ) + " img_matrix.getNLines()=" +
              Te2String( img_matrix.getNLines() ) + " Processing point [" + Te2String( curr_window_x_center, 1 ) +
              "," + Te2String( curr_window_y_center, 1 ) + "] ");
          
            imgMatrixValue1 = 0;
          }
            
          img_features_matrix( maximasListIndex, ( curr_y * 
            params.corrWindowsWidth_ ) + curr_x ) = imgMatrixValue1;

        }
      }
      
      /* Normalizing the generated window by subtracting its mean
         and dividing by its standard deviation */      
      
      if( params.normalizeWindows_ )
      {
        curr_wind_mean = 0.0;
        
        for( curr_x = 0 ; curr_x < img_features_matrix_cols ; 
          ++curr_x ) {
          
          curr_wind_mean += img_features_matrix( maximasListIndex,
            curr_x );
        }
        
        curr_wind_mean /= ( (double)img_features_matrix_cols  );
        
        curr_wind_stddev = 0.0;  
        
        for( curr_x = 0 ; curr_x < img_features_matrix_cols ; 
          ++curr_x ) {
          
          curr_wind_stddev_aux = img_features_matrix( 
            maximasListIndex, curr_x ) - curr_wind_mean;
            
          curr_wind_stddev += ( curr_wind_stddev_aux *
            curr_wind_stddev_aux );
        }      
        
        curr_wind_stddev /= ( (double)img_features_matrix_cols  );
        curr_wind_stddev = sqrt( curr_wind_stddev );
        
        if( curr_wind_stddev == 0.0 ) {
          for( curr_x = 0 ; curr_x < img_features_matrix_cols ; 
            ++curr_x ) {
            
            double& curr_value = img_features_matrix( 
              maximasListIndex, curr_x );
            
            curr_value -= curr_wind_mean;
          } 
        } else {
          for( curr_x = 0 ; curr_x < img_features_matrix_cols ; 
            ++curr_x ) {
            
            double& curr_value = img_features_matrix( 
              maximasListIndex, curr_x );
            
            curr_value -= curr_wind_mean;
            curr_value /= curr_wind_stddev;
          }
        }
      }
      
      /* Finished !! */
     
      if( progress.Increment() )
      {
        TEAGN_LOGERR( "Canceled by the user" );
        return;
      }
      
      ++maximasListIndex;
      ++maximIt;
    }
  }    
  
  params.returnValue_ = true;
}

bool TePDIMMIOMatching::matchFeatures( FeatMatchingMethod matching_method,
  const TePDIMatrix< double >& img1featMtx,
  const MaximasListT& img1_maximas_list,
  const TePDIMatrix< double >& img2featMtx,
  const MaximasListT& img2_maximas_list,
  TeCoordPairVect& tiePointsVec )
{
  if( ( img1featMtx.GetLines() == 0 ) || ( img2featMtx.GetLines() == 0 ) )
  {
    return true;
  }
  
  TEAGN_DEBUG_CONDITION( ( img1featMtx.GetLines() == 
    img1_maximas_list.size() ), "Size mismatch" )
  TEAGN_DEBUG_CONDITION( ( img2featMtx.GetLines() == 
    img2_maximas_list.size() ), "Size mismatch" )    
  TEAGN_DEBUG_CONDITION( ( img1featMtx.GetColumns() == 
    img2featMtx.GetColumns() ), "Size mismatch" )
    
  tiePointsVec.clear();
  
  // Init matching matrix
  
  TePDIMatrix<double> matchMatrix;
  TEAGN_TRUE_OR_RETURN( matchMatrix.Reset( img1_maximas_list.size(),
    img2_maximas_list.size(), TePDIMatrix<double>::RAMMemPol ),
    "Error allocating memory" );  
  
  const unsigned int mMCols = matchMatrix.GetColumns();
  const unsigned int mMLines = matchMatrix.GetLines();  
  
  {
    const double negativeInfinite = DBL_MAX * (-1.0);
    double* linePtr = 0;
    unsigned int col = 0;
    
    for( unsigned int line =0 ; line < mMLines ; ++line )
    {
      linePtr = matchMatrix[ line ];
      
      for( col =0 ; col < mMCols ; ++col )
        linePtr[ col ] = negativeInfinite;
    }
  }
  
  const unsigned long int maxJobs = jobsMan_.getMaxSimulJobs() ?
      jobsMan_.getMaxSimulJobs() : 1;
  
  switch( matching_method )
  {
    case EuclidianDistMethod :
    {
      // Spawning jobs
      
      std::vector< CalcEuclidianDistanceMtxParams > jobsParamsVec;  
      
      CalcEuclidianDistanceMtxParams auxPars;      
      auxPars.img1FeatMtxPtr_ = &img1featMtx;
      auxPars.img2FeatMtxPtr_ = &img2featMtx;
      auxPars.matchMatrixPtr_ = &matchMatrix;
      auxPars.progressEnabled_ = progress_enabled_;
      auxPars.glbMutexptr_ = &globalMutex_;
      auxPars.returnValue_ = false;
      
      TeFunctionCallThreadJob dummyJob;
      dummyJob.jobFunc_ = calcEuclidianDistanceMtx;
      
      unsigned int jobidx = 0;
      jobsParamsVec.resize( maxJobs );
      
      for( jobidx = 0 ; jobidx < maxJobs ; ++jobidx )
      {
        jobsParamsVec[ jobidx ] = auxPars;
        dummyJob.jobFuncParsPtr_ = &( jobsParamsVec[ jobidx ] );
        
        jobsMan_.executeJob( dummyJob );
      }
      
      // Waiting jobs to finish
      
      jobsMan_.waitAllToFinish();
      
      for( jobidx = 0 ; jobidx < maxJobs ; ++jobidx )
      {
        TEAGN_TRUE_OR_RETURN( jobsParamsVec[ jobidx ].returnValue_, 
          "Error calculating euclidian distance" );
      }      
      
      break;
    }    
    case NormCrossCorrMethod :
    {
      // Spawning jobs
      
      std::vector< CalcCCorrelationMtxParams > jobsParamsVec;  
      
      CalcCCorrelationMtxParams auxPars;      
      auxPars.img1FeatMtxPtr_ = &img1featMtx;
      auxPars.img2FeatMtxPtr_ = &img2featMtx;
      auxPars.matchMatrixPtr_ = &matchMatrix;
      auxPars.progressEnabled_ = progress_enabled_;
      auxPars.glbMutexptr_ = &globalMutex_;
      auxPars.returnValue_ = false;
      
      TeFunctionCallThreadJob dummyJob;
      dummyJob.jobFunc_ = calcCCorrelationMtx;

      unsigned int jobidx = 0;
      jobsParamsVec.resize( maxJobs );
      
      for( jobidx = 0 ; jobidx < maxJobs ; ++jobidx )
      {
        jobsParamsVec[ jobidx ] = auxPars;
        dummyJob.jobFuncParsPtr_ = &( jobsParamsVec[ jobidx ] );
        
        jobsMan_.executeJob( dummyJob );
      }
      
      // Waiting jobs to finish
      
      jobsMan_.waitAllToFinish();
      
      for( jobidx = 0 ; jobidx < maxJobs ; ++jobidx )
      {
        TEAGN_TRUE_OR_RETURN( jobsParamsVec[ jobidx ].returnValue_, 
          "Error calculating euclidian distance" );
      }      
      
      break;
    }
    default :
    {
      TEAGN_LOG_AND_THROW( "Invalid method")
      break;
    }
  }
  
  // Finding best matchings
  
  // Feature 1 desires to match with the feature 2 with the index
  // indicated by this vector
  std::vector< unsigned int> feat1Tofeat2( matchMatrix.GetLines(),
    matchMatrix.GetColumns() );  
  
  // Feature 2 desires to match with the feature 1 with the index
  // indicated by this vector
  std::vector< unsigned int> feat2Tofeat1( matchMatrix.GetColumns(),
    matchMatrix.GetLines() );     
  
  switch( matching_method )
  {
    case EuclidianDistMethod :
    {  
      // minimum values found on each column
      std::vector< double > colsMinsVec( matchMatrix.GetColumns(), 
        DBL_MAX );
      
      // minimum values found on each line
      std::vector< double > linesMinsVec( matchMatrix.GetLines(), 
        DBL_MAX );
      
      double const* linePtr = 0;
      unsigned int mMLine =0;
      unsigned int mMCol =0;
      
      TePDIPIManager progress( "Matching features",
        mMLines, progress_enabled_ );       
      
      for( mMLine =0 ; mMLine < mMLines ; ++mMLine )
      {
        linePtr = matchMatrix[ mMLine ];
        
        for( mMCol =0 ; mMCol < mMCols ; ++mMCol )
        {
          const double& value = linePtr[ mMCol ];
          
          if( value < linesMinsVec[ mMLine ] )
          {
            linesMinsVec[ mMLine ] = value;
            feat1Tofeat2[ mMLine ] = mMCol;
          }
          
          if( value < colsMinsVec[ mMCol ] )
          {
            colsMinsVec[ mMCol ] = value;
            feat2Tofeat1[ mMCol ] = mMLine;
          }          
        }
        
        if( progress.Increment() )
        {
          TEAGN_LOGERR( "Canceled by the user" );
          return false;
        }           
      }
      
      break;
    }
    case NormCrossCorrMethod :
    {
     // maximum values found on each column
      std::vector< double > colsMaxsVec( matchMatrix.GetColumns(), 
        (-1.0) * DBL_MAX );
      
      // maximum values found on each line
      std::vector< double > linesMaxsVec( matchMatrix.GetLines(), 
        (-1.0) * DBL_MAX );
      
      double const* linePtr = 0;
      unsigned int mMLine =0;
      unsigned int mMCol =0;
      
      TePDIPIManager progress( "Matching features",
        mMLines, progress_enabled_ );       
      
      for( mMLine =0 ; mMLine < mMLines ; ++mMLine )
      {
        linePtr = matchMatrix[ mMLine ];
        
        for( mMCol =0 ; mMCol < mMCols ; ++mMCol )
        {
          const double& value = linePtr[ mMCol ];
          
          if( value > linesMaxsVec[ mMLine ] )
          {
            linesMaxsVec[ mMLine ] = value;
            feat1Tofeat2[ mMLine ] = mMCol;
          }
          
          if( value > colsMaxsVec[ mMCol ] )
          {
            colsMaxsVec[ mMCol ] = value;
            feat2Tofeat1[ mMCol ] = mMLine;
          }          
        }
        
        if( progress.Increment() )
        {
          TEAGN_LOGERR( "Canceled by the user" );
          return false;
        }           
      }
      
      break;
    }
    default :
    {
      TEAGN_LOG_AND_THROW( "Invalid method")
      break;
    }
  }
  
  // Generating tie-points
  
  {
    MaximasListT::const_iterator mList1It = img1_maximas_list.begin();
    MaximasListT::const_iterator mList2It;
    TeCoordPair dummyCPair;
    
    for( unsigned int feat1Tofeat2_idx = 0 ; feat1Tofeat2_idx < 
      feat1Tofeat2.size() ; ++feat1Tofeat2_idx )
    {
      // feature1 wants to match with the feature 2 with index
      // feat1Tofeat2[ feat1Tofeat2_idx ]
      if( feat1Tofeat2[ feat1Tofeat2_idx ] < feat2Tofeat1.size() )
      {
        // does feature 2 wants to match back ??
        if( feat2Tofeat1[ feat1Tofeat2[ feat1Tofeat2_idx ] ] ==
          feat1Tofeat2_idx )
        { // Great !!! we hava a match
        
          // move an iterator to the correct maximas list 2 position
          
          mList2It = img2_maximas_list.begin();
          for( unsigned int maxList2dx = 0 ; maxList2dx < 
            feat1Tofeat2[ feat1Tofeat2_idx ] ; ++maxList2dx )
          {
            ++mList2It;
          }
          
          // Generating the new tie-point
          
          dummyCPair.pt1.x( mList1It->first );
          dummyCPair.pt1.y( mList1It->second );
          dummyCPair.pt2.x( mList2It->first );
          dummyCPair.pt2.y( mList2It->second );
          
          tiePointsVec.push_back( dummyCPair );
        }
      }
      
      ++mList1It;
    }
  }
  
  return true;
}

void TePDIMMIOMatching::calcEuclidianDistanceMtx( void * paramsPtr )
{
  CalcEuclidianDistanceMtxParams& params = 
    *((CalcEuclidianDistanceMtxParams*)paramsPtr );
  
  params.returnValue_ = false;
  
  TEAGN_DEBUG_CONDITION( params.matchMatrixPtr_->GetLines() ==
    params.img1FeatMtxPtr_->GetLines(), "Size mismatch" )
  TEAGN_DEBUG_CONDITION( params.matchMatrixPtr_->GetColumns() ==
    params.img2FeatMtxPtr_->GetLines(), "Size mismatch" )
  TEAGN_DEBUG_CONDITION( params.img1FeatMtxPtr_->GetColumns() ==
    params.img2FeatMtxPtr_->GetColumns(), "Size mismatch" )    
    
  TePDIMatrix< double >& matchMatrix = *params.matchMatrixPtr_;
  const TePDIMatrix< double >& img1FeatMtx = *(params.img1FeatMtxPtr_);
  const TePDIMatrix< double >& img2FeatMtx = *(params.img2FeatMtxPtr_);
  const unsigned int matchMatrixLines = matchMatrix.GetLines();
  const unsigned int matchMatrixCols = matchMatrix.GetColumns();
  unsigned int notProcLine = 0;
  unsigned int notProcCol = 0;
  const double negativeInfinite = DBL_MAX * (-1.0);
  double const* feat1Ptr = 0;
  double const* feat2Ptr = 0;
  double eucDist = 0;
  double diff = 0;
  unsigned int featCol = 0;
  const unsigned int featSize = img1FeatMtx.GetColumns();
  double* mMLinePtr = 0;
  
  // Loocking for a not processed cell
  
  TePDIPIManager progress( "Calculating Euclidian distance matrix",
    matchMatrixLines, params.progressEnabled_ );  
  
  for( notProcLine = 0 ; notProcLine < matchMatrixLines ; ++notProcLine )
  {
    mMLinePtr = matchMatrix[ notProcLine ];
    
    for( notProcCol = 0 ; notProcCol < matchMatrixCols ; ++notProcCol )
    {
      params.glbMutexptr_->lock();
      
      double& mMElem = mMLinePtr[ notProcCol ];
      
      if( mMElem == negativeInfinite )
      {
        // mark as under processing 
        mMElem = 0;
        
        params.glbMutexptr_->unLock();
        
        // Calculating distance
        
        feat1Ptr = img1FeatMtx[ notProcLine ];
        feat2Ptr = img2FeatMtx[ notProcCol ];
        
        eucDist = 0;
        
        for( featCol = 0 ; featCol < featSize ; ++featCol )
        {
          diff = feat1Ptr[ featCol ] - feat2Ptr[ featCol ];
          
          diff *= diff;
          
          eucDist += diff;
        }
        
        mMElem = sqrt( eucDist );
      }
      else
      {
        params.glbMutexptr_->unLock();
      }
    }
    
    if( progress.Increment() )
    {
      TEAGN_LOGERR( "Canceled by the user" );
      return;
    }    
  }
  
  params.returnValue_ = true;
}

void TePDIMMIOMatching::calcCCorrelationMtx( void * paramsPtr )
{
  CalcCCorrelationMtxParams& params = 
    *((CalcCCorrelationMtxParams*)paramsPtr );
  
  params.returnValue_ = false;
  
  TEAGN_DEBUG_CONDITION( params.matchMatrixPtr_->GetLines() ==
    params.img1FeatMtxPtr_->GetLines(), "Size mismatch" )
  TEAGN_DEBUG_CONDITION( params.matchMatrixPtr_->GetColumns() ==
    params.img2FeatMtxPtr_->GetLines(), "Size mismatch" )
  TEAGN_DEBUG_CONDITION( params.img1FeatMtxPtr_->GetColumns() ==
    params.img2FeatMtxPtr_->GetColumns(), "Size mismatch" )    
    
  TePDIMatrix< double >& matchMatrix = *params.matchMatrixPtr_;
  const TePDIMatrix< double >& img1FeatMtx = *(params.img1FeatMtxPtr_);
  const TePDIMatrix< double >& img2FeatMtx = *(params.img2FeatMtxPtr_);
  const unsigned int matchMatrixLines = matchMatrix.GetLines();
  const unsigned int matchMatrixCols = matchMatrix.GetColumns();
  unsigned int notProcLine = 0;
  unsigned int notProcCol = 0;
  const double negativeInfinite = DBL_MAX * (-1.0);
  double const* feat1Ptr = 0;
  double const* feat2Ptr = 0;
  double sumAA = 0;
  double sumBB = 0;
  double ccorrelation = 0;
  double cc_norm = 0;
  unsigned int featCol = 0;
  const unsigned int featSize = img1FeatMtx.GetColumns();
  double* mMLinePtr = 0;
  
  // Loocking for a not processed cell
  
  TePDIPIManager progress( "Calculating correlation matrix",
    matchMatrixLines, params.progressEnabled_ );  
  
  for( notProcLine = 0 ; notProcLine < matchMatrixLines ; ++notProcLine )
  {
    mMLinePtr = matchMatrix[ notProcLine ];
    
    for( notProcCol = 0 ; notProcCol < matchMatrixCols ; ++notProcCol )
    {
      params.glbMutexptr_->lock();
      
      double& mMElem = mMLinePtr[ notProcCol ];
      
      if( mMElem == negativeInfinite )
      {
        // mark as under processing 
        mMElem = 0;
        
        params.glbMutexptr_->unLock();
        
        feat1Ptr = img1FeatMtx[ notProcLine ];
        feat2Ptr = img2FeatMtx[ notProcCol ];        
        
        sumAA = 0;
        sumBB = 0;   
        
        for( featCol = 0 ; featCol < featSize ; ++featCol )
        {
          sumAA += feat1Ptr[ featCol ] * feat1Ptr[ featCol ];
          sumBB += feat2Ptr[ featCol ] * feat2Ptr[ featCol ];
        }
        
        cc_norm = sqrt( sumAA * sumBB );
        
        if( cc_norm == 0.0 )
        {
          mMElem = 0;
        }
        else
        {
          ccorrelation = 0;
          
          for( featCol = 0 ; featCol < featSize ; ++featCol )
          {
            ccorrelation += ( feat1Ptr[ featCol ] * feat2Ptr[ featCol ] ) / cc_norm;
          }
          
          mMElem = ABS( ccorrelation );
        }
      }
      else
      {
        params.glbMutexptr_->unLock();
      }
    }
    
    if( progress.Increment() )
    {
      TEAGN_LOGERR( "Canceled by the user" );
      return;
    }    
  }
  
  params.returnValue_ = true;
}

void TePDIMMIOMatching::matrix2Tiff( 
  const TePDIMtxDoubleAdptInt& input_matrix,
  const std::string& out_file_name,
  const MaximasListT& maxima_points )
{   
  TEAGN_TRUE_OR_THROW( ( ! out_file_name.empty() ), "Invalid file name" )
  TEAGN_TRUE_OR_THROW( ( input_matrix.getNLines() > 0 ), 
    "Invalid matrix lines" )
  TEAGN_TRUE_OR_THROW( ( input_matrix.getNCols() > 0 ), 
    "Invalid matrix cols" )
    
  double value = 0;
    
  TeRasterParams params;
  params.setNLinesNColumns( input_matrix.getNLines(),
    input_matrix.getNCols() );
  params.nBands( 1 );
  params.setDataType( TeUNSIGNEDCHAR, -1 );
  params.nBands( 1 );
  params.decoderIdentifier_ = "TIF";
  params.mode_ = 'c';
  params.fileName_ = out_file_name;
  
  TeRaster out_raster( params );
  TEAGN_TRUE_OR_THROW( out_raster.init(), "Error init raster" );
  
  for( unsigned int line = 0 ; 
    line < input_matrix.getNLines() ; ++line ) {
    for( unsigned int col = 0 ; 
      col < input_matrix.getNCols() ; 
      ++col ) 
    {
      input_matrix.getValue( line, col, value );
      TEAGN_TRUE_OR_THROW( out_raster.setElement( col, line, 
        MIN( value, 254.0 ), 0 ),
        "Error writing raster" )
    }  
  }
  
  /* Draw maxima points */
  
  MaximasListT::const_iterator maximas_it = 
    maxima_points.begin();
  MaximasListT::const_iterator maximas_it_end = 
    maxima_points.end();
    
  while( maximas_it != maximas_it_end ) {
    const unsigned int& x = maximas_it->first;
    
    TEAGN_TRUE_OR_THROW( ( ((int)x) < (int)input_matrix.getNCols() ),
      "Invalid maxima column" )
    const unsigned int& y = maximas_it->second;
    TEAGN_TRUE_OR_THROW( ( ((int)y) < (int)input_matrix.getNLines() ),
      "Invalid maxima line" )
    
    TEAGN_TRUE_OR_THROW( out_raster.setElement( x, y, 
      255.0, 0 ),
      "Error writing raster" )
      
    ++maximas_it;
  }  
}

void TePDIMMIOMatching::matrix2Tiff( 
  const TePDIMtxDoubleAdptInt& input_matrix,
  const std::string& out_file_name,
  const TeCoordPairVect& tiepoints,
  bool usePt1 )
{   
  TEAGN_TRUE_OR_THROW( ( ! out_file_name.empty() ), "Invalid file name" )
  TEAGN_TRUE_OR_THROW( ( input_matrix.getNLines() > 0 ), 
    "Invalid matrix lines" )
  TEAGN_TRUE_OR_THROW( ( input_matrix.getNCols() > 0 ), 
    "Invalid matrix cols" )
    
  double value = 0;
    
  TeRasterParams params;
  params.setNLinesNColumns( input_matrix.getNLines(),
    input_matrix.getNCols() );
  params.nBands( 1 );
  params.setDataType( TeUNSIGNEDCHAR, -1 );
  params.nBands( 1 );
  params.decoderIdentifier_ = "TIF";
  params.mode_ = 'c';
  params.fileName_ = out_file_name;
  
  TeRaster out_raster( params );
  TEAGN_TRUE_OR_THROW( out_raster.init(), "Error init raster" );
  
  for( unsigned int line = 0 ; 
    line < input_matrix.getNLines() ; ++line ) {
    for( unsigned int col = 0 ; 
      col < input_matrix.getNCols() ; 
      ++col ) 
    {
      input_matrix.getValue( line, col, value );
      TEAGN_TRUE_OR_THROW( out_raster.setElement( col, line, 
        MIN( value, 254.0 ), 0 ),
        "Error writing raster" )
    }  
  }
  
  /* Draw maxima points */
  
  TeCoordPairVect::const_iterator it = 
    tiepoints.begin();
  TeCoordPairVect::const_iterator it_end = 
    tiepoints.end();
  
  int tpx = 0;
  int tpy = 0;
    
  while( it != it_end ) {
    if( usePt1 )
    {
      tpx = (int)it->pt1.x();
      tpy = (int)it->pt1.y();
    }
    else
    {
      tpx = (int)it->pt2.x();
      tpy = (int)it->pt2.y();
    }
    
    TEAGN_TRUE_OR_THROW( ( tpx < (int)input_matrix.getNCols() ),
      "Invalid maxima column" )
    TEAGN_TRUE_OR_THROW( ( tpx >= 0 ),
      "Invalid maxima column" )
    TEAGN_TRUE_OR_THROW( ( tpy < (int)input_matrix.getNLines() ),
      "Invalid maxima line" )
    TEAGN_TRUE_OR_THROW( ( tpy >= 0 ),
      "Invalid maxima line" )           
    
    TEAGN_TRUE_OR_THROW( out_raster.setElement( tpx, tpy, 
      255.0, 0 ),
      "Error writing raster" )
      
    ++it;
  }  
}

void TePDIMMIOMatching::features2Tiff( 
  unsigned int corr_window_width,
  const MaximasListT& img_maxima_points,
  TePDIMatrix< double >& img_features_matrix,
  const std::string& filenameaddon )
{
  TEAGN_TRUE_OR_THROW( ( img_features_matrix.GetLines() ==
    img_maxima_points.size() ), 
    "Invalid matrix lines" )
  TEAGN_TRUE_OR_THROW( ( img_features_matrix.GetColumns() ==
    ( corr_window_width * corr_window_width ) ), 
    "Invalid matrix columns" )
  TEAGN_TRUE_OR_THROW( ( corr_window_width > 0 ),
    "Invalid corr_window_width" )   
    
  MaximasListT::const_iterator maxIt = img_maxima_points.begin();

  for( unsigned int curr_wind_index = 0 ; 
    curr_wind_index < img_features_matrix.GetLines() ;
    ++curr_wind_index ) {
    
    // finding feature min and max
    
    double featMin = DBL_MAX;
    double featMax = -1.0 * featMin;
    
    for( unsigned int fcol = 0 ; fcol < img_features_matrix.GetColumns();
      ++fcol )
    {
      if( img_features_matrix( curr_wind_index, fcol ) < featMin )
      {
        featMin = img_features_matrix( curr_wind_index, fcol );
      }
      if( img_features_matrix( curr_wind_index, fcol ) > featMax )
      {
        featMax = img_features_matrix( curr_wind_index, fcol );
      }
    }
    
    double featRange = ( featMin != featMax ) ? ( featMax - featMin ) : 1.0;
    double featMultFac = 255.0 / featRange;
    
    // saving output
    
    TeRasterParams params;
    params.setNLinesNColumns( corr_window_width,
      corr_window_width );
    params.nBands( 1 );
    params.setDataType( TeUNSIGNEDCHAR, -1 );
    params.nBands( 1 );
    params.decoderIdentifier_ = "TIF";
    params.mode_ = 'c';
    params.fileName_ = filenameaddon + "_" + Te2String( maxIt->first, 0 ) + 
      "_" + Te2String( maxIt->second, 0 ) + ".tif";
    
    TeRaster out_raster( params );
    TEAGN_TRUE_OR_THROW( out_raster.init(), "Error init raster" );
    
    for( unsigned int line = 0 ; line < corr_window_width ; ++line ) {
      for( unsigned int col = 0 ; col < corr_window_width ; ++col ) {

        TEAGN_TRUE_OR_THROW( out_raster.setElement( col, line, 
          ( img_features_matrix( curr_wind_index, ( line * 
          corr_window_width ) + col ) - featMin ) * featMultFac, 0 ), 
          "Error writing raster" )
      }  
    }
    
    ++maxIt;
  }
}

bool TePDIMMIOMatching::bicubicResampleMatrix( const TePDIMtxDoubleAdptInt& inputMatrix,
  unsigned int outLines, unsigned int outCols, bool progressEnabled,
  TeMutex& globalMutex, double outMinValue, double outMaxValue,
  TePDIMtxDoubleAdptInt& outputMatrix )
{
  globalMutex.lock();
  
  if( ! outputMatrix.reset( outLines, outCols ) )
  {
    globalMutex.unLock();
    
    TEAGN_LOGERR( "Output matrix reset error" );
  }
  
  globalMutex.unLock();
  
  const unsigned int inLines = inputMatrix.getNLines();
  const unsigned int inCols = inputMatrix.getNCols();
  const int lastInLineIdx = inLines - 1;
  const int lastInColIdx = inCols - 1;
  const double yResFact = ((double)outLines) / ((double)inLines);
  const double xResFact = ((double)outCols) / ((double)inCols);
  const double bicubic_columns_bound = ((double)inCols) - 2;
  const double bicubic_lines_bound = ((double)inLines) - 2;
  const double bicubic_kernel_parameter = -1.0;
  
  double inputLine = 0;
  double inputCol = 0;
  unsigned int outLine = 0;
  unsigned int outCol = 0;
  double value = 0;
  unsigned int  bicubic_grid_input_line = 0;
  unsigned int  bicubic_grid_input_col = 0;
  unsigned int  bicubic_buffer_line = 0;
  unsigned int  bicubic_buffer_col = 0;  
  double bicubic_offset_x;
  double bicubic_offset_y;
  double bicubic_h_weights[4];
  double bicubic_v_weights[4];  
  double bicubic_h_weights_sum = 0;
  double bicubic_v_weights_sum = 0;
  double bicubic_int_line_accum = 0;
  double bicubic_int_lines_values[4];
  int correctedInputLine = 0;
  int correctedInputCol = 0;
  
  TePDIPIManager progress( "Resampling", outLines, 
     progressEnabled );  
  
  for( outLine = 0 ; outLine < outLines ; ++outLine )
  {
    inputLine =  ((double)outLine) / yResFact;
    
    for( outCol = 0 ; outCol < outCols ; ++outCol )
    {
      inputCol =  ((double)outCol) / xResFact;
      
      if( ( inputCol < 1.0 ) || ( inputLine < 1.0 ) || ( inputCol >= 
        bicubic_columns_bound ) || ( inputLine >= bicubic_lines_bound ) ) 
      {
        /* Near neighborhood interpolation will be used */
        
        correctedInputLine = (int)TeRound( inputLine );
        correctedInputLine = MAX( 0, correctedInputLine );
        correctedInputLine = MIN( lastInLineIdx, correctedInputLine );
        
        correctedInputCol = (int)TeRound( inputCol );
        correctedInputCol = MAX( 0, correctedInputCol );
        correctedInputCol = MIN( lastInColIdx, correctedInputCol );        
        
        inputMatrix.getValue( correctedInputLine, correctedInputCol, value );
        outputMatrix.setValue( outLine , outCol, value );
      } else {
        bicubic_grid_input_line = ( (unsigned int)floor( inputLine ) ) - 1;
        bicubic_grid_input_col = ( (unsigned int)floor( inputCol ) ) - 1;
        
        /* Bicubic weights calcule for the required position */
        
        bicubic_offset_x = inputCol - (double)( bicubic_grid_input_col + 1 );
        bicubic_offset_y = inputLine - (double)( bicubic_grid_input_line + 1 );    
        
        bicubic_h_weights[0] = BICUBIC_KERNEL( 1.0 + bicubic_offset_x, 
          bicubic_kernel_parameter );
        bicubic_h_weights[1] = BICUBIC_KERNEL( bicubic_offset_x, 
          bicubic_kernel_parameter );
        bicubic_h_weights[2] = BICUBIC_KERNEL( 1.0 - bicubic_offset_x, 
          bicubic_kernel_parameter );
        bicubic_h_weights[3] = BICUBIC_KERNEL( 2.0 - bicubic_offset_x, 
          bicubic_kernel_parameter );
          
        bicubic_v_weights[0] = BICUBIC_KERNEL( 1.0 + bicubic_offset_y, 
          bicubic_kernel_parameter );
        bicubic_v_weights[1] = BICUBIC_KERNEL( bicubic_offset_y, 
          bicubic_kernel_parameter );
        bicubic_v_weights[2] = BICUBIC_KERNEL( 1.0 - bicubic_offset_y, 
          bicubic_kernel_parameter );
        bicubic_v_weights[3] = BICUBIC_KERNEL( 2.0 - bicubic_offset_y, 
          bicubic_kernel_parameter );
          
        bicubic_h_weights_sum = bicubic_h_weights[0] + bicubic_h_weights[1] +
          bicubic_h_weights[2] + bicubic_h_weights[3];
        bicubic_v_weights_sum = bicubic_v_weights[0] + bicubic_v_weights[1] +
          bicubic_v_weights[2] + bicubic_v_weights[3];
        
        /* interpolating the value */
        
        for( bicubic_buffer_line = 0 ; bicubic_buffer_line < 4 ; 
          ++bicubic_buffer_line) {
          
          bicubic_int_line_accum = 0;
          
          for( bicubic_buffer_col = 0 ; bicubic_buffer_col < 4 ; 
            ++bicubic_buffer_col ) 
          {
            inputMatrix.getValue( bicubic_grid_input_line + bicubic_buffer_line, 
                bicubic_grid_input_col + bicubic_buffer_col,
                value );            
            
            bicubic_int_line_accum += value * 
              bicubic_h_weights[ bicubic_buffer_col ];
          }
          
          bicubic_int_lines_values[ bicubic_buffer_line ] = 
            bicubic_int_line_accum / bicubic_h_weights_sum;
        }
        
        value = bicubic_int_lines_values[ 0 ] * bicubic_v_weights[ 0 ] +
          bicubic_int_lines_values[ 1 ] * bicubic_v_weights[ 1 ] +
          bicubic_int_lines_values[ 2 ] * bicubic_v_weights[ 2 ] +
          bicubic_int_lines_values[ 3 ] * bicubic_v_weights[ 3 ];
        value = value / bicubic_v_weights_sum;
        
        if( value > outMaxValue )
        {
          outputMatrix.setValue( outLine , outCol, outMaxValue );
        }
        else if( value < outMinValue )
        {
          outputMatrix.setValue( outLine , outCol, outMinValue );
        }
        else
        {  
          outputMatrix.setValue( outLine , outCol, value );
        }
      }
    }
    
    if( progress.Increment() )
    {
      TEAGN_LOGMSG( "Canceled by the user" );
      return false;
    }
  }
 
  return true;
}


