#include "TePDIGeoMosaic.hpp"
#include "TePDIBlending.hpp"
#include "TePDIUtils.hpp"
#include <TeAgnostic.h>

#include <TeVectorRemap.h>


TePDIGeoMosaic::TePDIGeoMosaic()
{
}


TePDIGeoMosaic::~TePDIGeoMosaic()
{
}


void TePDIGeoMosaic::ResetState( const TePDIParameters& )
{
}


bool TePDIGeoMosaic::CheckParameters( const TePDIParameters& parameters ) const
{
  /* Checking input_raster1 */
  
  TePDITypes::TePDIRasterPtrType input_raster1;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_raster1", 
    input_raster1 ),
    "Missing parameter: input_raster1" );
  TEAGN_TRUE_OR_RETURN( input_raster1.isActive(),
    "Invalid parameter: input_raster1 inactive" );
  TEAGN_TRUE_OR_RETURN( input_raster1->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: input_raster1 not ready" );    
    
  /* Checking input_raster2 */
  
  TePDITypes::TePDIRasterPtrType input_raster2;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_raster2", 
    input_raster2 ),
    "Missing parameter: input_raster2" );
  TEAGN_TRUE_OR_RETURN( input_raster2.isActive(),
    "Invalid parameter: input_raster2 inactive" );
  TEAGN_TRUE_OR_RETURN( input_raster2->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: input_raster2 not ready" );    
    
  /* channels1 parameter checking */

  std::vector< int > channels1;
  TEAGN_TRUE_OR_RETURN( 
    parameters.GetParameter( "channels1", channels1 ), 
    "Missing parameter: channels1" );
  for( unsigned int channels1_index = 0 ; 
    channels1_index < channels1.size() ; 
    ++channels1_index ) {
    
    TEAGN_TRUE_OR_RETURN( ( channels1[ channels1_index ] >= 0 ) &&
      ( channels1[ channels1_index ] < input_raster1->nBands() ),
      "Invalid parameter: channels1" );
  }
  
  /* channels2 parameter checking */

  std::vector< int > channels2;
  TEAGN_TRUE_OR_RETURN( 
    parameters.GetParameter( "channels2", channels2 ), 
    "Missing parameter: channels2" );
  TEAGN_TRUE_OR_RETURN( ( channels2.size() == channels1.size() ),
    "Size mismatch between channels1 and channels2" );
  for( unsigned int channels2_index = 0 ; 
    channels2_index < channels2.size() ; 
    ++channels2_index ) {
    
    TEAGN_TRUE_OR_RETURN( ( channels2[ channels2_index ] >= 0 ) &&
      ( channels2[ channels2_index ] < input_raster2->nBands() ),
      "Invalid parameter: channels2" );
  }     
    
  /* Checking output_raster */
  
  TePDITypes::TePDIRasterPtrType output_raster;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "output_raster", 
    output_raster ),
    "Missing parameter: output_raster" );
  TEAGN_TRUE_OR_RETURN( output_raster.isActive(),
    "Invalid parameter: output_raster inactive" );
  TEAGN_TRUE_OR_RETURN( output_raster->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: output_raster not ready" );

  /* Checking for blending type */
  
  std::string blending_type;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "blending_type", 
    blending_type ), "Missing parameter: blending_type" );
    
  /* Checking rasters projections */
  
  TEAGN_TRUE_OR_RETURN( ( input_raster1->params().projection() != 0 ),
    "Missing input_raster1 projection" );
  TEAGN_TRUE_OR_RETURN( 
    ( input_raster1->params().projection()->name() != "NoProjection" ),
    "Invalid input_raster1 projection" );  
  TEAGN_TRUE_OR_RETURN( ( input_raster2->params().projection() != 0 ),
    "Missing input_raster2 projection" );
  TEAGN_TRUE_OR_RETURN( 
    ( input_raster2->params().projection()->name() != "NoProjection" ),
    "Invalid input_raster2 projection" );      
    
  return true;
}


bool TePDIGeoMosaic::RunImplementation()
{
  TePDITypes::TePDIRasterPtrType input_raster1;
  params_.GetParameter( "input_raster1", input_raster1 );

  TePDITypes::TePDIRasterPtrType input_raster2;
  params_.GetParameter( "input_raster2", input_raster2 );
  
  std::vector< int > channels1;
  params_.GetParameter( "channels1", channels1 );

  std::vector< int > channels2;
  params_.GetParameter( "channels2", channels2 ); 

  TePDITypes::TePDIRasterPtrType output_raster;
  params_.GetParameter( "output_raster", output_raster );
  
  /* Input rasters swap using pixel resolution as reference */
  
  if( params_.CheckParameter< int >( "keep_best_res" ) ) {
    
    double raster1_x_res = 0;
    double raster1_y_res = 0;
    TePDIUtils::getGeodeticPixelRes( input_raster1->params(), 
      raster1_x_res, raster1_y_res );    
      
    double raster2_x_res = 0;
    double raster2_y_res = 0;
    TePDIUtils::getGeodeticPixelRes( input_raster2->params(), 
      raster2_x_res, raster2_y_res );      
    
    if( ( ( raster2_x_res <= raster1_x_res ) &&
        ( raster2_y_res < raster1_y_res ) ) ||
      ( ( raster2_x_res < raster1_x_res ) &&
        ( raster2_y_res <= raster1_y_res ) ) ) 
    {
    
      TePDITypes::TePDIRasterPtrType temp_raster_ptr = input_raster1;
      input_raster1 = input_raster2;
      input_raster2 = temp_raster_ptr;
    }
  }
  
  /* Dumyy value definition */
 
  const bool output_raster_uses_dummy = output_raster->params().useDummy_;
  double output_raster_dummy = 0;
  if( output_raster_uses_dummy ) {
    output_raster_dummy = output_raster->params().dummy_[ 0 ];
  }
  
  bool user_required_dummy = false;
  double user_dummy = 0;
  if( params_.CheckParameter< double >( "dummy_value" ) ) {
    
    params_.GetParameter( "dummy_value", user_dummy );
    user_required_dummy = true;
  }
  
  bool must_use_dummy = false;
  double output_dummy_value = 0;
  if( user_required_dummy ) {
    must_use_dummy = true;
    output_dummy_value = user_dummy;
  } else if( output_raster_uses_dummy ) {
    must_use_dummy = true;
    output_dummy_value = output_raster_dummy;
  }
  
  /* Bringing input_raster2 to the same projection and resolution,
     if needed */
     
  TePolygon input_raster2_pol_proj1;
  
  if( ( input_raster1->params().projection()->name() != 
        input_raster2->params().projection()->name() ) ||
      ( input_raster1->params().resx_ != 
        input_raster2->params().resx_ ) ||        
      ( input_raster1->params().resy_ != 
        input_raster2->params().resy_ ) ) {
        
    /* input_raster2 reprojection */
        
    TePDITypes::TePDIRasterPtrType new_input_raster2;
    TeRasterParams new_input_raster2_params = input_raster2->params();
    new_input_raster2_params.setNLinesNColumns( 10, 10 );
    TEAGN_TRUE_OR_RETURN( TePDIUtils::TeAllocRAMRaster( new_input_raster2, 
      new_input_raster2_params, TePDIUtils::TePDIUtilsAutoMemPol ), 
      "Unable to allocate new_input_raster2" );
      
    TEAGN_TRUE_OR_RETURN( TePDIUtils::reprojectRaster( input_raster2,
      *( input_raster1->projection() ), input_raster1->params().resx_,
      input_raster1->params().resy_, new_input_raster2 ),
      "Raster reprojection error" );
    
    /* updating input_raster2_pol_proj1 */

    TePolygon input_raster2_pol_proj2;
    
    TEAGN_TRUE_OR_RETURN( TePDIUtils::buildDetailedBBox( input_raster2,
      input_raster2_pol_proj2 ), 
      "Unable to build raster2 detailed bounding box" );
    
    TeVectorRemap( input_raster2_pol_proj2, input_raster2->projection(),
      input_raster2_pol_proj1, input_raster1->projection() );         
      
    /* swapping raster */
      
    input_raster2 = new_input_raster2;
  } else {
    TeBox input_raster2_box_proj2 = input_raster2->params().boundingBox();
    input_raster2_pol_proj1 = polygonFromBox( input_raster2_box_proj2 );
  }
  
  /* Finding the offset between the two boxes */
  
  double raster2_pol_offset_x = 0;
  double raster2_pol_offset_y = 0;
  {
    TeBox pol1_box_indexed;
    TeBox input_raster_box = input_raster1->params().box();
    TePDIUtils::MapCoords2RasterIndexes( 
      input_raster_box, input_raster1, pol1_box_indexed );
          
    double p1_min_x = MIN( pol1_box_indexed.x1(),
      pol1_box_indexed.x2() );
    double p1_min_y = MIN( pol1_box_indexed.y1(),
      pol1_box_indexed.y2() );      
      
    TeBox pol2_box_indexed;
    TePDIUtils::MapCoords2RasterIndexes( input_raster2_pol_proj1.box(),
      input_raster1, pol2_box_indexed );
          
    double p2_min_x = MIN( pol2_box_indexed.x1(),
      pol2_box_indexed.x2() );
    double p2_min_y = MIN( pol2_box_indexed.y1(),
      pol2_box_indexed.y2() );       

    raster2_pol_offset_x = p2_min_x - p1_min_x;
    raster2_pol_offset_y = p2_min_y - p1_min_y;
  }
    
  /* Creating blending parameters */
    
  TePDIParameters blend_params = params_;
  blend_params.SetParameter( "input_raster1", input_raster1 );
  blend_params.SetParameter( "input_raster2", input_raster2 );
  
  TeBox input_raster_box = input_raster1->params().boundingBox();
  TePDITypes::TePDIPolygonPtrType raster1_pol_ptr( new TePolygon );
  *raster1_pol_ptr = polygonFromBox( input_raster_box );
  blend_params.SetParameter( "raster1_pol_ptr",  raster1_pol_ptr );
    
  TePDITypes::TePDIPolygonPtrType raster2_pol_ptr( 
    &input_raster2_pol_proj1, true );
  blend_params.SetParameter( "raster2_pol_ptr", raster2_pol_ptr );
  
  blend_params.SetParameter( "raster2_pol_offset_x", raster2_pol_offset_x );
  blend_params.SetParameter( "raster2_pol_offset_y", raster2_pol_offset_y );
  
  TePDIBlending blend_instance;
  TEAGN_TRUE_OR_RETURN( blend_instance.Reset( blend_params ), 
    "Blending instance reset error" );
  
  return blend_instance.Apply();
}

