/**
 * @file    SBMLConvert.cpp
 * @brief   Performs conversion between SBML levels
 * @author  Ben Bornstein and Sarah Keating
 * 
 * <!--------------------------------------------------------------------------
 * This file is part of libSBML.  Please visit http://sbml.org for more
 * information about SBML, and the latest version of libSBML.
 *
 * Copyright (C) 2009-2011 jointly by the following organizations: 
 *     1. California Institute of Technology, Pasadena, CA, USA
 *     2. EMBL European Bioinformatics Institute (EBML-EBI), Hinxton, UK
 *  
 * Copyright (C) 2006-2008 by the California Institute of Technology,
 *     Pasadena, CA, USA 
 *  
 * Copyright (C) 2002-2005 jointly by the following organizations: 
 *     1. California Institute of Technology, Pasadena, CA, USA
 *     2. Japan Science and Technology Agency, Japan
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation.  A copy of the license agreement is provided
 * in the file named "LICENSE.txt" included with this software distribution
 * and also available online as http://sbml.org/software/libsbml/license.html
 * ---------------------------------------------------------------------- -->*/

#include <sbml/util/List.h>
#include <sbml/math/ASTNode.h>

#include <sbml/Model.h>
#include <sbml/KineticLaw.h>
#include <sbml/Compartment.h>
#include <sbml/SpeciesReference.h>

#include <sbml/validator/constraints/IdList.h>

LIBSBML_CPP_NAMESPACE_BEGIN
/** @cond doxygen-libsbml-internal **/

static const char* ASSIGNED_COMPARTMENT = "AssignedName";


/*
 * Converts the model to a from SBML L1 to L2.  Most of the necessary
 * changes occur during the various writeAttributes() methods, however
 * there are some difference between L1 and L2 that require the underlying
 * Model to be changed.
 */
void 
Model::convertL1ToL2 ()
{
  addModifiers();

  addConstantAttribute();
}

/* convert from L1 to L3 */
void 
Model::convertL1ToL3 ()
{
  addModifiers();

  addConstantAttribute();

  setSpatialDimensions();

  addDefinitionsForDefaultUnits();

  assignRequiredValues();
}


/* convert from L2 to L3 */
void 
Model::convertL2ToL3 ()
{
  addDefinitionsForDefaultUnits();

  setSpeciesReferenceConstantValueAndStoichiometry();

  convertStoichiometryMath();

  assignRequiredValues();
}


/*
 * Converts the model to a from SBML L2 to L1.  Most of the necessary
 * changes occur during the various writeAttributes() methods, however
 * there are some difference between L1 and L2 that require the underlying
 * Model to be changed.
 */
void 
Model::convertL2ToL1 (bool strict)
{
  //
  // Level 2 allows a model to be specified without a Compartment.  However
  // this is not valid in Level 1.  Thus if a L2 model has no Compartment
  // one must be included 
  //
  if (getNumCompartments() == 0)
  {
    createCompartment()->setId(ASSIGNED_COMPARTMENT);

  }

  /* make sure underlying model is correct */
  if (strict)
  {
    removeMetaId();
    removeSBOTerms(strict);
    removeHasOnlySubstanceUnits();
  }
}

/* convert from L1 to L3 */
void 
Model::convertL3ToL1 ()
{
  dealWithModelUnits();
}


/* convert from L1 to L3 */
void 
Model::convertL3ToL2 (bool strict)
{
  dealWithModelUnits();

  dealWithStoichiometry();

  dealWithEvents(strict);
}


void
Model::setSpeciesReferenceConstantValueAndStoichiometry()
{
  for (unsigned int i = 0; i < getNumReactions(); i++)
  {
    Reaction *r = getReaction(i);
    unsigned int j;
    for (j = 0; j < r->getNumReactants(); j++)
    {
      if (!(r->getReactant(j)->isSetStoichiometryMath()))
      {
        r->getReactant(j)->setConstant(true);
        if (!(r->getReactant(j)->isSetStoichiometry()))
        {
          r->getReactant(j)->setStoichiometry(1);
        }
      }
      else
      {
        r->getReactant(j)->setConstant(false);
      }
    }
    for (j = 0; j < r->getNumProducts(); j++)
    {
      if (!(r->getProduct(j)->isSetStoichiometryMath()))
      {
        r->getProduct(j)->setConstant(true);
        if (!(r->getProduct(j)->isSetStoichiometry()))
        {
          r->getProduct(j)->setStoichiometry(1);
        }
      }
      else
      {
        r->getProduct(j)->setConstant(false);
      }
    }
  }
}
/* adds species referred to in a KineticLaw to the ListOfModifiers
 * this will only be applicable when up converting an L1 model
 */
void 
Model::addModifiers ()
{
  //
  // Level 2/3 has a listOfModifiers associated with a Reaction
  // which are not listed in a L1 Model.
  // For each symbol in the Reaction's KineticLaw,
  // that symbol is a modifier iff:
  //
  //   1. It is defined as a Species in the Model
  //   2. It is not a Reactant or Product in this Reaction.
  //
  // Thus modifiers must be added where appropriate.
  //
  const char *id;

  unsigned int size;
  unsigned int n, l;

  const ASTNode *node;
  List          *names;
  KineticLaw* kl;

  for (n = 0; n < getNumReactions(); n++)
  {
    kl = getReaction(n)->getKineticLaw();

    if (kl == NULL || kl->isSetMath() == false) continue;
   
    node  = kl->getMath();
    names = node->getListOfNodes((ASTNodePredicate) ASTNode_isName);
    size  = names->getSize();

    for (l = 0; l < size; l++)
    {
      node = (ASTNode *) names->get(l);
      id   = node->getName();

      // 1. It is an AST_NAME (not AST_NAME_TIME), and
      if (node->getType() != AST_NAME) continue;

      // 2. It refers to a Species in this Model, and
      if (id == NULL || getSpecies(id) == NULL) continue;

      // 3. It is not a Reactant, Product, or (already) a Modifier
      if (getReaction(n)->getReactant(id) != NULL) continue;
      if (getReaction(n)->getProduct (id) != NULL) continue;
      if (getReaction(n)->getModifier(id) != NULL) continue;

      getReaction(n)->createModifier()->setSpecies(id);
    }

    delete names;
  }
}


/* declares constant = false for any L1 compartment/parameter
 * assigned by a rule
 */
void
Model::addConstantAttribute()
{
  unsigned int n;
  // parameters and compartments are declared to have constant=true
  // by default. Since in L1 the constant attribute didnt exist 
  // parameters/compartments that are the subjcet of rules must have
  // the value changed

  for ( n = 0; n < getNumParameters(); n++)
  {
    if (getRule(getParameter(n)->getId()) != NULL)
    {
      getParameter(n)->setConstant(false);
    }
  }

  for ( n = 0; n < getNumCompartments(); n++)
  {
    if (getRule(getCompartment(n)->getId()) != NULL)
    {
      getCompartment(n)->setConstant(false);
    }
  }
}


/* in L1 spatialDimensions did not exist as an attribute
 * but was considered to be '3'
 * L3 does not require the attribute and will
 * only record it is officially set
 */
void
Model::setSpatialDimensions(double dims)
{
  for (unsigned int n = 0; n < getNumCompartments(); n++)
  {
    getCompartment(n)->setSpatialDimensions(dims);
  }
}

/* in L1 and L2 there were built in values for key units
 * such as 'volume', 'length', 'area', 'substance' and 'time'
 * In L3 these have been removed - thus if a model uses one of these
 * it needs a unitDefinition to define it
 */
void
Model::addDefinitionsForDefaultUnits()
{
  /* create a list of unit values */
  IdList unitsUsed;
  unsigned int n;
  bool implicitVolume = false;
  bool implicitArea = false;
  bool implicitLength = false;
  bool implicitSubstance = false;

  for (n = 0; n < getNumCompartments(); n++)
  {
    if (getCompartment(n)->isSetUnits())
    {
      unitsUsed.append(getCompartment(n)->getUnits());
    }
    else
    {
      if (getCompartment(n)->getSpatialDimensions() == 3)
      {
        implicitVolume = true;
        getCompartment(n)->setUnits("volume");
      }
      else if (getCompartment(n)->getSpatialDimensions() == 2)
      {
        implicitArea = true;
        getCompartment(n)->setUnits("area");
      }
      else if (getCompartment(n)->getSpatialDimensions() == 1)
      {
        implicitLength = true;
        getCompartment(n)->setUnits("length");
      }
    }
  }

  for (n = 0; n < getNumSpecies(); n++)
  {
    if (getSpecies(n)->isSetSubstanceUnits())
    {
      unitsUsed.append(getSpecies(n)->getSubstanceUnits());
    }
    else
    {
      implicitSubstance = true;
      getSpecies(n)->setSubstanceUnits("substance");
    }
 
    if (getSpecies(n)->isSetSpatialSizeUnits())
      unitsUsed.append(getSpecies(n)->getSpatialSizeUnits());
  }

  for (n = 0; n < getNumParameters(); n++)
  {
    if (getParameter(n)->isSetUnits())
      unitsUsed.append(getParameter(n)->getUnits());
  }

  if (getUnitDefinition("volume") == NULL 
    && (unitsUsed.contains("volume") || implicitVolume))
  {
    UnitDefinition * ud = createUnitDefinition();
    ud->setId("volume");
    Unit * u = ud->createUnit();
    u->setKind(UnitKind_forName("litre"));
    u->setScale(0);
    u->setExponent(1.0);
    u->setMultiplier(1.0);
  }

  if (getUnitDefinition("substance") == NULL 
    && (unitsUsed.contains("substance") || implicitSubstance))
  {
    UnitDefinition * ud = createUnitDefinition();
    ud->setId("substance");
    Unit * u = ud->createUnit();
    u->setKind(UnitKind_forName("mole"));
    u->setScale(0);
    u->setExponent(1.0);
    u->setMultiplier(1.0);
  }

  if (getUnitDefinition("area") == NULL
    && (unitsUsed.contains("area") || implicitArea))
  {
    UnitDefinition * ud = createUnitDefinition();
    ud->setId("area");
    Unit * u = ud->createUnit();
    u->setKind(UnitKind_forName("metre"));
    u->setScale(0);
    u->setExponent(2.0);
    u->setMultiplier(1.0);
  }

  if (getUnitDefinition("length") == NULL
    && (unitsUsed.contains("length") || implicitLength))
  {
    UnitDefinition * ud = createUnitDefinition();
    ud->setId("length");
    Unit * u = ud->createUnit();
    u->setKind(UnitKind_forName("metre"));
    u->setScale(0);
    u->setExponent(1.0);
    u->setMultiplier(1.0);
  }

  if (getUnitDefinition("time") == NULL)
  {
    UnitDefinition * ud = createUnitDefinition();
    ud->setId("time");
    Unit * u = ud->createUnit();
    u->setKind(UnitKind_forName("second"));
    u->setScale(0);
    u->setExponent(1.0);
    u->setMultiplier(1.0);

    setTimeUnits("time");
  }
  else
  {
    setTimeUnits("time");
  }

}

void
Model::convertParametersToLocals(unsigned int level, unsigned int version)
{
  for (unsigned int i = 0; i < getNumReactions(); i++)
  {
    Reaction *r = getReaction(i);
    if (r->isSetKineticLaw())
    {
      KineticLaw *kl = r->getKineticLaw();
      for (unsigned int j = 0; j < kl->getNumParameters(); j++)
      {
        LocalParameter *lp = new LocalParameter(level, version);
        (*lp) = *(kl->getParameter(j));
        kl->addLocalParameter(lp);
      }
    }
  }
}
/* the new strict setters mean that for a conversion to L2 to
 * take place the model needs to think it still l1 for
 * some actions and think it is already L2 for others
 */
void 
Model::removeParameterRuleUnits (bool strict)
{
  if (strict == true)
  {
    /* in L1 a parameterRule coulkd specify units
     * for a strict conversion this attribute should be unset
     */
    for (unsigned int n = 0; n < getNumParameters(); n++)
    {
      if (getRule(getParameter(n)->getId()) != NULL)
      {
        getRule(getParameter(n)->getId())->unsetUnits();
      }
    }
  }
}

/* converting to l1 any metaid attributes should be removed */
void
Model::removeMetaId()
{
  unsigned int n, i;

  unsetMetaId();
  
  for (n = 0; n < getNumUnitDefinitions(); n++)
  {
    getUnitDefinition(n)->unsetMetaId();
    for (i = 0; i < getUnitDefinition(n)->getNumUnits(); i++)
    {
      getUnitDefinition(n)->getUnit(i)->unsetMetaId();
    }
  }

  for (n = 0; n < getNumCompartments(); n++)
  {
    getCompartment(n)->unsetMetaId();
  }

  for (n = 0; n < getNumSpecies(); n++)
  {
    getSpecies(n)->unsetMetaId();
  }

  for (n = 0; n < getNumParameters(); n++)
  {
    getParameter(n)->unsetMetaId();
  }

  for (n = 0; n < getNumRules(); n++)
  {
    getRule(n)->unsetMetaId();
  }

  for (n = 0; n < getNumReactions(); n++)
  {
    getReaction(n)->unsetMetaId();
    for (i = 0; i < getReaction(n)->getNumReactants(); i++)
    {
      getReaction(n)->getReactant(i)->unsetMetaId();
    }
    for (i = 0; i < getReaction(n)->getNumReactants(); i++)
    {
      getReaction(n)->getProduct(i)->unsetMetaId();
    }
    if (getReaction(n)->isSetKineticLaw())
    {
      getReaction(n)->getKineticLaw()->unsetMetaId();
    }
  }
}


/* converting to l1 or l2v1 any sboTerm attributes should be removed */

void
Model::removeSBOTerms(bool strict)
{
  unsigned int n, i;

  if (strict == true)
  {
    unsetSBOTerm();
    
    for (n = 0; n < getNumUnitDefinitions(); n++)
    {
      getUnitDefinition(n)->unsetSBOTerm();
      for (i = 0; i < getUnitDefinition(n)->getNumUnits(); i++)
      {
        getUnitDefinition(n)->getUnit(i)->unsetSBOTerm();
      }
    }

    for (n = 0; n < getNumCompartments(); n++)
    {
      getCompartment(n)->unsetSBOTerm();
    }

    for (n = 0; n < getNumSpecies(); n++)
    {
      getSpecies(n)->unsetSBOTerm();
    }

    for (n = 0; n < getNumParameters(); n++)
    {
      getParameter(n)->unsetSBOTerm();
    }

    for (n = 0; n < getNumRules(); n++)
    {
      getRule(n)->unsetSBOTerm();
    }

    for (n = 0; n < getNumReactions(); n++)
    {
      getReaction(n)->unsetSBOTerm();
      for (i = 0; i < getReaction(n)->getNumReactants(); i++)
      {
        getReaction(n)->getReactant(i)->unsetSBOTerm();
        if (getReaction(n)->getReactant(i)->isSetStoichiometryMath())
        {
          getReaction(n)->getReactant(i)->getStoichiometryMath()->unsetSBOTerm();
        }
      }
      for (i = 0; i < getReaction(n)->getNumProducts(); i++)
      {
        getReaction(n)->getProduct(i)->unsetSBOTerm();
        if (getReaction(n)->getProduct(i)->isSetStoichiometryMath())
        {
          getReaction(n)->getProduct(i)->getStoichiometryMath()->unsetSBOTerm();
        }
      }
      for (i = 0; i < getReaction(n)->getNumModifiers(); i++)
      {
        getReaction(n)->getModifier(i)->unsetSBOTerm();
      }
      if (getReaction(n)->isSetKineticLaw())
      {
        getReaction(n)->getKineticLaw()->unsetSBOTerm();
      }
    }

    for (n = 0; n < getNumFunctionDefinitions(); n++)
    {
      getFunctionDefinition(n)->unsetSBOTerm();
    }

    for (n = 0; n < getNumEvents(); n++)
    {
      getEvent(n)->unsetSBOTerm();
      for (i = 0; i < getEvent(n)->getNumEventAssignments(); i++)
      {
        getEvent(n)->getEventAssignment(i)->unsetSBOTerm();
      }
      if (getEvent(n)->isSetTrigger())
      {
        getEvent(n)->getTrigger()->unsetSBOTerm();
      }
      if (getEvent(n)->isSetDelay())
      {
        getEvent(n)->getDelay()->unsetSBOTerm();
      }
    }
  }
}

/* converting to l1 any hasOnlySubstanceUnits attributes should be removed */
void
Model::removeHasOnlySubstanceUnits()
{
  for (unsigned int i = 0; i < getNumSpecies(); i++)
  {
    getSpecies(i)->setHasOnlySubstanceUnits(false);
  }
}

/* converting to l2v2 some sboTerm attributes should be removed */

void
Model::removeSBOTermsNotInL2V2(bool strict)
{
  unsigned int n, i;

  if (strict == true)
  {
    for (n = 0; n < getNumUnitDefinitions(); n++)
    {
      getUnitDefinition(n)->unsetSBOTerm();
      for (i = 0; i < getUnitDefinition(n)->getNumUnits(); i++)
      {
        getUnitDefinition(n)->getUnit(i)->unsetSBOTerm();
      }
    }

    for (n = 0; n < getNumCompartments(); n++)
    {
      getCompartment(n)->unsetSBOTerm();
    }

    for (n = 0; n < getNumSpecies(); n++)
    {
      getSpecies(n)->unsetSBOTerm();
    }

    for (n = 0; n < getNumCompartmentTypes(); n++)
    {
      getCompartmentType(n)->unsetSBOTerm();
    }

    for (n = 0; n < getNumSpeciesTypes(); n++)
    {
      getSpeciesType(n)->unsetSBOTerm();
    }


    for (n = 0; n < getNumReactions(); n++)
    {
      for (i = 0; i < getReaction(n)->getNumReactants(); i++)
      {
        if (getReaction(n)->getReactant(i)->isSetStoichiometryMath())
        {
          getReaction(n)->getReactant(i)->getStoichiometryMath()->unsetSBOTerm();
        }
      }
      for (i = 0; i < getReaction(n)->getNumProducts(); i++)
      {
        if (getReaction(n)->getProduct(i)->isSetStoichiometryMath())
        {
          getReaction(n)->getProduct(i)->getStoichiometryMath()->unsetSBOTerm();
        }
      }
    }

    for (n = 0; n < getNumEvents(); n++)
    {
      if (getEvent(n)->isSetTrigger())
      {
        getEvent(n)->getTrigger()->unsetSBOTerm();
      }
      if (getEvent(n)->isSetDelay())
      {
        getEvent(n)->getDelay()->unsetSBOTerm();
      }
    }
  }
}

void
Model::convertStoichiometryMath()
{
  unsigned int n, j;
  Reaction * r;
  SpeciesReference *sr;
  unsigned int idCount = 0;
  char newid[15];
  std::string id;

  for (n = 0; n < getNumReactions(); n++)
  {
    r = getReaction(n);
    for (j = 0; j < r->getNumReactants(); j++)
    {
      sr = r->getReactant(j);
      if (sr->isSetStoichiometryMath())
      {
        if (!sr->isSetId())
        {
          sprintf(newid, "generatedId_%u", idCount);
          id.assign(newid);
          sr->setId(id);
          idCount++;
        }
        else
        {
          id = sr->getId();
        }
        sr->setConstant(false);

        AssignmentRule * ar = createAssignmentRule();
        ar->setVariable(id);
        if (sr->getStoichiometryMath()->isSetMath())
        {
          ar->setMath(sr->getStoichiometryMath()->getMath());
        }
      }
    }
    for (j = 0; j < r->getNumProducts(); j++)
    {
      sr = r->getProduct(j);
      if (sr->isSetStoichiometryMath())
      {
        if (!sr->isSetId())
        {
          sprintf(newid, "generatedId_%u", idCount);
          id.assign(newid);
          sr->setId(id);
          idCount++;
        }
        else
        {
          id = sr->getId();
        }

        sr->setConstant(false);
        AssignmentRule * ar = createAssignmentRule();
        ar->setVariable(id);
        if (sr->getStoichiometryMath()->isSetMath())
        {
          ar->setMath(sr->getStoichiometryMath()->getMath());
        }
      }
    }
  }
}

void
Model::removeDuplicateTopLevelAnnotations()
{
  unsigned int i, n;
  this->removeDuplicateAnnotations();

  if (getNumFunctionDefinitions() > 0)
  {
    getListOfFunctionDefinitions()->removeDuplicateAnnotations();
    for (i = 0; i < getNumFunctionDefinitions(); i++)
    {
      getFunctionDefinition(i)->removeDuplicateAnnotations();
    }
  }
  if (getNumUnitDefinitions() > 0)
  {
    getListOfUnitDefinitions()->removeDuplicateAnnotations();
    for (i = 0; i < getNumUnitDefinitions(); i++)
    {
      getUnitDefinition(i)->removeDuplicateAnnotations();
      getUnitDefinition(i)->getListOfUnits()->removeDuplicateAnnotations();
      for (n = 0; n < getUnitDefinition(i)->getNumUnits(); n++)
      {
        getUnitDefinition(i)->getUnit(n)->removeDuplicateAnnotations();
      }
    }
  }
  if (getNumCompartmentTypes() > 0)
  {
    getListOfCompartmentTypes()->removeDuplicateAnnotations();
    for (i = 0; i < getNumCompartmentTypes(); i++)
    {
      getCompartmentType(i)->removeDuplicateAnnotations();
    }
  }
  if (getNumSpeciesTypes() > 0)
  {
    getListOfSpeciesTypes()->removeDuplicateAnnotations();
    for (i = 0; i < getNumSpeciesTypes(); i++)
    {
      getSpeciesType(i)->removeDuplicateAnnotations();
    }
  }
  if (getNumCompartments() > 0)
  {
    getListOfCompartments()->removeDuplicateAnnotations();
    for (i = 0; i < getNumCompartments(); i++)
    {
      getCompartment(i)->removeDuplicateAnnotations();
    }
  }
  if (getNumSpecies() > 0)
  {
    getListOfSpecies()->removeDuplicateAnnotations();
    for (i = 0; i < getNumSpecies(); i++)
    {
      getSpecies(i)->removeDuplicateAnnotations();
    }
  }
  if (getNumParameters() > 0)
  {
    getListOfParameters()->removeDuplicateAnnotations();
    for (i = 0; i < getNumParameters(); i++)
    {
      getParameter(i)->removeDuplicateAnnotations();
    }
  }
  if (getNumInitialAssignments() > 0)
  {
    getListOfInitialAssignments()->removeDuplicateAnnotations();
    for (i = 0; i < getNumInitialAssignments(); i++)
    {
      getInitialAssignment(i)->removeDuplicateAnnotations();
    }
  }
  if (getNumConstraints() > 0)
  {
    getListOfConstraints()->removeDuplicateAnnotations();
    for (i = 0; i < getNumConstraints(); i++)
    {
      getConstraint(i)->removeDuplicateAnnotations();
    }
  }
  if (getNumRules() > 0)
  {
    getListOfRules()->removeDuplicateAnnotations();
    for (i = 0; i < getNumRules(); i++)
    {
      getRule(i)->removeDuplicateAnnotations();
    }
  }
  if (getNumReactions() > 0)
  {
    getListOfReactions()->removeDuplicateAnnotations();
    for (i = 0; i < getNumReactions(); i++)
    {
      Reaction * r = getReaction(i);
      r->removeDuplicateAnnotations();
      if (r->getNumReactants() > 0)
      {
        r->getListOfReactants()->removeDuplicateAnnotations();
        for (n = 0; n < r->getNumReactants(); n++)
        {
          r->getReactant(n)->removeDuplicateAnnotations();
        }
      }
      if (r->getNumProducts() > 0)
      {
        r->getListOfProducts()->removeDuplicateAnnotations();
        for (n = 0; n < r->getNumProducts(); n++)
        {
          r->getProduct(n)->removeDuplicateAnnotations();
        }
      }
      if (r->getNumModifiers() > 0)
      {
        r->getListOfModifiers()->removeDuplicateAnnotations();
        for (n = 0; n < r->getNumModifiers(); n++)
        {
          r->getModifier(n)->removeDuplicateAnnotations();
        }
      }
      if (r->isSetKineticLaw())
      {
        r->getKineticLaw()->removeDuplicateAnnotations();
        if (r->getKineticLaw()->getNumParameters() > 0)
        {
          r->getKineticLaw()->getListOfParameters()
                            ->removeDuplicateAnnotations();
          for (n = 0; n < r->getKineticLaw()->getNumParameters(); n++)
          {
            r->getKineticLaw()->getParameter(n)->removeDuplicateAnnotations();
          }
        }
      }
    }
  }
  if (getNumEvents() > 0)
  {
    getListOfEvents()->removeDuplicateAnnotations();
    for (i = 0; i < getNumEvents(); i++)
    {
      getEvent(i)->removeDuplicateAnnotations();
      if (getEvent(i)->getNumEventAssignments() > 0)
      {
        getEvent(i)->getListOfEventAssignments()->removeDuplicateAnnotations();
        for (n = 0; n < getEvent(i)->getNumEventAssignments(); n++)
        {
          getEvent(i)->getEventAssignment(n)->removeDuplicateAnnotations();
        }
      }
    }
  }
}

void
Model::assignRequiredValues()
{
  // when converting to L3 some attributes which have default values in L1/L2
  // but are required in L3 are not present or set
  unsigned int i, n;

  if (getNumUnitDefinitions() > 0)
  {
    for (i = 0; i < getNumUnitDefinitions(); i++)
    {
      for (n = 0; n < getUnitDefinition(i)->getNumUnits(); n++)
      {
        Unit *u = getUnitDefinition(i)->getUnit(n);
        if (!u->isSetExponent())
          u->setExponent(1.0);
        if (!u->isSetScale())
          u->setScale(0);
        if (!u->isSetMultiplier())
          u->setMultiplier(1.0);
      }
    }
  }
  
  if (getNumCompartments() > 0)
  {
    for (i = 0; i < getNumCompartments(); i++)
    {
      Compartment *c = getCompartment(i);
      c->setConstant(c->getConstant());
    }
  }
  if (getNumSpecies() > 0)
  {
    for (i = 0; i < getNumSpecies(); i++)
    {
      Species * s = getSpecies(i);
      s->setBoundaryCondition(s->getBoundaryCondition());
      s->setHasOnlySubstanceUnits(s->getHasOnlySubstanceUnits());
      s->setConstant(s->getConstant());
    }
  }
  if (getNumParameters() > 0)
  {
    for (i = 0; i < getNumParameters(); i++)
    {
      Parameter * p = getParameter(i);
      p->setConstant(p->getConstant());
    }
  }
  if (getNumReactions() > 0)
  {
    for (i = 0; i < getNumReactions(); i++)
    {
      Reaction * r = getReaction(i);
      r->setFast(r->getFast());
      r->setReversible(r->getReversible());
      if (r->getNumReactants() > 0)
      {
        for (n = 0; n < r->getNumReactants(); n++)
        {
          SpeciesReference *sr = r->getReactant(n);
          if (sr->isSetStoichiometryMath())
          {
            sr->setConstant(false);
          }
          else
          {
            sr->setConstant(true);
          }
        }
      }
      if (r->getNumProducts() > 0)
      {
        for (n = 0; n < r->getNumProducts(); n++)
        {
          SpeciesReference *sr = r->getProduct(n);
          if (sr->isSetStoichiometryMath())
          {
            sr->setConstant(false);
          }
          else
          {
            sr->setConstant(true);
          }
        }
      }
    }
  }
  if (getNumEvents() > 0)
  {
    for (i = 0; i < getNumEvents(); i++)
    {
      Event * e = getEvent(i);
      e->setUseValuesFromTriggerTime(e->getUseValuesFromTriggerTime());

      if (e->isSetTrigger())
      {
        Trigger *t = e->getTrigger();
        t->setPersistent(true);
        t->setInitialValue(true);
      }
    }
  }

}

void
Model::dealWithEvents(bool strict)
{
  // if strict conversion want to unset L3 prioirty
  if (strict == true)
  {
    if (getNumEvents() > 0)
    {
      for (unsigned int i = 0; i < getNumEvents(); i++)
      {
        Event * e = getEvent(i);
        e->unsetPriority();
      }
    }
  }
}

void
Model::dealWithModelUnits()
{
  if (isSetVolumeUnits())
  {
    std::string volume = getVolumeUnits();
    UnitDefinition * ud = removeUnitDefinition(volume);
    if (ud != NULL)
    {
      ud->setId("volume");
    }
    else
    {
      Unit *u = new Unit(getSBMLNamespaces());
      u->initDefaults();
      u->setKind(UnitKind_forName(volume.c_str()));
      ud = new UnitDefinition(getSBMLNamespaces());
      ud->setId("volume");
      ud->addUnit(u);
    }
    addUnitDefinition(ud);
  }
  if (isSetAreaUnits())
  {
    std::string area = getAreaUnits();
    UnitDefinition * ud = removeUnitDefinition(area);
    if (ud != NULL)
    {
      ud->setId("area");
    }
    else
    {
      Unit *u = new Unit(getSBMLNamespaces());
      u->initDefaults();
      u->setKind(UnitKind_forName(area.c_str()));
      ud = new UnitDefinition(getSBMLNamespaces());
      ud->setId("area");
      ud->addUnit(u);
    }
    addUnitDefinition(ud);
  }
  if (isSetLengthUnits())
  {
    std::string length = getLengthUnits();
    UnitDefinition * ud = removeUnitDefinition(length);
    if (ud != NULL)
    {
      ud->setId("length");
    }
    else
    {
      Unit *u = new Unit(getSBMLNamespaces());
      u->initDefaults();
      u->setKind(UnitKind_forName(length.c_str()));
      ud = new UnitDefinition(getSBMLNamespaces());
      ud->setId("length");
      ud->addUnit(u);
    }
    addUnitDefinition(ud);
  }
  if (isSetSubstanceUnits())
  {
    std::string substance = getSubstanceUnits();
    UnitDefinition * ud = removeUnitDefinition(substance);
    if (ud != NULL)
    {
      ud->setId("substance");
    }
    else
    {
      Unit *u = new Unit(getSBMLNamespaces());
      u->initDefaults();
      u->setKind(UnitKind_forName(substance.c_str()));
      ud = new UnitDefinition(getSBMLNamespaces());
      ud->setId("substance");
      ud->addUnit(u);
    }
    addUnitDefinition(ud);
  }
  if (isSetTimeUnits())
  {
    std::string time = getTimeUnits();
    UnitDefinition * ud = removeUnitDefinition(time);
    if (ud != NULL)
    {
      ud->setId("time");
    }
    else
    {
      Unit *u = new Unit(getSBMLNamespaces());
      u->initDefaults();
      u->setKind(UnitKind_forName(time.c_str()));
      ud = new UnitDefinition(getSBMLNamespaces());
      ud->setId("time");
      ud->addUnit(u);
    }
    addUnitDefinition(ud);
  }
}

void
Model::dealWithStoichiometry()
{
  unsigned int idCount = 0;
  char newid[15];
  std::string id;
  for (unsigned int i = 0; i < getNumReactions(); i++)
  {
    Reaction *r = getReaction(i);
    unsigned int j;
    for (j = 0; j < r->getNumReactants(); j++)
    {
      SpeciesReference *sr = r->getReactant(j);
      if (!(sr->isSetStoichiometry()))
      {
        if (!(sr->isSetId()))
        {
          // no stoichiometry and no id to set the stoichiometry
          // replace with stoichiometryMath using a parameter with no value
          sprintf(newid, "parameterId_%u", idCount);
          id.assign(newid);
          idCount++;
          Parameter *p = createParameter();
          p->setId(id);
          p->setConstant(false);

          StoichiometryMath *sm = sr->createStoichiometryMath();
          if (sm != NULL)
          {
            ASTNode *ast = SBML_parseFormula(id.c_str());
            sm->setMath(ast);
          }
        }
        else
        {
          // id is set it could be used by initialAssignment
          // used by rule
          // not used
          if (getInitialAssignment(sr->getId()) != NULL)
          {
            // use stoichiometryMath instead
            StoichiometryMath *sm = sr->createStoichiometryMath();
            if (sm != NULL)
            {
              sm->setMath(getInitialAssignment(sr->getId())->getMath());
              removeInitialAssignment(sr->getId());
            }
          }
          else if (getRule(sr->getId()) != NULL)
          {
            // use stoichiometryMath instead
            StoichiometryMath *sm = sr->createStoichiometryMath();
            if (sm != NULL)
            {
              sm->setMath(getRule(sr->getId())->getMath());
              removeRule(sr->getId());
            }
          }
          //else if (getRateRule(sr->getId()) != NULL)
          //{
          //  // use stoichiometryMath instead
          //  StoichiometryMath *sm = sr->createStoichiometryMath();
          //  if (sm != NULL)
          //  {
          //    sm->setMath(getRateRule(sr->getId())->getMath());
          //    removeRateRule(sr->getId());
          //  }
          //}
          else
          {
            // is set but not used 
            // use StoichiometryMath with parameter with no value
            sprintf(newid, "parameterId_%u", idCount);
            id.assign(newid);
            idCount++;
            Parameter *p = createParameter();
            p->setId(id);
            p->setConstant(false);

            StoichiometryMath *sm = sr->createStoichiometryMath();
            if (sm != NULL)
            {
              ASTNode *ast = SBML_parseFormula(id.c_str());
              sm->setMath(ast);
            }
          }
        }
      }
      else
      {
        // stoichiometry is set
        if (sr->isSetId())
        {
          // id is set it could be used by initialAssignment
          // used by rule
          // not used
          if (getInitialAssignment(sr->getId()) != NULL)
          {
            // use stoichiometryMath instead
            StoichiometryMath *sm = sr->createStoichiometryMath();
            if (sm != NULL)
            {
              sm->setMath(getInitialAssignment(sr->getId())->getMath());
              removeInitialAssignment(sr->getId());
            }
          }
          else if (getRule(sr->getId()) != NULL)
          {
            // use stoichiometryMath instead
            StoichiometryMath *sm = sr->createStoichiometryMath();
            if (sm != NULL)
            {
              sm->setMath(getRule(sr->getId())->getMath());
              removeRule(sr->getId());
            }
          }
          //else if (getRateRule(sr->getId()) != NULL)
          //{
          //  // use stoichiometryMath instead
          //  StoichiometryMath *sm = sr->createStoichiometryMath();
          //  if (sm != NULL)
          //  {
          //    sm->setMath(getRateRule(sr->getId())->getMath());
          //    removeRateRule(sr->getId());
          //  }
          //}
        }
        // no id set - do not need to do anything
      }
    }
    for (j = 0; j < r->getNumProducts(); j++)
    {
      SpeciesReference *sr = r->getProduct(j);
      if (!(sr->isSetStoichiometry()))
      {
        if (!(sr->isSetId()))
        {
          // no stoichiometry and no id to set the stoichiometry
          // replace with stoichiometryMath using a parameter with no value
          sprintf(newid, "parameterId_%u", idCount);
          id.assign(newid);
          idCount++;
          Parameter *p = createParameter();
          p->setId(id);
          p->setConstant(false);

          StoichiometryMath *sm = sr->createStoichiometryMath();
          if (sm != NULL)
          {
            ASTNode *ast = SBML_parseFormula(id.c_str());
            sm->setMath(ast);
          }
        }
        else
        {
          // id is set it could be used by initialAssignment
          // used by rule
          // not used
          if (getInitialAssignment(sr->getId()) != NULL)
          {
            // use stoichiometryMath instead
            StoichiometryMath *sm = sr->createStoichiometryMath();
            if (sm != NULL)
            {
              sm->setMath(getInitialAssignment(sr->getId())->getMath());
              removeInitialAssignment(sr->getId());
            }
          }
          else if (getRule(sr->getId()) != NULL)
          {
            // use stoichiometryMath instead
            StoichiometryMath *sm = sr->createStoichiometryMath();
            if (sm != NULL)
            {
              sm->setMath(getRule(sr->getId())->getMath());
              removeRule(sr->getId());
            }
          }
          //else if (getRateRule(sr->getId()) != NULL)
          //{
          //  // use stoichiometryMath instead
          //  StoichiometryMath *sm = sr->createStoichiometryMath();
          //  if (sm != NULL)
          //  {
          //    sm->setMath(getRateRule(sr->getId())->getMath());
          //    removeRateRule(sr->getId());
          //  }
          //}
          else
          {
            // is set but not used 
            // use StoichiometryMath with parameter with no value
            sprintf(newid, "parameterId_%u", idCount);
            id.assign(newid);
            idCount++;
            Parameter *p = createParameter();
            p->setId(id);
            p->setConstant(false);

            StoichiometryMath *sm = sr->createStoichiometryMath();
            if (sm != NULL)
            {
              ASTNode *ast = SBML_parseFormula(id.c_str());
              sm->setMath(ast);
            }
          }
        }
      }
      else
      {
        // stoichiometry is set
        if (sr->isSetId())
        {
          // id is set it could be used by initialAssignment
          // used by rule
          // not used
          if (getInitialAssignment(sr->getId()) != NULL)
          {
            // use stoichiometryMath instead
            StoichiometryMath *sm = sr->createStoichiometryMath();
            if (sm != NULL)
            {
              sm->setMath(getInitialAssignment(sr->getId())->getMath());
              removeInitialAssignment(sr->getId());
            }
          }
          else if (getRule(sr->getId()) != NULL)
          {
            // use stoichiometryMath instead
            StoichiometryMath *sm = sr->createStoichiometryMath();
            if (sm != NULL)
            {
              sm->setMath(getRule(sr->getId())->getMath());
              removeRule(sr->getId());
            }
          }
          //else if (getRateRule(sr->getId()) != NULL)
          //{
          //  // use stoichiometryMath instead
          //  StoichiometryMath *sm = sr->createStoichiometryMath();
          //  if (sm != NULL)
          //  {
          //    sm->setMath(getRateRule(sr->getId())->getMath());
          //    removeRateRule(sr->getId());
          //  }
          //}
        }
        // no id set - do not need to do anything
      }
    }
  }
}
/** @endcond **/

LIBSBML_CPP_NAMESPACE_END


