/******************************************************************************************************************************************
 cgraphics.c
******************************************************************************************************************************************/

#include <stdlib.h>

#include "cwidget.h"

//-----------------------------------------------------------------------------------------------------------------------------------------
// simple types
//-----------------------------------------------------------------------------------------------------------------------------------------

// point
TPoint::TPoint (const SInt32 xx, const SInt32 yy) : x(xx), y(yy)
{ }
                                                                                                                                                
TPoint::TPoint (const TPoint &p) : x(p.x), y(p.y)
{ }
                                                                                                                                                
// size
TSize::TSize (const SInt32 ww, const SInt32 hh) : w(ww), h(hh)
{ }
                                                                                                                                                
TSize::TSize (const TSize &s) : w(s.w), h(s.h)
{ }
                                                                                                                                                
// bounds
TBounds::TBounds (const SInt32 xx, const SInt32 yy, const SInt32 ww, const SInt32 hh) : TPoint (xx, yy), TSize (ww, hh)
{ }
                                                                                                                                                
TBounds::TBounds (const TPoint &p, const TSize &s) : TPoint (p), TSize (s)
{ }
                                                                                                                                                
TBounds::TBounds (const TSize &s, const TPoint &p) : TPoint (p), TSize (s)
{ }
                                                                                                                                                
TBounds::TBounds (const TBounds &b) : TPoint (b.x, b.y), TSize (b.w, b.h)
{ }
                                                                                                                                                
TBounds & TBounds::operator = (const TBounds &b)
{
        x = b.x; y = b.y; w = b.w; h = b.h;
        return *this;
}

TBounds & TBounds::operator = (const TSize &s)
{
	w = s.w; h = s.h;
	return *this;
}

TBounds & TBounds::operator = (const TPoint &p)
{
	x = p.x; y = p.y;
	return *this;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// CGraphics
//-----------------------------------------------------------------------------------------------------------------------------------------

// metaclass code resolution
RESOLVE_CAPSULE_METACLASS (CGraphics);

// constructor
CGraphics::CGraphics     (CWidget *inGtkolWidget, const TBounds &inInvalidated)
	  :m_Drawable    ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ? 
			   inGtkolWidget -> GetGtkWidget() -> window : NULL),
	   m_Layout      ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ? 
		 	  ::gtk_widget_create_pango_layout (inGtkolWidget -> GetGtkWidget(), NULL) : NULL),
	   m_Drawn       (false),
	   m_Invalidated (inInvalidated),
	   m_Painted     (),
	   m_GC	         ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ? 
			  ::gdk_gc_new (inGtkolWidget -> GetGtkWidget() -> window) : NULL),
	   m_Style	 ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ? 
			   inGtkolWidget -> GetGtkWidget() -> style : NULL)
{ }

// constructor
CGraphics::CGraphics     (GtkWidget *inGtkWidget, const TBounds &inInvalidated)
	  :m_Drawable    ((inGtkWidget == NULL) ? NULL : inGtkWidget -> window),
	   m_Layout      ((inGtkWidget == NULL) ? NULL : ::gtk_widget_create_pango_layout (inGtkWidget, NULL)),
	   m_Drawn       (false),
	   m_Invalidated (inInvalidated),
	   m_Painted     (),
	   m_GC	         ((inGtkWidget == NULL) ? NULL : ::gdk_gc_new (inGtkWidget -> window)),
	   m_Style	 (inGtkWidget != NULL ? inGtkWidget-> style : NULL)
	   
{ }

// constructor
CGraphics::CGraphics     (GdkDrawable *inGdkDrawable, GtkStyle *inGtkStyle, PangoLayout *inPangoLayout, const TBounds &inInvalidated)
	  :m_Drawable    (inGdkDrawable),
	   m_Layout      (inPangoLayout),
	   m_Drawn       (false),
	   m_Invalidated (inInvalidated),
	   m_Painted     (),
	   m_GC	         ((inGdkDrawable == NULL) ? NULL : ::gdk_gc_new (inGdkDrawable)),
	   m_Style	 (inGtkStyle)
{ }

// destructor
CGraphics::~CGraphics ()
{ 
	if (m_Layout != NULL) ::g_object_unref (m_Layout);
	if (m_GC     != NULL) ::gdk_gc_unref   (m_GC);
}

// Drawn
Bool CGraphics::GetDrawn () const
{
	return m_Drawn;
}

// painted bounds ?
TBounds CGraphics::GetPainted () const
{
	// painted bounds with one margin
	return TBounds (m_Painted.x-1, m_Painted.y-1, m_Painted.w+2, m_Painted.h+2);
}

// invalidated bounds
TBounds CGraphics::GetInvalidated () const
{
	return m_Invalidated;
}

// get string size
TSize CGraphics::GetStringSize (const CString &inString) const
{
	// size out
	TSize outSize;

	// be paranoïd
	if (m_Layout == NULL) return outSize;

	// set the text of the layout
	::pango_layout_set_text (m_Layout, inString.Get(), inString.GetLength());

	// get string size
	::pango_layout_get_size (m_Layout, (int*)&outSize.w, (int*)&outSize.h);

	// devide (pango units -> device units)
	outSize.w /= PANGO_SCALE;
	outSize.h /= PANGO_SCALE;

	// ok
	return outSize;
}

// draw a string
void CGraphics::DrawString (const CString &inString, const TPoint &inPoint)
{
	// be paranoïd
	if (m_Drawable == NULL || m_GC == NULL || m_Layout == NULL) return;

	// set the text of the layout
	::pango_layout_set_text (m_Layout, inString.Get(), inString.GetLength());

	// draw the layout
	::gdk_draw_layout (m_Drawable, m_GC, inPoint.x, inPoint.y, m_Layout);

	// get string size
	TSize inSize (GetStringSize(inString));

	// keep painted bounds
	m_Painted.x = m_Painted.x > 0 ? min (m_Painted.x, inPoint.x) : inPoint.x;
	m_Painted.y = m_Painted.y > 0 ? min (m_Painted.y, inPoint.y) : inPoint.y;
	TPoint Pp (m_Painted.x+m_Painted.w, m_Painted.y+m_Painted.h);
	TPoint Ps (inPoint.x+inSize.w, inPoint.y+inSize.h);
	TPoint Pm (max(Pp.x,Ps.x), max(Pp.y,Ps.y));
	m_Painted.w = Pm.x - m_Painted.x;
	m_Painted.h = Pm.y - m_Painted.y;

	// state flag
	m_Drawn = true;
}

// draw a line
void CGraphics::DrawLine (const TPoint &inP1, const TPoint &inP2)
{
	// be paranoïd
	if (m_Drawable == NULL || m_GC == NULL) return;

	// draw the line
	::gdk_draw_line (m_Drawable, m_GC, inP1.x, inP1.y, inP2.x, inP2.y);

	// line width
	GdkGCValues inGdkGCValues; ::gdk_gc_get_values (m_GC, &inGdkGCValues);

	// keep painted bounds
	TPoint inPmn (min(inP1.x,inP2.x), min(inP1.y,inP2.y));
	TPoint inPmx (max(inP1.x,inP2.x), max(inP1.y,inP2.y));
	m_Painted.x = m_Painted.x > 0 ? min (m_Painted.x, inPmn.x-inGdkGCValues.line_width) : inPmn.x-inGdkGCValues.line_width;
	m_Painted.y = m_Painted.y > 0 ? min (m_Painted.y, inPmn.y-inGdkGCValues.line_width) : inPmn.y-inGdkGCValues.line_width;
	TPoint Pp (m_Painted.x+m_Painted.w, m_Painted.y+m_Painted.h);
	TPoint Pm (max(inPmx.x+inGdkGCValues.line_width,Pp.x), max(inPmx.y+inGdkGCValues.line_width,Pp.y));
	m_Painted.w = Pm.x-m_Painted.x; m_Painted.w = m_Painted.w < inGdkGCValues.line_width ? inGdkGCValues.line_width : m_Painted.w;
	m_Painted.h = Pm.y-m_Painted.y; m_Painted.h = m_Painted.h < inGdkGCValues.line_width ? inGdkGCValues.line_width : m_Painted.h;

	// state flag
	m_Drawn = true;
}

// draw pixbuf
void CGraphics::DrawPixbuf (CPixbuf *inPixbuf, const TBounds &inBounds, const TPoint &inPoint)
{
	// check pointer
	if (m_Drawable == NULL || m_GC == NULL || inPixbuf == NULL || inPixbuf -> GetPixbuf() == NULL) return;

	// draw the pixbuf
	::gdk_draw_pixbuf (m_Drawable, m_GC, inPixbuf -> GetPixbuf(), inBounds.x, inBounds.y, inPoint.x, inPoint.y, inBounds.w, inBounds.h, 
			   GDK_RGB_DITHER_MAX, inPoint.x, inPoint.y);

	// keep painted bounds
	m_Painted.x = m_Painted.x > 0 ? min (m_Painted.x, inBounds.x) : inBounds.x;
	m_Painted.y = m_Painted.y > 0 ? min (m_Painted.y, inBounds.y) : inBounds.y;
	if (m_Painted.x + m_Painted.w < inBounds.x + inBounds.w) m_Painted.w += inBounds.x + inBounds.w - (m_Painted.x + m_Painted.w);
	if (m_Painted.y + m_Painted.h < inBounds.y + inBounds.h) m_Painted.h += inBounds.y + inBounds.h - (m_Painted.y + m_Painted.h);

	// state flag
	m_Drawn = true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// paint focus
//-----------------------------------------------------------------------------------------------------------------------------------------
void CGraphics::PaintFocus (const TBounds &inBounds, const GtkStateType inGtkStateType)
{
	// check pointer
	if (m_Drawable == NULL || m_Style == NULL) return;

	// paint focus
	::gtk_paint_focus (m_Style, m_Drawable, inGtkStateType, NULL, NULL, NULL, inBounds.x, inBounds.y, inBounds.w, inBounds.h);

	// keep painted bounds
	m_Painted.x = m_Painted.x > 0 ? min (m_Painted.x, inBounds.x) : inBounds.x;
	m_Painted.y = m_Painted.y > 0 ? min (m_Painted.y, inBounds.y) : inBounds.y;
	if (m_Painted.x + m_Painted.w < inBounds.x + inBounds.w) m_Painted.w += inBounds.x + inBounds.w - (m_Painted.x + m_Painted.w);
	if (m_Painted.y + m_Painted.h < inBounds.y + inBounds.h) m_Painted.h += inBounds.y + inBounds.h - (m_Painted.y + m_Painted.h);

	// state flag
	m_Drawn = true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// draw rectangle
//-----------------------------------------------------------------------------------------------------------------------------------------
void CGraphics::DrawRectangle (const TBounds &inBounds, const bool inFill)
{
	// check pointer
	if (m_Drawable == NULL || m_GC == NULL) return;

	// draw it
	::gdk_draw_rectangle (m_Drawable, m_GC, inFill, inBounds.x, inBounds.y, inBounds.w, inBounds.h);

	// keep painted bounds
	m_Painted.x = m_Painted.x > 0 ? min (m_Painted.x, inBounds.x) : inBounds.x;
	m_Painted.y = m_Painted.y > 0 ? min (m_Painted.y, inBounds.y) : inBounds.y;
	if (m_Painted.x + m_Painted.w < inBounds.x + inBounds.w) m_Painted.w += inBounds.x + inBounds.w - (m_Painted.x + m_Painted.w);
	if (m_Painted.y + m_Painted.h < inBounds.y + inBounds.h) m_Painted.h += inBounds.y + inBounds.h - (m_Painted.y + m_Painted.h);

	// state flag
	m_Drawn = true;
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// CClothPainter
//-----------------------------------------------------------------------------------------------------------------------------------------

// metaclass
RESOLVE_CAPSULE_METACLASS (CClothPainter);

// constructor
CClothPainter::CClothPainter (CWidget *inGtkolWidget)
	      :m_Drawable    ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ?
			      inGtkolWidget -> GetGtkWidget() -> window : NULL),
	       m_GC	     ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ?
			      ::gdk_gc_new (inGtkolWidget -> GetGtkWidget() -> window) : NULL),
	       m_Pixmap	     ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ?
			      ::gdk_pixmap_new (inGtkolWidget -> GetGtkWidget() -> window, 
						inGtkolWidget -> GetGtkWidget() -> allocation.width,
                                                inGtkolWidget -> GetGtkWidget() -> allocation.height, 
						(inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ?
			      inGtkolWidget -> GetGtkWidget() -> style -> depth : -1) : NULL),
	       m_Cloth	     ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ?
			      ::gdk_pixmap_new (inGtkolWidget -> GetGtkWidget() -> window, 
						inGtkolWidget -> GetGtkWidget() -> allocation.width,
                                                inGtkolWidget -> GetGtkWidget() -> allocation.height, 
						(inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ?
			      inGtkolWidget -> GetGtkWidget() -> style -> depth : -1) : NULL),
	       m_Layout	     ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ?
			      ::gtk_widget_create_pango_layout (inGtkolWidget -> GetGtkWidget(), NULL) : NULL),
	       m_Size	     ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ?
			      TSize (inGtkolWidget -> GetGtkWidget() -> allocation.width, 
				     inGtkolWidget -> GetGtkWidget() -> allocation.height) : 
			      TSize ()),
	       m_Style	     ((inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL) ?
			      inGtkolWidget -> GetGtkWidget() -> style : NULL)
{ 
	// initialize the orginal cloth from the targeted drawable
	if (m_GC != NULL && m_Pixmap != NULL) ::gdk_draw_drawable (m_Cloth, m_GC, m_Drawable, 0, 0, 0, 0, -1, -1);
}

// constructor
CClothPainter::CClothPainter (GdkDrawable *inGdkDrawable, GtkStyle *inStyle, PangoLayout *inPangoLayout, const TSize &inSize)
	      :m_Drawable    (inGdkDrawable),
	       m_GC	     ((inGdkDrawable == NULL) ? NULL : ::gdk_gc_new     (inGdkDrawable)),
	       m_Pixmap	     ((inGdkDrawable == NULL) ? NULL : ::gdk_pixmap_new (inGdkDrawable, inSize.w, inSize.h, 
										 inStyle != NULL ? inStyle -> depth : -1)),
	       m_Cloth 	     ((inGdkDrawable == NULL) ? NULL : ::gdk_pixmap_new (inGdkDrawable, inSize.w, inSize.h, 
										 inStyle != NULL ? inStyle -> depth : -1)),
	       m_Layout	     (inPangoLayout),
	       m_Size	     (inSize),
	       m_Style	     (inStyle)	 
{ 
	// initialize the orginal cloth from the targeted drawable
	if (m_GC != NULL && m_Pixmap != NULL) ::gdk_draw_drawable (m_Cloth, m_GC, m_Drawable, 0, 0, 0, 0, -1, -1);
}

// constructor
CClothPainter::CClothPainter (GtkWidget *inGtkWidget)
	      :m_Drawable    ((inGtkWidget == NULL) ? NULL : inGtkWidget -> window),
	       m_GC	     ((inGtkWidget == NULL) ? NULL : ::gdk_gc_new (inGtkWidget -> window)),
	       m_Pixmap	     ((inGtkWidget == NULL) ? NULL : ::gdk_pixmap_new (inGtkWidget -> window, inGtkWidget -> allocation.width, 
									       inGtkWidget -> allocation.height, 
									       inGtkWidget != NULL ? inGtkWidget -> style -> depth : -1)),
	       m_Cloth 	     ((inGtkWidget == NULL) ? NULL : ::gdk_pixmap_new (inGtkWidget -> window, inGtkWidget -> allocation.width, 
									       inGtkWidget -> allocation.height, 
									       inGtkWidget != NULL ? inGtkWidget -> style -> depth : -1)),
	       m_Layout	     ((inGtkWidget == NULL) ? NULL : ::gtk_widget_create_pango_layout (inGtkWidget, NULL)),
	       m_Size	     ((inGtkWidget == NULL) ? TSize() : TSize(inGtkWidget -> allocation.width, inGtkWidget -> allocation.height)),
	       m_Style	     ((inGtkWidget != NULL) ? inGtkWidget -> style : NULL)
{
	// initialize the orginal cloth from the targeted drawable
	if (m_GC != NULL && m_Pixmap != NULL) ::gdk_draw_drawable (m_Cloth, m_GC, m_Drawable, 0, 0, 0, 0, -1, -1);
}

// destructor
CClothPainter::~CClothPainter ()
{
	if (m_Pixmap != NULL) ::gdk_pixmap_unref (m_Pixmap);
	if (m_Cloth  != NULL) ::gdk_pixmap_unref (m_Cloth);
	if (m_Layout != NULL) ::g_object_unref   (m_Layout);
	if (m_GC     != NULL) ::gdk_gc_unref     (m_GC);
}

// instanciate a graphics tool
CGraphics * CClothPainter::CreateGraphics () const
{
	// initialize the pixmap to paint on from the orginal cloth
	::gdk_draw_drawable (m_Pixmap, m_GC, m_Cloth, 0, 0, 0, 0, -1, -1);

	// returns a tool to paint on the pixmap cloth
	return new CGraphics (m_Pixmap, m_Style, (m_Layout != NULL) ? ::pango_layout_copy(m_Layout) : NULL, TBounds(m_Size));
}

// release a graphics tool
void CClothPainter::DeleteGraphics (CGraphics *inGraphics) const
{
	// check pointer...
	if (inGraphics != NULL)
	{
		// get the painted bounds of the graphics
		TBounds inBounds (inGraphics -> GetPainted());

		// paint the pixmap cloth worked on on the targeted drawable
		::gdk_draw_drawable (m_Drawable, m_GC, m_Pixmap, inBounds.x, inBounds.y, inBounds.x, inBounds.y, inBounds.w, inBounds.h);
	
		// deletes the instance
		delete inGraphics;
	}
	// in doubt...
	else
	{
		// paint the whole pixmap cloth worked on on the targeted drawable
		::gdk_draw_drawable (m_Drawable, m_GC, m_Pixmap, 0, 0, 0, 0, -1, -1);
	}
}

//-----------------------------------------------------------------------------------------------------------------------------------------
// CPixbuf
//-----------------------------------------------------------------------------------------------------------------------------------------

// metaclass
RESOLVE_CAPSULE_METACLASS (CPixbuf);

// constructor
CPixbuf::CPixbuf  (const TSize &inSize, const TPixelFormat inPixelFormat)
	:m_Pixbuf (NULL),
	 m_Name   ()
{
	// analyse desired format
	switch (inPixelFormat)
	{
		case PIXELFORMAT_8  : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, 8,  inSize.w, inSize.h); break;
		case PIXELFORMAT_16 : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, 16, inSize.w, inSize.h); break;
		case PIXELFORMAT_24 : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, 24, inSize.w, inSize.h); break;
		case PIXELFORMAT_32 : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, true,  8,  inSize.w, inSize.h); break;
	}
}

// constructor
CPixbuf::CPixbuf  (CWidget *inGtkolWidget, const TBounds &inBounds)
	:m_Pixbuf (NULL),
	 m_Name   ()
{
	// check pointer in
	if (inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL && inGtkolWidget -> GetGtkWidget() -> window != NULL) 

		// create the pixbuf from the drawable (defaults to 8 bits pixel format output)
		m_Pixbuf = ::gdk_pixbuf_get_from_drawable (NULL, inGtkolWidget -> GetGtkWidget() -> window , ::gdk_colormap_get_system(), 
							   inBounds.x, inBounds.y, 0, 0, inBounds.w, inBounds.h);
}

// constructor
CPixbuf::CPixbuf  (CWidget *inGtkolWidget, const TPixelFormat inPixelFormat, const TBounds &inBounds)
	:m_Pixbuf (NULL),
	 m_Name   ()
{
	// analyse desired format and allocate the pixbuf first
	switch (inPixelFormat)
	{
		case PIXELFORMAT_8  : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, 8,  inBounds.w, inBounds.h); break;
		case PIXELFORMAT_16 : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, 16, inBounds.w, inBounds.h); break;
		case PIXELFORMAT_24 : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, 24, inBounds.w, inBounds.h); break;
		case PIXELFORMAT_32 : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, true,  8,  inBounds.w, inBounds.h); break;
	}

	// check pointer in
	if (inGtkolWidget != NULL && inGtkolWidget -> GetGtkWidget() != NULL && inGtkolWidget -> GetGtkWidget() -> window != NULL) 

		// create the pixbuf from the drawable 
		::gdk_pixbuf_get_from_drawable (m_Pixbuf, inGtkolWidget -> GetGtkWidget() -> window, ::gdk_colormap_get_system(), 
						inBounds.x, inBounds.y, 0, 0, inBounds.w, inBounds.h);
}

// constructor
CPixbuf::CPixbuf  (GdkDrawable *inGdkDrawable, const TBounds &inBounds)
	:m_Pixbuf (NULL),
	 m_Name   ()
{
	// check pointer in
	if (inGdkDrawable != NULL) 

		// create the pixbuf from the drawable (8 bits pixel format output)
		m_Pixbuf = ::gdk_pixbuf_get_from_drawable (NULL, inGdkDrawable, ::gdk_colormap_get_system(), inBounds.x, inBounds.y, 
							   0, 0, inBounds.w, inBounds.h);
}

// constructor
CPixbuf::CPixbuf  (GdkDrawable *inGdkDrawable, const TPixelFormat inPixelFormat, const TBounds &inBounds)
	:m_Pixbuf (NULL),
	 m_Name   ()
{
	// analyse desired format and allocate the pixbuf so
	switch (inPixelFormat)
	{
		case PIXELFORMAT_8  : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, 8,  inBounds.w, inBounds.h); break;
		case PIXELFORMAT_16 : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, 16, inBounds.w, inBounds.h); break;
		case PIXELFORMAT_24 : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, false, 24, inBounds.w, inBounds.h); break;
		case PIXELFORMAT_32 : m_Pixbuf = ::gdk_pixbuf_new (GDK_COLORSPACE_RGB, true,  8,  inBounds.w, inBounds.h); break;
	}

	// check pointer in
	if (inGdkDrawable != NULL) 

		// fill in the allocated pixbuf from the specified drawable
		::gdk_pixbuf_get_from_drawable (m_Pixbuf, inGdkDrawable, ::gdk_colormap_get_system(), inBounds.x, inBounds.y, 
						0, 0, inBounds.w, inBounds.h);
}

// function called to free the pixels
static void PixbufDataDestructor (guchar *inData, gpointer)
{
	// check...
	if (inData != NULL) delete [] inData;
}

// constructor
CPixbuf::CPixbuf  (const UInt8 *inData, const TSize &inSize, const UInt16 inRowBytes, const TPixelFormat inPixelFormat)
	:m_Pixbuf (NULL),
	 m_Name   ()
{
	// check pointer in
	if (inData != NULL)

		// allocate the pixbuf
		m_Pixbuf = ::gdk_pixbuf_new_from_data (inData, GDK_COLORSPACE_RGB, (inPixelFormat == PIXELFORMAT_32) ? true : false, 
					    	      (inPixelFormat == PIXELFORMAT_32) ? 8 : (int)inPixelFormat, inSize.w, inSize.h,
						       inRowBytes, PixbufDataDestructor, NULL);
}

// constructor
CPixbuf::CPixbuf  (const CString &inFileName)
	:m_Pixbuf (NULL),
	 m_Name   (inFileName)
{
	GError *inError = NULL;//new GError;
	m_Pixbuf = ::gdk_pixbuf_new_from_file (m_Name.Get(), &inError);
	//delete inError;
}

// constructor
CPixbuf::CPixbuf  (const char *inName)
	:m_Pixbuf (NULL),
	 m_Name   (CString("id:")+inName)
{
	GtkWidget *inCellView = ::gtk_cell_view_new ();
	m_Pixbuf = ::gtk_widget_render_icon (inCellView, inName, GTK_ICON_SIZE_BUTTON, NULL);
	::gtk_widget_destroy (inCellView);
}

// copy constructor
CPixbuf::CPixbuf  (const CPixbuf &inPixbuf)
	:m_Pixbuf ((inPixbuf.m_Pixbuf != NULL) ? ::gdk_pixbuf_copy (inPixbuf.m_Pixbuf) : NULL),
	 m_Name   (inPixbuf.m_Name)
{ }

// destructor
CPixbuf::~CPixbuf ()
{
	// release the gdk pixbuf
	if (m_Pixbuf != NULL) ::g_object_unref (m_Pixbuf);
}

// pixel format
TPixelFormat CPixbuf::GetPixelFormat () const
{
	// out result
	TPixelFormat outPixelFormat = PIXELFORMAT_8;

	// check pointer...
	if (m_Pixbuf == NULL) return outPixelFormat;

	// check pixbuf channels...
	switch (::gdk_pixbuf_get_n_channels (m_Pixbuf))
	{
		case 1 : outPixelFormat = PIXELFORMAT_8;  break;
		case 2 : outPixelFormat = PIXELFORMAT_16; break;
		case 3 : outPixelFormat = PIXELFORMAT_24; break;
		case 4 : outPixelFormat = PIXELFORMAT_32; break;
	}

	// ok
	return outPixelFormat;
}

// pixbuf size
TSize CPixbuf::GetSize () const
{
	// return value
	TSize outSize;

	// check pointer
	if (m_Pixbuf == NULL) return outSize;

	// get size
	outSize.w = ::gdk_pixbuf_get_width  (m_Pixbuf);
	outSize.h = ::gdk_pixbuf_get_height (m_Pixbuf);

	// ok
	return outSize;
}

// row bytes
UInt16 CPixbuf::GetRowBytes () const
{
	// ok ?
	return (m_Pixbuf != NULL) ? ::gdk_pixbuf_get_rowstride (m_Pixbuf) : 0;
}

// direct buffer access
UInt8 * CPixbuf::GetBaseAddress () const
{
	// ok ?
	return (m_Pixbuf != NULL) ? ::gdk_pixbuf_get_pixels (m_Pixbuf) : NULL;
}

// gdk pointer
GdkPixbuf * CPixbuf::GetPixbuf () const
{
	// ok
	return m_Pixbuf;
}

CString CPixbuf::GetName () const
{
	return m_Name;
}

// thanks to "Brensenham algorithm", draws a line on the specified buffer from a point to an other one
void CPixbuf::DrawLine (const TPoint &inP1, const TPoint &inP2, UInt8 *inData, const TPixelFormat inPixelFormat,
                        const UInt16 inRowBytes, const UInt32 inColor, const Float32 inAlpha)
{
	// let's get extrem lines addresses...
	UInt8 *begLine = inData + inP1.y * inRowBytes;
	UInt8 *endLine = inData + inP2.y * inRowBytes;

	// get distances between the two points
	UInt32 dx = abs (inP2.x - inP1.x);
	UInt32 dy = abs (inP2.y - inP1.y);

	// get signs to go through axes...
	SInt8 sx = (inP1.x < inP2.x) ? 1 : ((inP1.x > inP2.x) ? -1 : 0);
	SInt8 sy = (inP1.y < inP2.y) ? 1 : ((inP1.y > inP2.y) ? -1 : 0);

	// so wich is the axe to go through ?
	Bool doChange = false; if (dy > dx) { UInt32 tmp = dx; dx = dy; dy = tmp; doChange = true; }

	// just get beginning points and prepare for algorithm...
	SInt32 x = inP1.x;
	SInt32 y = inP1.y;
	SInt32 v = 2 * dy - dx;

	// pixel format analyse
	switch (inPixelFormat)
	{
		// 8 bits per pixel
		case PIXELFORMAT_8 :
		{
			// get extrem pixels
			UInt8 *begPixel = begLine + inP1.x;
			UInt8 *endPixel = endLine + inP2.x;

			// draw extrem pixels
			*begPixel = UInt8((inColor&0xFF000000)>>24) | UInt8((inColor&0x00FF0000)>>16) | UInt8((inColor&0x0000FF00)>>8) |
				    UInt8( inColor&0x000000FF);
			*endPixel = *begPixel;

			// draw the other points of the line...
			if (doChange)
			{
				for (SInt32 i=dx; i>0; i--)
				{
					for (; v>=0; v-=2*dx) x += sx; y += sy; v += 2*dy;
					*(inData + y * inRowBytes + x) = UInt8((inColor&0xFF000000)>>24) | 
									 UInt8((inColor&0x00FF0000)>>16) | 
									 UInt8((inColor&0x0000FF00)>>8)  | 
									 UInt8( inColor&0x000000FF);
				}
			}
			else
			{
				for (SInt32 i=dx; i>0; i--)
				{
					for (; v>=0; v-=2*dx) y += sy; x += sx; v += 2*dy;
					*(inData + y * inRowBytes + x) = UInt8((inColor&0xFF000000)>>24) | 
									 UInt8((inColor&0x00FF0000)>>16) | 
									 UInt8((inColor&0x0000FF00)>>8)  | 
									 UInt8( inColor&0x000000FF);
				}
			}
		}
		break;

		// 16 bits per pixel
		case PIXELFORMAT_16 :
		{
			// get extrem pixels
			UInt16 *begPixel = reinterpret_cast <UInt16 *> (begLine) + inP1.x;
			UInt16 *endPixel = reinterpret_cast <UInt16 *> (endLine) + inP2.x;

			// draw extrem pixels
			UInt16 r = UInt16((inColor&0x00FF0000)>>19);
			UInt16 g = UInt16((inColor&0x0000FF00)>>10);
			UInt16 b = UInt16((inColor&0x000000FF)>>3);
			*begPixel = (r<<11) | (g<<5) | b;
			*endPixel = *begPixel;
			
			// draw the other points of the line...
			if (doChange)
			{
				for (SInt32 i=dx; i>0; i--)
				{
					for (; v>=0; v-=2*dx) x += sx; y += sy; v += 2*dy;
					UInt16 *Pixel = reinterpret_cast <UInt16 *> (inData + y * inRowBytes) + x;
					UInt16 r = UInt16((inColor&0x00FF0000)>>19);
					UInt16 g = UInt16((inColor&0x0000FF00)>>10);
					UInt16 b = UInt16((inColor&0x000000FF)>>3);
					*Pixel = (r<<11) | (g<<5) | b;
				}
			}
			else
			{
				for (SInt32 i=dx; i>0; i--)
				{
					for (; v>=0; v-=2*dx) y += sy; x += sx; v += 2*dy;
					UInt16 *Pixel = reinterpret_cast <UInt16 *> (inData + y * inRowBytes) + x;
					UInt16 r = UInt16((inColor&0x00FF0000)>>19);
					UInt16 g = UInt16((inColor&0x0000FF00)>>10);
					UInt16 b = UInt16((inColor&0x000000FF)>>3);
					*Pixel = (r<<11) | (g<<5) | b;
				}
			}
		}
		break;

		// 24 bits per pixel
		case PIXELFORMAT_24 :
		{
			// get extrem pixels
			UInt8 *begPixel = begLine + inP1.x*3;
			UInt8 *endPixel = endLine + inP2.x*3;

			// draw extrem pixels
			*begPixel++ = UInt8((inColor&0x00FF0000)>>16);
			*endPixel++ = UInt8((inColor&0x00FF0000)>>16);
			*begPixel++ = UInt8((inColor&0x0000FF00)>>8);
			*endPixel++ = UInt8((inColor&0x0000FF00)>>8);
			*begPixel   = UInt8( inColor&0x000000FF);
			*endPixel   = UInt8( inColor&0x000000FF);

			// draw the other points of the line...
			if (doChange)
			{
				for (SInt32 i=dx; i>0; i--)
				{
					for (; v>=0; v-=2*dx) x += sx; y += sy; v += 2*dy;
					UInt8 *Pixel = inData + y * inRowBytes + x*3;
					*Pixel++ = UInt8((inColor&0x00FF0000)>>16);
					*Pixel++ = UInt8((inColor&0x0000FF00)>>8);
					*Pixel   = UInt8( inColor&0x000000FF);
				}
			}
			else
			{
				for (SInt32 i=dx; i>0; i--)
				{
					for (; v>=0; v-=2*dx) y += sy; x += sx; v += 2*dy;
					UInt8 *Pixel = inData + y * inRowBytes + x*3;
					*Pixel++ = UInt8((inColor&0x00FF0000)>>16);
					*Pixel++ = UInt8((inColor&0x0000FF00)>>8);
					*Pixel   = UInt8( inColor&0x000000FF);
				}
			}
		}
		break;

		// 32 bits per pixel
		case PIXELFORMAT_32 :
		{
			// get extrem pixels
			UInt32 *begPixel = reinterpret_cast <UInt32 *> (begLine) + inP1.x;
			UInt32 *endPixel = reinterpret_cast <UInt32 *> (endLine) + inP2.x;

			// draw extrem pixels
			*begPixel = inColor;
			*endPixel = inColor;
		
			// draw the other points of the line...
			if (doChange)
			{
				for (SInt32 i=dx; i>0; i--)
				{
					for (; v>=0; v-=2*dx) x += sx; y += sy; v += 2*dy;
					*(reinterpret_cast <UInt32 *> (inData + y * inRowBytes) + x) = inColor;
				}
			}
			else
			{
				for (SInt32 i=dx; i>0; i--)
				{
					for (; v>=0; v-=2*dx) y += sy; x += sx; v += 2*dy;
					*(reinterpret_cast <UInt32 *> (inData + y * inRowBytes) + x) = inColor;
				}
			}
		}
		break;
	}
}

// draw a rectangle
void CPixbuf::DrawRectangle (const TBounds &inRectangle, UInt8 *inData, const TPixelFormat inPixelFormat, const UInt16 inRowBytes,
                             const UInt32 inColor, const Float32 inAlpha)
{
	// get points !
	TPoint P1 (inRectangle.x, inRectangle.y);
	TPoint P2 (inRectangle.x + inRectangle.w, inRectangle.y);
	TPoint P3 (inRectangle.x, inRectangle.y + inRectangle.h);
	TPoint P4 (inRectangle.x + inRectangle.w, inRectangle.y + inRectangle.h);

	// just call to draw lines !
	CPixbuf::DrawLine (P1, P2, inData, inPixelFormat, inRowBytes, inColor, inAlpha);
	CPixbuf::DrawLine (P3, P4, inData, inPixelFormat, inRowBytes, inColor, inAlpha);
	CPixbuf::DrawLine (P1, P3, inData, inPixelFormat, inRowBytes, inColor, inAlpha);
	CPixbuf::DrawLine (P2, P4, inData, inPixelFormat, inRowBytes, inColor, inAlpha);
}
