/******************************************************************************************************************************************
 citemfieldvalue.h

 CClass
 |--CSerialized
    |--CMetaModule
       |--CObject 
          |--CItemFieldValue		generic  class of field value for hierarchical or components list view
             |--CItemFieldValueString	string   field value representation
             |--CItemFieldValueBoolean	boolean  field value representation
	     |--CItemFieldValueProgress	progress field value representation
             |--CItemFieldValuePixbuf	pixbuf   field value representation
	     |--CItemFieldValueCombo	combo	 field value representation TODO
	     |--CItemFieldValuePack	generic  class interpretated as a field linker i.e. two field values packed into the same layout

 Simple field value representation; those definitions are used to handle fields on various view models (tree views and list views). All
 those views are declared as friend to directly access the item field value parameters; when the programmer requests a specific item field
 value API function, the item field value instance sends its owner a state modification notification so that the model can handle specific
 gui modifications. THERE IS ANY HANDLING ABOUT GUI MODIFICATIONS IN THE FOLLOWING VALUE CONTAINER DECLARATIONS. It is the owner view
 responsability to assign the m_Owner and and m_Listener attributes of the item field value instance when affected in order to receive
 notifications as needed.
******************************************************************************************************************************************/

#ifndef __CITEMFIELDVALUE_H__
#define __CITEMFIELDVALUE_H__

#include "cgraphics.h"
#include "cobject.h"

// what's commin' next...
class CItemFieldValue;
class CItemFieldValueString;
class CItemFieldValueBoolean;
class CItemFieldValuePixbuf;
class CItemFieldValueCombo;
class CItemFieldValueProgress;
class CItemFieldValuePack;

//-----------------------------------------------------------------------------------------------------------------------------------------
// field values declaration
//-----------------------------------------------------------------------------------------------------------------------------------------
typedef NServices::TBuffer <CItemFieldValue *> CItemFieldValues;

//-----------------------------------------------------------------------------------------------------------------------------------------
// CItemFieldValueAPIListener
//-----------------------------------------------------------------------------------------------------------------------------------------
class CItemFieldValueAPIListener : public CObjectListener
{
	// instanciation section
	public :

		CItemFieldValueAPIListener		 ();
		virtual ~CItemFieldValueAPIListener	 ();

	// generic item field value state modification
	public :

		// called when the CItemFieldValue field value state is modified (only when invoking the field value dedicated API)
		virtual void			OnStateChange		(CObject *inItemFieldValue)	{ }

	// metaclass association
	SECTION_GENERIC_METACLASS;
};

// metaclass and class tag declaration
DECLARE_GENERIC_METACLASS ('_ifv', CItemFieldValueAPIListener, CObjectListener);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CItemFieldValue abstract class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CItemFieldValue : public CObject
{
	// instanciation section
	public :

		CItemFieldValue			(const bool inVisible=true);
		virtual ~CItemFieldValue	() =0;

	// cobject redefinition
	protected :

		virtual const CMetaClass *	ListenerMustBe		() const;

	// public item field value functions
	public :

		// field value owner access
		CObject *			GetOwner		() const;

		// visible field state access
		void				SetVisible		(const bool);
		bool				GetVisible		() const;

	// CSerialized redefinition
	public :

		// item field value xml serialization
		virtual void			Serialize		(CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE;

	// generic operators definition, defined only for polymorphic logical operators properties : not all of the given attributes will
	// have a signication on the derived instances, anyway those concerned should be overloaded
	public :
		
		// nothing done by default on the CItemFieldValue
		virtual CItemFieldValue &	operator =		(const CString &);
		virtual CItemFieldValue &	operator =		(const CItemFieldValueString &);

		// if not overloaded, sets the visible state of the field value
		virtual CItemFieldValue &	operator =		(const bool);
		virtual CItemFieldValue &	operator =		(const CItemFieldValueBoolean &);

		// nothing done by default on the CItemFieldValue
		virtual CItemFieldValue &	operator =		(const float);
		virtual CItemFieldValue &	operator =		(const CItemFieldValueProgress &);

		// nothing done by default on the CItemFieldValue
		virtual CItemFieldValue &	operator =		(const CPixbuf *);

		// significant only on CItemFieldValueBoolean
		friend bool			operator !		(const CItemFieldValue &);

	// protected attribute section
	protected :

		// item field value owner object
		CObject *			m_Owner;

		// visible state of the field value
		bool				m_Visible;

		// friends of us...
		friend class			CTreeView;
		friend class			CTreeViewItem;
		friend class			CTreeViewItemFieldValueAPIListener;
		friend class			CComboBox;
		friend class			CComboBoxItem;
		friend class			CComboBoxItemFieldValueAPIListener;
		friend class			CComboBoxEntry;
		friend class			CIconView;
		friend class			CIconViewItem;
		friend class			CIconViewItemFieldValueAPIListener;

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass declaration
DECLARE_GENERIC_METACLASS ('itfv', CItemFieldValue, CObject);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CItemFieldValuePack abstract class
//-----------------------------------------------------------------------------------------------------------------------------------------
class CItemFieldValuePack : private CItemFieldValue
{
	// instanciation section
	public :

		CItemFieldValuePack		();
		virtual ~CItemFieldValuePack	() =0;

		// metaclass association
		SECTION_GENERIC_METACLASS;
};

// metaclass declaration
DECLARE_GENERIC_METACLASS ('itfk', CItemFieldValuePack, CItemFieldValue);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CItemFieldValueString
//-----------------------------------------------------------------------------------------------------------------------------------------
class CItemFieldValueString : public CItemFieldValue
{
	// instanciation section
	public :

		CItemFieldValueString		(const CString &inFieldValue=CString(), const bool inEnabled=false, 
						 const bool inVisible=true);
		virtual ~CItemFieldValueString	();

	// public item field value functions
	public :

		// string item field value value access
		void				SetFieldValue		(const CString &);
		CString				GetFieldValue		() const;

		// string item field value enable state access
		void				SetEnabled		(const bool);
		bool				GetEnabled		() const;

	// CSerialized redefinition
	public :

		// item field value xml serialization
		virtual void			Serialize		(CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE;

	// operator section
	public :

		// field value affectation
		virtual CItemFieldValue &	operator =		(const CString &);
		virtual CItemFieldValue &	operator =		(const CItemFieldValueString &);

		// enabled/editable state affectation
		virtual CItemFieldValue &	operator =		(const bool);
		virtual CItemFieldValue &	operator =		(const CItemFieldValueBoolean &);

	// protected attributes section
	protected :

		// editable attribute and field value
		bool				m_Enabled;
		CString				m_FieldValue;

		// friends of us...
		friend class			CTreeView;
		friend class			CTreeViewItem;
		friend class			CComboBox;
		friend class			CComboBoxItem;
		friend class			CComboBoxEntry;
		friend class			CIconView;
		friend class			CIconViewItem;

		// metaclass associtation
		SECTION_DYNAMIC_METACLASS;
};

// metaclass declaration
DECLARE_DYNAMIC_METACLASS ('itfs', CItemFieldValueString, CItemFieldValue);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CItemFieldValueBoolean
//-----------------------------------------------------------------------------------------------------------------------------------------
class CItemFieldValueBoolean : public CItemFieldValue
{
	// instanciation section
	public :

		CItemFieldValueBoolean		(const bool inFieldValue=true, const bool inEnabled=true, const bool inVisible=true);
		virtual ~CItemFieldValueBoolean	();

	// public item field value functions
	public :

		// boolean item field value value access
		void				SetFieldValue		(const bool);
		bool				GetFieldValue		() const;

		// string item field value enable state access
		void				SetEnabled		(const bool);
		bool				GetEnabled		() const;

	// CSerialized redefinition
	public :

		// item field value xml serialization
		virtual void			Serialize		(CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE;

	// operator section
	public :

		// field value affectation
		virtual CItemFieldValue &	operator =		(const bool);
		virtual CItemFieldValue &	operator =		(const CItemFieldValueBoolean &);

	// protected attributes section
	protected :

		// activatable state and field value
		bool				m_Enabled;
		bool				m_FieldValue;

		// friends of us...
		friend class			CTreeView;
		friend class			CTreeViewItem;
		friend class			CComboBox;
		friend class			CComboBoxItem;

		// metaclass associtation
		SECTION_DYNAMIC_METACLASS;
};

// metaclass declaration
DECLARE_DYNAMIC_METACLASS ('itfb', CItemFieldValueBoolean, CItemFieldValue);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CItemFieldValueProgress
//-----------------------------------------------------------------------------------------------------------------------------------------
class CItemFieldValueProgress : public CItemFieldValue
{
	// instanciation section
	public :

		CItemFieldValueProgress		(const float inFieldValue=0.0f, const CString &inCaption=CString(),
						 const bool inVisible=true);
		virtual ~CItemFieldValueProgress();

	// public item field value functions
	public :

		// item field value access [0..1]
		void				SetFieldValue		(const float);
		float				GetFieldValue		() const;

		// item considered caption
		void				SetCaption		(const CString &);
		CString				GetCaption		() const;

	// CSerialized redefinition
	public :

		// item field value xml serialization
		virtual void			Serialize		(CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE;
	
	// operator
	public :

		// caption affectation
		virtual CItemFieldValue &	operator =		(const CString &);
		virtual CItemFieldValue &	operator =		(const CItemFieldValueString &);

		// field value affectation
		virtual CItemFieldValue &	operator =		(const float);
		virtual CItemFieldValue &	operator =		(const CItemFieldValueProgress &);

	// protected attributes
	protected :

		// field value, caption
		float				m_FieldValue;
		CString				m_Caption;

		// friends of us...
		friend class			CTreeView;
		friend class			CTreeViewItem;
		friend class			CComboBox;
		friend class			CComboBoxItem;	

		// metaclass associtation
		SECTION_DYNAMIC_METACLASS;
};

// metaclass declaration
DECLARE_DYNAMIC_METACLASS ('itfg', CItemFieldValueProgress, CItemFieldValue);

//-----------------------------------------------------------------------------------------------------------------------------------------
// CItemFieldValuePixbuf
//-----------------------------------------------------------------------------------------------------------------------------------------
class CItemFieldValuePixbuf : public CItemFieldValue
{
	// instanciation section
	public :

		CItemFieldValuePixbuf		(const CPixbuf *inFieldValue=NULL, const bool inVisible=true);
		virtual ~CItemFieldValuePixbuf	();

	// public item field value functions
	public :

		// item field value access
		void				SetFieldValue		(const CPixbuf *);
		CPixbuf *			GetFieldValue		() const;

	// CSerialized redefinition
	public :

		// item field value xml serialization
		virtual void			Serialize		(CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE;

	// operator
	public :

		// field value affectation
		virtual CItemFieldValue &	operator =		(const CPixbuf *);

	// protected attributes section
	protected :

		// field value
		CPixbuf *			m_FieldValue;

		// friends of us...
		friend class			CTreeView;
		friend class			CTreeViewItem;
		friend class			CComboBox;
		friend class			CComboBoxItem;
		friend class			CComboBoxEntry;
		friend class			CIconView;
		friend class			CIconViewItem;

		// metaclass associtation
		SECTION_DYNAMIC_METACLASS;
};

// metaclass declaration
DECLARE_DYNAMIC_METACLASS ('itfp', CItemFieldValuePixbuf, CItemFieldValue);







/*************************************************
  	!!! SECTION NOT HANDLED YET !!!
**************************************************/





//-----------------------------------------------------------------------------------------------------------------------------------------
// CItemFieldValueCombo
//-----------------------------------------------------------------------------------------------------------------------------------------
class CItemFieldValueCombo : public CItemFieldValue
{
/*
	// instanciation section
	public :

		CItemFieldValueCombo		(const CItemFieldValues &inFieldValues=CItemFieldValues(1,new CItemFieldValueString()),
						 const bool inEntry=false, const bool inVisible=true) THROWABLE;
		virtual ~CItemFieldValueCombo	();

	// CSerialized redefinition
	public :

		// item field value xml serialization
		virtual void			Serialize		(CXMLElementNode *&ioXMLElementNode, const int inMode) THROWABLE;

	// protected attributes
	protected :

		// combo handled model and attributes
		bool				m_Entry;
		CMetaClasses			m_Model;
		CItemFieldValues		m_FieldValues;
		GtkListStore *			m_GtkListStore;

		// friends of us...
		friend class			CTreeView;
		friend class			CTreeViewItem;
*/
		// metaclass associtation
		SECTION_DYNAMIC_METACLASS;
};

// metaclass declaration
DECLARE_DYNAMIC_METACLASS ('itfc', CItemFieldValueCombo, CItemFieldValue);

//-----------------------------------------------------------------------------------------------------------------------------------------
// metaclasses list definition, for commodity, what's used on models for identification
//-----------------------------------------------------------------------------------------------------------------------------------------
const static CMetaClasses _IFVString_ 	(1, __metaclass(CItemFieldValueString));
const static CMetaClasses _IFVBoolean_ 	(1, __metaclass(CItemFieldValueBoolean));
const static CMetaClasses _IFVProgress_	(1, __metaclass(CItemFieldValueProgress));
const static CMetaClasses _IFVPixbuf_ 	(1, __metaclass(CItemFieldValuePixbuf));
const static CMetaClasses _IFVCombo_ 	(1, __metaclass(CItemFieldValueCombo));
const static CMetaClasses _IFVPack_ 	(1, __metaclass(CItemFieldValuePack));

#endif
