/*
 * libgksuui -- Gtk+ widget and convenience functions for requesting passwords
 * Copyright (C) 2004 Gustavo Noronha Silva
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "defines.h"
#include "../config.h"

#include "gksuui-dialog.h"

static void
gksuui_dialog_class_init (GksuuiDialogClass *klass);

static void
gksuui_dialog_init (GksuuiDialog *gksuui_dialog);

GType
gksuui_dialog_get_type (void)
{
  static GType type = 0;

  if (type == 0)
    {
      static const GTypeInfo info =
	{
	  sizeof (GksuuiDialogClass), /* size of class */
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc) gksuui_dialog_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof (GksuuiDialog), /* size of object */
	  0, /* n_preallocs */
	  (GInstanceInitFunc) gksuui_dialog_init /* instance_init */
	};
      type = g_type_register_static (gtk_dialog_get_type (),
				     "GksuuiDialogType",
				     &info, 0);
    }

  return type;
}

static void
gksuui_dialog_class_init (GksuuiDialogClass *klass)
{
}

static void
gksuui_dialog_init (GksuuiDialog *gksuui_dialog)
{
  GtkDialog *dialog;
  GtkWidget *hbox; /* aditional hbox for 'password: entry' label */

  /* 
     make sure we're using UTF-8 and getting our locale files
     from the right place
  */
  bindtextdomain(PACKAGE_NAME, LOCALEDIR);
  bind_textdomain_codeset (PACKAGE_NAME, "UTF-8");
  
  gtk_widget_push_composite_child ();

  /* dialog window */
  dialog = GTK_DIALOG(gksuui_dialog);

  gksuui_dialog->main_vbox = dialog->vbox;

  gtk_window_set_title (GTK_WINDOW(gksuui_dialog), "");
  gtk_dialog_set_has_separator (GTK_DIALOG(gksuui_dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER(gksuui_dialog), 6);
  gtk_box_set_spacing (GTK_BOX(gksuui_dialog->main_vbox), 12);

  /* center window */
  gtk_window_set_default_size (GTK_WINDOW(gksuui_dialog), 250, 100);
  gtk_window_set_position (GTK_WINDOW(gksuui_dialog), GTK_WIN_POS_CENTER);

  /* the action buttons */
  /*  the cancel button  */
  gksuui_dialog->cancel_button = gtk_dialog_add_button (dialog,
						      GTK_STOCK_CANCEL,
						      GTK_RESPONSE_CANCEL);
  /*  the ok button  */
  gksuui_dialog->ok_button = gtk_dialog_add_button (dialog,
						  _("Continue"),
						  GTK_RESPONSE_OK);
  gtk_widget_grab_default (gksuui_dialog->ok_button);


  /* hbox */
  gksuui_dialog->hbox = gtk_hbox_new (FALSE, 12);
  gtk_container_set_border_width (GTK_CONTAINER(gksuui_dialog->hbox), 6);
  gtk_box_pack_start (GTK_BOX(gksuui_dialog->main_vbox),
		      gksuui_dialog->hbox, TRUE, TRUE, 0);
  gtk_widget_show (gksuui_dialog->hbox);

  /* image */
  gksuui_dialog->image = 
    gtk_image_new_from_stock (GTK_STOCK_DIALOG_AUTHENTICATION,
			      GTK_ICON_SIZE_DIALOG);
  gtk_misc_set_alignment (GTK_MISC(gksuui_dialog->image), 0.5, 0);
  gtk_box_pack_start (GTK_BOX(gksuui_dialog->hbox), gksuui_dialog->image, 
		      FALSE, FALSE, 0);
  gtk_widget_show (gksuui_dialog->image);

  /* vbox for label and entry */
  gksuui_dialog->entry_vbox = gtk_vbox_new (FALSE, 12);
  gtk_box_pack_start (GTK_BOX(gksuui_dialog->hbox), gksuui_dialog->entry_vbox, 
		      TRUE, TRUE, 0);
  gtk_widget_show (gksuui_dialog->entry_vbox);

  /* label */
  gksuui_dialog->label = gtk_label_new (_("<span weight=\"bold\" size=\"larger\">"
					  "Type the root password.</span>\n"));
  gtk_label_set_justify (GTK_LABEL(gksuui_dialog->label), 
			 GTK_JUSTIFY_CENTER);
  gtk_label_set_use_markup (GTK_LABEL(gksuui_dialog->label), TRUE);
  gtk_label_set_line_wrap (GTK_LABEL(gksuui_dialog->label), TRUE);
  gtk_misc_set_alignment (GTK_MISC(gksuui_dialog->label), 0.0, 0);
  gtk_box_pack_start (GTK_BOX(gksuui_dialog->entry_vbox), 
		      gksuui_dialog->label, TRUE, TRUE, 0);
  gtk_widget_show (gksuui_dialog->label);

  /* hbox for entry and label */
  hbox = gtk_hbox_new (FALSE, 6);
  gtk_box_pack_start (GTK_BOX (gksuui_dialog->entry_vbox), hbox, 
		      TRUE, TRUE, 0);
  gtk_widget_show (hbox);

  /* entry label */
  gksuui_dialog->prompt_label = gtk_label_new (_("Password:"));
  gtk_box_pack_start (GTK_BOX(hbox), gksuui_dialog->prompt_label, 
		      FALSE, FALSE, 0);
  gtk_widget_show (gksuui_dialog->prompt_label);
  
  /* entry */
  gksuui_dialog->entry = gtk_entry_new();
  g_signal_connect_swapped (G_OBJECT(gksuui_dialog->entry), "activate",
			    G_CALLBACK(gtk_button_clicked), 
			    gksuui_dialog->ok_button);
  gtk_entry_set_visibility(GTK_ENTRY(gksuui_dialog->entry), FALSE);
  gtk_entry_set_invisible_char(GTK_ENTRY(gksuui_dialog->entry), 0x25cf); 
  gtk_box_pack_start (GTK_BOX (hbox), gksuui_dialog->entry, 
		      TRUE, TRUE, 0);
  gtk_widget_show (gksuui_dialog->entry);
  gtk_widget_grab_focus(gksuui_dialog->entry);

  gtk_widget_pop_composite_child ();
}

/**
 * gksuui_dialog_new:
 * 
 * Creates a new #GksuuiDialog.
 *
 * Returns: the new #GksuuiDialog
 */
GtkWidget*
gksuui_dialog_new (void)
{
  return GTK_WIDGET (g_object_new (GKSUUI_TYPE_DIALOG, NULL));
}

/**
 * gksuui_dialog_set_message:
 * @dialog: the dialog on which to set the message
 * @message: the message to be set on the dialog
 * 
 * Sets the message that is displayed to the user when
 * requesting a password. You can use Pango markup to
 * modify font attributes.
 *
 */
void
gksuui_dialog_set_message (GksuuiDialog *dialog, gchar *message)
{
  GtkWidget *label = dialog->label;

  gtk_label_set_markup (GTK_LABEL(label), message);
}

/**
 * gksuui_dialog_get_message:
 * @dialog: the dialog from which to get the message
 * 
 * Gets the current message that the dialog will use
 * when run.
 *
 * Returns: a pointer to the string containing the
 * message. You need to make a copy of the string to
 * keep it.
 */
const gchar*
gksuui_dialog_get_message (GksuuiDialog *dialog)
{
  GtkWidget *label = dialog->label;

  return gtk_label_get_text (GTK_LABEL(label));
}

/**
 * gksuui_dialog_set_icon:
 * @dialog: the dialog on which the icon will be set
 * @icon: a #GdkPixbuf from which to set the image
 *
 * Sets the icon that will be shown on the dialog. Should
 * probably not be used, as the default icon is the default
 * authorization icon.
 */
void
gksuui_dialog_set_icon (GksuuiDialog *dialog, GdkPixbuf *icon)
{
  GtkWidget *image = dialog->image;

  gtk_image_set_from_pixbuf (GTK_IMAGE(image), icon);
}

/**
 * gksuui_dialog_get_icon:
 * @dialog: the dialog from which the icon should be
 * got
 *
 * Gets the #GtkImage which is currently defined as the
 * icon for the authorization dialog.
 *
 * Returns: a #GtkWidget which is the #GtkImage
 */
GtkWidget*
gksuui_dialog_get_icon (GksuuiDialog *dialog)
{
  return dialog->image;
}

/**
 * gksuui_dialog_get_password:
 * @dialog: the dialog from which to get the message
 * 
 * Gets the password typed by the user on the dialog.
 * This is a convenience function to grab the password
 * easily from the dialog after calling gtk_dialog_run ()
 *
 * Returns: a newly allocated string containing the password
 */
gchar*
gksuui_dialog_get_password (GksuuiDialog *dialog)
{
  GtkEditable *entry = GTK_EDITABLE(dialog->entry);

  return gtk_editable_get_chars (entry, 0, -1);
}

/**
 * gksuui_dialog_set_prompt:
 * @dialog: the dialog on which to set the prompt
 * @prompt: the prompt to be set on the dialog
 * 
 * Sets the prompt that is displayed to the user when
 * requesting a password. You can use Pango markup to
 * modify font attributes.
 *
 */
void
gksuui_dialog_set_prompt (GksuuiDialog *dialog, gchar *prompt)
{
  GtkWidget *label = dialog->prompt_label;

  gtk_label_set_markup (GTK_LABEL(label), prompt);
}

/**
 * gksuui_dialog_get_prompt:
 * @dialog: the dialog from which to get the prompt
 * 
 * Gets the current prompt that the dialog will use
 * when run.
 *
 * Returns: a pointer to the string containing the
 * prompt. You need to make a copy of the string to
 * keep it.
 */
const gchar*
gksuui_dialog_get_prompt (GksuuiDialog *dialog)
{
  GtkWidget *label = dialog->prompt_label;

  return gtk_label_get_text (GTK_LABEL(label));
}
