/*******************************************************************************************************************************************
 cfile.h
*******************************************************************************************************************************************/

#include "cfile.h"

//------------------------------------------------------------------------------------------------------------------------------------------
// metaclass resolution
//------------------------------------------------------------------------------------------------------------------------------------------
RESOLVE_CAPSULE_METACLASS (CFile);

//------------------------------------------------------------------------------------------------------------------------------------------
// constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CFile::CFile      (const CString &inFileName, const CString &inMode)
      :CClass     (),
       m_FileName (inFileName),
       m_File     (NULL),
       m_FileMode (inMode)
{
        m_File = ::fopen (m_FileName.Get(), m_FileMode.Get());
}

//------------------------------------------------------------------------------------------------------------------------------------------
// copy constructor
//------------------------------------------------------------------------------------------------------------------------------------------
CFile::CFile  	  (const CFile &inFile)
      :CClass 	  (),
       m_FileName (inFile.m_FileName),
       m_File	  (NULL),
       m_FileMode (inFile.m_FileMode)
{
	m_File = ::fopen (m_FileName.Get(), m_FileMode.Get());
}

//------------------------------------------------------------------------------------------------------------------------------------------
// destructor
//------------------------------------------------------------------------------------------------------------------------------------------
CFile::~CFile ()
{
	if (m_File != NULL) ::fclose (m_File);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// existence test
//------------------------------------------------------------------------------------------------------------------------------------------
TFile CFile::Exists (const CString &inFileName)
{
	return CFile (inFileName, CString("rb")).GetType ();
}

//------------------------------------------------------------------------------------------------------------------------------------------
// file name
//------------------------------------------------------------------------------------------------------------------------------------------
CString CFile::GetName () const
{
	CString outFileName (m_FileName);
	size_t EndIndex = 0;
	while (outFileName.Find ('/', 0, &EndIndex)) outFileName.Delete (0, EndIndex+1);
	return outFileName;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// path
//------------------------------------------------------------------------------------------------------------------------------------------
CString CFile::GetPath () const
{
	size_t EndIndex = 0, StartIndex = 0;
	while (m_FileName.Find ('/', StartIndex, &EndIndex)) StartIndex = EndIndex + 1; 
	CString outPath (m_FileName);
	outPath.Delete (EndIndex+1, outPath.GetLength()-EndIndex);
	return outPath;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// file type
//------------------------------------------------------------------------------------------------------------------------------------------
TFile CFile::GetType () const
{
	if (m_File == NULL) return FILE_ERROR;
	struct stat inStat;
        ::fstat (::fileno(m_File), &inStat);
	if (S_ISREG(inStat.st_mode)) return FILE_REGULAR;
	if (S_ISDIR(inStat.st_mode)) return FILE_DIRECTORY;
	if (S_ISLNK(inStat.st_mode)) return FILE_LINK;
	return FILE_ERROR;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// the file size
//------------------------------------------------------------------------------------------------------------------------------------------
UInt32 CFile::GetSize () const
{
	if (m_File == NULL) return 0L;
	struct stat inStat;
        ::fstat (::fileno(m_File), &inStat);
        return static_cast <UInt32> (inStat.st_size);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// the file buffer
//------------------------------------------------------------------------------------------------------------------------------------------
FILE * CFile::GetFile () const
{
	return const_cast <FILE *> (m_File);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// file access
//------------------------------------------------------------------------------------------------------------------------------------------
CString CFile::GetMode() const
{
	return m_FileMode;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// write 
//------------------------------------------------------------------------------------------------------------------------------------------
size_t CFile::Write (const void *inData, const size_t inSize, const size_t inLength)
{
	return m_File != NULL ? ::fwrite (const_cast <void *> (inData), inSize, inLength, m_File) : 0;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// read
//------------------------------------------------------------------------------------------------------------------------------------------
size_t CFile::Read (void *outData, const size_t inSize, const size_t inLength)
{
	return m_File != NULL ? ::fread (outData, inSize, inLength, m_File) : 0;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// flush
//------------------------------------------------------------------------------------------------------------------------------------------
void CFile::Flush ()
{
	if (m_File != NULL) ::fflush (m_File);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// seek
//------------------------------------------------------------------------------------------------------------------------------------------
void CFile::Seek (const long inOffset, const int inWhence)
{
	if (m_File != NULL) ::fseek (m_File, inOffset, inWhence);
}

//------------------------------------------------------------------------------------------------------------------------------------------
// tell
//------------------------------------------------------------------------------------------------------------------------------------------
long CFile::Tell () const
{
	return m_File != NULL ? ::ftell (const_cast <FILE *> (m_File)) : -1;
}

//------------------------------------------------------------------------------------------------------------------------------------------
// truncation
//------------------------------------------------------------------------------------------------------------------------------------------
void CFile::Truncate (const long inSize)
{
	if (m_File != NULL) ::ftruncate (::fileno(m_File), inSize);
}
