/**
 * @file libgalago-gtk/galago-gtk-presence-menu-item.c Presence menu item widget
 *
 * @Copyright (C) 2004-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-presence-menu-item.h>
#include <libgalago-gtk/galago-gtk-presence-icon.h>
#include <libgalago-gtk/galago-gtk-presence-label.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <gtk/gtk.h>

struct _GalagoGtkPresenceMenuItemPriv
{
	GalagoAccount *account;
	GalagoPerson *person;

	GtkWidget *label;

	gulong account_destroy_id;
	gulong person_destroy_id;
};

static void galago_gtk_presence_menu_item_class_init(GalagoGtkPresenceMenuItemClass *klass);
static void galago_gtk_presence_menu_item_init(GalagoGtkPresenceMenuItem *menu_item);
static void galago_gtk_presence_menu_item_finalize(GObject *obj);
static void galago_gtk_presence_menu_item_destroy(GtkObject *obj);

static GtkMenuItemClass *parent_class = NULL;

GType
galago_gtk_presence_menu_item_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkPresenceMenuItemClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_presence_menu_item_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkPresenceMenuItem),
			0,
			(GInstanceInitFunc)galago_gtk_presence_menu_item_init
		};

		type = g_type_register_static(GTK_TYPE_IMAGE_MENU_ITEM,
									  "GalagoGtkPresenceMenuItem", &info, 0);
	}

	return type;
}

static void
galago_gtk_presence_menu_item_class_init(GalagoGtkPresenceMenuItemClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	object_class  = GTK_OBJECT_CLASS(klass);

	gobject_class->finalize = galago_gtk_presence_menu_item_finalize;

	object_class->destroy = galago_gtk_presence_menu_item_destroy;
}

static void
galago_gtk_presence_menu_item_init(GalagoGtkPresenceMenuItem *menu_item)
{
	menu_item->priv = g_new0(GalagoGtkPresenceMenuItemPriv, 1);
}

static void
galago_gtk_presence_menu_item_finalize(GObject *obj)
{
	GalagoGtkPresenceMenuItem *menu_item;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_MENU_ITEM(obj));

	menu_item = GALAGO_GTK_PRESENCE_MENU_ITEM(obj);

	g_free(menu_item->priv);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_presence_menu_item_destroy(GtkObject *obj)
{
	GalagoGtkPresenceMenuItem *menu_item;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_MENU_ITEM(obj));

	menu_item = GALAGO_GTK_PRESENCE_MENU_ITEM(obj);

	GALAGO_GTK_DISCONNECT_HANDLER(menu_item->priv->account,
								  menu_item->priv->account_destroy_id);

	menu_item->priv->account = NULL;

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

static void
account_destroy_cb(GalagoPresence *presence,
				   GalagoGtkPresenceMenuItem *menu_item)
{
	galago_gtk_presence_menu_item_set_account(menu_item, NULL);
}

static void
person_destroy_cb(GalagoPresence *presence,
				  GalagoGtkPresenceMenuItem *menu_item)
{
	galago_gtk_presence_menu_item_set_person(menu_item, NULL);
}

GtkWidget *
galago_gtk_presence_menu_item_new(GalagoAccount *account)
{
	GalagoGtkPresenceMenuItem *menu_item;
	GtkWidget *label;

	menu_item = g_object_new(GALAGO_GTK_TYPE_PRESENCE_MENU_ITEM, NULL);

	label = galago_gtk_presence_label_new();
	gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);
	gtk_container_add(GTK_CONTAINER(menu_item), label);
	gtk_widget_show(label);

	menu_item->priv->label = label;

	galago_gtk_presence_menu_item_set_account(menu_item, account);

	return GTK_WIDGET(menu_item);
}

void
galago_gtk_presence_menu_item_set_account(GalagoGtkPresenceMenuItem *menu_item,
										  GalagoAccount *account)
{
	g_return_if_fail(menu_item != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_MENU_ITEM(menu_item));

	if (menu_item->priv->account == account)
		return;

	if (menu_item->priv->person != NULL)
		galago_gtk_presence_menu_item_set_person(menu_item, NULL);

	GALAGO_GTK_DISCONNECT_HANDLER(menu_item->priv->account,
								  menu_item->priv->account_destroy_id);

	menu_item->priv->account = account;

	if (account != NULL)
	{
		GtkWidget *icon;

		icon = galago_gtk_presence_icon_new();
		galago_gtk_presence_icon_set_account(GALAGO_GTK_PRESENCE_ICON(icon),
											 account);
		galago_gtk_presence_icon_set_size(GALAGO_GTK_PRESENCE_ICON(icon),
										  GTK_ICON_SIZE_MENU);
		galago_gtk_presence_icon_set_precedence(GALAGO_GTK_PRESENCE_ICON(icon),
												GALAGO_GTK_ICON_PROTOCOL);

		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menu_item), icon);

		menu_item->priv->account_destroy_id =
			g_signal_connect(G_OBJECT(account), "destroy",
							 G_CALLBACK(account_destroy_cb), menu_item);
	}

	galago_gtk_presence_label_set_account(
		GALAGO_GTK_PRESENCE_LABEL(menu_item->priv->label), account);
}

void
galago_gtk_presence_menu_item_set_person(GalagoGtkPresenceMenuItem *menu_item,
										 GalagoPerson *person)
{
	g_return_if_fail(menu_item != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_MENU_ITEM(menu_item));

	if (menu_item->priv->person == person)
		return;

	if (menu_item->priv->account != NULL)
		galago_gtk_presence_menu_item_set_account(menu_item, NULL);

	GALAGO_GTK_DISCONNECT_HANDLER(menu_item->priv->person,
								  menu_item->priv->person_destroy_id);

	menu_item->priv->person = person;

	if (person != NULL)
	{
		GtkWidget *icon;

		icon = galago_gtk_presence_icon_new();
		galago_gtk_presence_icon_set_person(GALAGO_GTK_PRESENCE_ICON(icon),
											person);
		galago_gtk_presence_icon_set_size(GALAGO_GTK_PRESENCE_ICON(icon),
										  GTK_ICON_SIZE_MENU);
		galago_gtk_presence_icon_set_precedence(GALAGO_GTK_PRESENCE_ICON(icon),
												GALAGO_GTK_ICON_PROTOCOL);

		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menu_item), icon);

		menu_item->priv->person_destroy_id =
			g_signal_connect(G_OBJECT(person), "destroy",
							 G_CALLBACK(person_destroy_cb), menu_item);
	}

	galago_gtk_presence_label_set_person(
		GALAGO_GTK_PRESENCE_LABEL(menu_item->priv->label), person);
}

void
galago_gtk_presence_menu_item_set_dim_idle(
	GalagoGtkPresenceMenuItem *menu_item, gboolean dim_idle)
{
	g_return_if_fail(menu_item != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_MENU_ITEM(menu_item));

	galago_gtk_presence_label_set_dim_idle(
		GALAGO_GTK_PRESENCE_LABEL(menu_item->priv->label), dim_idle);
}

void
galago_gtk_presence_menu_item_set_show_idle_times(
	GalagoGtkPresenceMenuItem *menu_item, gboolean show_idle_times)
{
	g_return_if_fail(menu_item != NULL);
	g_return_if_fail(GALAGO_GTK_IS_PRESENCE_MENU_ITEM(menu_item));

	galago_gtk_presence_label_set_show_idle_times(
		GALAGO_GTK_PRESENCE_LABEL(menu_item->priv->label), show_idle_times);
}

GalagoAccount *
galago_gtk_presence_menu_item_get_account(
	const GalagoGtkPresenceMenuItem *menu_item)
{
	g_return_val_if_fail(menu_item != NULL, NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_MENU_ITEM(menu_item), NULL);

	return menu_item->priv->account;
}

GalagoPerson *
galago_gtk_presence_menu_item_get_person(
	const GalagoGtkPresenceMenuItem *menu_item)
{
	g_return_val_if_fail(menu_item != NULL, NULL);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_MENU_ITEM(menu_item), NULL);

	return menu_item->priv->person;
}

gboolean
galago_gtk_presence_menu_item_get_dim_idle(
	const GalagoGtkPresenceMenuItem *menu_item)
{
	g_return_val_if_fail(menu_item != NULL, FALSE);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_MENU_ITEM(menu_item), FALSE);

	return galago_gtk_presence_label_get_dim_idle(
		GALAGO_GTK_PRESENCE_LABEL(menu_item->priv->label));
}

gboolean
galago_gtk_presence_menu_item_get_show_idle_times(
	const GalagoGtkPresenceMenuItem *menu_item)
{
	g_return_val_if_fail(menu_item != NULL, FALSE);
	g_return_val_if_fail(GALAGO_GTK_IS_PRESENCE_MENU_ITEM(menu_item), FALSE);

	return galago_gtk_presence_label_get_show_idle_times(
		GALAGO_GTK_PRESENCE_LABEL(menu_item->priv->label));
}
