/* vim:sw=4:sts=4
 * Library to use the Gtk2 widget library from Lua 5.1
 * Copyright (C) 2007 Wolfgang Oertl
 *
 * Functions for the widget's meta table.
 *
 * Exported symbols:
 *  luagtk_index
 *  luagtk_newindex
 */

#include "luagtk.h"
#include <lauxlib.h>	    /* luaL_error */
#include <string.h>	    /* strlen, strncmp, strcpy, memset, memcpy */


/**
 * Override?  This can be one defined in the gtk_methods[] array, i.e.
 * implemented in C, or one defined in override.lua; these functions
 * must also be inserted into the gtk table.
 *
 * Input stack: 1=unused, [2]=key, [-2]=dest metatable, [-1]=curr metatable
 * Output stack: same on failure (returns 0), or an additional item (returns 1)
 */
static int _check_override(lua_State *L, const char *name)
{
    lua_getglobal(L, "gtk");	// gtk
    lua_pushstring(L, name);	// gtk name
    lua_rawget(L, -2);		// gtk item/null
    lua_remove(L, -2);		// item/null

    if (lua_isnil(L, -1)) {
	lua_pop(L, 1);
	return 0;
    }

    lua_pushvalue(L, 2);	// mt mt item key
    lua_pushvalue(L, -2);	// mt mt item key item
    lua_rawset(L, -5);		// mt mt item	-- was: 3
    return 1;
}

/**
 * An entry has been found.  Store this fact in the object's meta table,
 * and then handle the entry.
 *
 * Called from: _find_element.
 *
 * Stack: [1]=widget [2]=key ... [-2]=dest mt [-1]=curr mt
 * Output stack: meta entry added.
 */
static int _found_function(lua_State *L, const char *name, struct func_info *fi)
{
    struct meta_entry *me = (struct meta_entry*) lua_newuserdata(L,
	sizeof(*me) + strlen(name) + 1);
    memset(me, 0, sizeof(*me));
    memcpy(&me->fi, fi, sizeof(me->fi));
    me->fi.name = me->name;
    strcpy(me->name, name);
    lua_pushvalue(L, 2);
    lua_pushvalue(L, -2);   // ... dest mt, curr mt, me, key, me
    lua_rawset(L, -5);	    // was:3
    return 2;		    // ... dest mt, curr mt, me
}

/**
 * Look at the current metatable.  If it contains the desired item, copy it
 * into the base meta table and return 2.
 * 
 * Input stack: 1=widget, 2=key, [-2]=dest metatable, [-1]=curr metatable
 *
 * @param L  lua_State
 * @param recursed  true if current metatable is not the widget's metatable.
 */
static int _fe_check_metatable(lua_State *L, int recursed)
{
    lua_pushvalue(L, 2);
    lua_rawget(L, -2);	    // was: 4

    if (lua_isnil(L, -1)) {
	lua_pop(L, 1);
	return 0;
    }

    if (recursed) {
	lua_pushvalue(L, 2);	    // mt mt value key
	lua_pushvalue(L, -2);	    // mt mt value key value    -- was:5
	lua_rawset(L, -5);	    // mt mt value  -- was:3
    }

    // return 2 for meta entry (only this is allowed in the metatable)
    return 2;
}


/**
 * Maybe the element being looked for is an attribute of the Gtk object,
 * i.e. part of its C structure.
 *
 * Stack: [1]=widget [2]=key [-2]=dest mt [-1]=curr mt
 */
static int _fe_check_struct(lua_State *L, const char *attr_name)
{
    const struct struct_elem *se = NULL;
    const struct type_info *si = NULL;

    lua_pushliteral(L, "_struct");
    lua_rawget(L, -2);		    // was:4
    if (!lua_isnil(L, -1)) {
	si = (struct type_info*) lua_topointer(L, -1);
	se = find_attribute(si, attr_name);
    }
    lua_pop(L, 1);

    if (!se)
	return 0;

    /* Found. Create an appropriate meta entry. */
    struct meta_entry *me = (struct meta_entry*) lua_newuserdata(L,
	sizeof(*me) + strlen(attr_name) + 1);
    memset(me, 0, sizeof(*me));

    me->type_idx = si - type_list;
    me->se = se;
    strcpy(me->name, attr_name);
    lua_pushvalue(L, 2);	    // mt mt me key
    lua_pushvalue(L, -2);	    // mt mt me key me
    lua_rawset(L, -5);		    // mt mt me	    // was:3
    return 2;
}

/**
 * The class being looked at may implement interfaces; look for these
 * functions, too.
 *
 * Returns: 0=nothing found, 1=found something
 * Stack: [1]=widget [2]=key ... [-2]=dest mt [-1]=curr mt
 */
static int _fe_check_interfaces(lua_State *L, const char *attr_name)
{
    guint n_interfaces;
    int i, rc=0;
    const char *class_name;
    GType *gtypes;
    char tmp_name[80];
    struct func_info fi;

    /* retrieve the class type, an element of the meta table */
    lua_pushliteral(L, "_gtktype");
    lua_rawget(L, -2);	    // was: 4
    int gtk_type = lua_tonumber(L, -1);
    lua_pop(L, 1);

    gtypes = g_type_interfaces(gtk_type, &n_interfaces);

    for (i=0; i<n_interfaces; i++) {
	class_name = g_type_name(gtypes[i]);

	if (luagtk_make_func_name(tmp_name, sizeof(tmp_name), class_name,
	    attr_name))
	    break;

	// an override might exist - use it.
	if (_check_override(L, tmp_name)) {
	    rc = 1;
	    break;
	}

	if (find_func(tmp_name, &fi)) {
	    rc = _found_function(L, tmp_name, &fi);
	    if (rc == 2) {
		// Try to find the structure for this interface.  This is not
		// always possible, e.g. GtkEditableIface doesn't exist.  If
		// found, writing to it is possible.
		char *iface_name = g_malloc(strlen(class_name) + 10);
		sprintf(iface_name, "%sIface", class_name);
		const struct type_info *ti = find_struct(iface_name, 1);
		g_free(iface_name);	// ok
		struct meta_entry *me = (struct meta_entry*) lua_topointer(L,
		    -1);
		if (ti)
		    me->iface_type_idx = ti - type_list;
		me->iface_type_id = gtypes[i];
	    }
	    break;
	}
    }

    g_free(gtypes); // ok
    return rc;
}


/**
 * Look in the widget's environment for a given key.
 *
 * @luaparam stack[1]  The widget
 * @luaparam stack[2]  Key
 * @luaparam stack[-1]  Metatable of the widget
 * @luareturn  The value, or nothing
 */
static int _fe_check_env(lua_State *L)
{
    lua_getfenv(L, 1);		// ... env
    lua_pushvalue(L, 2);		// ... env k
    lua_rawget(L, -2);		// ... env value/nil
    lua_remove(L, -2);		// ... value/nil
    if (!lua_isnil(L, -1))
	return 1;

    lua_pop(L, 1);			// stack back to how it was
    return 0;
}

/**
 * Look for a function with the desired name.
 *
 * Input stack: [1]=widget [2]=key [-2]=widget's metatable [-1]=curr metatable
 * Returns 0 if not found, >0 otherwise.
 */
static int _fe_check_function(lua_State *L, int recursed)
{
    const char *class_name, *attr_name;
    char tmp_name[80];
    struct func_info fi;

    /* retrieve the class name, an element of the meta table */
    lua_pushliteral(L, "_classname");
    lua_rawget(L, -2);		    // was:4
    class_name = lua_tostring(L, -1);
    lua_pop(L, 1);		    // still has a ref, won't be freed.
    if (!class_name)
	return luaL_error(L, "internal error: metatable of a widget has "
	    "no _classname attribute.");

    /* check for a (not yet mapped) function? */
    attr_name = lua_tostring(L, 2);
    if (luagtk_make_func_name(tmp_name, sizeof(tmp_name), class_name,
	attr_name))
	return 0;

    if (_check_override(L, tmp_name))
	return 1;

    if (find_func(tmp_name, &fi))
	return _found_function(L, tmp_name, &fi);

    /* maybe a gdk_ function for GdkSomething objects? */
    /* XXX - HACK */
    if (!recursed && !strncmp(class_name, "Gdk", 3)) {
	sprintf(tmp_name, "gdk_%s", attr_name);
	if (find_func(tmp_name, &fi))
	    return _found_function(L, tmp_name, &fi);
    }

    return 0;
}


/**
 * Search a widget's class and parent classes for an attribute.
 *
 * @luaparam stack[1] widget
 * @luaparam stack[2] key
 * @luaparam stack[-2] mt1  The widget's metatable
 * @luaparam stack[-1] mt2  Metatable of widget or one of its parents.
 */
static int _fe_recurse(lua_State *L, int must_exist)
{
    int recursed = 0, rc;
    const char *attr_name;
    struct func_info fi;
    char tmp_name[80];

    attr_name = lua_tostring(L, 2);

    for (;;) {
	// may be stored in the metatable from previous lookups (optimization),
	// or an item with arbitrary key stored by the user, possibly
	// overriding functions.
	rc = _fe_check_metatable(L, recursed);
	if (rc)
	    return rc;

	// catch read accesses to structure elements
	rc = _fe_check_struct(L, attr_name);
	if (rc)
	    return rc;

	// may be a function name in the form gtk_some_thing_attr_name
	rc = _fe_check_function(L, recursed);
	if (rc)
	    return rc;

	// check functions of all interfaces of the class
	rc = _fe_check_interfaces(L, attr_name);
	if (rc)
	    return rc;

	// replace [-1] with base class, if any.
	lua_pushliteral(L, "_parent");
	lua_rawget(L, -2);	// was: 4
	if (lua_isnil(L, -1)) {
	    lua_pop(L, 1);
	    break;
	}

	lua_remove(L, -2);	// was: 4
	recursed = 1;
    }

    /* last try - maybe it is g_object_xxx */
    sprintf(tmp_name, "g_object_%s", attr_name);
    if (_check_override(L, tmp_name))
	return 1;
    if (find_func(tmp_name, &fi))
	return _found_function(L, tmp_name, &fi);

    /* Give up.  Note that this is not an error when called from
     * gtk_newindex.  Shows the last checked class. */
    if (must_exist) {
	struct widget *w = (struct widget*) lua_topointer(L, 1);
	luaL_error(L, "[gtk] %s.%s not found.", WIDGET_NAME(w), attr_name);
    }
    return 0;
}




/**
 * Look for a method or attribute of the given object.
 *
 * It handles accesses to methods and attributes found in this class or any
 * base class.  Once the method or attribute has been found, it is inserted
 * into the widget's table to avoid looking it up again.
 *
 * Input Stack: 1=widget, 2=key
 * Output Stack: depends on the return value.
 *
 * @param L  lua_State
 * @param must_exist  Set to 1 to print an error message on failure
 * @return
 *	0	nothing found
 *	1	found an entry or function (returned on the stack)
 *	2	found a meta entry (meta entry returned)
 *	-1	other error
 */
static int _find_element(lua_State *L, int must_exist)
{
    struct widget *w;
    const char *attr_name;

    /* check arguments. */
    if (lua_type(L, 1) != LUA_TUSERDATA)
	return luaL_error(L, "find_element called on something other than "
	    "userdata.\n");

    if (!lua_getmetatable(L, 1))
	return luaL_error(L, "find_element called with a userdata without "
	    "metatable - can't be a widget.\n");

    attr_name = luaL_checkstring(L, 2);

    // can't be NULL - it's not a lightuserdata.
    w = (struct widget*) lua_topointer(L, 1);

    if (!w->p) {
	printf("%s access to %s.%s on NULL object\n",
	    msgprefix, WIDGET_NAME(w), attr_name);
	lua_pop(L, 1);
	return -1;
    }

    // Stack: [1]=w [2]=key [-1]=mt.  Have a look at the widget's environment.
    if (_fe_check_env(L))
	return 1;

    /* Duplicate the metatable; [-2] is the metatable of the widget, and [-1]
     * is the current metatable as we ascend the object hierarchy. */
    lua_pushvalue(L, -1);

    /* stack: 1=widget, 2=key, -2=destination metatable, -1=current metatable */
    return _fe_recurse(L, must_exist);
}





/**
 * Given a pointer to a structure and the description of the desired element,
 * push a value onto the Lua stack with this item.
 *
 * Returns the number of pushed items, i.e. 1 on success, 0 on failure.
 */
static int _push_attribute(lua_State *L, const struct type_info *si,
    const struct struct_elem *se, unsigned char *ptr)
{
    const struct ffi_type_map_t *arg_type;
    const struct type_info *ti;

    ti = type_list + se->type_idx;
    arg_type = ffi_type_map + ti->fundamental_id;
    if (arg_type->struct2lua_idx)
	return ffi_type_struct2lua[arg_type->struct2lua_idx](L, se, ptr);

    return luaL_error(L, "%s unhandled attribute type %s (%s.%s)\n",
	msgprefix, FTYPE_NAME(arg_type), TYPE_NAME(si),
	STRUCT_ELEM_NAME(se));
}

/**
 * After a method has been looked up, this function is called to do the
 * invoction of the corresponding gtk function.
 *
 * Note that this is intended to be used in a closure with the upvalue(1)
 * being a reference to the meta entry for the function call.
 */
static int l_call_func(lua_State *L)
{
    struct meta_entry *me = (struct meta_entry*) lua_topointer(L,
	lua_upvalueindex(1));
    return luagtk_call(L, &me->fi, 1);
}


/**
 * A meta entry is on the top of the stack; use it to retrieve the method
 * pointer or attribute value.
 *
 * Stack: 1=widget, 2=key, 3=dest metatable, 4=current metatable,... meta entry
 */
static int _read_meta_entry(lua_State *L)
{
    /* an override -- just return it */
    if (lua_iscfunction(L, -1) || lua_isfunction(L, -1))
	return 1;

    /* For functions, set up a c closure with one upvalue, which is the pointer
     * to the meta entry. */
    const struct meta_entry *me = lua_topointer(L, -1);
    if (me->type_idx == 0) {
	lua_pushlightuserdata(L, (void*) me);
	lua_pushcclosure(L, l_call_func, 1);
	return 1;
    }

    /* otherwise, handle attribute access */
    struct widget *w = (struct widget*) lua_topointer(L, 1);
    return _push_attribute(L, type_list + me->type_idx, me->se, w->p);
}


/**
 * __index function for the metatable used for userdata (widgets).  This is
 * to access a method or an attribute of the class, or a value stored by
 * the user with an arbitrary key.
 *
 * @luaparam widget  The Lua object (a userdata of type struct widget) to
 *   examine
 * @luaparam key  The key to look up
 * @luareturn  The resulting object, or nothing in case of failure.  It can
 *   be a function, a meta entry describing a structure element, or any
 *   object that the user stored.
 */
int luagtk_index(lua_State *L)
{
    int rc;

    rc = _find_element(L, 1);

    /* Stack: 1=widget, 2=key, 3=metatable, 4=metatable,
     * 5=func or meta entry (if found) */
    switch (rc) {
	case 0:
	case 1:
	    return rc;
	
	case 2:
	    /* meta entry */
	    return _read_meta_entry(L);
	
	default:
	    printf("%s invalid return code %d from find_element\n", msgprefix,
		rc);
	    return 0;
    }
}

/**
 * Try to overwrite a function, which is possible if it is in the virtual
 * table of an interface.
 */
static int _try_overwrite_function(lua_State *L, int index)
{
    const struct meta_entry *me = lua_topointer(L, -1);
    struct widget *w = (struct widget*) lua_topointer(L, 1);
    const char *name = lua_tostring(L, 2);

    // only virtual functions of an interface can be set.
    if (!me->iface_type_idx)
	return luaL_error(L, "%s overwriting method %s.%s not supported.",
	    msgprefix, WIDGET_NAME(w), name);

    const struct type_info *si = type_list + me->iface_type_idx;
    const struct struct_elem *se = find_attribute(si, name);
    if (G_UNLIKELY(!se))
	return luaL_error(L, "%s attribute %s.%s not found",
	    msgprefix, TYPE_NAME(si), name);

    // type_idx is actually the offset into struct_strings for the
    // function signature.
    void *func = luagtk_make_closure(L, index,
	(const unsigned char*) TYPE_NAME(type_list + se->type_idx));
    lua_pushlightuserdata(L, func);

    const struct type_info *ti = type_list + se->type_idx;
    const struct ffi_type_map_t *arg_type = ffi_type_map + ti->fundamental_id;
    if (!arg_type->lua2struct_idx)
	return luaL_error(L, "%s can't set closure %s.%s - not implemented.",
	    msgprefix, TYPE_NAME(si), name);

    // set the function pointer in the object's interface table.
    void *p = G_TYPE_INSTANCE_GET_INTERFACE(w->p, me->iface_type_id, void);
    ffi_type_lua2struct[arg_type->lua2struct_idx](L, si, se, p, -1);
    lua_pop(L, 1);

    // not used.
    return 0;
}


/**
 * Assignment to an attribute of a structure.  Must not be a built-in
 * method, but basically could be...
 * Stack: 1=widget, 2=key, ... [-1]=meta entry
 *
 * @param index  Lua stack position where the value is at
 */
static int _write_meta_entry(lua_State *L, int index)
{
    const struct meta_entry *me = lua_topointer(L, -1);
    struct widget *w = (struct widget*) lua_topointer(L, 1);

    /* the meta entry must describe a structure element, not a method. */
    if (G_UNLIKELY(me->type_idx == 0))
	return _try_overwrite_function(L, index);

    /* write to attribute using a type-specific handler */
    const struct type_info *ti = type_list + me->se->type_idx;
    const struct ffi_type_map_t *arg_type = ffi_type_map + ti->fundamental_id;
    if (arg_type->lua2struct_idx)
	return ffi_type_lua2struct[arg_type->lua2struct_idx](L,
	    type_list + me->type_idx, me->se, w->p, index);

    /* no write operation defined for this type */
    return luaL_error(L, "%s can't write %s.%s (unsupported type %s)",
	msgprefix, WIDGET_NAME(w), STRUCT_ELEM_NAME(me->se),
	FTYPE_NAME(arg_type));
}



/**
 * Set existing attributes of an object, or arbitrary values.
 * The environment of the userdata will be used to store additional values.
 *
 * Input stack: 1=widget, 2=key, 3=value
 */
int luagtk_newindex(lua_State *L)
{
    /* check parameters */
    if (lua_gettop(L) != 3) {
	printf("%s gtk_newindex not called with 3 parameters\n", msgprefix);
	return 0;
    }

    /* Is this an attribute of the underlying object? */
    int rc = _find_element(L, 0);

    switch (rc) {
	case -1:
	    return 0;

	case 2:
	    _write_meta_entry(L, 3);
	    return 0;
    }

    /* Not found, or existing entry in the object's environment table.  In both
     * cases store the value in the environment table. */

    lua_getfenv(L, 1);				// w k v env

    /* Is this the default empty table?  If so, create a new one private to
     * this object. */
    lua_getglobal(L, "gtk");
    lua_getfield(L, -1, LUAGTK_EMPTYATTR);	// w k v env gtk ea
    if (lua_equal(L, -1, -3)) {
	lua_newtable(L);			// w k v env gtk ea t
	lua_pushvalue(L, -1);			// w k v env gtk ea t t
	lua_setfenv(L, 1);			// w k v env gtk ea t
    } else {
	lua_pop(L, 2);				// w k v env
    }

    /* the top of the stack now has the table where to put the data */
    lua_replace(L, 1);				// env k v [...]
    lua_settop(L, 3);				// env k v
    lua_rawset(L, 1);				// env 

    return 0;
}


