/* vim:sw=4:sts=4
 * Library to use the Gtk2 widget library from Lua 5.1
 * Copyright (C) 2007 Wolfgang Oertl
 *
 * Handle Gtk widgets.
 *
 * Exported symbols:
 *   luagtk_get_widget
 *   luagtk_check_widget
 */

#include "luagtk.h"
#include <lauxlib.h>	    // luaL_checktype
#include <string.h>

static int _make_widget(lua_State *L, void *p, int type_idx, int flags);
static int _get_widget_meta(lua_State *L, int type_idx);

/**
 * The Lua stack should contain a proxy object at the given stack position,
 * verify this.
 *
 * @param L  Lua state
 * @param index  Stack position
 * @return  Pointer to the widget, or NULL on error.
 */
struct widget *luagtk_check_widget(lua_State *L, int index)
{
    // must be a userdata
    if (lua_type(L, index) != LUA_TUSERDATA)
	return NULL;

    // must have a metatable
    lua_getmetatable(L, index);
    if (lua_isnil(L, -1)) {
	lua_pop(L, 1);
	return NULL;
    }

    // the metatable must have this entry
    lua_getfield(L, -1, "_classname");
    if (lua_isnil(L, -1)) {
	lua_pop(L, 2);
	return NULL;
    }

    lua_pop(L, 2);
    return (struct widget*) lua_topointer(L, index);
}


// update/remove the entry in widgets
static void _set_widget_pointer(lua_State *L, void *p, int ref, int old_ref)
{
    lua_getglobal(L, "gtk");
    lua_getfield(L, -1, LUAGTK_WIDGETS);	// gtk gtk.widgets

    // check that the entry in widgets currently points to old_ref.  If not,
    // don't update.
    if (old_ref) {
	lua_pushlightuserdata(L, p);
	lua_rawget(L, -2);
	if (lua_tointeger(L, -1) != old_ref) {
	    /*
	    fprintf(stderr, "NOT setting widget[%p] = %d (%d != %d)\n", p, ref,
		lua_tointeger(L, -1), old_ref);
	    */
	    lua_pop(L, 3);
	    return;
	}
	lua_pop(L, 1);
    }

    lua_pushlightuserdata(L, p);
    if (ref == 0)
	lua_pushnil(L);
    else
	lua_pushinteger(L, ref);
    lua_rawset(L, -3);
    lua_pop(L, 2);
}

/**
 * Get the ref_nr for the widget at the address "p".
 *
 * @return  the reference_nr, or -1 if not found.
 */
static int _get_widget_ref(lua_State *L, void *p)
{
    int ref_nr = -1;

    lua_getglobal(L, "gtk");			// gtk
    lua_getfield(L, -1, LUAGTK_WIDGETS);	// gtk gtk.widgets
    lua_pushlightuserdata(L, p);		// gtk gtk.widgets p
    lua_rawget(L, -2);				// gtk gtk.widgets ref
    if (!lua_isnil(L, -1))
	ref_nr = lua_tonumber(L, -1);
    lua_pop(L, 3);				// stack empty again

    return ref_nr;
}


/**
 * A Lua widget is to be garbage collected, but it is not the only entry for
 * this memory location.  Remove this one from the circular list.
 *
 * @param w       The widget to release
 *
 * w2: widget being looked at
 * w3: the one before w2
 */
static void _alias_unlink(lua_State *L, struct widget *w)
{
    struct widget *w2 = w;
    int curr_ref, have_ref = 0;

    // what is the current reference in widgets?
    curr_ref = _get_widget_ref(L, w->p);

    // Find the item w2 of the circular list, which is just before w.  This
    // involves walking the whole list until w2->next == w.  At the same time
    // check whether any of the items on the list currently holds the reference
    // from gtk.widgets.
    for (;;) {
	if (w2->own_ref == curr_ref)
	    have_ref = 1;
	if (w2->next == w)
	    break;
	w2 = w2->next;
    }

    // remove w from the list
    w2->next = (w->next == w2) ? NULL : w->next;

    // If this group "owns" the entry in gtk.widgets, set it to w2, if it
    // currently points to w.  Note that w2 might not be present in
    // gtk.widget_aliases either due to GC.
    if (have_ref && w2->own_ref != curr_ref)
	_set_widget_pointer(L, w2->p, w2->own_ref, 0);
}


/**
 * When a Lua object is garbage collected, decrease the reference count of
 * the associated Gtk object, too.  This may cause the Gtk object to be freed.
 *
 * Note that for a given Gtk object, multiple Lua objects can exist, having
 * different types (which should be related to each other, of course).  Each
 * such Lua proxy object can have multiple Lua references, but holds only
 * one Gtk reference.
 */
static int l_widget_gc(struct lua_State *L)
{
    struct widget *w = (struct widget*) lua_topointer(L, 1);

    // sanity check
    if (!w) {
	printf("%s Error: l_widget_gc on a NULL pointer\n", msgprefix);
	return 0;
    }

    // The pointer must not be NULL, unless it is deleted.
    if (!w->p && !w->is_deleted) {
	printf("%s Error: l_widget_gc: pointer is NULL (%p, %s)\n",
	    msgprefix, w, WIDGET_NAME(w));
	return 0;
    }

    // optionally show some debugging info
    if (G_UNLIKELY(runtime_flags & RUNTIME_DEBUG_MEMORY)) {
	// find the entry in widgets
	int ref_nr = _get_widget_ref(L, w->p);
	int ref_count = luagtk_get_refcount(w);
	struct widget_type *wt = luagtk_get_widget_type(w);

	// Lua object address - Gtk object address - widget_type -
	//   current reference counter - class name -
	//   reference in gtk.widget_aliases for this Lua object -
	//   ref of next alias (if applicable) -
	//   reference for the address w->p in gtk.widgets
	fprintf(stderr, "%p %p %5d GC %s refcnt=%d %s - %d %d\n", w, w->p,
	    w->own_ref,
	    wt->name, ref_count, WIDGET_NAME(w),
	    w->next ? w->next->own_ref : 0, ref_nr);
    }

    // If other aliases exist, remove this one from the linked list; otherwise,
    // unset the entry in gtk.widgets.
    if (w->next)
	_alias_unlink(L, w);

    // w->own_ref is 0 for stack objects, which don't have an entry in
    // gtk.widgets anyway.
    else if (w->own_ref)
	_set_widget_pointer(L, w->p, 0, w->own_ref);

    // decrease the refcount of the Gtk/Gdk object
    luagtk_dec_refcount(w);
    return 0;
}


/**
 * A meta table for a Gtk class has been created on the stack.  Now
 * try to find the Gtk base class and call _get_widget_meta for it, too.
 *
 * Input stack: metaclass
 * Output stack: metaclass
 *
 * @return 1 on success, 0 otherwise
 */
static int _get_widget_meta_parent(lua_State *L, GType type_nr)
{
    const char *parent_name;
    const struct type_info *sptr2;
    GTypeQuery query;
    query.type_name = NULL;
    int rc, type_idx;

    /* determine the name of the parent class, if any */
    type_nr = g_type_parent(type_nr);
    if (!type_nr)
	return 1;

    parent_name = g_type_name(type_nr);
    if (!parent_name) {
	fprintf(stderr, "%s Unknown GType %ld\n", msgprefix, (long int)type_nr);
	return 1;
    }

    /* Get LuaGtk description of this structure.  It might not exist, as
     * abstract, empty base classes like GInitiallyUnowned or GBoxed are
     * not known to LuaGtk. */
    sptr2 = find_struct(parent_name, 0);
    if (!sptr2)
	return 1;

    type_idx = sptr2 - type_list;
    rc = _get_widget_meta(L, type_idx);
    if (rc == 1) {
	/* add _parent -- used by gtk_index */
	lua_pushliteral(L, "_parent");	    // meta parentmeta name
	lua_insert(L, -2);		    // meta name parentmeta
	lua_rawset(L, -3);		    // meta
    }

    return 1;
}

/**
 * Test two widgets for equality.  As just a single proxy object should
 * exist for a given widget, this shouldn't be required; but in certain
 * situations this could happen, so here it is.
 */
static int l_widget_compare(lua_State *L)
{
    struct widget *w1 = (struct widget*) lua_topointer(L, 1);
    struct widget *w2 = (struct widget*) lua_topointer(L, 2);
    return w1->p == w2->p;
}

static const luaL_reg widget_methods[] = {
    { "__index",    luagtk_index },
    { "__newindex", luagtk_newindex },
    { "__tostring", luagtk_tostring },
    { "__gc",	    l_widget_gc },
    { "__eq",	    l_widget_compare },
    { NULL, NULL }
};

/**
 * Given the structure nr, retrieve or create the metaclass for this type of
 * widget.  If the given widget type has a base class, recurse to make that,
 * too.
 *
 * Stack input: nothing
 * Returns: 0 on error, or 1 on success.
 * Stack output: on success: the metaclass; otherwise, nothing.
 */
static int _get_widget_meta(lua_State *L, int type_idx)
{
    const struct type_info *ti = STRUCT_INFO(type_idx);
    const char *type_name = TYPE_NAME(ti);

    lua_getglobal(L, "gtk");
    lua_getfield(L, -1, LUAGTK_METATABLES);
    lua_remove(L, -2);				// _meta_tables
    lua_pushstring(L, type_name);		// _meta_tables name
    lua_rawget(L, -2);				// _meta_tables meta|nil

    if (!lua_isnil(L, -1)) {
	lua_remove(L, -2);			// meta
	return 1;
    }
    lua_pop(L, 1);				// _meta_tables

    /* The meta table for this structure (i.e. class) doesn't exist yet.
     * Create it with __index, _classname, _struct, _parent, and store
     * in _meta_tables. */
    lua_newtable(L);				// _meta_tables t
    lua_pushstring(L, type_name);		// _meta_tables t name
    lua_pushvalue(L, -2);			// _meta_tables t name t
    lua_rawset(L, -4);				// _meta_tables t
    lua_remove(L, -2);				// t

    luaL_register(L, NULL, widget_methods);

    /* store the structure number and the class name */
    lua_pushliteral(L, "_struct");
    lua_pushlightuserdata(L, (void*) ti);
    lua_rawset(L, -3);
    lua_pushliteral(L, "_classname");
    lua_pushstring(L, type_name);
    lua_rawset(L, -3);

    /* Determine GTk type number.  Gdk classes like GdkEvent are not found. */
    GType type_nr = luagtk_g_type_from_name(type_name);
    if (!type_nr)
	return 1;

    lua_pushliteral(L, "_gtktype");
    lua_pushnumber(L, type_nr);
    lua_rawset(L, -3);

    return _get_widget_meta_parent(L, type_nr);
}



/**
 * A widget for a given Gtk object (identified by its address) has been found.
 * Check the address, and the type.
 *
 * Returns:
 *  0 ... success
 *  1 ... error (NIL on top of stack, return that)
 *  2 ... type mismatch; need to create new Lua widget
 *
 * Lua stack:
 *  gtk.widgets gtk.widgets_aliases w
 *
 * w may be replaced with another widget (alias), but otherwise the Lua stack
 * remains unchanged.
 */
static int _get_widget_check(lua_State *L, void *widget, int type_idx)
{
    struct widget *w = (struct widget*) lua_topointer(L, -1), *w_start;

    if (!w) {
	printf("%p ERROR: _get_widget_check with nil\n", w);
	return 1;
    }

    w_start = w;
    do {
	// internal check
	if (w->p != widget)
	    return luaL_error(L, "%s internal error: Lua widget %p should "
		"point to %p, but points to %p", msgprefix, w, widget, w->p);

	// don't care about the type?  Always OK
	if (!type_idx)
	    return 0;

	/* Verify that the type matches.  It is possible to have different
	 * widget types at the same address, e.g. GdkEvent. */
	if (type_idx == w->type_idx)
	    return 0;

	// no chained next item - failure
	w = w->next;
	if (!w)
	    break;

	lua_pop(L, 1);
	lua_rawgeti(L, -1, w->own_ref);
    } while (w != w_start);

    // No more chained entries exist.  Return the error to luagtk_get_widget,
    // which can then choose to add another alias.
    return 2;
}


/**
 * Determine whether p points to something on the stack.
 *
 * I don't want to create reusable Lua proxy objects for Gtk/Gdk objects on the
 * stack.  Such objects are usually given to callbacks (e.g. a GdkEvent), don't
 * have refcounting, and will go away when returning from the callback.
 *
 * The user must still take care not to keep the Lua proxy object around.
 * Accessing the same address again (with luagtk_get_widget) will return a new
 * Lua object.
 *
 * @param p  Pointer to a memory location
 * @return  true if p is on the stack
 */
static inline int _is_on_stack(void *p)
{
    volatile char c[30];    // large enough so it can't be in registers.
    long int ofs = ((char*)p) - c;
    if (ofs < 0)
	ofs = -ofs;
    return ofs < 36000;
}


/**
 * A widget has been found for the given address.  It is on the Lua stack;
 * check that it matches the requested type.  If not, make a new alias.
 *
 * Lua stack: [-3]widgets [-2]widget_aliases [-1]w
 */
static void _reuse_widget(lua_State *L, void *p, int type_idx, int flags)
{
    // match, or complete failure -> return.  Doesn't change Lua stack.
    if (_get_widget_check(L, p, type_idx) != 2)
	return;

    // This widget obviously already exists, so unset FLAG_NEW_OBJECT.
    int w2_ref = _make_widget(L, p, type_idx, flags & ~FLAG_NEW_OBJECT);

    if (G_UNLIKELY(w2_ref < 0)) {
	lua_pop(L, 1);			    // replace w with nil
	lua_pushnil(L);
	return;
    }
	
    // widgets widget_aliases w1 w2
    if (w2_ref > 0) {
	// the old widget, that already existed (with the wrong type)
	struct widget *w1 = (struct widget*) lua_topointer(L, -2);

	// this is the new widget
	struct widget *w2 = (struct widget*) lua_topointer(L, -1);

	// add to circular list
	w2->next = w1->next ? w1->next : w1;
	w1->next = w2;

	if (G_UNLIKELY(runtime_flags & RUNTIME_DEBUG_MEMORY))
	    fprintf(stderr, "%p %p alias %s for %p %s\n",
		w2, w2->p, WIDGET_NAME(w2), w1, WIDGET_NAME(w1));
    }

    lua_remove(L, -2);				// widgets w_a w2
}



/**
 * Create or find a Lua proxy object for the widget or structure at the given
 * address to luagtk for later usage in Lua scripts.  If this widget was
 * already registered with a different type_idx, create a new alias.
 *
 * @param L  Lua state
 * @param p  Pointer to the widget or structure
 * @param type_idx  Type of the widget at *p, or 0 for auto detection.
 * @param flags  FLAG_NEW_OBJECT if this is a newly allocated/created object;
 *	FLAG_ALLOCATED if this object was created by g_slice_alloc and doesn't
 *	have refcounting.
 *
 * If we already have a Lua object for this widget, do NOT increase the
 * refcount of the widget.  Only the Lua object has a new reference.
 *
 * You can call this function from C code to make existing Gtk widgets
 * available to Lua code: luagtk_get_widget(L, widget_ptr, 0, 0);
 */
void luagtk_get_widget(lua_State *L, void *p, int type_idx, int flags)
{
    // NULL pointers are turned into nil.
    if (!p) {
	lua_pushnil(L);
	return;
    }

    // translate the address to a reference in the aliases table
    lua_getglobal(L, "gtk");
    lua_getfield(L, -1, LUAGTK_WIDGETS);
    lua_getfield(L, -2, LUAGTK_ALIASES);	// gtk gtk.widgets gtk.aliases
    lua_remove(L, -3);				// widgets w_a

    lua_pushlightuserdata(L, p);		// widgets w_a *w
    lua_rawget(L, -3);				// widgets w_a ref/nil

    // if found, look up the ref number in wiget_aliases.
    if (!lua_isnil(L, -1)) {
	lua_rawget(L, -2);			// widgets w_a w/nil
	if (!lua_isnil(L, -1)) {
	    _reuse_widget(L, p, type_idx, flags);
	    goto ex;
	}
    }

    lua_pop(L, 1);

    // Either the address isn't a key in gtk.widgets, or the reference
    // number isn't in gtk.widget_aliases.  The latter may happen when an
    // entry in widget_aliases is removed by GC (weak values!), but
    // l_widget_gc hasn't been called on it yet.

    // returns ref=0 if the widget is on the stack.
    int ref = _make_widget(L, p, type_idx, flags);
    if (ref > 0) {
	// new entry in widgets table
	_set_widget_pointer(L, p, ref, 0);

	if (G_UNLIKELY(runtime_flags & RUNTIME_DEBUG_MEMORY
	    && !lua_isnil(L, -1))) {
	    struct widget *w = (struct widget*) lua_topointer(L, -1);
	    int ref_count = luagtk_get_refcount(w);
	    struct widget_type *wt = luagtk_get_widget_type(w);
	    fprintf(stderr, "%p %p %5d new %s %d %s\n", w, w->p, ref,
		wt->name, ref_count, WIDGET_NAME(w));
	}
    }

ex:
    lua_remove(L, -2);				// widgets w/nil
    lua_remove(L, -2);				// w/nil
}

/**
 * Determine the type of the widget
 *
 * @param p            Pointer to the structure
 * @param type_idx     (output) where to store the detected struct nr
 * @return
 *  0 ... error
 *  1 ... found a structure; type_idx is now set.
 */
static int _determine_object_type(lua_State *L, void *p, int *type_idx)
{
    GType type_nr = G_TYPE_FROM_INSTANCE(p);
    const char *type_name;
    const struct type_info *ti;

    for (;;) {
	// g_type_query might not succeed if the type isn't completely
	// initialized. use g_type_name instead.
	type_name = g_type_name(type_nr);

	if (!type_name)
	    return luaL_error(L, "invalid widget at %p (type %d)", p,
		(int) type_nr);

	// This actually refers to a GEnumClass - a collection of possible
	// values of an ENUM, not to a specific value.  So this is not useful!
	if (G_TYPE_IS_ENUM(type_nr) || G_TYPE_IS_FLAGS(type_nr)) {
	    // lua_pushnumber(L, * (int*) p);
	    return 0;
	}

	ti = find_struct(type_name, 1);
	
	/* found? if so, perform an integrity check */
	if (ti) {
	    const char *name = TYPE_NAME(ti);
	    if (strcmp(name, type_name))
		return luaL_error(L, "[gtk] internal error: type names don't "
		    "match: %s - %s", name, type_name);
	    
	    *type_idx = ti - type_list;
	    return 1;
	}

	/* This class is not known. Maybe a base class is known? */
	GType parent_type = g_type_parent(type_nr);
	if (!parent_type)
	    return luaL_error(L, "[gtk] g_type_parent failed on GType %s (%d)",
		type_name, type_nr);

	/* Parent found; try again with this.  Happens with GdkGCX11,
	 * which is private, but can be used as GdkGC. */
	type_nr = parent_type;
    }

    /* NOT REACHED */
    return 0;
}


/**
 * Push a new widget (struct widget) onto the stack.
 *
 * @param p  pointer to the object to make the Lua object for; must not be NULL.
 * @param type_idx  what the pointer type is; 0 for auto detect
 * @param flags  see luagtk_get_widget
 * @return -1 on error (nothing pushed), 0 (stack object) or >0 (normal, ref
 *   to an entry in gtk.widget_aliases)
 *
 * Lua stack: unless -1 is returned, a new widget is pushed.
 */
static int _make_widget(lua_State *L, void *p, int type_idx, int flags)
{
    struct widget *w;
    const struct type_info *ti;

    /* If the structure number is not given, the object must be derived from
     * GObject; otherwise, the result is undefined (probably SEGV). */
    if (!type_idx) {
	int rc = _determine_object_type(L, p, &type_idx);
	if (!rc)
	    return -1;
    }

    if (type_idx <= 0 || type_idx >= type_count)
	return luaL_error(L, "%s invalid type_idx %d in _make_widget",
	    msgprefix, type_idx);

    ti = type_list + type_idx;


    /* make new Lua object with meta table */
    w = (struct widget*) lua_newuserdata(L, sizeof(*w));
    memset(w, 0, sizeof(*w));
    w->p = p;
    w->type_idx = type_idx;

    /* determine which widget type to use. */
    luagtk_guess_widget_type(L, w, flags);

    /* set metatable - shared among widgets of the same type */
    _get_widget_meta(L, type_idx);		// w meta
    lua_setmetatable(L, -2);			// w

    /* Set the environment to an empty table - used to store data with
     * arbitrary keys for a specific object.  The env can't be nil.  To avoid
     * having an unused table for each widget, use the same for all and replace
     * with a private table when the first data is stored.
     */
    lua_getglobal(L, "gtk");			// w gtk
    lua_getfield(L, -1, LUAGTK_EMPTYATTR);	// w gtk emptyattr
    lua_setfenv(L, -3);				// w gtk

    // Increase refcount (but not always - see ffi2lua_struct_ptr).  flags
    // may have FLAG_NEW_OBJECT set.
    luagtk_inc_refcount(w, flags);

    // stack objects neither get an entry in gtk.widgets, nor in
    // gtk.widget_aliases.
    if (_is_on_stack(p)) {
	w->own_ref = 0;
	lua_pop(L, 1);				// w
	return 0;
    }

    // Store it in the widgets aliases table, using the next index.  Can't
    // use luaL_ref, because the widget IDs must not be reused.
    lua_getfield(L, -1, LUAGTK_ALIASES);	// w gtk aliases
    lua_remove(L, -2);				// w w_a
    lua_rawgeti(L, -1, 0);			// w w_a idx
    int ref = lua_tonumber(L, -1) + 1;
    lua_pushnumber(L, ref);			// w w_a idx next_idx
    lua_rawseti(L, -3, 0);			// w w_a idx
    lua_pushvalue(L, -3);			// w w_a idx w
    lua_rawseti(L, -3, ref);			// w w_a idx
    lua_pop(L, 2);				// w

    w->own_ref = ref;
    return ref;
}

