/*
 * dlock.c - AIX lslk lock functions
 *
 * Vic Abell
 * Purdue University Computing Center
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dlock.c,v 1.10 99/11/10 15:00:05 abe Exp $";
#endif


#include "lslk.h"


/*
 * Local static variables
 */

static struct xutsname Xname;		/* structure to receive machine ID */


/*
 * Local definitions
 */

#define	KMEM		"/dev/kmem"
#define	SYSIDTABINCR	32		/* local sysid table increment */


/*
 * Local function prototypes
 */

_PROTOTYPE(static unsigned long lkupsysid,(unsigned int s));
_PROTOTYPE(static int savelock,(struct inode *i, struct filock *fp));
_PROTOTYPE(static void process_lock_list,(struct inode *i, KA_T ll));


/*
 * gather_lock_info() -- gather lock information
 */

void
gather_lock_info()
{
	struct hinode *hb;
	struct hinode *hba;
	struct filock fl;
	int ft, hx;
	struct inode i, *ip;
	unsigned len;
	int nb;
/*
 * Determine the inode hash bucket table size and address.
 */

#if	AIXV<=3250
	hba = (struct hinode *)Knl[X_HINODE].n_value;
	nb = NHINO;
#else	/* AIXV>3250 */
	if (kread((KA_T)Knl[X_HINODE].n_value, (char *)&hba, sizeof(hba))
	||  hba == (KA_T)NULL) {
	    (void) fprintf(stderr,
		"%s: can't read inode hash bucket address (%#x) from %#x\n",
		Pn, hba, Knl[X_HINODE].n_value);
	    Exit(1);
	}
	nb = 0;
	if (Nl[X_NHINO].n_value == (long)0
	||  kread((KA_T)Nl[X_NHINO].n_value, (char *)&nb, sizeof(nb))
	||  nb < 1) {
	    (void) fprintf(stderr,
		"%s: can't read inode hash bucket count (%d) from %#x\n",
		Pn, nb, Nl[X_NHINO].n_value);
	    Exit(1);
	}
#endif	/* AIXV<=3250 */

/*
 * Allocate space for a local copy of the inode hash bucket table and
 * read it from kernel memory.
 */
	len = (unsigned)(nb * sizeof(struct hinode));
	if ((hb = (struct hinode *)malloc(len)) == (struct hinode *)NULL) {
	    (void) fprintf(stderr,
		"%s: can't allocate %d bytes for inode hash bucket table\n",
		Pn, len);
	    Exit(1);
	}
	if (kread((KA_T)hba, (char *)hb, len)) {
	    (void) fprintf(stderr,
		"%s: can't read %d byte inode hash table from %#x\n",
		Pn, len, hba);
	}
/*
 * Loop through the inode hash buckets.
 */
	for (hx = 0; hx < nb; hba++, hx++) {

	/*
	 * Loop through the inodes of a hash bucket.
	 */
	    for (ft = 1, ip = hb[hx].hi_forw;
	         ip && (ft || ip != (struct inode *)hba);
	         ft = 0, ip = i.i_forw)
	    {
		if (kread((KA_T)ip, (char *)&i, sizeof(i))) {
		    if (Owarn)
			(void) fprintf(stderr,
			    "%s: premature exit from inode chain\n", Pn);
		    return;
		}
		if (i.i_count)
		    process_lock_list(&i, (KA_T)i.i_gnode.gn_filocks);
	    }
	}
}


/*
 * get_cmdnm() -- get command name
 */

char *
get_cmdnm(lp)
	struct llock_info *lp;		/* local lock structure */
{
	struct procinfo p;
	static struct userinfo u;
/*
 * Fake the procinfo structure for the process and get the userinfo
 * structure associated with it.
 */
	p.pi_pid = lp->pid;
	if (getuser(&p, sizeof(p), &u, sizeof(u)) == 0)
	    return(u.ui_comm);
	return("(unknown)");
}


/*
 * get_kernel_access() -- get access to kernel information
 */

void
get_kernel_access()
{

/*
 * Open access to kernel memory.
 */
	if ((Kd = open(KMEM, O_RDONLY, 0)) < 0) {
	    (void) fprintf(stderr, "%s: can't open %s: %s\n",
		Pn, KMEM, strerror(errno));
	    Exit(1);
	}
/*
 * Drop setgid permission.
 */

#if	defined(WILLDROPGID)
	(void) dropgid();
#else	/* !defined(WILLDROPGGID) */
# if	AIXV>3250
	if (Nmlst && !isreadable(Nmlst, 1))
	    Exit(1);
# endif	/* AIXV>3250 */
#endif	/* defined(WILLDROPGGID) */

/*
 * Convert kernel variable names to addresses and check the result.
 */
	if (knlist(Knl, X_KNL_NUM, sizeof(struct nlist)) != 0) {
	    (void) fprintf(stderr, "%s: knlist() failed\n", Pn);
	    Exit(1);
	}

#if	AIXV>3250
/*
 * Get kernel variable addresses from Nmlst or N_UNIX, too.
 */
	if (nlist(Nmlst ? Nmlst : N_UNIX, Nl)) {
	    (void) fprintf(stderr, "%s: can't nlist(%s)\n", Pn,
		Nmlst ? Nmlst : N_UNIX);
	    Exit(1);
	}
#endif	/* AIXV>3250 */

}


/*
 * initialize() -- initialize
 */

void
initialize()
{
	(void) get_kernel_access();
	if (unamex(&Xname) < 0) {
	    (void) fprintf(stderr, "%s: can't get machine ID: %s\n",
		Pn, strerror(errno));
	    Exit(1);
	}
}


/*
 * kread() -- read kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel address */
	char *buf;			/* local receiving buffer address */
	int len	;			/* length to read */
{
	int br;

	if (lseek(Kd, addr, L_SET) == (off_t)-1L)
		return(-1);
	br = read(Kd, buf, len);
	return((br == len) ? 0 : 1);
}


/*
 * lkupsysid() - look up sysid in system's lm_sysid chain
 */

static unsigned long
lkupsysid(s)
	unsigned int s;			/* l_sysid target */
{
	int i;
	KA_T ka;
	MALLOC_S len;
	static int ncp = 0;
	static int ncpa = 0;
	struct lm_sysid {		/* not available in an AIX header
					 * file, although it should be */
	    struct lm_sysid *next;
	    int refct;
	    int protocol;
	    struct sockaddr_in addr;
	    char *name;
	    short sysid;
	} lms;
	struct nfs_syms {		/* not available in an AIX header
					 * file, although it should be */
	    char ident[8];
	    void *dp[30];
	    struct lm_sysid **lm_sysids;
	} ns;
	struct sysid_entry {		/* local sysid table entry */
	    short sysid;
	    int protocol;
	    struct sockaddr_in addr;
	};
	static struct sysid_entry *st = (struct sysid_entry *)NULL;
	static int sts = 0;
/*
 * Check the sysid table status.
 */
	if (sts == -1)

	/*
	 * The table can't be loaded, so return the argument.
	 */
	    return(unsigned long)s;
	else if (sts == 0) {

	/*
	 * Load the table.
	 */

	    if (!Knl[X_NFS_SYMS].n_value
	    ||  kread((KA_T)Knl[X_NFS_SYMS].n_value, (char *)&ns, sizeof(ns)))
	    {
		sts = -1;
		return(unsigned long)s;
	    }
	    if (!ns.lm_sysids
	    ||  kread((KA_T)ns.lm_sysids, (char *)&ka, sizeof(ka))) {
		sts = -1;
		return(unsigned long)s;
	    }
	/*
	 * Examine the kenel lm_sysid chain.
	 */
	    for (; ka; ka = (KA_T)lms.next) {
		if (kread(ka, (char *)&lms, sizeof(lms)))
		    break;
	    /*
	     * Skip duplicates.
	     */
		for (i = 0; i < ncp; i++) {
		    if (lms.sysid == st[i].sysid
		    &&  lms.protocol == st[i].protocol)
			break;
		}
		if (i < ncp)
		    continue;
	    /*
	     * Allocate local sysid table space.
	     */
		if (ncp >= ncpa) {
		    ncpa += SYSIDTABINCR;
		    len = (MALLOC_S)(ncpa * sizeof(struct sysid_entry));
		    if (st)
			st = (struct sysid_entry *)realloc((MALLOC_P *)st, len);
		    else
			st = (struct sysid_entry *)malloc(len);
		    if (!st) {
			(void) fprintf(stderr,
			    "%s: can't allocate %d sysid table bytes\n",
			    Pn, len);
			Exit(1);
		    }
		}
	    /*
	     * Fill a new local sysid table entry.
	     */
		st[ncp].sysid = (unsigned int)lms.sysid;
		st[ncp].protocol = lms.protocol;
		st[ncp++].addr = lms.addr;
	    }
	}
/*
 * Search the loaded sysid table.
 */
	for (i == 0; i < ncp; i++) {
	    if (st[i].sysid == s)
		return((unsigned long)st[i].addr.sin_addr.s_addr);
	}
	return((unsigned long)s);
}


/*
 * print_dev() -- print device number
 */

char *
print_dev(lp)
	struct llock_info *lp;		/* local lock structure */
{
	static char buf[128];

	(void) sprintf(buf, "%d,%d", major(lp->dev), minor(lp->dev));
	return(buf);
}


/*
 * process_lock_list() -- process lock list
 */

static void
process_lock_list(ip, ll)
	struct inode *ip;		/* local inode */
	KA_T ll;			/* lock list address */
{
	struct filock fl;
	KA_T fla;

	if (!(fla = ll))
	    return;
	do {

	/*
	 * Read and process the chain of data_lock structures.
	 */
	    if (kread(fla, (char *)&fl, sizeof(fl)))
		return;
	    if (savelock(ip, &fl)) {
		if (is_lock_sel()) {
		    NLockU++;
		    Lp = (struct llock_info *)NULL;
		}
	    }
	    fla = (KA_T)fl.next;
	} while (fla && fla != ll);
}


/*
 * savelock() -- save lock information
 */

static int
savelock(ip, fp)
	struct inode *ip;		/* inode structure pointer */
	struct filock *fp;		/* filock structure pointer */
{

/*
 * Ignore the request if the PID is zero.
 *
 * Allocate a local lock structure.
 */
	if (!fp->set.l_pid)
	    return(0);
	(void) alloc_llock();
/*
 * Save: inode number; device; process ID; and size.
 */
	Lp->inum = ip->i_number;
	Lp->dev = ip->i_dev;
	Lp->pid = (unsigned long)fp->set.l_pid;
	Lp->sz = (unsigned long)ip->i_size;
	Lp->szs = 1;
/*
 * Save the lock description: mandatory status; type; start; and
 * length or end.
 */
	if (ENF_LOCK(ip->i_mode))
	    Lp->mand = 1;
	else
	    Lp->mand = 0;
	Lp->type = (int) fp->set.l_type;
	Lp->ls = Lp->ss = Lp->ws = 1;
	Lp->len = (unsigned long)fp->set.l_len;
	Lp->start = (unsigned long)fp->set.l_start;
	Lp->whence = (unsigned long)fp->set.l_whence;
/*
 * If the l_sysid element of the flock structure is non-zero,
 * this is a remote lock.
 */
	if (fp->set.l_sysid != (unsigned long)0
	&&  fp->set.l_sysid != Xname.nid) {
	    Lp->src = 1;
	    Lp->hn = (char *)NULL;
	    Lp->iap = alloc_in_addr();
	    if (!*((unsigned char *)&fp->set.l_sysid))
	        Lp->iap->s_addr = lkupsysid(fp->set.l_sysid);
	    else
	        Lp->iap->s_addr = (unsigned long)fp->set.l_sysid;
	} else
	    Lp->src = 0;
	return(1);
}
