# Copyright (C) 2008-2010 LottaNZB Development Team
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA.

from kiwi.ui.delegates import SlaveDelegate
from kiwi.ui.dialogs import error

from lottanzb.hellaconfig import Server
from lottanzb.util import has_ssl_support, _

class ConfigView(SlaveDelegate):
    gladefile = "standalone_config_view"
    
    lotta_config_fields = ["hellanzb_command"]
    hella_config_fields = ["address", "port", "username", "password", "ssl"]
    
    def __init__(self, mode):
        self.mode = mode
        
        SlaveDelegate.__init__(self)
        
        for widget in [self.address, self.port, self.hellanzb_command]:
            widget.mandatory = True
        
        if not self.mode.hella_config.servers:
            self.mode.hella_config.add_server(Server(id=_("Default Server")))
        
        self.server = self.mode.hella_config.servers[0]
        
        self.lotta_proxy = self.add_proxy(self.mode.config, self.lotta_config_fields)
        self.hella_proxy = self.add_proxy(self.server, self.hella_config_fields)
        
        self.auth_required.set_active(self.server.needs_authentication())
        
        # The toggle event isn't emitted if needs_authentication() is False.
        # That's why it's emitted manually here.
        self.auth_required.emit("toggled")
    
    def on_auth_required__toggled(self, widget):
        required = widget.read()
        
        self.username_label.set_sensitive(required)
        self.password_label.set_sensitive(required)
        
        for widget in [self.username, self.password]:
            widget.mandatory = required
            
            if not required:
                widget.set_text("")
            
            widget.set_sensitive(required)
            widget.validate()

    def on_ssl__toggled(self, widget):
        if widget.read() and not has_ssl_support():
            title = _("OpenSSL for Python not installed")
            detailed = _("In order to use SSL, you need to install the Python "
                "bindings for OpenSSL. The package is usually called "
                "python-openssl.")
            
            error(title, detailed, self.toplevel.get_toplevel())
            
            # The `Server` object won't do this on its own.
            self.server.ssl = False
        
        # We need to manually read the new `ssl` and `port` value from the model
        # as they might have been changed while the model was updating the `ssl`
        # and therefore didn't receive any notifications. 
        # It's ugly, but it works.
        self.hella_proxy.update("ssl")
        self.hella_proxy.update("port")
