/* Copyright (C) 2000-2003 Markus Lausser (sgop@users.sf.net)
   This is free software distributed under the terms of the
   GNU Public License.  See the file COPYING for details. */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <sys/utsname.h>

#include "support.h"
#include "global.h"
#include "lopster.h"
#include "commands.h"
#include "chat.h"
#include "server.h"
#include "whois.h"
#include "connection.h"
#include "handler.h"
#include "scheme.h"
#include "string_list.h"
#include "hotlist.h"
#include "browse.h"
#include "exec.h"
#include "transfer.h"
#include "utils.h"
#include "search.h"

static net_t* active_net = NULL;
static chat_page_t* active_page = NULL;
static int all_servers;

#define CHECK_NET \
  if (!active_net) {\
    client_message("Error", "No Network specified!");\
    return;\
  } else if (active_net->active_server && active_net->active_server->status != SERVER_ONLINE) {\
    client_message("Error", "You are not connected to that Network!");\
    return;\
  }
  
#define CHECK_TEXT_PAGE \
  if (!active_page || (active_page->type == P_CHANNEL)) return;

#define COMMAND_ERROR(val, text) \
  if (val) {\
    client_message("Error", "%s", text);\
    return;\
  }

#define CHECK_TYPE(typ, text) \
  if (!active_page || (active_page->type != typ)) {\
    client_message("Error", "%s", text);\
    return;\
  }

char *FolderNames(int no) {
  switch (no) {
  case 0:
    return "Chat Commands";
  case 1:
    return "User Modification Commands";
  case 2:
    return "Infomation Commands";
  case 3:
    return "Misc Commands";
  case 4:
    return "Client Commands";
  case 5:
    return "Channel Operator Commands";
  case 6:
    return "Moderator Commands";
  case 7:
    return "Admin Commands";
  default:
    return "Unspecified command";
  }
}

int command_new(int id, char *iname, char *syntax, char *ides,
		user_level_t ilevel, folder_t ifolder,
		COMMAND((*ihandler)))
{
  command_t *new_comm;

  if (!iname) {
    g_warning("no command name given");
    return 0;
  }
  if (command_is_command(iname)) {
    g_warning("chat command already exists [%s]", iname);
    return 0;
  }
  new_comm = (command_t *) g_malloc(sizeof(command_t));
  new_comm->id = id;
  new_comm->name = g_strdup(iname);
  new_comm->syntax = g_strdup(syntax);
  new_comm->description = g_strdup(ides);
  new_comm->is_alias = 0;
  new_comm->level = ilevel;
  new_comm->folder = ifolder;
  new_comm->handler = ihandler;

  global.commands = g_list_append(global.commands, new_comm);

  return 1;
}

int command_rename(int iid, char *iname) {
  command_t *comm;

  (void)iid;
  comm = command_is_command(iname);

  if (!comm) {
    g_warning("chat command does not exists");
    return 0;
  }

  g_free(comm->name);
  comm->name = g_strdup(iname);

  return 1;
}

GList *command_search(char *prefix) {
  GList *result = NULL;
  command_t *comm;
  int len = strlen(prefix);
  GList* dlist;

  for (dlist = global.commands; dlist; dlist = dlist->next) {
    comm = dlist->data;
    if (!strncasecmp(prefix, comm->name, len))
      result = g_list_append(result, comm);
  }

  return result;
}

command_t *command_is_command(char *iname) {
  command_t *comm;
  char *name;
  GList* dlist;

  if (!iname)
    return NULL;
  if (*iname == '/')
    name = iname + 1;
  else
    name = iname;

  for (dlist = global.commands; dlist; dlist = dlist->next) {
    comm = dlist->data;
    if (!g_strcasecmp(name, comm->name))
      return comm;
  }

  return NULL;
}

char *command_get_alias_string(int id)
{
  command_t *comm;
  static char result[1024];
  GList* dlist;

  result[0] = 0;
  for (dlist = global.commands; dlist; dlist = dlist->next) {
    comm = dlist->data;
    if ((comm->id == id) && (comm->is_alias)) {
      if (*result) strcat(result, " ");
      strcat(result, comm->name);
    }
  }

  return result;
}

int command_make_alias(char *iname, char *alias)
{
  command_t *comm;
  command_t *new_alias;
  char temp[1024];

  if (!iname) {
    g_warning("no command name given");
    return 0;
  }
  comm = command_is_command(alias);
  if (comm) {
    client_message("Error", "Command does already exists");
    return 0;
  }
  if (!alias) {
    client_message("Error", "No alias name given");
    return 0;
  }
  comm = command_is_command(iname);
  if (!comm) {
    client_message("Error", "Command does not exists");
    return 0;
  }

  new_alias = (command_t *) g_malloc(sizeof(command_t));
  new_alias->id = comm->id;
  new_alias->name = g_strdup(alias);
  new_alias->syntax = g_strdup_printf("see %s", iname);
  strcpy(temp, "This is an alias for <");
  strcat(temp, iname);
  strcat(temp, ">");
  new_alias->description = g_strdup(temp);
  new_alias->is_alias = 1;
  new_alias->level = comm->level;
  new_alias->folder = comm->folder;
  new_alias->handler = comm->handler;
  global.commands = g_list_append(global.commands, new_alias);

  strcpy(temp, "New alias: ");
  strcat(temp, alias);
  strcat(temp, " := ");
  strcat(temp, iname);
  client_message("Message", "%s", temp);

  return 1;
}

void command_delete_alias_for(int id)
{
  command_t *comm;
  int i1;

  i1 = 0;
  while (i1 < (int)g_list_length(global.commands))
  {
    comm = (command_t *) (g_list_nth(global.commands, i1)->data);
    if ((comm->id == id) && (comm->is_alias)) {
      global.commands = g_list_remove(global.commands, comm);
    } else {
      i1++;
    }
  }
}

void command_delete_all()
{
  command_t *comm;
  GList* dlist;

  for (dlist = global.commands; dlist; dlist = dlist->next) {
    comm = (command_t *) (dlist->data);
    g_free(comm->name);
    g_free(comm->syntax);
    g_free(comm->description);
    g_free(comm);
  }
  g_list_free(global.commands);
  global.commands = NULL;
}

void handle_lopster_command(char *data) {
  char* command;
  char* server_id;
  char* pos;
  command_t *comm;
  int temp;

  if (global.commands == NULL) {
    g_warning("no commands declared!");
    return;
  }
  if (data[0] != '/') {
    return;
  } else
    data++;

  active_page = global.current_page;
  command = arg(data, 0);

  pos = arg(NULL, 1);
  if (pos && *pos == ':') {
    server_id = arg(pos+1, 0);
    temp = atoi(server_id);
    if (temp > 0) {
      active_net = g_list_nth_data(global.net_active, temp-1);
      all_servers = 0;
    } else if (temp == -1) {
      active_net = NULL;
      all_servers = 1;
    } else {
      active_net = NULL;
      all_servers = 0;
    }
    pos = arg(NULL, 1);
    //    printf("[%s] [%s] [%s]\n", command, server_id, pos);
  } else if (active_page) {
    active_net = active_page->net;
  } else active_net = NULL;

  if (command) {
    comm = command_is_command(command);
    COMMAND_ERROR(!comm, "Unknown command! Type /help for a list of commands.");
    comm->handler(pos);
  }
}

COMMAND(lopster_none)
{
  (void)data;
  g_warning("command with id C_NONE");
}

COMMAND(lopster_whisper)
{
  char *nick;
  char *message;
  
  CHECK_NET;
  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "You have to specify a nickname!");
  message = arg(NULL, 1);
  COMMAND_ERROR(!message, "Whisper what?");

  send_private(active_net, nick, message, 1);
}

COMMAND(lopster_whois) {
  char *nick;

  CHECK_NET;

  nick = arg(data, 0);
  if (!nick) {
    if (global.user_reply.user)
      whois_request(active_net, global.user_reply.user, 
		    WHOIS_SHOW);
    else
      whois_request(active_net, active_net->user.username,
		    WHOIS_SHOW);
  } else {
    if (!strncasecmp(nick, "-text", 5)) {
      nick = arg(NULL, 0);
      if (!nick) {
	if (global.user_reply.user)
	  whois_request(active_net, 
			global.user_reply.user, WHOIS_TEXT);
	else
	  whois_request(active_net, active_net->user.username,
			WHOIS_TEXT);
      } else {
	whois_request(active_net, nick, WHOIS_TEXT);
      }
    } else {
      whois_request(active_net, nick, WHOIS_SHOW);
    }
  }
}

COMMAND(lopster_emote) {
  char *message;

  CHECK_NET;

  message = arg(data, 1);
  COMMAND_ERROR(!message, "Please specify an emote message!");
  if (!active_page) return;

  if (active_page->type == P_PUBLIC)
    send_public_emote(active_page, message);
  if (active_page->type == P_PRIVATE)
    send_private_emote(active_net, active_page->name, message);
}

COMMAND(lopster_join) {
  char *channel;

  channel = arg(data, 0);
  COMMAND_ERROR(!channel, "Join what channel?");
  if (all_servers) {
    join_channel(NULL, channel);
  } else {
    CHECK_NET;
    join_channel(active_net, channel);
  }
}

COMMAND(lopster_autojoin) {
  char *channel;
  net_group_t* ng;
  unsigned int max = 0;
  net_t* net;
  chat_page_t* page;
  int mode;
  GList* dlist;
  GList* dlist2;
  GList* dlist3;

  channel = arg(data, 0);

  if (!channel) {
    for (dlist = global.net_groups; dlist; dlist = dlist->next) {
      ng = dlist->data;
      for (dlist2 = ng->nets; dlist2; dlist2 = dlist2->next) {
	net = dlist2->data;
	if (!net->auto_join) continue;
	if (strlen(net->name) > max) max = strlen(net->name);
      }
    }

    page = chat_page_get_printable();
    chat_print_time_stamp(page, M_PUBLIC);
    chat_print_prefix(page, 1);
    chat_print_text(page, M_PUBLIC, "text", "[");
    chat_print_text(page, M_PUBLIC, "message", "Channel Autojoin List");
    chat_print_text(page, M_PUBLIC, "text", "]\n");
    
    for (dlist = global.net_groups; dlist; dlist = dlist->next) {
      ng = dlist->data;
      for (dlist2 = ng->nets; dlist2; dlist2 = dlist2->next) {
	net = dlist2->data;
	if (!net->auto_join) continue;
	chat_print_time_stamp(page, M_PUBLIC);
	chat_print_prefix(page, 1);
	chat_print_text(page, M_PUBLIC, "user", "[");
	chat_print_network(page, M_PUBLIC, net, max, 0);
	chat_print_text(page, M_PUBLIC, "user", "]");
	for (dlist3 = net->auto_join; dlist3; dlist3 = dlist3->next) {
	  channel = dlist3->data;
	  chat_print_text(page, M_PUBLIC, "text", " ");
	  chat_print_channel(page, M_PUBLIC, channel, net);
	}
	chat_print_text(page, M_PUBLIC, "text", "\n");
      }
    }
  } else {
    CHECK_NET;
    if (*channel == '-') {
      channel++;
      mode = 0;
    } else if (*channel == '+') {
      channel++;
      mode = 1;
    } else mode = 1;
    COMMAND_ERROR(*channel == 0, "Argument too short");

    if (mode == 0) {
      char* text = is_string_in_list(active_net->auto_join, channel);
      COMMAND_ERROR(!text, "Not in autojoin list");
      active_net->auto_join =
	g_list_remove(active_net->auto_join, text);
      g_free(text);
    } else {
      char* text = is_string_in_list(active_net->auto_join, channel);
      COMMAND_ERROR(text, "Already in autojoin list");
      active_net->auto_join =
	g_list_append(active_net->auto_join, g_strdup(channel));
    }
    server_save();
  }
}

COMMAND(lopster_raw) {
  char *args;

  CHECK_NET;

  args = arg(data, 1);
  COMMAND_ERROR(!args, "You have to specify an argument!");
  command_send(active_net, CMD_RAW, args);
}

static void
help_print_folder(chat_page_t* page, folder_t folder, int all) {
  char* temp_str;
  GList* dlist;
  command_t* command;
  int cnt = 1;
  char str[1024];
  int i1;
  user_level_t level;

  level = server_get_highest_level();
  
  chat_print_time_stamp(page, M_PUBLIC);
  chat_print_prefix(page, 1);
  //  temp_str = g_strdup(FolderNames(folder));
  chat_print_text(page, M_PUBLIC, "error", FolderNames(folder));
  //  g_free(temp_str);
  cnt = 0;
  for (dlist = global.commands; dlist; dlist = dlist->next) {
    command = dlist->data;
    if (command->folder != folder) continue;
    if (!all && (command->level > level)) continue;

    if (cnt % 5 == 0) {
      chat_print_text(page, M_PUBLIC, "text", "\n");
      chat_print_time_stamp(page, M_PUBLIC);
      chat_print_prefix(page, 1);
      chat_print_text(page, M_PUBLIC, "text", "  ");
    } else {
      chat_print_text(page, M_PUBLIC, "text", " ");
    }
    //      chat_print("message", "/");
    sprintf(str, "%%K[%%n%s", command->name);
    for (i1 = strlen(command->name); i1 < 10; i1++) strcat(str, " ");
    strcat(str, "%K]");
    temp_str = cparse(str);
    chat_print_colored(page, M_PUBLIC, "message", temp_str);
    cnt++;
  }
  chat_print_text(page, M_PUBLIC, "text", "\n");
}

COMMAND(lopster_help) {
  char *com;
  int i2;
  command_t *command;
  int all;
  user_level_t level;
  chat_page_t* page;
  int old_val;

  level = server_get_highest_level();

  page = chat_page_get_printable();

  old_val = global.options.strip_colors;
  global.options.strip_colors = 0;
  
  com = arg(data, 0);
  if (!com || !strcasecmp(com, "all")) {
    all = (com != NULL);
    chat_print_time_stamp(page, M_PUBLIC);
    chat_print_prefix(page, 1);
    chat_print_text(page, M_PUBLIC, "text", "[");
    chat_print_text(page, M_PUBLIC, "message", "Available Commands");
    chat_print_text(page, M_PUBLIC, "text", "]\n");
    for (i2 = 0; i2 < NO_FOLDERS; i2++) {
      if (!all) {
	if ((i2 == F_MOD) && ((int) level < (int)L_MOD))
	  continue;
	if ((i2 == F_ADMIN) && ((int) level < (int)L_ADMIN))
	  continue;
      }
      help_print_folder(page, i2, all);
    }
    chat_print_time_stamp(page, M_PUBLIC);
    chat_print_prefix(page, 1);
    chat_print_current("text", "Type ");
    chat_print_current("message", "/help <command>");
    chat_print_current("text", " for more information!\n");
  } else {
    command = command_is_command(com);
    if (command) {
      chat_print_time_stamp(page, M_PUBLIC);
      chat_print_prefix(page, 1);
      chat_print_text(page, M_PUBLIC, "error", "Help for command");
      chat_print_text(page, M_PUBLIC, "message", " /");
      chat_print_text(page, M_PUBLIC, "message", command->name);
      chat_print_text(page, M_PUBLIC, "text", "\n");
      client_message("Syntax", "%s", command->syntax);
      client_message("Description", "%s", command->description);
      if ((int) command->level > L_ADMIN)
	if (level > L_ADMIN)
	  client_message("Access", "Elite (and you are!)");
	else
	  client_message("Access", "Elite (not you!)");
      else if ((int) command->level > L_MOD)
	if (level > L_MOD)
	  client_message("Access", "Admin or higher (and you are!)");
	else
	  client_message("Access", "Admin or higher (not you!)");
      else if ((int) command->level > L_USER)
	if (level > L_USER)
	  client_message("Access",
			 "Moderator or higher (and you are!)");
	else
	  client_message("Access", "Moderator or higher (not you!)");
      else if ((int) command->folder == F_OP)
	client_message("Access", "All channel operators");
      else
	client_message("Access", "All users");
    } else {
      client_message("Error", "This is not a command!");
    }
  }
  global.options.strip_colors = old_val;
}

unsigned long pop_time(char** string) {
  char* pos;
  unsigned long result = 0;

  if (!string || !(*string)) return 0;
  if (!isdigit(**string)) return 0;

  pos = *string;
  while (isdigit(*pos)) {
    result *= 10;
    result += *pos -'0';
    pos++;
  }

  switch (*pos) {
  case 'm': 
  case 'M': 
    result *= 60;
    break;
  case 'h': 
  case 'H': 
    result *= 60*60;
    break;
  case 'D': 
  case 'd': 
    result *= 60*60*24;
    break;
  case 'W': 
  case 'w': 
    result *= 60*60*24*7;
    break;
  case 'Y': 
  case 'y': 
    result *= 60*60*24*365;
    break;
  case ' ': 
  case 's': 
    break;
  default:
    return 0;
  }
  
  while (*pos && (*pos != ' ')) pos++;
  while (*pos == ' ') pos++;

  if (*pos) *string = pos;
  else *string = NULL;

  return result;
}

COMMAND(lopster_ban) {
  char *nick;
  char *message;
  unsigned long timeout;

  CHECK_NET;

  nick = arg(data, 0);
  if (!nick) {
    show_bans(active_net, NULL);
    return;
  }

  message = arg(NULL, 1);
  if (!message) {
    ban_user(active_net, nick, NULL, 0);
  } else {
    timeout = pop_time(&message);
    if (message) {
      ban_user(active_net, nick, message, timeout);
    } else {
      ban_user(active_net, nick, "banned for some time", timeout);
    }
  }
}

COMMAND(lopster_chlevel) {
  char *arg1;
  char *arg2;
  chat_page_t* page;

  CHECK_NET;

  arg1 = arg(data, 0);
  if (!arg1) {
    COMMAND_ERROR(active_page && (active_page->type != P_PUBLIC),
		  "Please specify a channel or switch to a channel and specify a level!");
    command_send(active_net, CMD_GET_CHAN_LEVEL, active_page->name);
    return;
  }
  arg2 = arg(NULL, 0);
  if (!arg2) {
    if (level2int(arg1) >= 0) {
      COMMAND_ERROR(active_page && (active_page->type != P_PUBLIC),
		    "Please specify a channel or switch to a channel!");
      channel_set_level(active_page, NULL, level2int(arg1), 1);
    } else {
      command_send(active_net, CMD_GET_CHAN_LEVEL, arg1);
    }
  } else {
    page = chat_page_search(active_net, arg1, P_PUBLIC, 1);
    COMMAND_ERROR(!page, "That's not a valid channel!");
    channel_set_limit(page, NULL, level2int(arg1), 1);
  }
}

COMMAND(lopster_clear) {
  char *where;

  CHECK_TEXT_PAGE;

  where = arg(data, 0);
  if (where) {
    if (!g_strcasecmp(where, "TOP")) {
      chat_page_clear(active_page, 0);
    } else if (!g_strcasecmp(where, "BOTTOM")) {
      chat_page_clear(active_page, 1);
    } else {
      client_message("Error", "Clear what?");
    }
  } else {
    chat_page_clear(active_page, 0);
    chat_page_clear(active_page, 1);
  }
}

COMMAND(lopster_cloak) {
  (void)data;

  CHECK_NET;
  
  command_send(active_net, CMD_CLOAK);
  whois_request(active_net, active_net->user.username, WHOIS_NONE);
}

COMMAND(lopster_conn) {
  (void)data;
  net_group_start_connect(NULL);
}

COMMAND(lopster_dataport) {
  char *nick;
  char *port;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Missing argument in command /dataport !");

  port = arg(NULL, 0);
  if (!port) {
    create_upload_port(atoi(nick), TRUE);
  } else {
    CHECK_NET;
    command_send(active_net, CMD_ALTER_PORT, nick, atoi(port));
  }
}

COMMAND(lopster_disc) {
  (void)data;
  net_group_disconnect(NULL);
}

COMMAND(lopster_email)
{
  char *email;

  email = arg(data, 0);
  COMMAND_ERROR(!email, "You have to specify an e-mail address!");
  command_send(NULL, CMD_CHANGE_EMAIL, email);
}

COMMAND(lopster_kick) {
  char *arg1;
  char *arg2;
  char *arg3;

  CHECK_NET;
  arg1 = arg(data, 0);
  COMMAND_ERROR(!arg1, "Please specify a channel or a user or switch to a channel");

  if (user_search_full(active_page, arg1)) {
    // check is not neccessary, as user_search_full() only returns
    // true if we are in a channel
    CHECK_TYPE(P_PUBLIC, "You have to specify a channel!");
    arg2 = arg(NULL, 1);
    if (arg2) convert_quotes(arg2);
    command_send(active_net, CMD_KICK,
		 active_page->name, arg1, arg2, NULL);
  } else {
    arg2 = arg(NULL, 0);
    arg3 = arg(NULL, 1);
    COMMAND_ERROR(!arg2, "You have to specify a user");
    if (arg3) convert_quotes(arg3);
    command_send(active_net, CMD_KICK, arg1, arg2, arg3);
  }
}

COMMAND(lopster_kill) {
  char *nick;
  char *reason;

  CHECK_NET;
  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Who should be killed?");

  reason = arg(NULL, 1);
  if (reason) convert_quotes(reason);
  command_send(active_net, CMD_KILL, nick, reason);
}

COMMAND(lopster_level) {
  char *nick;
  char *level;

  CHECK_NET;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Please specify a user!");

  level = arg(NULL, 0);
  if (!level) {
    COMMAND_ERROR(level2int(nick) >= 0,
		  "If you are sure that you like to change your own user level, then use: /raw 606 <yournick> <level>");
    COMMAND_ERROR(TRUE, "Please sepcify a user level!");
  } else {
    command_send(active_net, CMD_SET_USERLEVEL, nick, level);
  }
}

COMMAND(lopster_chlist) {
  (void)data;
  CHECK_NET;
  channel_list_get(active_net);
}

COMMAND(lopster_motd) {
  (void)data;
  CHECK_NET;
  command_send(active_net, CMD_MOTD, NULL);
}

COMMAND(lopster_muzzle) {
  char *nick;
  char *reason;

  CHECK_NET;
  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Who should be muzzled?");

  reason = arg(NULL, 1);
  if (reason) convert_quotes(reason);
  command_send(active_net, CMD_MUZZLE, nick, reason);
}

COMMAND(lopster_wallop) {
  char *message;

  CHECK_NET;
  message = arg(data, 1);
  COMMAND_ERROR(!message, "Do you want to send an operator message or NOT?");
  send_wallop(active_net, message);
}

COMMAND(lopster_part) {
  char* reason;
  
  reason = arg(data, 1);
  if (active_page) chat_page_leave(active_page, reason);
}

COMMAND(lopster_password) {
  char *nick;
  char *passwd;
  char *command;

  CHECK_NET;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "You need an argument!");

  passwd = arg(NULL, 0);
  if (!passwd) {
    if (strcmp(active_net->user.password, nick)) {
      command_send(active_net, CMD_CHANGE_PASS, nick);
      command =
	g_strdup_printf("Your password was changed to [%s]", nick);
      strcpy(active_net->user.password, nick);
      client_message("Message", "%s", command);
      g_free(command);
      server_save();
    } else {
      client_message("Message", "Your password was not changed");
    }
  } else {
    command_send(active_net, CMD_ALTER_PASS, nick, passwd);
  }
}

COMMAND(lopster_ping) {
  char *nick;
  user_timestamp_t *stamp;
  chat_page_t* page;

  CHECK_NET;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Please specify a nickname!");

  stamp = timestamp_search(global.userstamp, nick);
  if (!stamp) {
    stamp = timestamp_new(nick);
    global.userstamp = g_list_append(global.userstamp, stamp);
  } else
    timestamp_touch(stamp);

  page = chat_page_get_printable();
  chat_print_time_stamp(page, M_PUBLIC);
  chat_print_prefix(page, 1);
  chat_print_text(page, M_PUBLIC, "message", "Pinging ");
  chat_print_text(page, M_PUBLIC, "user", "<");
  chat_print_nick(page, M_PUBLIC, nick, active_net);
  chat_print_text(page, M_PUBLIC, "user", ">\n");
  
  command_send(active_net, CMD_PING, nick);
}

COMMAND(lopster_quit) {
  (void)data;
  global_exit();
}

COMMAND(lopster_speed) {
  char *nick;
  char *speed;

  CHECK_NET;

  nick = arg(data, 0);
  
  COMMAND_ERROR(!nick, "Please specify a nickname!");
  speed = arg(NULL, 0);
  if (!speed) {
    if (isdigit(*nick)) {
      command_send(active_net, CMD_CHANGE_SPEED, nick);
      client_message("LineSpeed",
		     "Your linespeed was changed to %s!", nick);
      global.linespeed = atoi(nick);
    } else
      speed_request(active_net, nick);
  } else {
    command_send(active_net, CMD_ALTER_SPEED, nick, speed);
  }
}

COMMAND(lopster_stats) {
  (void)data;
  
  CHECK_NET;
  command_send(active_net, CMD_SERVER_STATS);
}

COMMAND(lopster_topic) {
  char *topic;

  CHECK_NET;
  
  CHECK_TYPE(P_PUBLIC, "Switch to a channel!");
  topic = arg(data, 1);
  
  if (!topic) {
    command_send(active_net, CMD_GET_TOPIC, active_page->name);
  } else {
    if (global.options.parse_colors) topic = cparse(topic);
    channel_set_topic(active_page, NULL, topic, 1);
  }
}

COMMAND(lopster_unban) {
  char *nick;
  char *message;

  CHECK_NET;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Who should be unbanned?");

  message = arg(NULL, 1);
  if (message) convert_quotes(message);

  command_send(active_net, CMD_UNBAN, nick, message);
}

COMMAND(lopster_unmuzzle) {
  char *nick;
  char *reason;

  CHECK_NET;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Who should be unmuzzled?");
  reason = arg(NULL, 1);
  if (reason) convert_quotes(reason);
  command_send(active_net, CMD_UNMUZZLE, nick, reason);
}

COMMAND(lopster_version) {
  (void)data;

  CHECK_NET;
  
  command_send(active_net, CMD_SERVER_VERSION, NULL);
}

COMMAND(lopster_scheme) {
  char *scheme;
  char *command;
  scheme_t *scheme1;

  scheme = arg(data, 1);
  if (!scheme) {
    command =
      g_strdup_printf("Current scheme is [%s]", global.scheme->name);
    client_message("Message", "%s", command);
    g_free(command);
  } else {
    scheme1 = scheme_load(scheme);
    if (scheme1) {
      if (global.scheme) scheme_destroy(global.scheme);
      global.scheme = scheme1;
      client_message("Message", "Scheme [%s] loaded!",
		     global.scheme->name);
    } else {
      client_message("Message", "Loading scheme [%s] failed!",
		     scheme);
    }
  }
}

COMMAND(lopster_alias) {
  char *alias;
  char *command;

  alias = arg(data, 0);
  COMMAND_ERROR(!alias, "Please give an alias name!");
  command = arg(NULL, 0);
  COMMAND_ERROR(!command, "Please give an command name!");
  command_make_alias(command, alias);
}

COMMAND(lopster_afk) {
  char *message;

  message = arg(data, 1);
  set_afk(message);
  if (message && *message) afkreason_add(message);
}

COMMAND(lopster_public) {
  char *message;

  message = arg(data, 1);
  COMMAND_ERROR(!message, "Want to send _nothing_?");
  CHECK_TYPE(P_PUBLIC, "/public not supported in this page");

  send_public(global.current_page, message);
}

COMMAND(lopster_reply) {
  char *message;

  COMMAND_ERROR(!global.user_reply.user, "Noone to reply!");

  message = arg(data, 1);
  COMMAND_ERROR(!message, "Want to reply _nothing_?");
  COMMAND_ERROR(!g_list_find(global.net_active, global.user_reply.net),
		"You are not longer on that user's server");
  send_private(global.user_reply.net, 
	       global.user_reply.user, message, 1);
}

COMMAND(lopster_private) {
  char *nick;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Yes, private mode, but to whom?");
  create_private_page(active_net, nick);
}

COMMAND(lopster_ignore) {
  char *nick;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Missing nickname!");

  if (string_list_search(LIST_IGNORE, nick))
    COMMAND_ERROR(!nick, "User is already ignored");

  ignore_add(nick);
}

COMMAND(lopster_chclear) {
  char *channel;
  char* reason;

  CHECK_NET;
  
  channel = arg(data, 0);
  reason = arg(NULL, 1);

  if (!channel) {
    COMMAND_ERROR((active_page && (active_page->type != P_PUBLIC)),
		  "/chclear is not supported without argument in this page!");
    command_send(active_net, CMD_CLEAR_CHANNEL, active_page->name, NULL);
  } else {
    if (!strcmp(channel, ".")) {
      CHECK_TYPE(P_PUBLIC, "This is not a channel page");
      channel = active_page->name;
    }
    command_send(active_net, CMD_CLEAR_CHANNEL, channel, reason);
  }
}

COMMAND(lopster_unignore) {
  char *nick;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Who should be unignored?");

  if (!string_list_search(LIST_IGNORE, nick))
    COMMAND_ERROR(!nick, "User is not ignored");

  ignore_remove(nick);
}

COMMAND(lopster_chban) {
  char *nick;
  char *channel;
  char *reason;
  GtkCList *clist;
  channel_user_t* cu;
  int row;

  CHECK_NET;

  channel = arg(data, 0);

  if (!channel) {
    CHECK_TYPE(P_PUBLIC, "Trying to ban a user from a channel? Better type /help chban !");
    show_bans(active_net, active_page->name);
    return;
  }

  if (active_page && (active_page->type == P_PUBLIC)) {
    clist = GTK_CLIST(active_page->online);
    cu = search_user_in_list(clist, channel, &row);
    if (cu) {
      nick = channel;
      channel = active_page->name;
    } else nick = arg(NULL, 0);
  } else {
    nick = arg(NULL, 0);
    cu = NULL;
  }

  COMMAND_ERROR(!nick, "Missing Nickname!");

  reason = arg(NULL, 1);
  if (reason) convert_quotes(reason);

  if (!strcmp(channel, ".")) {
    CHECK_TYPE(P_PUBLIC, "This is not a channel page");
    channel = active_page->name;
  }

  command_send(active_net, CMD_CHAN_BAN, channel, nick, reason);
}

COMMAND(lopster_chunban) {
  char *nick;
  char *channel;
  char *reason;
  GtkCList *clist;
  channel_user_t* cu;
  int row;

  CHECK_NET;

  channel = arg(data, 0);

  COMMAND_ERROR(!channel, "Try /help chunban !");

  if (active_page && (active_page->type == P_PUBLIC)) {
    clist = GTK_CLIST(active_page->online);
    cu = search_user_in_list(clist, channel, &row);
    if (cu) {
      nick = channel;
      channel = active_page->name;
    } else nick = arg(NULL, 0);
  } else {
    nick = arg(NULL, 0);
    cu = NULL;
  }

  COMMAND_ERROR(!nick, "Missing Nickname!");

  reason = arg(NULL, 1);
  if (reason) convert_quotes(reason);

  if (!strcmp(channel, ".")) {
    CHECK_TYPE(P_PUBLIC, "This is not a channel page");
    channel = active_page->name;
  }
  
  command_send(active_net, CMD_CHAN_UNBAN, channel, nick, reason);
}

COMMAND(lopster_nuke) {
  char *nick;

  CHECK_NET;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Specify a user!");
  command_send(active_net, CMD_NUKE, nick);
}

COMMAND(lopster_unnuke) {
  char *nick;

  CHECK_NET;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Specify a user!");
  command_send(active_net, CMD_UNNUKE, nick);
}

COMMAND(lopster_global) {
  char *message;

  CHECK_NET;
  message = arg(data, 1);
  COMMAND_ERROR(!message, "Try: /help global");
  send_global(active_net, message);
}

COMMAND(lopster_cversion) {
  char* t2 = NULL;
  struct utsname uts;
  char* out;
  int send;
  char str[1204];

  print_time(str, global.current_time - global.start_time);

  out = arg(data, 0);
  if (!out || strcasecmp(out, "-out")) send = 0;
  else send = 1;
  
  if (uname(&uts) < 0) {
    if (INTERN_MICRO_VERSION)
      t2 = g_strdup_printf(VERSION_STRING".%d [Uptime %s]",
			   INTERN_MICRO_VERSION, str);
    else
      t2 = g_strdup_printf(VERSION_STRING" [Uptime %s]", str);
  } else {
    if (INTERN_MICRO_VERSION)
      t2 = g_strdup_printf(VERSION_STRING".%d (%s %s) [Uptime: %s]",
			   INTERN_MICRO_VERSION,
			   uts.sysname, uts.release, str);
    else
      t2 = g_strdup_printf(VERSION_STRING" (%s %s) [Uptime: %s]",
			   uts.sysname, uts.release, str);
  }
  
  if (send) {
    CHECK_NET;
    COMMAND_ERROR(!active_page || !active_page->net,
		  "You have to switch to a valid page");
    if (active_page->type == P_PUBLIC) {
      send_public(active_page, t2);
    } else if (active_page->type == P_PRIVATE) {
      send_private(active_net, active_page->name, t2, 1);
    } else if (!strcmp("Wallop", active_page->name)) {
      send_wallop(active_net, t2);
    } else if (!strcmp("Global", active_page->name)) {
      send_global(active_net, t2);
    } else {
      COMMAND_ERROR(1, "You have to switch to a valid page");
    }
  } else {
    client_message("Message", "%s", t2);
  }
  g_free(t2);
}

COMMAND(lopster_echo) {
  char *message;
  char *text2;

  message = arg(data, 1);
  COMMAND_ERROR(!message, "Echo what?");

  if (global.options.parse_colors)
    text2 = cparse(message);
  else text2 = message;
  chat_print_current_ln("text", text2);
}

COMMAND(lopster_chlimit) {
  char *channel;
  char *limit;
  chat_page_t* page;

  CHECK_NET;

  channel = arg(data, 0);
  if (!channel) {
    CHECK_TYPE(P_PUBLIC,
	       "Please specify a channel or switch to a channel");
    command_send(active_net, CMD_GET_CHAN_LIMIT, active_page->name);
    return;
  }
  limit = arg(NULL, 0);
  if (!limit) {
    if (isdigit(*channel)) {
      CHECK_TYPE(P_PUBLIC,
		 "Please specify a channel or switch to a channel");
      channel_set_limit(active_page, NULL, atoi(channel), 1);
    } else {
      command_send(active_net, CMD_GET_CHAN_LIMIT, channel);
    }
  } else {
    page = chat_page_search(active_net, channel, P_PUBLIC, 1);
    COMMAND_ERROR(!page, "That's not a valid channel!");
    channel_set_limit(page, NULL, atoi(limit), 1);
  }
}

COMMAND(lopster_sping) {
  char *server;
  user_timestamp_t *stamp;

  CHECK_NET;

  server = arg(data, 0);
  COMMAND_ERROR(!server, "Pinging, yes, but what server?");

  stamp = timestamp_search(global.userstamp, server);
  if (!stamp) {
    stamp = timestamp_new(server);
    global.userstamp = g_list_append(global.userstamp, stamp);
  } else
    timestamp_touch(stamp);

  client_message("Message", "Pinging server %s", server);
  command_send(active_net, CMD_PING_SERVER, server);
}

COMMAND(lopster_friend) {
  char *nick;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "You have to specify a user!");
  toggle_friend(nick);
}

COMMAND(lopster_enemy) {
  char *nick;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "You have to specify a user!");
  toggle_enemy(nick);
}

COMMAND(lopster_hotadd) {
  char *nick;
  char *des;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "You have to specify a user!");
  des = arg(NULL, 1);
  hotlist_add(nick, des, 1);
}

COMMAND(lopster_hotrem) {
  char *nick;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "You have to specify a user!");
  hotlist_remove(nick, 1);
}

COMMAND(lopster_guser) {
  char *message;

  CHECK_NET;

  message = arg(data, 1);
  if (global.user_win) gtk_widget_show(global.user_win);
  if (!message) {
    show_global_users(active_net, "");
  } else {
    show_global_users(active_net, message);
  }
}

COMMAND(lopster_sconfig) {
  char *var;
  char *val;

  CHECK_NET;

  var = arg(data, 0);
  if (!var) {
    command_send(active_net, CMD_GET_SERVER_VARS);
    return;
  }
  val = arg(NULL, 0);
  if (!val) {
    command_send(active_net, CMD_GET_SERVER_VAR, var);
  } else {
    command_send(active_net, CMD_SET_SERVER_VAR, var, val);
  }
}

COMMAND(lopster_op) {
  char *nick;
  char *channel;

  CHECK_NET;

  channel = arg(data, 0);
  if (!channel) {
    CHECK_TYPE(P_PUBLIC,
	       "Please specify a channel or a user switch to a channel");
    command_send(active_net, CMD_LIST_OP, active_page->name);
    return;
  }
  nick = arg(NULL, 0);
  if (!nick) {
    if (active_page && (active_page->type == P_PUBLIC)) {
      if (user_search_full(active_page, channel)) {
	nick = channel;
	channel = active_page->name;
      } else if (!strcmp(channel, ".")) {
	channel = active_page->name;
      } else {
	COMMAND_ERROR(TRUE, "arg1 is not a valid channel");
      }
    }
  } else if (!strcmp(channel, ".")) {
    CHECK_TYPE(P_PUBLIC, "This page is not a valid channel");
    channel = active_page->name;
  }
  if (nick)
    command_send(active_net, CMD_OP, channel, nick);
  else
    command_send(active_net, CMD_LIST_OP, channel);
}

COMMAND(lopster_deop) {
  char *nick;
  char *channel;

  CHECK_NET;

  channel = arg(data, 0);
  COMMAND_ERROR(!channel, "Please specify a channel or a user switch to a channel");
  nick = arg(NULL, 0);
  if (!nick) {
    CHECK_TYPE(P_PUBLIC,
	       "Please specify a channel or a user switch to a channel");
    COMMAND_ERROR(!user_search_full(active_page, channel),
		  "Could not find specified user in channel");
    nick = channel;
    channel = active_page->name;
  } else if (!strcmp(channel, ".")) {
    CHECK_TYPE(P_PUBLIC, "This page is not a valid channel");
    channel = active_page->name;
  }
  command_send(active_net, CMD_DEOP, channel, nick);
}

COMMAND(lopster_chvoice) {
  char *nick;
  char *channel;

  CHECK_NET;

  channel = arg(data, 0);
  if (!channel) {
    CHECK_TYPE(P_PUBLIC,
	       "Please specify a channel or a user switch to a channel");
    command_send(active_net, CMD_CHAN_VOICE, active_page->name);
    return;
  }
  nick = arg(NULL, 0);
  if (!nick) {
    if (active_page && (active_page->type == P_PUBLIC)) {
      if (user_search_full(active_page, channel)) {
	nick = channel;
	channel = active_page->name;
      } else if (!strcmp(channel, ".")) {
	channel = active_page->name;
      } else {
	COMMAND_ERROR(TRUE, "arg1 is not a valid channel");
      }
    }
  } else if (!strcmp(channel, ".")) {
    CHECK_TYPE(P_PUBLIC, "This page is not a valid channel");
    channel = active_page->name;
  }
  if (nick)
    command_send(active_net, CMD_CHAN_VOICE, channel, nick);
  else
    command_send(active_net, CMD_LIST_VOICE, channel);
}

COMMAND(lopster_chunvoice) {
  char *nick;
  char *channel;

  CHECK_NET;

  channel = arg(data, 0);
  COMMAND_ERROR(!channel, "Please specify a channel or a user switch to a channel");

  nick = arg(NULL, 0);
  if (!nick) {
    CHECK_TYPE(P_PUBLIC,
	       "Please specify a channel or a user switch to a channel");
    COMMAND_ERROR(!user_search_full(active_page, channel),
		  "Could not find user in this channel");
    nick = channel;
    channel = active_page->name;
  } else if (!strcmp(channel, ".")) {
    CHECK_TYPE(P_PUBLIC, "This page is not a valid channel");
    channel = active_page->name;
  }

  command_send(active_net, CMD_CHAN_UNVOICE, channel, nick);
}

COMMAND(lopster_links) {
  char *action = NULL;

  CHECK_NET;

  if (data) action = arg(data, 0);

  if (!action) {
    if (!active_net->links || !data) {
      client_message("Message", "Evaluating linked server list");
      network_links_get(active_net);
    } else {
      network_links_print(active_net);
    }
  } else if (!g_strcasecmp(action, "GET")) {
    client_message("Message", "Evaluating linked server list");
    network_links_get(active_net);
  } else if (!g_strcasecmp(action, "PRINT")) {
    network_links_print(active_net);
  } else {
    client_message("Error", "Unknown action for /links");
  }
}

COMMAND(lopster_invite) {
  char *nick;
  char *channel;

  CHECK_NET;

  channel = arg(data, 0);
  COMMAND_ERROR(!channel, "You have to specify at least a user!");

  nick = arg(NULL, 0);
  if (!nick) {
    CHECK_TYPE(P_PUBLIC, 
	       "Please specify a channel or switch to a channel!");
    nick = channel;
    channel = active_page->name;
  } else if (!strcmp(channel, ".")) {
    CHECK_TYPE(P_PUBLIC, "This page is not a valid channel");
    channel = active_page->name;
  }

  command_send(active_net, CMD_CHAN_INVITE, channel, nick);
}

COMMAND(lopster_chmuzzle) {
  char *nick;
  char *channel;
  char *reason;
  GtkCList *clist;
  channel_user_t* cu;
  int row;

  CHECK_NET;

  channel = arg(data, 0);

  if (!channel) {
    CHECK_TYPE(P_PUBLIC, "Try /help chban !");
    command_send(active_net, CMD_CHAN_LIST_MUZZLE, active_page->name);
    return;
  }

  if (active_page && (active_page->type == P_PUBLIC)) {
    clist = GTK_CLIST(active_page->online);
    cu = search_user_in_list(clist, channel, &row);
    if (cu) {
      nick = channel;
      channel = active_page->name;
    } else nick = arg(NULL, 0);
  } else {
    nick = arg(NULL, 0);
    cu = NULL;
  }

  COMMAND_ERROR(!nick, "Missing Nickname!");

  reason = arg(NULL, 1);
  if (reason) convert_quotes(reason);

  if (!strcmp(channel, ".")) {
    CHECK_TYPE(P_PUBLIC, "This is not a channel page");
    channel = active_page->name;
  }

  command_send(active_net, CMD_CHAN_MUZZLE,
	       channel, nick, reason);
}

COMMAND(lopster_chunmuzzle) {
  char *nick;
  char *channel;
  char *reason;
  GtkCList *clist;
  channel_user_t* cu;
  int row;

  CHECK_NET;

  channel = arg(data, 0);

  COMMAND_ERROR(!channel, "Try /help chunban !");

  if (active_page && (active_page->type == P_PUBLIC)) {
    clist = GTK_CLIST(active_page->online);
    cu = search_user_in_list(clist, channel, &row);
    if (cu) {
      nick = channel;
      channel = active_page->name;
    } else nick = arg(NULL, 0);
  } else {
    nick = arg(NULL, 0);
    cu = NULL;
  }

  COMMAND_ERROR(!nick, "Missing Nickname!");

  reason = arg(NULL, 1);
  if (reason) convert_quotes(reason);

  if (!strcmp(channel, ".")) {
    CHECK_TYPE(P_PUBLIC, "This is not a channel page");
    channel = active_page->name;
  }
  
  command_send(active_net, CMD_CHAN_UNMUZZLE, channel, nick, reason);
}


COMMAND(lopster_usermode) {
  char *message;

  CHECK_NET;

  message = arg(data, 1);
  if (!message) {
    command_send(active_net, CMD_GET_USER_MODE);
  } else {
    command_send(active_net, CMD_SET_USER_MODE, message);
  }
}

COMMAND(lopster_sstats) {
  char* arg1;

  CHECK_NET;
  arg1 = arg(data, 0);
  server_stats_get(active_net, arg1);
}

COMMAND(lopster_chmode) {
  char *channel;
  char *mode;
  int chmode1 = 0;
  int chmode2 = 0;
  chat_page_t* page = NULL;

  CHECK_NET;

  channel = arg(data, 0);
  if (!channel) {
    CHECK_TYPE(P_PUBLIC, 
	       "Please specify a channel or switch to a channel!");
    command_send(active_net, CMD_GET_CHAN_MODE, active_page->name);
    return;
  }
  mode = arg(NULL, 0);

  if (!strcmp(channel, ".")) {
    CHECK_TYPE(P_PUBLIC, "Channel is not valid!");
    page = active_page;
  }
  
  if (!mode) {
    if ((*channel == '+') || (*channel == '-')) {
      CHECK_TYPE(P_PUBLIC, 
		 "Please specify a channel or switch to a channel!");
      mode = channel;
      page = active_page;
    }
  }

  if (!page) 
    page = chat_page_search(active_net, channel, P_PUBLIC, 1);
  
  COMMAND_ERROR(!page, "That's not a valid channel!");

  if (chmode2int_lop(mode, &chmode1, &chmode2)) {
    channel_set_mode(page, NULL, chmode1, chmode2, 1);
  } else {
    command_send(active_net, CMD_GET_CHAN_MODE, channel);
  }
}

COMMAND(lopster_chwallop) {
  char *message;

  CHECK_NET;
  
  message = arg(data, 1);
  COMMAND_ERROR(!message, "You want say something, but what?");
  CHECK_TYPE(P_PUBLIC, "Switch to a channel page!");
  send_chwallop(active_page, message);
}

COMMAND(lopster_killghost) {
  char *ghost;
  char *passwd;
  char *command;

  CHECK_NET;

  ghost = arg(data, 0);
  COMMAND_ERROR(!ghost, "You have to specify the ghost!");
  passwd = arg(NULL, 0);
  if (!passwd) {
    command = g_strdup_printf("ghost %s %s", ghost,
			      active_net->user.password);
  } else {
    command = g_strdup_printf("ghost %s %s", ghost, passwd);
  }
  
  send_notice(active_net, "NickServ", command, 1);
  g_free(command);
}

COMMAND(lopster_locate) {
  char *nick;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "You have to specify a user!");;
  command_send(active_net, CMD_WHICH_SERVER, nick);
}

COMMAND(lopster_clients) {
  (void)data;

  CHECK_NET;
  show_clients(active_net);
}

// /exec by R00t
////////////////////////////////

COMMAND(lopster_exec) {
  exec_command(active_net, data);
}

COMMAND(lopster_whowas) {
  char *nick;

  CHECK_NET;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Whowas who?");
  command_send(active_net, CMD_WHO_WAS, nick);
}

COMMAND(lopster_rcolors) {
  int i1;
  char str[1024];
  int pos = 0;
  char *codes;

  codes = arg(data, 0);

  if (!codes) {
  } else if (!g_strcasecmp(codes, "-reset")) {
    g_free(global.rainbow_colors);
    global.rainbow_colors = g_strdup("RrpPBbcCgGYy");
  } else {
    for (i1 = 0; data[i1] && (i1 < 1023); i1++) {
      if (strchr("krgybmpcwKRGYBMPCWn", data[i1]))
	str[pos++] = data[i1];
    }
    str[pos] = 0;
    if (pos == 0)
      global.rainbow_colors = g_strdup("RrpPBbcCgGYy");
    else
    global.rainbow_colors = g_strdup(str);
  }
  client_message("Rainbow colors", "%s", global.rainbow_colors);
}

COMMAND(lopster_rainbow) {
  char *size;
  int isize;
  char *sep = NULL;
  char *message;
  char command[2048];
  int pos;
  int cpos;
  int cnt;
  int i1;
  int old_parse;

  CHECK_NET;
  CHECK_TEXT_PAGE;
  size = arg(data, 0);
  COMMAND_ERROR(!size, "Wrong format in command /rainbow!");
  if (!isdigit(*size)) {
    sep = size;
    size = arg(NULL, 0);
  }
  COMMAND_ERROR(!size, "Wrong format in command /rainbow!");
  message = arg(NULL, 1);
  COMMAND_ERROR(!message, "Wrong format in command /rainbow!");

  isize = atoi(size);
  if (isize < 1) isize = 1;
  pos = cnt = 0;
  cpos = strlen(global.rainbow_colors) - 1;
  for (i1 = 0; message[i1]; i1++) {
    if (message[i1] == '\n') {
      command[pos++] = message[i1];
      command[pos++] = '%';
      command[pos++] = global.rainbow_colors[cpos];
    } else {
      if ((cnt % isize) == 0) {
	if (sep) {
	  int temp = 0;
	  command[pos++] = '%';
	  command[pos++] = 'n';
	  while (sep[temp] != 0) {
	    command[pos++] = sep[temp];
	    temp++;
	  }
	}

	cpos++;
	if (global.rainbow_colors[cpos] == 0) cpos = 0;
	command[pos++] = '%';
	command[pos++] = global.rainbow_colors[cpos];
      }
      cnt++;
      command[pos++] = message[i1];
    }
  }
  command[pos++] = 0;

  // always parse rainbow colors
  old_parse = global.options.parse_colors;
  global.options.parse_colors = 1;
  if (global.current_page->type == P_PUBLIC) {
    send_public(active_page, command);
  } else if (global.current_page->type == P_PRIVATE) {
    send_private(active_net, active_page->name, command, 1);
  } else if (!strcmp("Wallop", active_page->name)) {
    send_wallop(active_net, command);
  } else if (!strcmp("Global", active_page->name)) {
    send_global(active_net, command);
  }
  global.options.parse_colors = old_parse;
}

COMMAND(lopster_browse) {
  char *nick;

  CHECK_NET;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "Browsing whom?");

  browse_user_files(active_net, nick);
}

COMMAND(lopster_send) {
  char *nick;
  char *file;

  CHECK_NET;

  nick = arg(data, 0);
  file = arg(NULL, 1);

  COMMAND_ERROR(!nick, "Missing nickname!");
  if (file)
    dcc_send_file(nick, file, active_net);
  else
    dcc_select_file(nick, active_net);
}

COMMAND(lopster_slink) {
  char *server;
  char *remote;
  char *command;

  CHECK_NET;

  server = arg(data, 0);
  remote = arg(NULL, 0);

  COMMAND_ERROR(!server, "Link a server or what?");

  if (remote) {
    command = g_strdup_printf("%s %s", server, remote);
    command_send(active_net, CMD_SERVER_CONNECT, command);
    g_free(command);
  } else {
    command_send(active_net, CMD_SERVER_CONNECT, data);
  }
}

COMMAND(lopster_sdelink) {
  char *server;
  char *reason;

  CHECK_NET;

  server = arg(data, 0);
  reason = arg(NULL, 1);
  COMMAND_ERROR(!server, "Delink a server or what?");

  if (reason) convert_quotes(reason);

  command_send(active_net, CMD_SERVER_DISCONNECT, server, reason);
}

COMMAND(lopster_skill) {
  char *server;
  char *reason;

  CHECK_NET;

  server = arg(data, 0);
  reason = arg(NULL, 1);
  COMMAND_ERROR(!server, "Kill what server?");

  if (reason) convert_quotes(reason);

  command_send(active_net, CMD_KILL_SERVER, server, reason);
}

COMMAND(lopster_sremove) {
  char *server;
  char *reason;

  CHECK_NET;

  server = arg(data, 0);
  reason = arg(NULL, 1);
  COMMAND_ERROR(!server, "Remove what server?");

  if (reason) convert_quotes(reason);

  command_send(active_net, CMD_SERVER_REMOVE, server, reason);
}

COMMAND(lopster_sreload) {
  CHECK_NET;

  command_send(active_net, CMD_SERVER_REHASH, data);
}

COMMAND(lopster_masskill) {
  char *ip;
  char *reason;

  CHECK_NET;

  ip = arg(data, 0);
  reason = arg(NULL, 1);
  COMMAND_ERROR(!ip, "Kill what ip?");

  if (reason) convert_quotes(reason);

  command_send(active_net, CMD_MASS_KILL, ip, reason);
}

COMMAND(lopster_shistogram) {
  char* dir;
  int direction = 0;

  CHECK_NET;
  dir = arg(data, 0);
  if (!dir) direction = 0;
  else if (!strcmp(dir, "-in")) direction = 0;
  else if (!strcmp(dir, "-out")) direction = 1;
  else COMMAND_ERROR(1, "Invalid argument");

  if (direction == 0) {
    client_message(NULL, "Requesting server's incoming traffic...");
    command_send(active_net, CMD_HISTOGRAM);
  } else {
    client_message(NULL, "Requesting server's outgoing traffic...");
    command_send(active_net, CMD_SHISTOGRAM);
  }
}

COMMAND(lopster_register) {
  char *user;
  char *passwd;
  char *email;
  char *level;

  CHECK_NET;

  user = arg(data, 0);
  passwd = arg(NULL, 0);
  email = arg(NULL, 0);
  level = arg(NULL, 0);

  if (!user) {
    command_send(active_net, CMD_REGISTER_USER,
		 active_net->user.username,
		 active_net->user.password,
		 "anon@anon", "User");
    return;
  }
  COMMAND_ERROR((!passwd || !email),
		"Specify password and E-mail.");
  if (level)
    command_send(active_net, CMD_REGISTER_USER,
		 user, passwd, email, level);
  else
    command_send(active_net, CMD_REGISTER_USER,
		 user, passwd, email, "User");
}

COMMAND(lopster_redirect) {
  char *user;
  char *server;
  char *port;

  CHECK_NET;

  user = arg(data, 0);
  server = arg(NULL, 0);
  port = arg(NULL, 0);

  COMMAND_ERROR(!port, "Wrong number of arguments.");

  command_send(active_net, CMD_REDIRECT, user, server, atoi(port));
}

COMMAND(lopster_cycle) {
  char *user;
  char *server;

  CHECK_NET;

  user = arg(data, 0);
  server = arg(NULL, 0);

  COMMAND_ERROR(!server, "Wrong number of arguments.");
  command_send(active_net, CMD_CYCLE, user, server);
}

COMMAND(lopster_limit) {
  char *tag;
  char *ip;
  
  CHECK_NET;

  tag = arg(data, 0);
  ip = arg(NULL, 1);

  if (!tag || !g_strcasecmp(tag, "list")) {
    client_message("Message", "Requesting limits on <%s>",
		   active_net->name);
    command_send(active_net, CMD_CLASS_LIST);
  } else if (!g_strcasecmp(tag, "add")) {
    COMMAND_ERROR(!ip, "No ip specified");
    command_send(active_net, CMD_CLASS_ADD, ip);
  } else if (!g_strcasecmp(tag, "remove")) {
    COMMAND_ERROR(!ip, "No ip specified");
    command_send(active_net, CMD_CLASS_REMOVE, ip);
  } else {
    client_message("Error", "Unknown action");
  }
}

COMMAND(lopster_iline) {
  char *tag;
  char *ip;

  CHECK_NET;

  tag = arg(data, 0);
  ip = arg(NULL, 1);
  if (!tag || !g_strcasecmp(tag, "list")) {
    client_message("Message", "Requesting ilines on <%s>",
		   active_net->name);
    command_send(active_net, CMD_ILINE_LIST);
  } else if (!g_strcasecmp(tag, "add")) {
    COMMAND_ERROR(!ip, "No ip specified");
    command_send(active_net, CMD_ILINE_ADD, ip);
  } else if (!g_strcasecmp(tag, "remove")) {
    COMMAND_ERROR(!ip, "No ip specified");
    command_send(active_net, CMD_ILINE_REMOVE, ip);
  } else {
    client_message("Error", "Unknown action");
  }
}

COMMAND(lopster_dline) {
  char *tag;
  char *ip;

  CHECK_NET;

  tag = arg(data, 0);
  ip = arg(NULL, 1);
  if (!tag || !g_strcasecmp(tag, "list")) {
    client_message("Message", "Requesting ilines on <%s>",
		   active_net->name);
    command_send(active_net, CMD_DLINE_LIST);
  } else if (!g_strcasecmp(tag, "add")) {
    COMMAND_ERROR(!ip, "No ip specified");
    command_send(active_net, CMD_DLINE_ADD, ip);
  } else if (!g_strcasecmp(tag, "remove")) {
    COMMAND_ERROR(!ip, "No ip specified");
    command_send(active_net, CMD_DLINE_REMOVE, ip);
  } else {
    client_message("Error", "Unknown action");
  }
}

COMMAND(lopster_eline) {
  char *tag;
  char *ip;

  CHECK_NET;

  tag = arg(data, 0);
  ip = arg(NULL, 1);
  if (!tag || !g_strcasecmp(tag, "list")) {
    client_message("Message", "Requesting ilines on <%s>",
		   active_net->name);
    command_send(active_net, CMD_ELINE_LIST);
  } else if (!g_strcasecmp(tag, "add")) {
    COMMAND_ERROR(!ip, "No ip specified");
    command_send(active_net, CMD_ELINE_ADD, ip);
  } else if (!g_strcasecmp(tag, "remove")) {
    COMMAND_ERROR(!ip, "No ip specified");
    command_send(active_net, CMD_ELINE_REMOVE, ip);
  } else {
    client_message("Error", "Unknown action");
  }
}

COMMAND(lopster_sreconfig) {
  CHECK_NET;
  
  COMMAND_ERROR(!data, "Which server configuration variable do you want to reset?");
  command_send(active_net, CMD_SERVER_RECONFIG, data);
}

COMMAND(lopster_pretend) {
  char* code;
  char* message;
  gint16 old_code;
  char* old_message;

  CHECK_NET;

  code = arg(data, 0);
  message = arg(NULL, 1);
  COMMAND_ERROR(!code, "Pretend what?");

  old_code = active_net->c_type;
  old_message = active_net->c_message;
  active_net->c_type = atoi(code);
  active_net->c_message = message;
  active_net->handle_command(active_net);
  active_net->c_type = old_code;
  active_net->c_message = old_message;
}

COMMAND(lopster_servers) {
  net_t *net;
  GList* dlist;
  int cnt;
  char text[1024];
  chat_page_t* page;
  int max_length = 0;
  char* option;
  int show_net = 0;
#ifdef HAVE_ZLIB
  int show_comp = 0;
#endif
  int show_stats = 0;
  int number_printed = 0;
  int show_only = 0;

  option = arg(data, 0);
  if (option) {
    if (isdigit(*option)) {
      show_only = atoi(option);
      option = arg(NULL, 0);
    }
    if (option) {
      while (*option) {
	if (*option == 'n') show_net = 1;
#ifdef HAVE_ZLIB
	else if (*option == 'c') show_comp = 1;
#endif
	else if (*option == 's') show_stats = 1;
	option++;
      }
    } else {
      // [default] show all info, if user doesnt specify options and
      // want only one server to be displayed
      show_net = show_stats = 1;
#ifdef HAVE_ZLIB
      show_comp = 1;
#endif
    }
  } else {
    // [default] show only netnames, if user doesnt specify options and
    // want all servers to be displayed
    show_net = 1;
  }

  page = chat_page_get_printable();
  if (!show_only) {
    for (dlist = global.net_active; dlist; dlist = dlist->next) {
      net = dlist->data;
      cnt = strlen(net->name);
      if (cnt > max_length) max_length = cnt;
    }
  }

  cnt = 1;
  for (dlist = global.net_active; dlist; dlist = dlist->next) {
    net = dlist->data;
    number_printed = 0;
    
    if (show_only && cnt != show_only) {
      cnt++;
      continue;
    }
    if (show_net) {
      chat_print_time_stamp(page, M_PUBLIC);
      chat_print_prefix(page, 1);

      chat_print_text(page, M_PUBLIC, "chat_background", "[");
      sprintf(text, "%2d", cnt);
      chat_print_colored(page, M_PUBLIC, "message", text);
      chat_print_text(page, M_PUBLIC, "chat_background", "] ");
      number_printed = 1;

      chat_print_text(page, M_PUBLIC, "chat_background", "[");
      sprintf(text, "%c", *(LevelShort(net->user.level)));
      chat_print_text(page, M_PUBLIC, "message", text);
      chat_print_text(page, M_PUBLIC, "chat_background", "] ");
      chat_print_text(page, M_PUBLIC, "user", "[");
      chat_print_network(page, M_PUBLIC, net, max_length, 0);
      chat_print_text(page, M_PUBLIC, "user", "] ");
      sprintf(text, "%s:%d\n", net->active_server->address, 
	      net->active_server->port);
      chat_print_text(page, M_PUBLIC, "message", text);
    }
#ifdef HAVE_ZLIB
    if (show_comp) {
      chat_print_time_stamp(page, M_PUBLIC);
      chat_print_prefix(page, 1);
      if (!number_printed) {
	chat_print_text(page, M_PUBLIC, "chat_background", "[");
	sprintf(text, "%2d", cnt);
	chat_print_colored(page, M_PUBLIC, "message", text);
	chat_print_text(page, M_PUBLIC, "chat_background", "] ");
	number_printed = 1;
      } else {
	chat_print_text(page, M_PUBLIC, "chat_background", "     ");
      }
      chat_print_text(page, M_PUBLIC, "text", "Compression (Level/In/Out)");
      chat_print_text(page, M_PUBLIC, "chat_background", " [");
      sprintf(text, "%d", net->compress);
      chat_print_text(page, M_PUBLIC, "message", text);
      chat_print_text(page, M_PUBLIC, "chat_background", "/");
      sprintf(text, "%.3f", net->rcvd_compressed/net->rcvd_uncompressed);
      chat_print_text(page, M_PUBLIC, "message", text);
      chat_print_text(page, M_PUBLIC, "chat_background", "/");
      sprintf(text, "%.3f", net->sent_compressed/net->sent_uncompressed);
      chat_print_text(page, M_PUBLIC, "message", text);
      chat_print_text(page, M_PUBLIC, "chat_background", "]\n");
    }
#endif
    if (show_stats) {
      chat_print_time_stamp(page, M_PUBLIC);
      chat_print_prefix(page, 1);
      if (!number_printed) {
	chat_print_text(page, M_PUBLIC, "chat_background", "[");
	sprintf(text, "%2d", cnt);
	chat_print_colored(page, M_PUBLIC, "message", text);
	chat_print_text(page, M_PUBLIC, "chat_background", "] ");
	number_printed = 1;
      } else {
	chat_print_text(page, M_PUBLIC, "chat_background", "     ");
      }
      chat_print_text(page, M_PUBLIC, "text", "Stats      (User/Files/GB)");
      chat_print_text(page, M_PUBLIC, "chat_background", " [");
      sprintf(text, "%5d", net->active_server->users);
      chat_print_text(page, M_PUBLIC, "message", text);
      chat_print_text(page, M_PUBLIC, "chat_background", "/");
      sprintf(text, "%8d", net->active_server->files);
      chat_print_text(page, M_PUBLIC, "message", text);
      chat_print_text(page, M_PUBLIC, "chat_background", "/");
      sprintf(text, "%7d", net->active_server->gigs);
      chat_print_text(page, M_PUBLIC, "message", text);
      chat_print_text(page, M_PUBLIC, "chat_background", "]\n");
    }
    
    cnt++;
  }
}

COMMAND(lopster_status) {
  char* user;

  user = arg(data, 0);

  COMMAND_ERROR(!user, "No user specified");
  send_user_status(NULL, user);
}

COMMAND(lopster_search) {
  send_search_request_from_string(data);
}

COMMAND(lopster_nick) {
  char *nick;

  CHECK_NET;

  nick = arg(data, 0);
  COMMAND_ERROR(!nick, "You need an argument!");

  if (strcmp(active_net->user.username, nick)) {
    command_send(active_net, CMD_CHANGE_NICK, nick);
    strcpy(active_net->user.username, nick);
    server_save();
  } else {
    client_message("Message", "Your nickname was not changed");
  }
}

COMMAND(lopster_urls) {
  GList* dlist;
  int cnt;
  chat_page_t* page;
  char text[128];

  (void)data;

  page = chat_page_get_printable();
  cnt = 0;
  for (dlist = global.urls; dlist; dlist = dlist->next) {
    char* url = dlist->data;

    chat_print_time_stamp(page, M_PUBLIC);
    chat_print_prefix(page, 1);

    chat_print_text(page, M_PUBLIC, "_background", "[");
    sprintf(text, "%03d", cnt);
    chat_print_text(page, M_PUBLIC, "message", text);
    chat_print_text(page, M_PUBLIC, "chat_background", "] ");

    chat_print_url(page, M_PUBLIC, url);
    chat_print_text(page, M_PUBLIC, "text", "\n");

    cnt++;
  }
}

#ifdef PROTOCOL_DEBUG
COMMAND(lopster_test) {
  (void)data;
  /*
  char* num;
  chat_page_t* page;
  GtkText* text;

  num = arg(data, 0);
  if (!num) {
    page = global.current_page;
    if (!page || !page->main) return;
    text = GTK_TEXT(page->main);
    printf("---Number of chars : %d\n",
	   gtk_text_get_length (text));
  } else {
    global.options.max_chat_buffer = atoi(num);
  }
  */
  return;
}
#endif
