/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_DIARY_HEADER
#define LIFEOGRAPH_DIARY_HEADER


#include "helpers.hpp"	// i18n headers
#include "diarydata.hpp"
#include "entry.hpp"


namespace LIFEO
{
const char						DB_FILE_HEADER[] = "LIFEOGRAPHDB";
const int						DB_FILE_VERSION_INT( 74 );
const int						DB_FILE_VERSION_INT_MIN( 56 );
const std::string::size_type	PASSPHRASE_MIN_SIZE( 4 );
const char						LOCK_SUFFIX[] = ".~LOCK~";


enum Result
{
	OK,
	ABORTED,
	SUCCESS,
	FAILURE,
	COULD_NOT_START,
	COULD_NOT_FINISH,
	WRONG_PASSWORD,
	APPEARENTLY_ENCRYTED_FILE,
	APPEARENTLY_PLAIN_FILE,
	INCOMPATIBLE_FILE,
	CORRUPT_FILE,
	EMPTY_DATABASE
};


typedef PoolEntries::iterator				Entryiter;
typedef PoolEntries::reverse_iterator		EntryIterReverse;
typedef PoolEntries::const_iterator			Entryciter;

typedef std::vector< Entry* >				Entryvector;
typedef std::vector< Entry* >::iterator		Entryiterv;


class Diary : public DiaryBase, public DiaryElement
{
	public:
		typedef unsigned int FilteringStatus;
		static const FilteringStatus FS_CLEAR 				= 0;
		static const FilteringStatus FS_FILTER_TEXT 		= 2;
		static const FilteringStatus FS_FILTER_FAVORITES 	= 4;
		static const FilteringStatus FS_FILTER_TAG 			= 8;
		static const FilteringStatus FS_FILTER				= 14;
		static const FilteringStatus FS_NEW 				= 1024;

		struct Shower { virtual void show( Diary& ) = 0; };
		static Shower			*shower;

								Diary( void );
								Diary( const std::string& );	// locks diary file
								~Diary( void );

		static Diary			*d;

		bool					is_old( void )
		{ return( m_read_version < DB_FILE_VERSION_INT ); }

		void					init_new( const std::string& );

		// DISK I/O
		LIFEO::Result			read_body( void );
		LIFEO::Result			read_header( void );

		LIFEO::Result			write( void );
		LIFEO::Result			write( const std::string& );
		LIFEO::Result			write_copy( const std::string&, const std::string& );
		LIFEO::Result			write_txt( const std::string& );

		virtual void			clear( void );

		int						get_size() const
		{ return m_entries.size(); }
		Glib::ustring			get_name( void ) const
		{ return Glib::filename_display_basename( m_path ); }
		Type					get_type( void ) const
		{ return IT_DIARY; }
		Glib::RefPtr< Gdk::Pixbuf >&
								get_icon( void ) const;

		Glib::ustring			get_list_str( void ) const
		{ return Glib::ustring::compose( "<b>%1</b>",
				Glib::Markup::escape_text(
						Glib::filename_display_basename( m_path ) ) ); }

		void					set_path( const std::string& );
		const std::string&		get_path( void ) const;
		bool					is_path_set( void ) const;

		bool					set_passphrase( const std::string& );
		void					clear_passphrase( void );
		std::string				get_passphrase( void ) const;
		bool					compare_passphrase( const std::string& ) const;
		bool					is_passphrase_set( void ) const;

		DiaryElement*			get_element( DEID );
		DiaryElement*			get_startup_elem( void );
		DiaryElement*			get_most_current_elem( void );
		DiaryElement*			get_prev_session_elem( void );

		Entry*					get_entry( const Date& );
		Entryvector*			get_entries( Date::date_t );
		PoolEntries&			get_entries( void )
		{ return m_entries; }
		bool					get_day_has_multiple_entries( const Date& );
		Entry*					get_entry_today( void );
		Entry*					get_entry_nextinday( const Date& );
		Entry*					get_entry_first( void );
		void					set_entry_date( Entry*, const Date& );
		//int						get_entrydayorder( const Entry * const ) const;

		bool					is_first( const Entry * const ) const;
		bool					is_last( const Entry * const ) const;
		SortingCriteria			get_sorting_criteria( void ) const
		{ return m_option_sorting_criteria; }
		void					set_sorting_criteria( SortingCriteria sc )
		{ m_option_sorting_criteria = sc; }

		void					set_filter_text( const Glib::ustring& );
		Glib::ustring			get_filter_text( void ) const
		{ return m_filter_text; }
		void					set_filter_tag( const Tag* );
		const Tag*				get_filter_tag( void ) const
		{ return m_filter_tag; }
		void					toggle_filter_favorites( void );

		FilteringStatus			get_filtering_status( void ) const
		{	return m_filtering_status; }
		void					set_filtering_status_applied( void )
		{	if( m_filtering_status & FS_NEW ) m_filtering_status -= FS_NEW; }

		int						replace_text( const Glib::ustring& );

		Entry*					create_entry( Date, const Glib::ustring& = "", bool = false );
		// adds a new entry to today even if there is already one or more:
		Entry*					add_today( void );
		virtual bool			dismiss_entry( Entry* );
		int						list_entries( void );

		PoolTags*				get_tags( void )
		{ return &m_tags; }
		CategoryTags*			create_tag_ctg( void );
		CategoryTags*			create_tag_ctg( const Glib::ustring& );
		void					dismiss_tag_ctg( CategoryTags* );
		Tag*					create_tag( const Glib::ustring&, CategoryTags* = NULL );
		void					dismiss_tag( Tag* );

		const PoolCategoriesChapters*
								get_chapter_ctgs( void ) const
		{ return &m_chapter_categories; }
		CategoryChapters*		get_current_chapter_ctg( void )
		{ return m_ptr2chapter_ctg_cur; }
		CategoryChapters*		create_chapter_ctg( void );
		CategoryChapters*		create_chapter_ctg( const Glib::ustring& );
		void					dismiss_chapter_ctg( CategoryChapters* );
		void					dismiss_chapter( Chapter* );
		bool					rename_chapter_ctg( CategoryChapters*, const Glib::ustring& );
		const CategoryChapters*	get_topics( void )
		{ return &m_topics; }
		Chapter*				add_topic( const Glib::ustring& );

		Date					get_free_chapter_order_temporal( void );
		Date					get_free_chapter_order_ordinal( void );

		bool					make_free_entry_order( Date& );

		bool					get_spellcheck( void ) const
		{ return m_option_spellcheck; }
		void					set_spellcheck( bool opt_spell )
		{ m_option_spellcheck = opt_spell; }
		bool					is_encrypted( void ) const
		{ return( ! m_passphrase.empty() ); }

		Theme*					create_theme( const Glib::ustring& );
		Theme*					duplicate_theme( const Theme &theme );
		bool					has_theme( void ) const
		{ return( m_default_theme != NULL ); }
		const PoolThemes&		get_themes( void ) const
		{ return m_themes; }
		PoolThemes&				get_themes( void )
		{ return m_themes; }
		void					set_default_theme( Theme *theme )
		{ m_default_theme = theme; }
		Theme*					get_default_theme( void ) const
		{ return m_default_theme; }
		void					dismiss_theme( Theme *theme );

		void					show( void );

		bool					import_tag( Tag* );
		bool					import_entryset( const PoolEntries&, bool, const Glib::ustring& );

	protected:
		std::string				m_path;
		std::string				m_passphrase;
		// content
		PoolEntries				m_entries;
		PoolTags				m_tags;
		PoolCategoriesTags		m_tag_categories;
		PoolCategoriesChapters	m_chapter_categories;
		CategoryChapters		*m_ptr2chapter_ctg_cur;
		CategoryChapters		m_topics;	// ordinal chapters
		PoolThemes				m_themes;
		Theme					*m_default_theme;

		DEID					m_startup_elem;
		DEID					m_last_elem;
		// options & flags
		bool					m_option_spellcheck;
		SortingCriteria			m_option_sorting_criteria;
		int						m_read_version;
		// filtering
		Glib::ustring			m_filter_text;
		const Tag				*m_filter_tag;
		FilteringStatus			m_filtering_status;

		LIFEO::Result			parse_db_body_text( std::stringstream& );
		LIFEO::Result			parse_db_body_text_56( std::stringstream& );
		bool					create_db_body_text( std::stringstream& );
		bool					create_db_header_text( std::stringstream&, bool );
		LIFEO::Result			read_plain( void );
		LIFEO::Result			read_encrypted( void );
		LIFEO::Result			write_plain( const std::string&, bool = false );
		LIFEO::Result			write_encrypted( const std::string& );

	private:
		std::ios::pos_type		BodyPosition;

	friend class Lifeograph;
	friend class EntryView;
	friend class DiaryView;
	friend class TagView;
	friend class CategoryTagsView;
	friend class ThemeView;
	friend class PanelDiary;
	friend class TagPanel;
	friend class WidgetPanelExp;
	friend class DialogImport;
};

} // end of namespace LIFEO

#endif

