/***********************************************************************************

	Copyright (C) 2009 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_TEXTVIEW_HEADER
#define LIFEOGRAPH_TEXTVIEW_HEADER

#include <deque>
#include <gtkmm/textview.h>
#include <gdkmm/cursor.h>

extern "C"
{
#include <gtkspell/gtkspell.h>
}

#include "helpers.hpp"
#include "undo.hpp"
#include "diarydata.hpp"
#include "entry.hpp"


namespace LIFEO
{

const int		UNDO_MERGE_TIMEOUT = 3;

enum LookingFor
{
	LF_NOTHING			= 0x1,
	LF_NEWLINE			= 0x2,
	LF_NONSPACE			= 0x4,
	LF_SPACE			= 0x8,		// space that will come eventually
	LF_TAB				= 0x10,
	LF_SPACE_IMMEDIATE	= 0x20,		// immediate space char (" ")

	LF_ASTERISK			= 0x40,		// bold
	LF_UNDERSCORE		= 0x80,		// italic
	LF_EQUALS			= 0x100,	// strikethrough
	LF_HASH				= 0x400,	// highlight
	LF_FORMATCHAR		= 0x5C0,	// all above

	LF_SLASH			= 0x800,
	LF_NUMBER			= 0x2000,
	LF_AT				= 0x4000,	// email
	LF_CHECKBOX			= 0x8000,

	LF_DOTYM			= 0x10000,
	LF_DOTMD			= 0x20000,
	LF_DOTDATE			= 0x30000,	// DOTMD | DOTYM

	LF_LESS				= 0x80000,	// tagging
	LF_MORE				= 0x100000,
	LF_TILDE			= 0x200000,

	LF_APPLY			= 0x1000000,
	LF_APPLY_LINK		= 0x3000000,
	LF_APPLY_LINK_DATE	= 0x5000000,
	LF_APPLY_MARKUP		= 0x7000000,
	LF_APPLY_HEADING	= 0x9000000,
	LF_APPLY_SUBHEADING	= 0xB000000,

	LF_APPLY_NUMBER		= 0xD000000,
	LF_APPLY_DOTMD		= 0xF000000,
	LF_APPLY_DOTYM		= 0x11000000,

	LF_APPLY_TAG		= 0x13000000,
	
	LF_APPLY_CHECK_CAN	= 0x51000000,
	LF_APPLY_CHECK_UNF	= 0x61000000,
	LF_APPLY_CHECK_FIN	= 0x71000000,

	LF_NUM_SPCIM_SLH	= LF_NUMBER|LF_SPACE_IMMEDIATE|LF_SLASH,
	LF_NUM_SPCIM_CHK	= LF_NUMBER|LF_SPACE_IMMEDIATE|LF_CHECKBOX
};

enum CharClass
{
	CC_NONE				= 0,
	CC_NUMBER			= 0x10,
	CC_ALPHA			= 0x20,
	CC_ALPHANUM			= 0x30,
	CC_SIGN				= 0x40,

	CC_SPACE			= 0x100,
	CC_TAB				= 0x200,
	CC_NEWLINE			= 0x400,
	CC_SEPARATOR		= 0x700
};

enum LinkStatus
{
	LS_OK,
	LS_ENTRY_UNAVAILABLE,
	LS_INVALID,		// separator: to check a valid entry link: linkstatus < LS_INVALID
	LS_CYCLIC,

	LS_FILE_OK,
	LS_FILE_INVALID,
	LS_FILE_UNAVAILABLE,
	LS_FILE_UNKNOWN
};


class UndoEdit : public Undoable
{
	public:
							UndoEdit ( int position,
									   const Glib::ustring& text,
									   UndoableType type )
		: Undoable( "placeholder", type ), m_position( position ), m_text( text ) { }

		unsigned int		get_position (void) const
		{
			return m_position;
		}

		Glib::ustring		get_text( void ) const
		{
			return m_text;
		}

	protected:
		virtual void		merge( Undoable* ) = 0;

		virtual void		undo( void ) = 0;
		virtual void		redo( void ) = 0;

		void				erase( void )
		{
			Gtk::TextIter iter_start = m_ptr2buffer->get_iter_at_offset( m_position );
			Gtk::TextIter iter_end = m_ptr2buffer->get_iter_at_offset( m_position +
													m_text.size() );
			m_ptr2buffer->erase( iter_start, iter_end );
		}

		void				insert( void )
		{
			Gtk::TextIter iterator = m_ptr2buffer->get_iter_at_offset( m_position );
			m_ptr2buffer->insert( iterator, m_text );
		}

		unsigned int		m_position;
		Glib::ustring		m_text;

	private:
		static Gtk::TextBuffer		*m_ptr2buffer;

	friend class TextbufferDiary;
};

class UndoInsert :public UndoEdit
{
	public:
							UndoInsert( int position, const Glib::ustring& text )
		: UndoEdit( position, text, UT_INSERT_TEXT ) { }

		bool				can_merge( const Undoable* action ) const
		{
			if( ( action->get_time_sec() - m_time.tv_sec ) < UNDO_MERGE_TIMEOUT )
				if( action->get_type() == m_type )
					if( dynamic_cast< const UndoInsert* >( action )->get_position()
							== m_position + m_text.size() )
					return true;
			return false;
		}

		void				merge (Undoable* action)
		{
			m_text += dynamic_cast< UndoInsert* >( action )->get_text();
		}

	protected:
		void				undo( void )
		{
			erase();
		}
		void				redo( void )
		{
			insert();
		}
};

class UndoErase : public UndoEdit
{
	public:
							UndoErase( int position, const Glib::ustring& text )
		: UndoEdit( position, text, UT_ERASE_TEXT ) { }

		bool				can_merge (const Undoable* action) const
		{
			if ( ( action->get_time_sec() - m_time.tv_sec ) < UNDO_MERGE_TIMEOUT )
				if ( action->get_type() == m_type )
				{
					const UndoErase * action_erase =
						dynamic_cast< const UndoErase* >( action );
					if ( action_erase->get_position() +
							action_erase->get_text().size()
								== m_position )
						return true;
				}
			return false;
		}

		void				merge (Undoable* action)
		{
			UndoErase * action_erase = dynamic_cast< UndoErase* >( action );
			m_text.insert( 0, action_erase->get_text() );
			m_position = action_erase->get_position();
		}

	protected:
		void				undo( void )
		{
			insert();
		}
		void				redo( void )
		{
			erase();
		}

};

class Link
{
	public:
		virtual						~Link();
		virtual void				go( void ) = 0;

		// position of link in the buffer:
		Glib::RefPtr< Gtk::TextMark >
									m_mark_start;
		Glib::RefPtr< Gtk::TextMark >
									m_mark_end;

	protected:
									Link(	const Glib::RefPtr< Gtk::TextMark >&,
											const Glib::RefPtr< Gtk::TextMark >& );
};

class LinkEntry : public Link
{
	public:
		typedef sigc::signal< void, Date >
									Signal_void_Date;

									LinkEntry(	const Glib::RefPtr< Gtk::TextMark >&,
												const Glib::RefPtr< Gtk::TextMark >&,
												Date,
												unsigned int = 0 );
		void						go( void );

		static Signal_void_Date		signal_activated( void )
		{ return m_signal_activated; }

		Date						m_date;
		unsigned int				m_order;	// a day can have multiple entries
	protected:
		static Signal_void_Date		m_signal_activated;
};

class LinkUri : public Link
{
	public:
									LinkUri(	const Glib::RefPtr< Gtk::TextMark >&,
												const Glib::RefPtr< Gtk::TextMark >&,
												const std::string& );
		void						go( void );

		std::string					m_url;
		static Gtk::TextBuffer		*m_ptr2buffer;
};

class LinkCheck : public Link
{
	public:
									LinkCheck( const Glib::RefPtr< Gtk::TextMark >&,
											   const Glib::RefPtr< Gtk::TextMark >&,
											   unsigned int = 0 );
		void						go( void );

		const static Glib::ustring	boxes;
		unsigned int				m_state_index;
};


class TextbufferDiary : public Gtk::TextBuffer
{
	public:
		typedef void ( TextbufferDiary::*ParsingApplierFn )( void );
	
//		typedef sigc::signal< LinkStatus, Date >
//									Signal_LinkStatus_Date;
		typedef sigc::signal< void, const Glib::ustring& >
									Signal_void_ustring;

									TextbufferDiary( void );
		void						set_textview( Gtk::TextView *ptr2textview )
		{ m_ptr2textview = ptr2textview; }
		void						handle_logout( void );
		void						handle_login( void );

		Link*						get_link( int ) const;

		void						set_richtext( Entry* );
		void						reparse( void );

		void						set_searchstr( const Glib::ustring& );
		bool						select_searchstr_previous( void );
		bool						select_searchstr_next( void );

		void						toggle_format(	Glib::RefPtr< Tag >,
													const Glib::ustring& );
		void						toggle_bold( void );
		void						toggle_italic( void );
		void						toggle_highlight( void );
		void						toggle_strikethrough( void );
		void						toggle_hide_markup( void );

		void						handle_indent( void );
		void						handle_unindent( void );

		void						add_bullet( void );
		void						add_checkbox( void );

		void						set_spellcheck( bool );
		void						set_theme( const Theme* );

//		Signal_LinkStatus_Date		signal_link_needs_checking( void )
//		{
//			return m_signal_link_needs_checking;
//		}

		void						handle_menu( Gtk::Menu* );

	protected:
		void						on_insert( const Gtk::TextIter&,
											   const Glib::ustring&,
											   int );
		void						on_erase( const Gtk::TextIter&,
											  const Gtk::TextIter& );

		void						on_apply_tag( const Glib::RefPtr< TextBuffer::Tag >&,
												  const Gtk::TextIter&,
												  const Gtk::TextIter& );

		void						on_remove_tag( const Glib::RefPtr< TextBuffer::Tag >&,
												   const Gtk::TextIter&,
												   const Gtk::TextIter& );

		void						process_space( void );
		bool						process_new_line( void );

		// PARSING
		void						parse( unsigned int, unsigned int );

		void						process_char( unsigned int, unsigned int,
												  unsigned int,
												  ParsingApplierFn,
												  CharClass );

		void						apply_markup( const Glib::RefPtr<Tag>& );

		void						parsing_triggerer_subheading( void );
		void						parsing_triggerer_bold( void );
		void						parsing_triggerer_italic( void );
		void						parsing_triggerer_strikethrough( void );
		void						parsing_triggerer_highlight( void );
		void						parsing_triggerer_link( void );
		void						parsing_triggerer_link_at( void );
		void						parsing_triggerer_link_date( void );
		void						parsing_triggerer_list( void );
		void						parsing_junction_list( void );

		void						parsing_applier_heading( void );
		void						parsing_applier_subheading( void );
		void						parsing_applier_bold( void );
		void						parsing_applier_italic( void );
		void						parsing_applier_strikethrough( void );
		void						parsing_applier_highlight( void );
		void						parsing_applier_link( void );
		void						parsing_applier_link_date( void );
		void						parsing_applier_check_can( void );
		void						parsing_applier_check_unf( void );
		void						parsing_applier_check_fin( void );

		void						parsing_applier_number( void );
		void						parsing_applier_dotym( void );
		void						parsing_applier_dotmd( void );

		Gtk::TextIter				parser_iter_start;
		Gtk::TextIter				parser_iter_current;
		Gtk::TextIter				parser_iter_word;
		CharClass					parser_char_last;
		gunichar					parser_char_current;
		Glib::ustring				parser_word_last;
		unsigned int				parser_int_last;
		Date						parser_date_link;
		std::deque< unsigned int >	parser_lookingfor;

		std::deque< unsigned int >	m_plan_date;
		std::deque< unsigned int >	m_plan_subheading;
		std::deque< unsigned int >	m_plan_bold;

		std::map< unsigned int, ParsingApplierFn >
									m_parsers;

		// LINKS
		void						clear_links( int, int );
		void						clear_links( void );

		// TAGS
		static Glib::RefPtr< Tag >	m_tag_heading;
		static Glib::RefPtr< Tag >	m_tag_subheading;
		static Glib::RefPtr< Tag >	m_tag_match;
		static Glib::RefPtr< Tag >	m_tag_markup;
		static Glib::RefPtr< Tag >	m_tag_bold;
		static Glib::RefPtr< Tag >	m_tag_italic;
		static Glib::RefPtr< Tag >	m_tag_strikethrough;
		static Glib::RefPtr< Tag >	m_tag_highlight;
		static Glib::RefPtr< Tag >	m_tag_link;
		static Glib::RefPtr< Tag >	m_tag_link_broken;
		static Glib::RefPtr< Tag >	m_tag_checkbox;

		// THEMING
		static Gdk::Color			m_theme_color_match;
		static Gdk::Color			m_theme_color_markup;
		static Gdk::Color			m_theme_color_link;
		static Gdk::Color			m_theme_color_linkbroken;

		// OTHER VARIABLES
		Glib::ustring				m_searchstr;
		bool						m_option_search;
		bool						m_flag_settextoperation;
		bool						m_flag_ongoingoperation;
		bool						m_flag_parsing;
		Entry						*m_ptr2entry;
		Gtk::TextView				*m_ptr2textview;
		GtkSpell					*m_ptr2spellobj;

		typedef std::list< Link* >	ListLinks;
		ListLinks					m_list_links;

	friend class DialogTheme;
	friend class EntryView;
	friend class LinkCheck;
	friend class TextviewDiary;
};


class TextviewDiary : public Gtk::TextView
{
	public:
									TextviewDiary( BaseObjectType*,
												   const Glib::RefPtr<Gtk::Builder>& );

		static TextbufferDiary		*m_buffer;

	protected:
		void						update_link( void );
		virtual bool				on_motion_notify_event( GdkEventMotion* );
		virtual bool				on_button_release_event( GdkEventButton* );
		virtual bool				on_key_press_event( GdkEventKey* );
		virtual bool				on_key_release_event( GdkEventKey* );
		//~ virtual void				on_populate_popup( Gtk::Menu* );
//		virtual void				on_style_changed( const Glib::RefPtr< Gtk::Style >& );

	private:
		const Gdk::Cursor			m_cursor_hand;
		const Gdk::Cursor			m_cursor_xterm;
		const Gdk::Cursor			*m_ptr2cursor_last;
		Link						*m_link_hovered;
};

} // end of namespace LIFEO

#endif
