/***********************************************************************************

	Copyright (C) 2007-2010 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "diarydata.hpp"
#include "lifeobase.hpp"


using namespace LIFEO;


bool
LIFEO::compare_listitems( DiaryElement *item_l, DiaryElement *item_r )
{
	return( item_l->get_date() > item_r->get_date() );
}

bool
LIFEO::compare_listitems_by_name( DiaryElementNamed *item_l, DiaryElementNamed *item_r )
{
	return( item_l->get_name() < item_r->get_name() );
}

bool
LIFEO::compare_names( const Glib::ustring &name_l, const Glib::ustring &name_r )
{
	return( name_l < name_r );
}

// DIARYELEMENT ================================================================
// STATIC MEMBERS
ListData::Colrec		*ListData::colrec;
bool					DiaryElement::FLAG_ALLOCATE_GUI_FOR_DIARY = true;

DiaryElement::DiaryElement( void )
{
	if( FLAG_ALLOCATE_GUI_FOR_DIARY )
		m_list_data = new ListData;
}

// NOTE: this function may be moved to Chapter class if no other class...
// ...derived from DiaryelementContainer is ever going to use it 
DiaryElement*
DiaryElementContainer::add_to_list( Glib::RefPtr< Gtk::TreeStore > &treestore_entries,
									const Gtk::TreeRow &row_diary )
{
	DiaryElement				*ptr2elem = NULL;
	bool						flag_row_added = false;
	Gtk::TreeRow				row_container, row_entry;

	for( SetDiaryElements::iterator	iter_item = m_items.begin();
		 iter_item != m_items.end();
		 ++iter_item )
	{
		ptr2elem = * iter_item;
		if( ptr2elem->get_filtered_out() == false )
		{
			// this is here to prevent empty folders when filtering
			if( flag_row_added == false )
			{
				row_container = * treestore_entries->append( row_diary.children() );
				// ptr must be first:
				row_container[ ListData::colrec->ptr ] = this;
				row_container[ ListData::colrec->id ] = get_1st_str();
				row_container[ ListData::colrec->info ] = get_2nd_str();
				row_container[ ListData::colrec->icon ] = get_icon();

				m_list_data->treepath = treestore_entries->get_path( row_container );

				flag_row_added = true;
			}

			Lifeobase::m_entrycount++;
			row_entry = * treestore_entries->append( row_container.children() );
			row_entry[ ListData::colrec->ptr ] = ptr2elem;
			row_entry[ ListData::colrec->id ] = ptr2elem->get_1st_str();
			row_entry[ ListData::colrec->info ] = ptr2elem->get_2nd_str();
			row_entry[ ListData::colrec->icon ] = ptr2elem->get_icon();

			ptr2elem->m_list_data->treepath = treestore_entries->get_path( row_entry );
		}
	}

	return( flag_row_added ? ptr2elem : NULL );
}

// TAG =========================================================================
// STATIC MEMBERS
Tag::Shower		*Tag::shower = NULL;

Tag::Tag( const Glib::ustring &name, CategoryTags *category )
	: DiaryElementContainer( name ), m_ptr2category( category )
{
	if( category != NULL )
		category->insert( this );
}

Tag::~Tag( void )
{
}

Glib::RefPtr< Gdk::Pixbuf >&
Tag::get_icon( void ) const
{
	return Lifeobase::icons->tag_16;
}

void
Tag::show( void )
{
	if( shower != NULL )
		shower->show( *this );
	else
		PRINT_DEBUG( "Tag has no graphical data!" );
}

void
Tag::set_name( const Glib::ustring &name )
{
	m_name = name;
}

void
Tag::set_category( CategoryTags *category_new )
{
	if( m_ptr2category )
		m_ptr2category->erase( this );
	if( category_new )
		category_new->insert( this );
	m_ptr2category = category_new;
}

// TAGPOOL =====================================================================
PoolTags::~PoolTags()
{
	for( iterator iter = begin(); iter != end(); ++iter )
		delete iter->second;
}

Tag*
PoolTags::get_tag( unsigned int tagorder )
{
	if( tagorder >= size() )
		return NULL;

	const_iterator iter = begin();
	advance( iter, tagorder );
	return iter->second;
}

Tag*
PoolTags::get_tag( const Glib::ustring &name )
{
	iterator iter = find( name );
	if( iter != end() )
		return( iter->second );
	else
		return NULL;
}

Tag*
PoolTags::create( const Glib::ustring &name, CategoryTags *category )
{
	iterator iter = find( name );
	if( iter != end() )
	{
		PRINT_DEBUG( "Tag already exists: " + name );
		return( iter->second );
	}
	Tag *tag = new Tag( name, category );
	insert( value_type( name, tag ) );
	return tag;
}

bool
PoolTags::rename( Tag *tag, const Glib::ustring &new_name )
{
	erase( tag->m_name );
	tag->m_name = new_name;
	return( insert( value_type( new_name, tag ) ).second );
}

void
PoolTags::clear( void )
{
	for( iterator iter = begin(); iter != end(); ++iter )
		delete iter->second;

	std::map< Glib::ustring, Tag*, FuncCompareStrings >::clear();
}

// CATEGORYTAGS ================================================================
// STATIC MEMBERS
CategoryTags::Shower		*CategoryTags::shower;

CategoryTags::CategoryTags( const Glib::ustring &name )
:	DiaryElementNamed( name ),
	std::set< Tag*, FuncCompareDiaryElemNamed >( compare_listitems_by_name ),
	m_flag_expanded( true )
{
}

Glib::RefPtr< Gdk::Pixbuf >&
CategoryTags::get_icon( void ) const
{
	return Lifeobase::icons->tag_category_16;
}

void
CategoryTags::show( void )
{
	if( shower != NULL )
		shower->show( *this );
	else
		PRINT_DEBUG( "Category has no graphical data!" );
}

// POOL OF DEFINED TAG CATEGORIES
PoolCategoriesTags::PoolCategoriesTags( void )
:	std::map< Glib::ustring, CategoryTags*, FuncCompareStrings >( compare_names )
{
}

PoolCategoriesTags::~PoolCategoriesTags()
{
	for( iterator iter = begin(); iter != end(); ++iter )
	{
		delete iter->second;
	}
}

void
PoolCategoriesTags::clear( void )
{
	for( iterator iter = begin(); iter != end(); ++iter )
	{
		delete iter->second;
	}

	std::map< Glib::ustring, CategoryTags*, FuncCompareStrings >::clear();
}

bool
PoolCategoriesTags::rename_category( CategoryTags *category, const Glib::ustring &new_name )
{
	erase( category->m_name );
	category->m_name = new_name;
	return( insert( value_type( new_name, category ) ).second );
}

// TAGSET
Tagset::~Tagset( void )
{
}

bool
Tagset::add( Tag *tag )
{
	if( insert( tag ).second ) // if did not exist previously
		return true;
	else
	{
		PRINT_DEBUG( " tagset already has the tag " + tag->get_name() );
		return false;
	}

}

bool
Tagset::checkfor_member( const Tag *tag ) const
{
	return( count( const_cast< Tag* >( tag ) ) > 0 );
}

const Tag*
Tagset::get_tag( unsigned int tagorder ) const
{
	unsigned int i = 0;

	for( Tagset::const_iterator iter = this->begin(); iter != this->end(); ++iter )
	{
		if( i == tagorder )
		{
			return( *iter );
		}
		i++;
	}
	return NULL;
}

// CHAPTER =====================================================================
// STATIC MEMBERS
Chapter::Shower		*Chapter::shower;

Chapter::Chapter( const Glib::ustring &name )
:	DiaryElementContainer( name ), m_date_begin( 0xFFFFFFFF ), m_flag_initialized( false ),
	m_flag_expanded( false )
{
}

Chapter::Chapter( const Glib::ustring &name, Date date )
	:	DiaryElementContainer( name ), m_date_begin( date ), m_flag_initialized( true ),
	m_flag_expanded( false )
{
}


Glib::RefPtr< Gdk::Pixbuf >&
Chapter::get_icon( void ) const
{
	return Lifeobase::icons->chapter_16;
}

void
Chapter::show( void )
{
	if( shower != NULL )
		shower->show( *this );
	else
		PRINT_DEBUG( "Chapter has no graphical data!" );

}

void
Chapter::set_name( const Glib::ustring &name )
{
	m_name = name;
}

Glib::ustring
Chapter::get_date_str( void ) const
{
	if( m_flag_initialized )
		return m_date_begin.format_string();
	else
		return "";
}

Date
Chapter::get_date( void ) const
{
	return m_date_begin;
}

void
Chapter::set_date( Date date )
{
	m_date_begin = date;
	m_flag_initialized = true;
}

// CHAPTER CATEGORY ============================================================
CategoryChapters::CategoryChapters( const Glib::ustring &name )
:	DiaryElementNamed( name ),
	std::set< Chapter*, FuncCompareDiaryElem >( compare_listitems )
{
}

CategoryChapters::~CategoryChapters()
{
	for( CategoryChapters::iterator iter = begin(); iter != end(); ++iter )
		delete( * iter );
}

// not used:
Glib::RefPtr< Gdk::Pixbuf >&
CategoryChapters::get_icon( void ) const
{
	PRINT_DEBUG( "This function should have never called: CategoryChapters::get_icon()" );
	return Lifeobase::icons->chapter_16;
}

Chapter*
CategoryChapters::add_chapter( const Glib::ustring &name )
{
	Chapter *chapter = new Chapter( name );
	insert( chapter );
	return chapter;
}

void
CategoryChapters::remove_chapter( Chapter *chapter )
{
	erase( chapter );
	delete chapter;
}

bool
CategoryChapters::rename_chapter( Chapter *chapter, const Glib::ustring &name )
{
	// Chapters must have unique begin dates not names
	chapter->set_name( name );
	return true;
}

bool
CategoryChapters::set_chapter_date( Chapter *chapter, const Glib::ustring &str_date )
{
	Glib::Date date;

	date.set_parse( str_date );

	if( ! date.valid() )
		return false;

	erase( chapter );
	chapter->set_date( Date( date.get_year(), date.get_month(), date.get_day() ) );
	insert( chapter );

	return true;
}

bool
CategoryChapters::set_chapter_date( Chapter *chapter, const Date &date )
{
	// TODO: check validity?

	erase( chapter );
	chapter->set_date( date );
	insert( chapter );

	return true;
}

void
CategoryChapters::clear( void )
{
	for( CategoryChapters::iterator iter = begin(); iter != end(); ++iter )
		delete( * iter );

	std::set< Chapter*, FuncCompareDiaryElem >::clear();
}

PoolCategoriesChapters::~PoolCategoriesChapters()
{
	for( PoolCategoriesChapters::iterator iter = begin(); iter != end(); ++iter )
		delete( iter->second );
}

void
PoolCategoriesChapters::clear( void )
{
	for( PoolCategoriesChapters::iterator iter = begin(); iter != end(); ++iter )
		delete( iter->second );

	std::map< Glib::ustring, CategoryChapters*, FuncCompareStrings >::clear();
}

// THEMES ======================================================================
Theme Theme::system_theme( _( "[system]" ) ); // static system theme
// STATIC MEMBERS
Theme::Shower		*Theme::shower = NULL;

Theme::Theme( const Glib::ustring &name )
:	DiaryElementNamed( name ), color_base( "white" ), color_text( "black" ),
	color_heading( "blue" ), color_subheading( "violet" ), color_highlight( "yellow" )
{

}

Theme::Theme( const Glib::ustring &name,
			  const Glib::ustring &str_font,
			  const std::string &str_base,
			  const std::string &str_text,
			  const std::string &str_heading,
			  const std::string &str_subheading,
			  const std::string &str_highlight )
:	DiaryElementNamed( name ), font( str_font ), color_base( str_base ), color_text( str_text ),
	color_heading( str_heading ), color_subheading( str_subheading ),
	color_highlight( str_highlight )
{
}

Glib::RefPtr< Gdk::Pixbuf >&
Theme::get_icon( void ) const
{
	return Lifeobase::icons->theme_16;
}

void
Theme::show( void )
{
	if( shower != NULL )
		shower->show( *this );
	else
		PRINT_DEBUG( "Theme has no graphical data!" );
}

PoolThemes::~PoolThemes()
{
	for( iterator iter = begin(); iter != end(); ++iter )
		delete iter->second;
}

void
PoolThemes::clear( void )
{
	for( iterator iter = begin(); iter != end(); ++iter )
		delete iter->second;

	std::map< Glib::ustring, Theme*, FuncCompareStrings >::clear();
}

bool
PoolThemes::rename_theme( Theme *theme, const Glib::ustring &new_name )
{
	erase( theme->get_name() );
	theme->set_name( new_name );
	return( insert( value_type( new_name, theme ) ).second );
}
