/* -*- mode: c++ -*-

  This file is part of the Life library

  Author(s): Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
       Date: 2005-09-03

  Copyright (C) 2005,2006 EPFL

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3.0 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/**
   \file points.hpp
   \author Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
   \date 2005-09-03
 */
#ifndef __points_H
#define __points_H 1

#include <boost/multi_index_container.hpp>
#include <boost/multi_index/member.hpp>
#include <boost/multi_index/mem_fun.hpp>
#include <boost/multi_index/ordered_index.hpp>

#include <life/lifemesh/geoelement.hpp>

namespace Life
{
namespace multi_index = boost::multi_index;

/// \cond detail
/*!
  \class Points
  \brief Points container class

  @author Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
  @see
*/
template<uint16_type nDim>
class Points
{
public:


    /** @name Typedefs
     */
    //@{

    typedef GeoElement0D<nDim> point_type;
    typedef multi_index::multi_index_container<
        point_type,
        multi_index::indexed_by<
            // sort by employee::operator<
            multi_index::ordered_unique<multi_index::identity<point_type> >,
            // sort by less<int> on marker
            multi_index::ordered_non_unique<multi_index::tag<detail::by_marker>,
                                            multi_index::const_mem_fun<point_type,
                                                                       Marker1 const&,
                                                                       &point_type::marker> >,

            // sort by less<int> on processId
            multi_index::ordered_non_unique<multi_index::tag<detail::by_pid>,
                                            multi_index::const_mem_fun<point_type,
                                                                       uint16_type,
                                                                       &point_type::processId> >,

            // sort by less<int> on boundary
            multi_index::ordered_non_unique<multi_index::tag<detail::by_location>,
                                            multi_index::const_mem_fun<point_type,
                                                                       bool,
                                                                       &point_type::isOnBoundary> >
            >
        > points_type;


    typedef typename points_type::iterator point_iterator;
    typedef typename points_type::const_iterator point_const_iterator;

    typedef typename points_type::template index<detail::by_marker>::type marker_points;
    typedef typename marker_points::iterator marker_point_iterator;
    typedef typename marker_points::const_iterator marker_point_const_iterator;

    typedef typename points_type::template index<detail::by_pid>::type pid_points;
    typedef typename pid_points::iterator pid_point_iterator;
    typedef typename pid_points::const_iterator pid_point_const_iterator;

    typedef typename points_type::template index<detail::by_location>::type location_points;
    typedef typename location_points::iterator location_point_iterator;
    typedef typename location_points::const_iterator location_point_const_iterator;



    //@}

    /** @name Constructors, destructor
     */
    //@{

    Points()
        :
        _M_points()
    {}

    Points( Points const & f )
        :
        _M_points( f._M_points )
    {}

    virtual ~Points()
    {}

    //@}

    /** @name Operator overloads
     */
    //@{

    Points& operator=( Points const& e )
    {
        if ( this != &e )
        {
            _M_points = e._M_points;
        }
        return *this;
    }

    //@}

    /** @name Accessors
     */
    //@{

    /**
     * \return the points container
     */
    points_type & points() { return _M_points; }

    /**
     * \return the points container
     */
    points_type const& points() const { return _M_points; }


    virtual bool isEmpty() const { return _M_points.empty(); }
    bool isBoundaryPoint( point_type const & e ) const { return _M_points.find( e )->isOnBoundary(); }
    bool isBoundaryPoint( size_type const & id ) const { return _M_points.find( point_type( id ) )->isOnBoundary(); }


    point_type const& point( size_type i ) const { return *_M_points.find( point_type( i ) ); };

    point_iterator beginPoint() { return _M_points.begin(); }
    point_const_iterator beginPoint() const { return _M_points.begin(); }
    point_iterator endPoint() { return _M_points.end(); }
    point_const_iterator endPoint() const { return _M_points.end(); }


    marker_point_iterator beginPointWithMarker( size_type m ) { return _M_points.template get<detail::by_marker>().lower_bound(m); }
    marker_point_const_iterator beginPointWithMarker( size_type m ) const { return _M_points.template get<detail::by_marker>().lower_bound(m); }
    marker_point_iterator endPointWithMarker( size_type m ) { return _M_points.template get<detail::by_marker>().upper_bound(m); }
    marker_point_const_iterator endPointWithMarker( size_type m ) const { return _M_points.template get<detail::by_marker>().upper_bound(m); }

    /**
     * get the points container by id
     *
     *
     * @return the point container by id
     */
    typename points_type::template nth_index<0>::type &
    pointsById()
    {
        return _M_points.template get<0>();
    }

    /**
     * get the points container by id
     *
     *
     * @return the point container by id
     */
    typename points_type::template nth_index<0>::type const&
    pointsById() const
    {
        return _M_points.template get<0>();
    }

    /**
     * get the points container using the marker view
     *
     *
     * @return the point container using marker view
     */
    marker_points &
    pointsByMarker()
    {
        return _M_points.template get<detail::by_marker>();
    }

    /**
     * get the points container using the marker view
     *
     *
     * @return the point container using marker view
     */
    marker_points const&
    pointsByMarker() const
    {
        return _M_points.template get<detail::by_marker>();
    }
    /**
     * get the points container using the location view
     *
     *
     * @return the point container using location view
     */
    location_points &
    pointsByLocation()
    {
        return _M_points.template get<detail::by_location>();
    }

    /**
     * get the points container using the location view
     *
     *
     * @return the point container using location view
     */
    location_points const&
    pointsByLocation() const
    {
        return _M_points.template get<detail::by_location>();
    }

    /**
     * get the begin() iterator on all the internal points
     *
     * @return the begin() iterator on all the internal points
     */
    location_point_iterator beginInternalPoint()
    {
        return _M_points.template get<detail::by_location>().lower_bound(INTERNAL);
    }
    /**
     * get the end() iterator on all the internal points
     *
     * @return the end() iterator on all the internal points
     */
    location_point_iterator endInternalPoint()
    {
        return _M_points.template get<detail::by_location>().upper_bound(INTERNAL);
    }

    /**
     * get the begin() iterator on all the internal points
     *
     * @return the begin() iterator on all the internal points
     */
    location_point_const_iterator beginInternalPoint() const
    {
        return _M_points.template get<detail::by_location>().lower_bound(INTERNAL);
    }

    /**
     * get the end() iterator on all the internal points
     *
     * @return the end() iterator on all the internal points
     */
    location_point_const_iterator endInternalPoint() const
    {
        return _M_points.template get<detail::by_location>().upper_bound(INTERNAL);
    }

    /**
     * get the begin() iterator on all the boundary points
     *
     * @return the begin() iterator on all the boundary points
     */
    location_point_iterator beginPointOnBoundary()
    {
        return _M_points.template get<detail::by_location>().lower_bound(ON_BOUNDARY);
    }
    /**
     * get the end() iterator on all the boundary points
     *
     * @return the end() iterator on all the boundary points
     */
    location_point_iterator endPointOnBoundary()
    {
        return _M_points.template get<detail::by_location>().upper_bound(ON_BOUNDARY);
    }

    /**
     * get the begin() iterator on all the boundary points
     *
     * @return the begin() iterator on all the boundary points
     */
    location_point_const_iterator beginPointOnBoundary() const
    {
        return _M_points.template get<detail::by_location>().lower_bound(ON_BOUNDARY);
    }

    /**
     * get the end() iterator on all the boundary points
     *
     * @return the end() iterator on all the boundary points
     */
    location_point_const_iterator endPointOnBoundary() const
    {
        return _M_points.template get<detail::by_location>().upper_bound(ON_BOUNDARY);
    }

    //@}

    /** @name  Mutators
     */
    //@{


    //@}

    /** @name  Methods
     */
    //@{

    /**
     * add a new point in the mesh
     * @param f a new point
     * @return the new point from the list
     */
    point_type const& addPoint( point_type const& f ) { return *_M_points.insert( f ).first; }

    //@}

private:

    points_type _M_points;
};
/// \endcond
} // Life
#endif /* __points_H */
