/*
 *  Copyright 1994-2012 Olivier Girondel
 *
 *  This file is part of lebiniou.
 *
 *  lebiniou is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  lebiniou is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with lebiniou. If not, see <http://www.gnu.org/licenses/>.
 */

#include "erlang.h"
#include "globals.h"
#include "pnglite.h"


#define PROTOCOL     '7'
#define MAX_PROTOCOL PROTOCOL

char proto = PROTOCOL;

/*
 * Packet format:
 *
 * [ packet size ]        4 bytes
 * [ protocol version ]   4 bytes, should be enough ;)
 * [ protocol specific ]  N bytes
 *
 * See the v?.c files for protocol specific format
 */

u_long id = 1216486988;
u_long options = BE_NONE;

/* erlang port plugin */
#define IN_FD      3
#define OUT_FD     4

#define EDEBUG 1

/* TODO: when we do not USE_FILE,
   remove xfwrite from protocols < v6 */
#define USE_FILE 0
FILE *in = NULL, *out = NULL;

static uint32_t png_size;
static GByteArray *png_bin = NULL;

extern void v1(const Context_t *);
extern void v2(const Context_t *);
extern void v3(const Context_t *);
extern void v4(const Context_t *);
extern void v5(const Context_t *);
extern void v6(const Context_t *);
extern void v7(const Context_t *);


uint32_t last_colormap;
uint32_t last_picture;
uint32_t last_sequence;

char send_colormap_update;
char send_picture_update;
char send_sequence_update;


static int set_nonblocking(int, int);
static Buffer8_t *cam;

void
create(__attribute__ ((unused)) Context_t *ctx)
{
  char *env;

  env = getenv("BINIOU_ERLANG_PROTO");
  if (NULL != env) {
    char v = env[0];
    if ((v < '1') || (v > MAX_PROTOCOL)) {
      printf("[!] Unknown protocol version '%c', setting to %d\n", v, PROTOCOL);
      proto = PROTOCOL;
    } else {
      printf("[i] erlang: setting protocol to %c\n", v);
      proto = v;
    }
  }

  last_colormap = last_picture = last_sequence = 0;

#if USE_FILE
  in = fdopen(IN_FD, "r");
  if (NULL == in)
    xperror("fdopen");
#if 0
  if (-1 == (flags = fcntl(in, F_GETFL, 0)))
    flags = 0;
  if (-1 == fcntl(in, F_SETFL, flags | O_NONBLOCK))
    xperror("fcntl");
#endif

  out = fdopen(OUT_FD, "w");
  if (NULL == out)
    xperror("fdopen");
#endif

  cam = Buffer8_new();
}


void
destroy(__attribute__ ((unused)) Context_t *ctx)
{
#if USE_FILE
  /* TODO fclose */
#endif
  Buffer8_delete(cam);
}


void
xfwrite(const void *ptr, size_t size, size_t nmemb, FILE *stream)
{
  size_t res;

  res = fwrite(ptr, size, nmemb, stream);
  if (res != nmemb)
    xerror("xfwrite: short write (%d vs %d)\n", res, nmemb);
}


static int
set_nonblocking(int fd, int on)
{
  int flags;

  /* If they have O_NONBLOCK, use the Posix way to do it */
#if defined(O_NONBLOCK)
  if (-1 == (flags = fcntl(fd, F_GETFL, 0)))
    flags = 0;
  // printf("POSIX fcntl(%d) flags= %d\n", on, flags); fflush(stdout);
  if (on)
    return fcntl(fd, F_SETFL, flags | O_NONBLOCK);
  else
    return fcntl(fd, F_SETFL, flags & ~O_NONBLOCK);
#else
  /* Otherwise, use the old way of doing it */
  flags = on;
  return ioctl(fd, FIOBIO, &flags);
#endif
}


ssize_t
read_exact(u_char *buf, const size_t len)
{
#if USE_FILE
  int res = fread((void *)buf, sizeof(char), len, in);

  return res;
#else
  ssize_t i;
  size_t got = 0;

  do {
    if ((i = read(IN_FD, buf+got, len-got)) <= 0)
      return i;
    got += i;
  } while (got < len);

  return len;
#endif
}


ssize_t
write_exact(const u_char *buf, const ssize_t len)
{
  ssize_t i, wrote = 0;

  do {
    if ((i = write(OUT_FD, buf+wrote, len-wrote)) <= 0)
      return i;
    wrote += i;
  } while (wrote < len);

  return len;
}


static void
_info(const char *fmt, ...)
{
  va_list ap;

  fprintf(stdout, "[I] ");
  va_start(ap, fmt);
  vfprintf(stdout, fmt, ap);
  va_end(ap);

  fflush(stdout);
}


#define BUFLEN 1024

void
info(const char *fmt, ...)
{
  char buf[BUFLEN+1];
  uint32_t proto = 7, proto2;
  uint32_t total, total2;
  int ret;
  u_char val;
  uint32_t len;
  va_list ap;

#if EDEBUG
  memset((void *)&buf, 0, (BUFLEN+1)*sizeof(char));
  va_start(ap, fmt);
  vsnprintf(buf, BUFLEN, fmt, ap);
  va_end(ap);

  _info("%s\n", buf);

  len = strlen(buf);

  /* send packet size */
  total = sizeof(uint32_t)+sizeof(u_char)+len*sizeof(char);
  total2 = htonl(total);
  ret = write_exact((const u_char *)&total2, sizeof(uint32_t));

  /* protocol version */
  proto2 = htonl(proto);
  ret = write_exact((const u_char *)&proto2, sizeof(uint32_t));

  val = INFO_CHAR;
  ret = write_exact(&val, sizeof(u_char));

  ret = write_exact((const u_char *)&buf, len*sizeof(char));

  /* XXX hack to make gcc happy
   * variable ‘ret’ set but not used [-Werror=unused-but-set-variable]
   */
  ret++;
#endif /* EDEBUG */
}


static void
create_or_skip_sequence(Context_t *ctx, const uint32_t _id, u_char *buff)
{
  uint32_t id = _id;
  Sequence_t *seq = Sequences_find(id);

  if (NULL == seq) { /* Unknown sequence let's create it ! */
    uint32_t lens, lens2;
    u_char size;
    int i;
    Plugin_t *p;

    // info("Creating new sequence %lu", id);
    Sequence_t *new = Sequence_new(id);

    /* lens plugin id or 0 */
    lens = *((uint32_t *)buff);
    lens2 = ntohl(lens);

    if (lens2) {
      p = Plugins_find(lens2);
      assert(NULL != p);
      new->lens = p;
    }

    /* size of the sequence */
    buff += sizeof(uint32_t);
    size = *buff;
    assert(size <= MAX_SEQ_LEN); /* sert a rien mais bon,
				  * "sanity check" preventif :) */
    buff++;
    for (i = 0; i < size; i++) {
      Layer_t *layer;
      uint32_t id2;
      uint8_t mode;

      /* plugin id */
      id = *((uint32_t *)buff);
      id2 = ntohl(id);
      p = Plugins_find(id2);
      assert(NULL != p);

      /* layer mode */
      buff += sizeof(uint32_t);
      mode = *buff;

      if (!(*p->options & BEQ_DISABLED)) {
	layer = Layer_new(p);
	layer->mode = mode;

	new->layers = g_list_append(new->layers, (gpointer)layer);
      }

      buff++;
    }

    Sequence_delete(ctx->sm->transient);
    ctx->sm->transient = new;
  }
}


static void
send_ack(__attribute__ ((unused)) Context_t *ctx, const u_char type, const u_char ack)
{
#define ACKLEN 5
  uint32_t proto = 7, proto2;
  uint32_t total, total2;
  u_char buff[ACKLEN] = {type, 'A', 'c', 'k', ack};
  int ret;

  /* send packet size */
  total = sizeof(uint32_t)+ACKLEN*sizeof(u_char);
  total2 = htonl(total);
  ret = write_exact((const u_char *)&total2, sizeof(uint32_t));

  /* protocol version */
  proto2 = htonl(proto);
  ret = write_exact((const u_char *)&proto2, sizeof(uint32_t));

  /* webcam */
  ret = write_exact(buff, ACKLEN*sizeof(u_char));

  /* XXX hack to make gcc happy
   * variable ‘ret’ set but not used [-Werror=unused-but-set-variable]
   */
  ret++;
}


static unsigned
webcam_png_write_callback(void *input, size_t size, size_t numel, __attribute__ ((unused)) void *user_pointer)
{
  g_byte_array_append(png_bin, input, (size*numel));
  png_size += size*numel;

  return numel;
}


static unsigned
webcam_png_read_callback(void *output, size_t size, size_t numel, void *data)
{
  size_t how_many = size*numel;

  memcpy(output, (const void *)(data+png_size*sizeof(u_char)), how_many);
  png_size += how_many;

  return numel;
}


static void
send_webcam(Context_t *ctx)
{
  uint32_t proto = 7, proto2;
  uint32_t total, total2;
  uint16_t width2, height2;
  int ret;
  u_char val;
  png_t png;

  /* prepare PNG binary */
  png_bin = g_byte_array_new();

  /* translate image to PNG using custom write callback */
  ret = png_open_write(&png, webcam_png_write_callback, NULL);
  if (PNG_NO_ERROR == ret) {
    Buffer8_t *src;
    u_char *data;

    png_size = 0;
    pthread_mutex_lock(&ctx->cam_mtx);
    src = ctx->buffers[CAM_IN_BUFFER];
    Buffer8_copy(src, cam);
    pthread_mutex_unlock(&ctx->cam_mtx);
    Buffer8_flip_v(cam);
    data = cam->buffer;
    ret = png_set_data(&png, WIDTH, HEIGHT, 8, PNG_GREYSCALE, data);

    if (PNG_NO_ERROR != ret)
      xerror("png_set_data: %s\n", png_error_string(ret));
  } else
    xerror("png_open_write: %s\n", png_error_string(ret));

  /* send packet size */
  total = sizeof(uint32_t)+sizeof(u_char)+2*sizeof(uint16_t)
    + png_size*sizeof(u_char);
  total2 = htonl(total);
  ret = write_exact((const u_char *)&total2, sizeof(uint32_t));

  /* protocol version */
  proto2 = htonl(proto);
  ret = write_exact((const u_char *)&proto2, sizeof(uint32_t));

  /* webcam */
  val = WEBCAM_CHAR;
  ret = write_exact(&val, sizeof(u_char));

  /* image width, height */
  width2 = htons(WIDTH);
  height2 = htons(HEIGHT);
  ret = write_exact((const u_char *)&width2, sizeof(uint16_t));
  ret = write_exact((const u_char *)&height2, sizeof(uint16_t));

  /* webcam PNG */
  ret = write_exact(png_bin->data, png_size*sizeof(u_char));
  g_byte_array_free(png_bin, TRUE);

  /* Ack */
  send_ack(ctx, WEBCAM_CHAR, 1);
}

static void
send_biniou(Context_t *ctx)
{
  uint32_t proto = 7, proto2;
  uint32_t total, total2;
  uint16_t width2, height2;
  int ret;
  u_char val;
  png_t png;

  /* prepare PNG binary */
  png_bin = g_byte_array_new();

  /* translate image to PNG using custom write callback */
  ret = png_open_write(&png, webcam_png_write_callback, NULL);
  if (PNG_NO_ERROR == ret) {
    u_char *data;

    png_size = 0;
    data = export_RGB_active_buffer(ctx, 1);
    ret = png_set_data(&png, WIDTH, HEIGHT, 8, PNG_TRUECOLOR, data);

    if (PNG_NO_ERROR != ret)
      xerror("png_set_data: %s\n", png_error_string(ret));
    xfree(data);
  } else
    xerror("png_open_write: %s\n", png_error_string(ret));

  /* send packet size */
  total = sizeof(uint32_t)+sizeof(u_char)+2*sizeof(uint16_t)
    + png_size*sizeof(u_char);
  total2 = htonl(total);
  ret = write_exact((const u_char *)&total2, sizeof(uint32_t));

  /* protocol version */
  proto2 = htonl(proto);
  ret = write_exact((const u_char *)&proto2, sizeof(uint32_t));

  /* biniou */
  val = BINIOU_CHAR;
  ret = write_exact(&val, sizeof(u_char));

  /* image width, height */
  width2 = htons(WIDTH);
  height2 = htons(HEIGHT);
  ret = write_exact((const u_char *)&width2, sizeof(uint16_t));
  ret = write_exact((const u_char *)&height2, sizeof(uint16_t));

  /* webcam PNG */
  ret = write_exact(png_bin->data, png_size*sizeof(u_char));
  g_byte_array_free(png_bin, TRUE);

  /* Ack */
  send_ack(ctx, BINIOU_CHAR, 255);
}


/* Process any command coming from the erlang side */
static int
command(Context_t *ctx)
{
  uint32_t pkt_size, pkt_size2;
  u_char *cmd;
  ssize_t res;
  int ret;

  ret = set_nonblocking(IN_FD, 1); /* FIXME do something with ret */

  res = read_exact((u_char *)&pkt_size, sizeof(uint32_t));
  if (-1 == res)
    return 0;

  ret = set_nonblocking(IN_FD, 0); /* FIXME do something with ret */

  pkt_size2 = ntohl(pkt_size);

  cmd = xcalloc(pkt_size2, sizeof(u_char));
  res = read_exact(cmd, sizeof(u_char)*pkt_size2);

  if (1 == res) { /* got one byte */
    if (STOP_CHAR == cmd[0]) {
      info("Erlang port: got STOP_CHAR");
      xfree(cmd);
      return -1;
    }

    if (WEBCAM_CHAR == cmd[0]) {
      send_webcam(ctx);
      xfree(cmd);
      return 1;
    }

    if (BINIOU_CHAR == cmd[0]) {
      send_biniou(ctx);
      xfree(cmd);
      return 1;
    }

    xerror("Unknown 1-byte command %d\n", cmd[0]);
  }

  if (2 == res) { /* got 2 bytes */
    if (WEBCAM_CHAR == cmd[0]) {
      switch (cmd[1]) {
      case 0: /* set local */
	ctx->cam = CAM_IN_BUFFER;
	break;

      case 1: /* set remote */
	ctx->cam = CAM_OUT_BUFFER;
	break;

      case 2: /* switch */
	ctx->cam = (ctx->cam == CAM_OUT_BUFFER) ? CAM_IN_BUFFER : CAM_OUT_BUFFER;
	break;

      default:
	xerror("Wrong 2-bytes command %d:%d\n", cmd[0], cmd[1]);
	break;
      }
      xfree(cmd);
      return 1;
    }

    xerror("Unknown 2-byte command %d\n", cmd[0]);
  }

  if (4 == res) { /* got command */
    if (CMD_CHAR == cmd[0]) {
      Event_t e;
      
      e.to = (enum RcptTo)cmd[1];
      e.cmd = (enum Command)cmd[2];
      e.arg0 = (enum Arg)cmd[3];
      
      Context_event(ctx, &e);

      xfree(cmd);
      return 1;
    } else
      xerror("Unknown 4-bytes command %d\n", cmd[0]);
  }

  if (UPDATE_CHAR == cmd[0]) { /* set picture/colormap/sequence */
    uint32_t *id, id2;

    id = (uint32_t *)&cmd[2];
    id2 = ntohl(*id);
    
    switch (cmd[1]) {
    case UPDATE_COLORMAP_CHAR:
      ctx->sm->next->cmap_id = id2;
      Context_set_colormap(ctx);
      last_colormap = id2;
      send_colormap_update = 0;
      break;
      
    case UPDATE_PICTURE_CHAR:
      ctx->sm->next->picture_id = id2;
      Context_set_picture(ctx);
      last_picture = id2;
      send_picture_update = 0;
      break;
      
    case UPDATE_SEQUENCE_CHAR:
      create_or_skip_sequence(ctx, id2, (&cmd[2]+sizeof(uint32_t)));
      Context_set_sequence(ctx, id2);
      last_sequence = id2;
      send_sequence_update = send_colormap_update = send_picture_update = 0;
      break;

    default:
      xerror("Wrong 6-byte command\n");
      break;
    }

    xfree(cmd);
    return 1;
  }

  if ((WEBCAM_CHAR == cmd[0]) && (pkt_size2 >= (sizeof(u_char)+2*sizeof(uint16_t)))) {
    uint16_t *width, width2;
    uint16_t *height, height2;
    png_t png;
    Buffer8_t *dst = ctx->buffers[CAM_OUT_BUFFER];

    width = (uint16_t *)&cmd[1];
    height = (uint16_t *)&cmd[3];
    width2 = ntohs(*width);
    height2 = ntohs(*height);

    png_size = 0;
    ret = png_open_read(&png, webcam_png_read_callback, (void *)&cmd[5]);

    if (PNG_NO_ERROR == ret) {
      if ((width2 == WIDTH) && (height2 == HEIGHT)) {
	ret = png_get_data(&png, dst->buffer);
	if (PNG_NO_ERROR != ret)
	  xerror("png_get_data: %s\n", png_error_string(ret));
      } else {
	Pixel_t temp[width2*height2];

	ret = png_get_data(&png, (u_char *)&temp);
	if (PNG_NO_ERROR != ret)
	  xerror("png_get_data: %s\n", png_error_string(ret));
	gray_scale(dst->buffer, width2, height2, temp);
      }
    } else
      xerror("%s:%d png_open_read: %s (%d)\n", __func__, __LINE__, png_error_string(ret), ret);

    Buffer8_flip_v(dst);
    xfree(cmd);
    /* Ack */
    send_ack(ctx, WEBCAM_CHAR, 0);
    return 1;
  } else {
    if (pkt_size2)
      xerror("Wrong webcam command (%lu bytes)\n", pkt_size2);
  }

  if (res)
    xerror("Got wrong packet length from erlang: %lu\n", res);

  return 0; /* not reached */
}


void
run(Context_t *ctx)
{
  uint32_t total, total2;
  int res;

  send_colormap_update =
    send_picture_update =
    send_sequence_update = 1;

  do
    res = command(ctx);
  while (res == 1);

  if (res == -1) {
    u_char c = STOP_CHAR;
    int ret;
    
    ctx->running = 0;

    /* ACK stop */
    /* send packet size */
    total = sizeof(char);
    total2 = htonl(total);
    ret = write_exact((const u_char *)&total2, sizeof(uint32_t));

    /* send STOP_CHAR */
    ret = write_exact((const u_char *)&c, sizeof(u_char));

    /* XXX keep gcc happy */
    ret++; /* <- remove this ! */
  } else {
    switch (proto) {
    case '1':
      v1(ctx);
      break;

    case '2':
      v2(ctx);
      break;

    case '3':
      v3(ctx);
      break;

    case '4':
      v4(ctx);
      break;

    case '5':
      v5(ctx);
      break;

    case '6':
      v6(ctx);
      break;

    case '7':
      v7(ctx);
      break;

    default:
      xerror("Unknown protocol version\n");
      break;
    }
  }
}
