/**
 * \file script-lua.c
 *
 * contains all the lua-scripting hooks.  everything is conditioned
 * on an #ifdef, so cpp substitutes dummies if no lua-hooks
 * are wanted at installation time.
 *
 * see script_lua.h
 * see fetchnews.c::getarticle()
 * see store.c::store_stream()
 *
 * see script-test-store.c for testing all this.
 *
 * clemens fischer <ino-news@spotteswoode.dnsalias.org>
 */

#include "leafnode.h"

#include <stdio.h>

#include <stdarg.h>
#include <stdlib.h>

/* for int eaccess(const char *pathname, int mode) */
#include <unistd.h>

#include <string.h>

#include "ln_log.h"
#include "mastring.h"

/* for "body_size", "owndn", "fqdn": seems overkill */
#include "leafnode.h"

#include "script.h"
#include "script-lua.h"

#include "lua.h"
#include "lauxlib.h"
#include "lualib.h"

static lua_State *L = NULL;

/* contained in "leafnode.h":
- maximum size of msg body to be cached in RAM
extern size_t body_size;
- hostname as configured, eg. for message-IDs
extern char * fqdn;
extern char * owndn;
*/
/* name of file containing script code */
extern char * script_file_lua;

/* -ino: XXX see: filterutil.c::int seconds_since(const char *date) */
extern int seconds_since(const char *);

/* this is the index of the current header in the header_table */
static /*lua_Integer*/ int header_key = 0;

/* this is the index of the header_table on the stack */
static int at_header_table = 0;

/* this is the index of the result_table on the stack */
static int at_result_table = 0;

/* flag: something went wrong: dont call me unless fixed */
static int fn_get_result_dontcall = 0;

static int the_tos = 0; /* current top of stack */

/* reset stack top and return that code */
#define reset_and_return(tos, code) \
    do { \
        lua_settop(L, (tos)); \
        return(code); \
    } while(0)

/* remove old results, adjusting other stack positions */
#define REMOVE_OLD_RESULT(stackpos) \
    do { \
        if ((stackpos) > 0) { \
            lua_remove(L, (stackpos)); \
            if (at_header_table > (stackpos)) at_header_table -= 1; \
            (stackpos) = 0; \
            the_tos = lua_gettop(L); \
        } \
    } while (0)

#define VALIDATE_AND_RETURN_RESULT(status, noerr_code, err_code) \
    do { \
        if (! status) { \
            /* now the tos has the result, it is evaluated later with \
             * get_result */ \
            if (lua_istable(L, -1)) { \
                at_result_table = lua_gettop(L); \
                fn_get_result_dontcall = 0; \
                return((script_return_t) (noerr_code)); \
            } \
        } \
        /*at_result_table = 0;*/ \
        fn_get_result_dontcall = 1; \
        return_val = script_simple_result(status); \
        /*reset_and_return(the_tos, return_val);*/ \
        reset_and_return(the_tos, (script_return_t) (err_code)); \
    } while (0)

#define VALID_MASTR(mastring) \
    (((mastring) != NULL) && (mastr_len(mastring) > 0))

/* if queried using "-1", returns current state of initialization, if
 * called with non-negative number, sets state to that number and
 * returns previous state.
 */
int script_init_ready_lua(int nu_wod)
{
    static int init_ready = 0;
    int state;

    state = init_ready;

    if (nu_wod >= 0) {
        init_ready = nu_wod;
    }
    return state;
}

/* stand-alone lua interpreter */
int main_mini_lua(void);
int
main_mini_lua(void)
{
    char buff[256];
    int err;
    lua_State *Lua = luaL_newstate();
    luaL_openlibs(Lua); /* _all_ standard libs */

    while (fgets(buff, sizeof(buff), stdin) != NULL)
    {
        err = luaL_loadbuffer(Lua, buff, strlen(buff), "line") ||
            lua_pcall(Lua, 0, 0, 0);
        if (err) {
            fprintf(stderr, "%s\n", lua_tostring(Lua, -1));
            lua_pop(Lua, 1);    /* pop error mess */
        }
    }

    lua_close(Lua);
    return 0;
}

/*
void lua_error_abort(lua_State *, const char *, ...);
void
lua_error_abort(lua_State *Lua, const char *fmt, ...)
{
    va_list argp;
    va_start(argp, fmt);
    vprintf(stderr, fmt, argp);
    va_end(argp);
    lua_close(Lua);
    exit(EXIT_FAILURE);
}
*/

/* stack dumper */
/*static*/ void stackDump(lua_State *);
/*static*/ void
stackDump(lua_State *Lua)
{
    int i;
    int top = lua_gettop(Lua);
    for (i=1; i <= top; i++) {
        int t = lua_type(Lua, i);
        switch (t)
        {
            case LUA_TSTRING:
                fprintf(stderr, "pos %3d: '%s'\n", i, lua_tostring(Lua, i));
                break;
            case LUA_TBOOLEAN:
                fprintf(stderr, "pos %3d: '%s'\n", i,
                        (lua_toboolean(Lua, i) ? "true" : "false"));
                break;
            case LUA_TNUMBER:
                fprintf(stderr, "pos %3d: %g\n", i, lua_tonumber(Lua, i));
                break;
            case LUA_TTABLE:
                /*
                While traversing a table, do not call lua_tolstring
                directly on a key, unless you know that the key is
                actually a string. Recall that lua_tolstring changes the
                value at the given index; this confuses the next call to
                lua_next.
                */
                /* table is in the stack at index 'i' */
                fprintf(stderr, "pos %3d: table with content:\n", i);
                lua_pushnil(Lua);  /* first key */
                while (lua_next(Lua, i) != 0) {
                    /* uses 'key' (at index -2) and 'value' (at index -1) */
                    fprintf(stderr, "key: %s - value: %s\n",
                          lua_typename(Lua, lua_type(Lua, -2)),
                          lua_typename(Lua, lua_type(Lua, -1)));
                    /* removes 'value'; keeps 'key' for next iteration */
                    lua_pop(Lua, -1);
                }
                break;
            default:
                fprintf(stderr, "pos %3d: a %s\n", i, lua_typename(Lua, t));
                break;
        }
    }
}

/*
 * whenever there's a string on the stack, we need to copy it, because
 * as soon as it's popped from the stack, its space will eventually be
 * reclaimed.  also, we cannot simply return literal strings, because
 * they bare no mark distinguishing them from malloc'ed stuff.  so we
 * make a copy here and rely on the upstream user to free the space.
 */
script_return_t
mk_mess(const char * string)
{
    char * errm = strdup(string);
    if (errm != NULL) {
        return((script_return_t) errm);
    }
    return((script_return_t) SCRIPT_ERROR_UNSPEC);
}

/*
 * checks simple script results, eg. non-table ones.  it pops one result
 * item from the stack, so you shouldn't be doing this for routines
 * whose values are retrieved using "script_fn_get_result*".
 *
 * NB: if returning a string, YOU are responsible for deallocating it
 * later!
 *
 * XXX what about groups like "007.001"?
 */
static script_return_t
script_simple_result(int lua_stat)
{
    const char * result = NULL;         /* space for string result */
    int result_int;                     /* result field is an int code */
    const char * result_string;         /* result as string */
    lua_Integer result_luaint;          /* result field is a lua int code */
    /*lua_Number result_number;*/       /* result field is a number */

    /* no error in previous lua callout */
    if (lua_stat == 0) {
        if (lua_isnumber(L, -1)) {
            /*result_number = lua_tonumber(L, -1);*/
            result_luaint = lua_tointeger(L, -1);
            lua_pop(L, 1);
            result_int = (int) result_luaint;
            if (result_int < SCRIPT_STATUS_MAX) {
                return((script_return_t) result_int);
            } else {
                /* result numeric, but undefined code */
                return((script_return_t) SCRIPT_API_RETURN_PROB);
            }
        }
        if (lua_isstring(L, -1)) {
            result_string = lua_tostring(L, -1);
            /* alloc space for result */
            result = (script_return_t) mk_mess(result_string);
            lua_pop(L, 1);
            return((script_return_t) result);
        }
        /* simple result must be number or string! */
        return((script_return_t) SCRIPT_API_RETURN_PROB);
    } else {
        /* did lua leave an error message for us? */
        if (lua_isstring(L, -1)) {
            result_string = lua_tostring(L, -1);
            fprintf(stderr, "%s\n", result_string);
            /* alloc space for err mess */
            result = (script_return_t) mk_mess(result_string);
            lua_pop(L, 1);
            return((script_return_t) result);
        }
        /* no description of the problem, return generic error number */
        return((script_return_t) SCRIPT_RUNTIME_PROB);
    }
    /* notreached */
}

unsigned long int
check_script_return_lua(
        script_return_t kukku,
        const char * script_name,
        const char * mid,
        script_cmd_t get_what,
        script_result_t * what)
{
    int return_code = 0;
    script_result_t script_result = SCRIPT_NOERROR;

    switch ((unsigned int) kukku)
    {
        case SCRIPT_UNAVAILABLE:
            return_code = 0;
            break;
        case SCRIPT_NOCHANGE:
        case SCRIPT_NOCHANGE_HEADER:
        case SCRIPT_NOCHANGE_BODY:
            return_code = (unsigned int) kukku;
            break;
        case SCRIPT_REJECT:
        case SCRIPT_IGNORE_ARTICLE:
        case SCRIPT_IGNORE_GROUP:
            ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
            "script: item %s: rejected by %s",
            mid, script_name);
            return_code = (unsigned int) kukku;
            break;
        case SCRIPT_ILLEGAL_SEQ:
            ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
            "script: item %s: script check_script_return_lua"
            " after %s called out-of-sequence",
            mid, script_name);
            /* cannot be ignored */
            return_code = (unsigned int) kukku;
            break;
        case SCRIPT_ILLEGAL_API_ARGS:
            ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
            "script: item %s: script %s got illegal arguments",
            mid, script_name);
            /* cannot be ignored */
            return_code = (unsigned int) kukku;
            break;
        case SCRIPT_API_RETURN_PROB:
        case SCRIPT_ERROR_UNSPEC:
        case SCRIPT_RUNTIME_PROB:
            ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
            "script: item %s: script %s returns undefined numeric result",
            mid, script_name);
            /* currently, user script errors are ignored */
            return_code = 0;
            break;
        case SCRIPT_WRONG_RESULT:
            ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
            "script: item %s: script %s returns wrong result type",
            mid, script_name);
            /* currently, user script errors are ignored */
            return_code = 0;
            break;
        case SCRIPT_NOERROR:
            if (get_what != SCRIPT_GET_NOTHING) {
                script_result = script_fn_get_result(get_what);
                switch ((unsigned int) script_result)
                {
                    case SCRIPT_IGNORE_GROUP:
                    case SCRIPT_IGNORE_ARTICLE:
                    case SCRIPT_REJECT:
                        ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
                        "script: item %s: rejected by %s",
                        mid, script_name);
                        return_code = (unsigned int) script_result;
                        break;
                    case SCRIPT_ILLEGAL_API_ARGS:
                        ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
                        "script: item %s: script %s got illegal arguments",
                        mid, script_name);
                        /* cannot be ignored */
                        return_code = (unsigned int) script_result;
                        break;
                    case SCRIPT_ILLEGAL_SEQ:
                        ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
                        "script: item %s: script fn_get_result"
                        " after %s called out-of-sequence",
                        mid, script_name);
                        /* cannot be ignored */
                        return_code = (unsigned int) script_result;
                        break;
                    case SCRIPT_API_RETURN_PROB:
                    case SCRIPT_ERROR_UNSPEC:
                    case SCRIPT_RUNTIME_PROB:
                        ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
                        "script: item %s: script %s returns undefined numeric result",
                        mid, script_name);
                        /* currently, user script errors are ignored */
                        return_code = 0;
                        break;
                    case SCRIPT_WRONG_RESULT:
                        ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
                        "script: item %s: script %s returns wrong result type",
                        mid, script_name);
                        /* currently, user script errors are ignored */
                        return_code = 0;
                        break;
                    case SCRIPT_NOCHANGE:
                    case SCRIPT_NOCHANGE_HEADER:
                    case SCRIPT_NOCHANGE_BODY:
                    case SCRIPT_UNAVAILABLE:
                    case SCRIPT_NOERROR:
                        return_code = 0;
                        break;
                    default:
                        if ((what != NULL) &&
                            ((unsigned int) script_result > SCRIPT_STATUS_MAX)) {
                            mastr_delete(* what);
                            * what = script_result;
                        }
                        break;
                }
            } else return_code = 0;
            break;
        default:
            if ((unsigned int) kukku > SCRIPT_STATUS_MAX) {
                ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
                       "script: item %s script %s message %s",
                       mid, script_name, kukku);
                /* text messages come malloc(3)ed */
                free(/*(void *)*/ kukku);
                return_code = 1;
            } else {
                ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
                    "script: item %s: script %s "
                    "returns undefined numeric result %d",
                    mid, script_name, (int) kukku);
                /* currently, user script errors are ignored */
                return_code = 0;
            }
            break;
    }
    return return_code;
}

/* every C-function gets its own local stack */
static int
ln_log_lua(lua_State *LuaC)
{
    lua_Integer severity;         /* severity */
    lua_Integer context;          /* context */
    const char * message;         /* message */
    int nargs = lua_gettop(LuaC); /* number of arguments */

    if (nargs != 3) {
        ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
               "%s: needs three arguments, got %d", LN_LOG_LUA, nargs);
        return 0;
    }
    if (!lua_isnumber(LuaC, 1)) {
        ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
               "%s: severity must be numeric", LN_LOG_LUA);
        return 0;
    }
    if (!lua_isnumber(LuaC, 2)) {
        ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
               "%s: context must be numeric", LN_LOG_LUA);
        return 0;
    }
    if (!lua_isstring(LuaC, 3)) {
        ln_log(LNLOG_SINFO, LNLOG_CARTICLE,
               "%s: message must be string", LN_LOG_LUA);
        return 0;
    }

    severity = lua_tointeger(LuaC, 1);
    context = lua_tointeger(LuaC, 1);
    message = lua_tostring(LuaC, 3);

    ln_log(severity, context, "%s", message);
    return 0;
}

#define LN_AGE_FMTERR LN_AGE_LUA ": date_string must be string"
#define LN_AGE_ARGERR LN_AGE_LUA ": need one argument: date_string"
/* -ino: XXX see: filterutil.c::int seconds_since(const char *date) */
static int
ln_age_lua(lua_State *LuaC)
{
    const char * datestr; /* date string from article header */
    int age_seconds;      /* age of article in seconds since epoch */
    int nargs = lua_gettop(LuaC); /* number of arguments */

    if (nargs != 1) {
        ln_log(LNLOG_SINFO, LNLOG_CARTICLE, LN_AGE_ARGERR);
        lua_pushnil(LuaC);
        lua_pushliteral(LuaC, LN_AGE_FMTERR);
        return 2;
    }
    if (!lua_isstring(LuaC, 1)) {
        ln_log(LNLOG_SINFO, LNLOG_CARTICLE, LN_AGE_FMTERR);
        lua_pushnil(LuaC);
        lua_pushliteral(LuaC, LN_AGE_FMTERR);
        return 2;
    }
    datestr = lua_tostring(LuaC, 1);
    age_seconds = seconds_since(datestr);
    if (age_seconds >= 1000 && age_seconds <= 1009) {
        lua_pushnil(LuaC);
        lua_pushliteral(LuaC, LN_AGE_FMTERR);
        return 2;
    }
    lua_pushinteger (LuaC, (lua_Integer) age_seconds);
    return 1;
}

const char *
mk_init_file_lua(const char * default_filename)
{
    if (script_file_lua && *script_file_lua)
        return((const char *) script_file_lua);
    else
        return default_filename;
}

static int
file_can_read(const char * filename)
{
    int acc_ok;

    /* For an unknown reason, this triggers:
       "warning: implicit declaration of function 'eaccess'"
       "warning: nested extern declaration of 'eaccess'"
       It is declared in "unistd.h", included above.
       See EUIDACCESS(3), EACCESS(3).
    */
#ifdef HAVE_EACCESS
    acc_ok = eaccess(filename, R_OK);
#else
    acc_ok = access(filename, R_OK);
#endif

    return (! acc_ok);
}

/*
lua_atpanic

[-0, +0, -]

lua_CFunction lua_atpanic (lua_State *L, lua_CFunction panicf);

Sets a new panic function and returns the old one.

If an error happens outside any protected environment, Lua calls a
panic function and then calls exit(EXIT_FAILURE), thus exiting the host
application. Your panic function may avoid this exit by never returning
(e.g., doing a long jump).

The panic function can access the error message at the top of the stack.
*/

/*
static int max_lua_errors = MAX_SCRIPT_ERRORS;
static int fn_panic(lua_State *L)
{
    fprintf(stderr,
            "PANIC: unprotected error in call to Lua API (%s)\n",
            lua_tostring(L, -1));
    return 0;
}
*/

int
script_ln_register_user_lua(const char * user)
{
    if (! script_init_ready_lua(-1)) return -1;

    lua_pushstring(L, user);
    lua_setglobal(L, LN_AUTHENTICATED_USER);
    return 0;
}

script_return_t
script_ln_chk_grp_access_lua(const char * cmd, const char * group)
{
    int call_err = 0;
    script_return_t return_val;

    /*printf("%s: cmd: %s, group: %s\n", __func__, cmd, group);*/
    if (! script_init_ready_lua(-1)) {
        return((script_return_t) SCRIPT_UNAVAILABLE);
    }

    the_tos = lua_gettop(L);

    lua_getglobal(L, LN_ENTER_GROUP);
    if (! lua_isfunction(L, -1)) {
        /* The function is not properly defined */
        lua_pop(L, 1);
        reset_and_return(the_tos, (script_return_t) SCRIPT_UNAVAILABLE);
    }

    lua_pushstring(L, cmd);
    lua_pushstring(L, group);

    call_err = lua_pcall(L, 2, 1, 0);
    return_val = script_simple_result(call_err);
    reset_and_return(the_tos, return_val);
    /* notreached */
}

script_return_t
script_ln_init_lua(void)
{
    int call_err = 0;
    const char * hooks_file;
    const char * errmess_string; /* result as string */

    L = luaL_newstate();
    if (L == NULL) {
        (void) script_init_ready_lua(0);
        return (script_return_t) SCRIPT_WRONG_RESULT;
        /* notreached */
    }

    /*(void)lua_atpanic(L, &fn_panic);*/

    luaL_openlibs(L);   /* _all_ standard libs */

    /* make std status return codes lua globals */
    lua_pushinteger(L, (lua_Integer) SCRIPT_NOERROR);
    lua_setglobal(L, "SCRIPT_NOERROR");
    lua_pushinteger(L, (lua_Integer) SCRIPT_REJECT);
    lua_setglobal(L, "SCRIPT_REJECT");
    lua_pushinteger(L, (lua_Integer) SCRIPT_NOCHANGE);
    lua_setglobal(L, "SCRIPT_NOCHANGE");
    lua_pushinteger(L, (lua_Integer) SCRIPT_NOCHANGE_HEADER);
    lua_setglobal(L, "SCRIPT_NOCHANGE_HEADER");
    lua_pushinteger(L, (lua_Integer) SCRIPT_NOCHANGE_BODY);
    lua_setglobal(L, "SCRIPT_NOCHANGE_BODY");
    lua_pushinteger(L, (lua_Integer) SCRIPT_UNAVAILABLE);
    lua_setglobal(L, "SCRIPT_UNAVAILABLE");
    lua_pushinteger(L, (lua_Integer) SCRIPT_IGNORE_ARTICLE);
    lua_setglobal(L, "SCRIPT_IGNORE_ARTICLE");
    lua_pushinteger(L, (lua_Integer) SCRIPT_IGNORE_GROUP);
    lua_setglobal(L, "SCRIPT_IGNORE_GROUP");
    lua_pushinteger(L, (lua_Integer) SCRIPT_ILLEGAL_SEQ);
    lua_setglobal(L, "SCRIPT_ILLEGAL_SEQ");
    lua_pushinteger(L, (lua_Integer) SCRIPT_ILLEGAL_API_ARGS);
    lua_setglobal(L, "SCRIPT_ILLEGAL_API_ARGS");
    lua_pushinteger(L, (lua_Integer) SCRIPT_API_RETURN_PROB);
    lua_setglobal(L, "SCRIPT_API_RETURN_PROB");
    lua_pushinteger(L, (lua_Integer) SCRIPT_ERROR_UNSPEC);
    lua_setglobal(L, "SCRIPT_ERROR_UNSPEC");
    lua_pushinteger(L, (lua_Integer) SCRIPT_WRONG_RESULT);
    lua_setglobal(L, "SCRIPT_WRONG_RESULT");

    /* the logging constants, see ln_log.h */
    lua_pushinteger(L, (lua_Integer) LNLOG_SCRIT);
    lua_setglobal(L, "LNLOG_SCRIT");
    lua_pushinteger(L, (lua_Integer) LNLOG_SERR);
    lua_setglobal(L, "LNLOG_SERR");
    lua_pushinteger(L, (lua_Integer) LNLOG_SWARNING);
    lua_setglobal(L, "LNLOG_SWARNING");
    lua_pushinteger(L, (lua_Integer) LNLOG_SNOTICE);
    lua_setglobal(L, "LNLOG_SNOTICE");
    lua_pushinteger(L, (lua_Integer) LNLOG_SINFO);
    lua_setglobal(L, "LNLOG_SINFO");
    lua_pushinteger(L, (lua_Integer) LNLOG_SDEBUG);
    lua_setglobal(L, "LNLOG_SDEBUG");
    lua_pushinteger(L, (lua_Integer) LNLOG_SMIN);
    lua_setglobal(L, "LNLOG_SMIN");
    lua_pushinteger(L, (lua_Integer) LNLOG_CTOP);
    lua_setglobal(L, "LNLOG_CTOP");
    lua_pushinteger(L, (lua_Integer) LNLOG_CSERVER);
    lua_setglobal(L, "LNLOG_CSERVER");
    lua_pushinteger(L, (lua_Integer) LNLOG_CGROUP);
    lua_setglobal(L, "LNLOG_CGROUP");
    lua_pushinteger(L, (lua_Integer) LNLOG_CARTICLE);
    lua_setglobal(L, "LNLOG_CARTICLE");
    lua_pushinteger(L, (lua_Integer) LNLOG_CALL);
    lua_setglobal(L, "LNLOG_CALL");

    /* export constants that may change at compile time */
    lua_pushstring(L, SCRIPT_UNAUTHENTICATED_USER);
    lua_setglobal(L, LN_UNAUTHENTICATED_USER);
    lua_pushstring(L, SCRIPT_UNAUTHENTICATED_USER);
    lua_setglobal(L, LN_AUTHENTICATED_USER);
    lua_pushstring(L, SCRIPT_CURRENT_GROUP);
    lua_setglobal(L, LN_CURRENTGROUP);

    /* export configuration items */
    lua_pushinteger(L, (lua_Integer) body_size);
    lua_setglobal(L, "LN_BODY_SIZE");
    /* FIXME! using "fqdn" causes a SIGSEGV! */
    /*lua_pushstring(L, owndn ? owndn : (fqdn ? fqdn : "LN_NO_HOSTNAME"));*/
    lua_pushstring(L, (owndn && owndn[0] != '\0') ?
            owndn : "script_ln_init_lua:NO_HOSTNAME");
    lua_setglobal(L, "LN_HOSTNAME");

    /* c-functions to be called from lua */
    lua_register(L, LN_LOG_LUA, ln_log_lua);
    lua_register(L, LN_AGE_LUA, ln_age_lua);

    hooks_file = mk_init_file_lua(CONFIG_SCRIPT_DEFAULT);
    /*printf("%s: scriptfile: %s\n", __func__, hooks_file ? hooks_file : "NULL");*/
    if (hooks_file == NULL) {
        (void) script_init_ready_lua(0);
        reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_DOFILE_CANTREAD));
    }

    the_tos = lua_gettop(L);

    if (file_can_read(hooks_file)) {
        call_err = luaL_dofile(L, hooks_file);
        if (! call_err) goto call_ln_init;

        /* some error */
        (void) script_init_ready_lua(0);
        if (lua_isstring(L, -1)) {
            errmess_string = lua_tostring(L, -1);
            fprintf(stderr, "%s\n", errmess_string);
            errmess_string = mk_mess(errmess_string);
            lua_pop(L, 1);
            return((script_return_t) errmess_string);
            /*return((script_return_t) mk_mess(ERRMESS_DOFILE_UNSPEC));*/
        }
        lua_close(L);
        switch (call_err)
        {
            case LUA_ERRFILE:
                /* cannot read file */
                return((script_return_t) mk_mess(ERRMESS_DOFILE_CANTREAD));
                /* notreached */
                break;
            case LUA_ERRSYNTAX:
                /* syntax error during pre-compilation */
                return((script_return_t) mk_mess(ERRMESS_DOFILE_SYNTAX));
                /* notreached */
                break;
            case LUA_ERRMEM:
                /* memory allocation error */
                return((script_return_t) mk_mess(ERRMESS_MEMORY_ALLOC));
                /* notreached */
                break;
            default:
                /* unknown error */
                return((script_return_t) mk_mess(ERRMESS_DOFILE_UNSPEC));
                /* notreached */
                break;
        }
        /* notreached */
    } else {
        (void) script_init_ready_lua(0);
        lua_close(L);
        return((script_return_t) mk_mess(ERRMESS_DOFILE_CANTREAD));
        /* notreached */
    }

call_ln_init:

    lua_getglobal(L, LN_INIT);
    if (! lua_isfunction(L, -1)) {
        /* The function is not properly defined */
        lua_pop(L, 1);
        reset_and_return(the_tos, (script_return_t) SCRIPT_UNAVAILABLE);
    }

    call_err = lua_pcall(L, 0, 0, 0);
    if (! call_err) {
        reset_and_return(the_tos, (script_return_t) SCRIPT_NOERROR);
    }
    (void) script_init_ready_lua(0);
    fprintf(stderr, "%s\n", lua_tostring(L, -1));
    lua_pop(L, 1);  /* pop error mess */
    switch (call_err)
    {
        case LUA_ERRRUN:
            /* runtime error */
            reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_RUNTIME));
            /* notreached */
            break;
        case LUA_ERRMEM:
            /* memory allocation error */
            reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_MEMORY_ALLOC));
            /* notreached */
            break;
        case LUA_ERRERR:
            /* error while running error handler */
            /* cannot happen without error handler */
            reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_ERR_HANDLER));
            /* notreached */
            break;
        default:
            reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_API_RETURN_PROB));
            /* notreached */
    }
    /* notreached */
}

script_return_t
script_fn_init_lua(void)
{
    int call_err = 0;
    const char * hooks_file;
    const char * errmess_string; /* result as string */

    L = luaL_newstate();
    if (L == NULL) {
        (void) script_init_ready_lua(0);
        return (script_return_t) SCRIPT_WRONG_RESULT;
        /* notreached */
    }

    /*(void)lua_atpanic(L, &fn_panic);*/

    luaL_openlibs(L);   /* _all_ standard libs */

    /* make std status return codes lua globals */
    lua_pushinteger(L, (lua_Integer) SCRIPT_NOERROR);
    lua_setglobal(L, "SCRIPT_NOERROR");
    lua_pushinteger(L, (lua_Integer) SCRIPT_REJECT);
    lua_setglobal(L, "SCRIPT_REJECT");
    lua_pushinteger(L, (lua_Integer) SCRIPT_NOCHANGE);
    lua_setglobal(L, "SCRIPT_NOCHANGE");
    lua_pushinteger(L, (lua_Integer) SCRIPT_NOCHANGE_HEADER);
    lua_setglobal(L, "SCRIPT_NOCHANGE_HEADER");
    lua_pushinteger(L, (lua_Integer) SCRIPT_NOCHANGE_BODY);
    lua_setglobal(L, "SCRIPT_NOCHANGE_BODY");
    lua_pushinteger(L, (lua_Integer) SCRIPT_UNAVAILABLE);
    lua_setglobal(L, "SCRIPT_UNAVAILABLE");
    lua_pushinteger(L, (lua_Integer) SCRIPT_IGNORE_ARTICLE);
    lua_setglobal(L, "SCRIPT_IGNORE_ARTICLE");
    lua_pushinteger(L, (lua_Integer) SCRIPT_IGNORE_GROUP);
    lua_setglobal(L, "SCRIPT_IGNORE_GROUP");
    lua_pushinteger(L, (lua_Integer) SCRIPT_ILLEGAL_SEQ);
    lua_setglobal(L, "SCRIPT_ILLEGAL_SEQ");
    lua_pushinteger(L, (lua_Integer) SCRIPT_ILLEGAL_API_ARGS);
    lua_setglobal(L, "SCRIPT_ILLEGAL_API_ARGS");
    lua_pushinteger(L, (lua_Integer) SCRIPT_API_RETURN_PROB);
    lua_setglobal(L, "SCRIPT_API_RETURN_PROB");
    lua_pushinteger(L, (lua_Integer) SCRIPT_ERROR_UNSPEC);
    lua_setglobal(L, "SCRIPT_ERROR_UNSPEC");
    lua_pushinteger(L, (lua_Integer) SCRIPT_WRONG_RESULT);
    lua_setglobal(L, "SCRIPT_WRONG_RESULT");

    /* the logging constants, see ln_log.h */
    lua_pushinteger(L, (lua_Integer) LNLOG_SCRIT);
    lua_setglobal(L, "LNLOG_SCRIT");
    lua_pushinteger(L, (lua_Integer) LNLOG_SERR);
    lua_setglobal(L, "LNLOG_SERR");
    lua_pushinteger(L, (lua_Integer) LNLOG_SWARNING);
    lua_setglobal(L, "LNLOG_SWARNING");
    lua_pushinteger(L, (lua_Integer) LNLOG_SNOTICE);
    lua_setglobal(L, "LNLOG_SNOTICE");
    lua_pushinteger(L, (lua_Integer) LNLOG_SINFO);
    lua_setglobal(L, "LNLOG_SINFO");
    lua_pushinteger(L, (lua_Integer) LNLOG_SDEBUG);
    lua_setglobal(L, "LNLOG_SDEBUG");
    lua_pushinteger(L, (lua_Integer) LNLOG_SMIN);
    lua_setglobal(L, "LNLOG_SMIN");
    lua_pushinteger(L, (lua_Integer) LNLOG_CTOP);
    lua_setglobal(L, "LNLOG_CTOP");
    lua_pushinteger(L, (lua_Integer) LNLOG_CSERVER);
    lua_setglobal(L, "LNLOG_CSERVER");
    lua_pushinteger(L, (lua_Integer) LNLOG_CGROUP);
    lua_setglobal(L, "LNLOG_CGROUP");
    lua_pushinteger(L, (lua_Integer) LNLOG_CARTICLE);
    lua_setglobal(L, "LNLOG_CARTICLE");
    lua_pushinteger(L, (lua_Integer) LNLOG_CALL);
    lua_setglobal(L, "LNLOG_CALL");

    /* export configuration items */
    lua_pushinteger(L, (lua_Integer) body_size);
    lua_setglobal(L, "LN_BODY_SIZE");
    /* FIXME! using "fqdn" causes a SIGSEGV! */
    /*lua_pushstring(L, owndn ? owndn : (fqdn ? fqdn : "LN_NO_HOSTNAME"));*/
    lua_pushstring(L, (owndn && owndn[0] != '\0') ?
            owndn : "script_fn_init_lua:NO_HOSTNAME");
    lua_setglobal(L, "LN_HOSTNAME");

    /* c-functions to be called from lua */
    lua_register(L, LN_LOG_LUA, ln_log_lua);
    lua_register(L, LN_AGE_LUA, ln_age_lua);

    hooks_file = mk_init_file_lua(CONFIG_SCRIPT_DEFAULT);
    if (hooks_file == NULL) {
        (void) script_init_ready_lua(0);
        reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_DOFILE_CANTREAD));
    }

    the_tos = lua_gettop(L);

    if (file_can_read(hooks_file)) {
        call_err = luaL_dofile(L, hooks_file);
        if (! call_err) goto call_fn_init;

        /* some error */
        (void) script_init_ready_lua(0);
        if (lua_isstring(L, -1)) {
            errmess_string = lua_tostring(L, -1);
            fprintf(stderr, "%s\n", errmess_string);
            errmess_string = mk_mess(errmess_string);
            lua_pop(L, 1);
            return((script_return_t) errmess_string);
            /*return((script_return_t) mk_mess(ERRMESS_DOFILE_UNSPEC));*/
        }
        lua_close(L);
        switch (call_err)
        {
            case LUA_ERRFILE:
                /* cannot read file */
                return((script_return_t) mk_mess(ERRMESS_DOFILE_CANTREAD));
                /* notreached */
                break;
            case LUA_ERRSYNTAX:
                /* syntax error during pre-compilation */
                return((script_return_t) mk_mess(ERRMESS_DOFILE_SYNTAX));
                /* notreached */
                break;
            case LUA_ERRMEM:
                /* memory allocation error */
                return((script_return_t) mk_mess(ERRMESS_MEMORY_ALLOC));
                /* notreached */
                break;
            default:
                /* unknown error */
                return((script_return_t) mk_mess(ERRMESS_DOFILE_UNSPEC));
                /* notreached */
                break;
        }
        /* notreached */
    } else {
        (void) script_init_ready_lua(0);
        lua_close(L);
        return((script_return_t) mk_mess(ERRMESS_DOFILE_CANTREAD));
        /* notreached */
    }

call_fn_init:

    lua_getglobal(L, FN_INIT);
    if (! lua_isfunction(L, -1)) {
        /* The function is not properly defined */
        lua_pop(L, 1);
        reset_and_return(the_tos, (script_return_t) SCRIPT_UNAVAILABLE);
    }

    call_err = lua_pcall(L, 0, 0, 0);
    if (! call_err) {
        reset_and_return(the_tos, (script_return_t) SCRIPT_NOERROR);
    }
    (void) script_init_ready_lua(0);
    fprintf(stderr, "%s\n", lua_tostring(L, -1));
    lua_pop(L, 1);  /* pop error mess */
    switch (call_err)
    {
        case LUA_ERRRUN:
            /* runtime error */
            reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_RUNTIME));
            /* notreached */
            break;
        case LUA_ERRMEM:
            /* memory allocation error */
            reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_MEMORY_ALLOC));
            /* notreached */
            break;
        case LUA_ERRERR:
            /* error while running error handler */
            /* cannot happen without error handler */
            reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_ERR_HANDLER));
            /* notreached */
            break;
        default:
            reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_API_RETURN_PROB));
            /* notreached */
    }
    /* notreached */
}

script_return_t
script_fn_finish_lua(int rc)
{
    if (! script_init_ready_lua(0))
        return((script_return_t) SCRIPT_UNAVAILABLE);

    lua_getglobal(L, FN_FINISH);
    if (lua_isfunction(L, -1)) {
        /* The function is defined */
        lua_pushinteger(L, (lua_Integer) rc);
        /* as this callout won't do anything besides side effects,
         * ignore returns */
        (void) lua_pcall(L, 1, 0, 0);
    }
    lua_close(L);
    return((script_return_t) SCRIPT_NOERROR);
}

script_return_t
script_fn_init_group_lua(char * groupname)
{
    int call_err = 0;
    script_return_t return_val;

    if (! script_init_ready_lua(-1)) {
        return((script_return_t) SCRIPT_UNAVAILABLE);
    }

    the_tos = lua_gettop(L);

    lua_getglobal(L, FN_INIT_GROUP);
    if (! lua_isfunction(L, -1)) {
        /* The function is not properly defined */
        lua_pop(L, 1);
        reset_and_return(the_tos, (script_return_t) SCRIPT_UNAVAILABLE);
    }

    lua_pushstring(L, groupname);

    call_err = lua_pcall(L, 1, 1, 0);
    return_val = script_simple_result(call_err);
    reset_and_return(the_tos, return_val);
    /* notreached */
}

script_return_t
script_fn_finish_group_lua(int rc)
{
    int call_err = 0;
    lua_Integer status_code = 0;
    script_return_t return_val;

    if (! script_init_ready_lua(-1))
        return((script_return_t) SCRIPT_UNAVAILABLE);

    /*
    if (at_header_table > 0) {
        lua_remove(L, at_header_table);
        at_header_table = 0;
    }
    header_key = 0;
    */

    the_tos = lua_gettop(L);

    lua_getglobal(L, FN_FINISH_GROUP);
    if (! lua_isfunction(L, -1)) {
        /* The function is not properly defined */
        lua_pop(L, 1);
        reset_and_return(the_tos, (script_return_t) SCRIPT_UNAVAILABLE);
    }

    status_code = (lua_Integer) rc;
    lua_pushinteger(L, status_code);

    call_err = lua_pcall(L, 1, 1, 0);
    return_val = script_simple_result(call_err);
    reset_and_return(the_tos, return_val);
    /* notreached */
}

/*
3.7 - Functions and Types

Here we list all functions and types from the C API in alphabetical
order. Each function has an indicator like this: [-o, +p, x]

The first field, o, is how many elements the function pops from the
stack. The second field, p, is how many elements the function pushes
onto the stack. (Any function always pushes its results after popping
its arguments.) A field in the form x|y means the function may push (or
pop) x or y elements, depending on the situation; an interrogation mark
'?' means that we cannot know how many elements the function pops/pushes
by looking only at its arguments (e.g., they may depend on what is on
the stack).

The third field, x, tells whether the function may throw
errors:
'-' means the function never throws any error;
'm' means the function may throw an error only due to not enough memory;
'e' means the function may throw other kinds of errors;
'v' means the function may throw an error on purpose.
*/

script_return_t
script_fn_init_article_lua(void)
{
    int call_err = 0;
    script_return_t return_val;

    if (! script_init_ready_lua(-1))
        return((script_return_t) SCRIPT_UNAVAILABLE);

    /* remove old header table, which should be at the first stack pos */
    /* remove old results, make sure at_header_table is adjusted. */
    /* -ino: XXX does the stack get out-of-sync wrt: at_header_table? */
    /* XXX stack ordering prob? */
    REMOVE_OLD_RESULT(at_result_table);
    if (at_header_table > 0) {
        lua_remove(L, at_header_table);
        at_header_table = 0;
    }

    /* make the table add_header needs for intermediate headers */
    lua_newtable(L);
    the_tos = lua_gettop(L);
    at_header_table = the_tos;

    header_key = 0;

    lua_getglobal(L, FN_INIT_ARTICLE);
    if (! lua_isfunction(L, -1)) {
        /* The function is not properly defined */
        lua_pop(L, 1);
        reset_and_return(the_tos, (script_return_t) SCRIPT_UNAVAILABLE);
    }

    call_err = lua_pcall(L, 0, 1, 0);
    return_val = script_simple_result(call_err);
    reset_and_return(the_tos, return_val);
    /* notreached */
}

script_return_t
script_fn_finish_article_lua(int rc)
{
    int call_err = 0;
    lua_Integer status_code = 0;
    script_return_t return_val;

    if (! script_init_ready_lua(-1))
        return((script_return_t) SCRIPT_UNAVAILABLE);

    header_key = 0;

    the_tos = lua_gettop(L);

    lua_getglobal(L, FN_FINISH_ARTICLE);
    if (! lua_isfunction(L, -1)) {
        /* The function is not properly defined */
        lua_pop(L, 1);
        reset_and_return(the_tos, (script_return_t) SCRIPT_UNAVAILABLE);
    }

    status_code = (lua_Integer) rc;
    lua_pushinteger(L, status_code);

    call_err = lua_pcall(L, 1, 1, 0);
    return_val = script_simple_result(call_err);
    reset_and_return(the_tos, return_val);
    /* notreached */
}

script_return_t
script_fn_add_header_lua(script_data_t header)
{
    if (! script_init_ready_lua(-1)) {
        return((script_return_t) SCRIPT_UNAVAILABLE);
    }

    /* will not add empty header lines */
    if (! VALID_MASTR(header)) return SCRIPT_NOERROR;

    the_tos = lua_gettop(L);

    if ((at_header_table <= 0) || (the_tos != at_header_table)) {
        fn_get_result_dontcall = 1;
        reset_and_return(the_tos, (script_return_t) SCRIPT_ILLEGAL_SEQ);
    }

    /* there should be enough space for some more headers */
    if (! lua_checkstack(L, FN_MIN_STACK_LUA)) {
        reset_and_return(the_tos, (script_return_t) mk_mess(ERRMESS_MEMORY_ALLOC));
    }

    /* add this particular header.  since leafnodes main data type is
     * a sized string, we can pretend it's a lua string, so it may
     * contain zeros and is eight-bit-transparent.  headers are
     * collected into a table, and its entries are "rawset" meaning no
     * metamethods will be triggered, for speed and robustness.
     */
    lua_pushlstring(L, mastr_str(header), mastr_len(header));
    header_key += 1;
    /*lua_pushinteger(L, header_key);*/
    /*lua_settable(L, at_header_table);*/
    lua_rawseti(L, at_header_table, header_key);
    return SCRIPT_NOERROR;
}

script_return_t
script_fn_add_body_lua(script_data_t body __unused)
{
    /* currently unneeded */
    if (! script_init_ready_lua(-1)) {
        return((script_return_t) SCRIPT_UNAVAILABLE);
    }
    /*
    if (! VALID_MASTR(body)) return SCRIPT_ILLEGAL_API_ARGS;
    the_tos = lua_gettop(L);
    lua_pushlstring(L, body, mastr_len(body));
    */
    return((script_return_t) SCRIPT_NOERROR);
}

script_return_t
script_fn_filter_header_table_lua()
{
    int call_err = 0;
    script_return_t return_val;

    if (! script_init_ready_lua(-1)) return (script_return_t) SCRIPT_UNAVAILABLE;

    the_tos = lua_gettop(L);

    lua_getglobal(L, FN_HEADERTABLE);
    if (! lua_isfunction(L, -1)) {
        /* The function is not properly defined */
        lua_pop(L, 1);
        header_key = 0;
        reset_and_return(the_tos, (script_return_t) SCRIPT_UNAVAILABLE);
    }

    /* the header table is removed later by finish_article_lua */
    if (at_header_table <= 0) {
        /*at_result_table = 0;*/
        fn_get_result_dontcall = 1;
        reset_and_return(the_tos, (script_return_t) SCRIPT_ILLEGAL_SEQ);
    }
    lua_pushvalue(L, at_header_table);

    call_err = lua_pcall(L, 1, 1, 0);
    VALIDATE_AND_RETURN_RESULT(call_err, SCRIPT_NOERROR, SCRIPT_WRONG_RESULT);
}

script_return_t
script_fn_filter_header_lua(script_data_t headers)
{
    int call_err = 0;
    script_return_t return_val;

    if (! script_init_ready_lua(-1)) return (script_return_t) SCRIPT_UNAVAILABLE;

    the_tos = lua_gettop(L);

    if (! VALID_MASTR(headers))
        reset_and_return(the_tos, (script_return_t) SCRIPT_ILLEGAL_API_ARGS);

    /* remove old results */
    REMOVE_OLD_RESULT(at_result_table);

    lua_getglobal(L, FN_HEADERTXT);
    if (! lua_isfunction(L, -1)) {
        /* The function is not properly defined */
        lua_pop(L, 1);
        reset_and_return(the_tos, (script_return_t) SCRIPT_UNAVAILABLE);
    }

    lua_pushlstring(L, mastr_str(headers), mastr_len(headers));

    call_err = lua_pcall(L, 1, 1, 0);
    VALIDATE_AND_RETURN_RESULT(call_err, SCRIPT_NOERROR, SCRIPT_WRONG_RESULT);
}

script_return_t
script_fn_filter_header_body_lua(script_data_t headers, script_data_t body)
{
    int call_err = 0;
    script_return_t return_val;

    if (! script_init_ready_lua(-1)) return (script_return_t) SCRIPT_UNAVAILABLE;

    the_tos = lua_gettop(L);

    if (! VALID_MASTR(headers))
        reset_and_return(the_tos, (script_return_t) SCRIPT_ILLEGAL_API_ARGS);
    if (! VALID_MASTR(body))
        reset_and_return(the_tos, (script_return_t) SCRIPT_ILLEGAL_API_ARGS);

    /* remove old results */
    REMOVE_OLD_RESULT(at_result_table);

    lua_getglobal(L, FN_HEADERTXT_BODYTXT);
    if (! lua_isfunction(L, -1)) {
        /* The function is not properly defined */
        lua_pop(L, 1);
        reset_and_return(the_tos, (script_return_t) SCRIPT_UNAVAILABLE);
    }

    lua_pushlstring(L, mastr_str(headers), mastr_len(headers));
    lua_pushlstring(L, mastr_str(body), mastr_len(body));

    call_err = lua_pcall(L, 2, 1, 0);
    VALIDATE_AND_RETURN_RESULT(call_err, SCRIPT_NOERROR, SCRIPT_WRONG_RESULT);
}

static const char * _result_name(script_cmd_t cmd);
static const char *
_result_name(script_cmd_t cmd)
{
    const char * unspecified = "ERROR: UNSPEC";

    switch (cmd)
    {
        case SCRIPT_GET_NOTHING:
            return("GET_NOTHING");
            /* notreached */
            break;
        case SCRIPT_GET_NEWSGROUPS:
            return("GET_NEWSGROUPS");
            /* notreached */
            break;
        case SCRIPT_GET_BODY:
            return("GET_BODY");
            /* notreached */
            break;
        case SCRIPT_GET_HEADER:
            return("GET_HEADER");
            /* notreached */
            break;
        default:
            return(unspecified);
            /* notreached */
    }
}

script_result_t
script_fn_get_result_lua(script_cmd_t cmd)
{
    int return_type = 0;
    mastr *result = NULL;               /* space for result */
    int result_int;                     /* result field is an int code */
    const char * result_string;         /* result as string */
    // lua_Integer result_luaint;       /* result field is a lua int code */
    lua_Number result_number;           /* result field is a number */
    const char * res_field;             /* what field to get */
    script_result_t std_response;       /* if nothing else, what to answer */

    /*
    fprintf(stderr, "script_fn_get_result_lua(%d), dontcall=%d\n",
            cmd, fn_get_result_dontcall);
    */
    if (fn_get_result_dontcall)
        return ((script_result_t) SCRIPT_UNAVAILABLE);

    the_tos = lua_gettop(L);
    if ((at_result_table <= 0) || (the_tos != at_result_table)) {
        fn_get_result_dontcall = 1;
        reset_and_return(the_tos, (script_result_t) SCRIPT_ILLEGAL_SEQ);
    }

    /* check return value:
     * must be table at stacktop with field "result"
     */
    return_type = lua_type(L, -1);
    if (return_type != LUA_TTABLE) {
        fn_get_result_dontcall = 1;
        REMOVE_OLD_RESULT(at_result_table);
        reset_and_return(the_tos, (script_result_t) SCRIPT_WRONG_RESULT);
    }
    lua_getfield(L, -1, RES_FIELD_RESULT);
    if (lua_isnil(L, -1)) {
        fn_get_result_dontcall = 1;
        REMOVE_OLD_RESULT(at_result_table);
        reset_and_return(the_tos, (script_result_t) SCRIPT_WRONG_RESULT);
    }

    /* the "result" field can be a specific number
     * or a status string, but it must be the OK code if
     * anything else is to be evaluated.
     */
    if (lua_isnumber(L, -1)) {
        result_number = lua_tonumber(L, -1);
        lua_pop(L, 1);
        result_int = (int) result_number;
        switch (result_int)
        {
            case SCRIPT_NOERROR:
                goto handle_cmd;
                /* notreached */
                break;
            case SCRIPT_REJECT:
            case SCRIPT_NOCHANGE:
            case SCRIPT_NOCHANGE_HEADER:
            case SCRIPT_NOCHANGE_BODY:
            case SCRIPT_UNAVAILABLE:
            case SCRIPT_IGNORE_ARTICLE:
            case SCRIPT_IGNORE_GROUP:
                reset_and_return(the_tos, (script_result_t) result_int);
                /* notreached */
                break;
            default:
                /* numeric results _MUST NOT_ be undefined */
                fn_get_result_dontcall = 1;
                REMOVE_OLD_RESULT(at_result_table);
                reset_and_return(the_tos, (script_result_t) SCRIPT_API_RETURN_PROB);
                /* notreached */
                break;
        }
    }
    /* XXX
     * currently, the result field can be a status message formatted as
     * a string, but this is discouraged, because that memory won't be
     * reclaimed/free()ed, thus leaked.
     */
    if (lua_isstring(L, -1)) {
        result_string = lua_tostring(L, -1);
        /* alloc space for result */
        result = mastr_new(4095l);
        (void)mastr_cpy(result, result_string);
        lua_pop(L, 1);
        reset_and_return(the_tos, result);
    }
    /* "result" must be a defined number or some string! */
    fn_get_result_dontcall = 1;
    REMOVE_OLD_RESULT(at_result_table);
    reset_and_return(the_tos, (script_result_t) SCRIPT_WRONG_RESULT);

    /* ok, it's a std result table.  again, the various fields can be
     * numeric codes or strings.  if the field asked for doesn't exist,
     * this is no error:  instead, NOCHANGE is returned.
     */

handle_cmd:

    switch (cmd)
    {
        case SCRIPT_GET_NOTHING:
            /*reset_and_return(the_tos, (script_result_t) SCRIPT_UNAVAILABLE);*/
            return((script_result_t) SCRIPT_UNAVAILABLE);
            /* notreached */
            break;
        case SCRIPT_GET_NEWSGROUPS:
            res_field = RES_FIELD_NEWSGROUPS;
            std_response = (script_result_t) SCRIPT_NOCHANGE;
            break;
        case SCRIPT_GET_BODY:
            res_field = RES_FIELD_BODY;
            std_response = (script_result_t) SCRIPT_NOCHANGE_BODY;
            break;
        case SCRIPT_GET_HEADER:
            res_field = RES_FIELD_HEADER;
            std_response = (script_result_t) SCRIPT_NOCHANGE_HEADER;
            break;
        default:
            fn_get_result_dontcall = 1;
            REMOVE_OLD_RESULT(at_result_table);
            reset_and_return(the_tos, (script_result_t) SCRIPT_ILLEGAL_API_ARGS);
            /* notreached */
    }

    /* Requested result field doesn't exist. This is no error:  user
     * might not have implemented some hook or didn't want to change
     * newsgroups (2), header (3) or body (4).
     */
    lua_getfield(L, -1, res_field);
    if (lua_isnil(L, -1)) {
        ln_log(LNLOG_SDEBUG, LNLOG_CARTICLE,
            "script_fn_get_result_lua: result item '%s' does not exist",
            _result_name(cmd));
        lua_pop(L, 1);
        return((script_result_t) SCRIPT_UNAVAILABLE);
    }

    if (lua_isnumber(L, -1)) {
        result_number = lua_tonumber(L, -1);
        lua_pop(L, 1);
        result_int = (int) result_number;
        switch (result_int)
        {
            case SCRIPT_NOERROR:
            case SCRIPT_REJECT:
            case SCRIPT_NOCHANGE:
            case SCRIPT_NOCHANGE_HEADER:
            case SCRIPT_NOCHANGE_BODY:
            case SCRIPT_UNAVAILABLE:
            case SCRIPT_IGNORE_ARTICLE:
            case SCRIPT_IGNORE_GROUP:
                return((script_result_t) result_int);
                /* notreached */
                break;
            /* XXX what about groups like "007.001"? */
            default:
                /* result numeric, but undefined code */
                fn_get_result_dontcall = 1;
                REMOVE_OLD_RESULT(at_result_table);
                reset_and_return(the_tos, (script_result_t) SCRIPT_API_RETURN_PROB);
                /* notreached */
                break;
        }
    }
    if (lua_isstring(L, -1)) {
        result_string = lua_tostring(L, -1);
        /* alloc space for result */
        result = mastr_new(4095l);
        (void)mastr_cpy(result, result_string);
        lua_pop(L, 1);
        reset_and_return(the_tos, result);
    }
    /* result field must be number or string! */
    fn_get_result_dontcall = 1;
    REMOVE_OLD_RESULT(at_result_table);
    reset_and_return(the_tos, (script_result_t) SCRIPT_WRONG_RESULT);
}

