/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * Linux Desktop Testing Project http://ldtp.freedesktop.org
 *
 * Author:
 *    Nagappan A <nagappan@gmail.com>
 *
 * Copyright 2004 - 2006 Novell, Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110, USA.
 */

#include "ldtp.h"
#include "ldtp-gui.h"
#include "ldtp-error.h"
#include "ldtp-logger.h"
#include "ldtp-command.h"
#include "ldtp-gui-comp.h"

extern gboolean ldtp_debug;

/*
  Sort column based on given index
*/  
static LDTPErrorCode
sort_column_index (Accessible *object, int col, FILE *log_fp)
{
	long n_cols;	
	long n_actions, i;
	gboolean flag = FALSE;
	LDTPErrorCode error;
	AccessibleTable *table;
	Accessible *header;
	AccessibleAction *action;
	char msg[256];
	char *name = NULL;

	table = Accessible_getTable (object);
	n_cols = AccessibleTable_getNColumns (table);
	g_print ("Number of columns: %ld\n", n_cols);

	if (col < 0 || col >= n_cols) {
		g_sprintf (msg, "Column %d doesnot exist in table", col);
		log_msg (LDTP_LOG_CAUSE, msg, log_fp);
		Accessible_unref (table);
		error =  (LDTP_ERROR_INVALID_COLUMN_INDEX_TO_SORT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	header = AccessibleTable_getColumnHeader (table, col);
	Accessible_unref (table);
  
	action = Accessible_getAction (header);
	Accessible_unref (header);
	n_actions = AccessibleAction_getNActions (action);

	for (i = 0; i < n_actions; i++) {
		name = AccessibleAction_getName (action, i);

		if (g_utf8_collate (name, "sort") == 0) {      
			AccessibleAction_doAction (action, i);
			flag = 1;
			break;
		}
	}

	Accessible_unref (action);
	SPI_freeString (name);

	if (flag == FALSE) { 
		error =  (LDTP_ERROR_UNABLE_TO_SORT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
	return  (LDTP_ERROR_SUCCESS);
}

/*
  Sort column based on given name
*/  
static LDTPErrorCode
sort_column (Accessible *object, char *row_text, FILE *log_fp)
{
	long n_cols;	
	int n_actions, i, j;
	LDTPErrorCode error;
	gboolean flag = FALSE;
	AccessibleTable *table;
	AccessibleAction *action;
	Accessible *header = NULL;
	char *name = NULL;

	table = Accessible_getTable (object);
	n_cols = AccessibleTable_getNColumns (table);
	g_print ("Number of columns: %ld\n", n_cols);

	for (i = 0; i < n_cols; i++) {	  
		char *name;
		header = AccessibleTable_getColumnHeader (table, i);
		name = Accessible_getName (header);

		if (g_utf8_collate (name, row_text) == 0) {      
			Accessible_unref (table);
			SPI_freeString (name);
			break;
		}
	
		Accessible_unref (header);			
	}	
  
	if (i == n_cols) {	  
		Accessible_unref (table);
		error =  (LDTP_ERROR_COLUMN_DOES_NOT_EXIST);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}

	action = Accessible_getAction (header);
	Accessible_unref (header);
	n_actions = AccessibleAction_getNActions (action);

	for (j = 0; j < n_actions; j++) {
		name = AccessibleAction_getName (action, j);

		if (g_utf8_collate (name, "sort") == 0) {      
			AccessibleAction_doAction (action, j);
			flag = TRUE;
			break;
		}
	}

	Accessible_unref (action);
	SPI_freeString (name);

	if (flag == FALSE) { 
		error =  (LDTP_ERROR_UNABLE_TO_SORT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
	return  (LDTP_ERROR_SUCCESS);
}  

static LDTPErrorCode
get_cell_value (LDTPClientContext *cctxt)
{
	long i;
	long row = 0, column = 0;
	long n_rows = 0, n_cols = 0;
	int  child_type;
	long child_count;
	LDTPErrorCode error;
	Accessible *tmp_child = NULL;
	GSList *l = cctxt->req->arg_list;

	AccessibleTable *table;
	Accessible *child_object;
	Accessible *object = cctxt->gui_handle->handle;

	if (l && l->data) {
		row = atol (l->data);
		l = l->next;
		if (l && l->data) {
			column = atol (l->data);
		}
	}

	table = Accessible_getTable (object);
	if (table) {
		n_rows = AccessibleTable_getNRows (table);
		n_cols = AccessibleTable_getNColumns (table);
	}

	if (n_rows < row) {
		g_print ("Row cout: %ld - Actual: %ld\n", row, n_rows);
		Accessible_unref (table);
		error =  (LDTP_ERROR_ACTUAL_ROW_COUNT_LESS_THAN_GIVEN_ROW_COUNT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
		return error;
	}
	if (n_cols < column) {
		g_print ("Column cout: %ld - Actual: %ld\n", column, n_cols);
		Accessible_unref (table);
		error =  (LDTP_ERROR_ACTUAL_COLUMN_COUNT_LESS_THAN_GIVEN_COLUMN_COUNT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
		return error;
	}

	g_print ("Tree table - number of rows: %ld - %ld\tColumn: %ld %ld\n", row, n_rows, column, n_cols);
	child_object = AccessibleTable_getAccessibleAt (table, row, column);
	child_count = Accessible_getChildCount (child_object);
	g_print ("Child count: %ld\n", child_count);

	for (i = 0; i < child_count; i++) {
		tmp_child = Accessible_getChildAtIndex (child_object, i);
		if (tmp_child) {
			char *name;
			name = Accessible_getName (tmp_child);
			if (name) {
				/*
				  FIXME: Currently ignoring the child,
				  if name is empty string
				*/
				if (g_ascii_strcasecmp (name, "") != 0) {
					g_print ("Debug: Cell value: %s - %ld - %ld\n", name, i, Accessible_getChildCount (tmp_child));
					Accessible_unref (child_object);
					child_object = tmp_child;
					SPI_freeString (name);
					break;
				}
				SPI_freeString (name);
			}
		}
		Accessible_unref (tmp_child);
	}

	cctxt->gui_handle->handle = child_object;
	if (child_object) {
		char *name;
		char *role;
		name = Accessible_getName (child_object);
		if (name) {
			g_print ("Cell value: %s - %ld\n", name, Accessible_getChildCount (child_object));
			SPI_freeString (name);
		}
		role = Accessible_getRoleName (child_object);
		if (role) {
			g_print ("Child type: %d %s\n", Accessible_getRole (child_object), role);
			SPI_freeString (role);
		}
	}
	if (child_object && Accessible_isComponent (child_object)) {
		child_type = get_object_type (child_object);
		if (child_type == SPI_ROLE_TABLE_CELL) {
			g_print ("Table cell text type: %d\n", Accessible_isText (child_object));
			if (Accessible_isText (child_object)) {
				g_print ("Table cell is a text object\n");
				cctxt->req->arg_list = g_slist_prepend (cctxt->req->arg_list,
									"0");
				cctxt->req->arg_list = g_slist_prepend (cctxt->req->arg_list,
									"0");
				error = text_main (cctxt, LDTP_CMD_GETTEXTVALUE);
				cctxt->req->arg_list = g_slist_remove_all (cctxt->req->arg_list,
									   "0");
				Accessible_unref (table);
				Accessible_unref (child_object);
				cctxt->gui_handle->handle = object;
				return error;
			}
			else {
				AccessibleStateSet* stateset;
				stateset = Accessible_getStateSet (child_object);
				if (AccessibleStateSet_contains (stateset, SPI_STATE_CHECKED))
					cctxt->resp->data = g_strdup_printf ("1");
				else
					cctxt->resp->data = g_strdup_printf ("0");
				cctxt->resp->data_len = strlen (cctxt->resp->data);
				AccessibleStateSet_unref (stateset);
				Accessible_unref (table);
				Accessible_unref (child_object);
				cctxt->gui_handle->handle = object;
				return  (LDTP_ERROR_SUCCESS);
			}
		}
	} //if
	g_print ("Child object: NOT component\n");
	Accessible_unref (table);
	Accessible_unref (child_object);
	cctxt->gui_handle->handle = object;
	error =  (LDTP_ERROR_GET_TREE_TABLE_CELL_FAILED);
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
	return error;
}

/*
  FIXME: Logic should be same as get_cell_value
*/

static LDTPErrorCode
verify_partial_cell_value (LDTPClientContext* cctxt)
{
	long row = 0, column = 0;
	long n_rows, n_cols;
	int  child_type;
	long child_count;
	LDTPErrorCode error;
	GSList *l = cctxt->req->arg_list;

	AccessibleTable *table;
	Accessible *child_object;
	Accessible *object = cctxt->gui_handle->handle;
  
	if (l && l->data) {
		row = atol (l->data);
		l = l->next;
		if (l && l->data) {
			column = atol (l->data);
		}
	}
        
	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);
 
	if (n_rows < row) {
		Accessible_unref (table);
		error =  (LDTP_ERROR_ACTUAL_ROW_COUNT_LESS_THAN_GIVEN_ROW_COUNT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
		return error;
	}
	if (n_cols < column) {
		Accessible_unref (table);
		error =  (LDTP_ERROR_ACTUAL_COLUMN_COUNT_LESS_THAN_GIVEN_COLUMN_COUNT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
		return error;
	}  

	g_print ("Tree table - number of rows:%ld\tColumn: %ld\n", n_rows, n_cols);
	child_object = AccessibleTable_getAccessibleAt (table, row, column);

	do { 
		child_count = Accessible_getChildCount (child_object);
		if (child_count > 0) {
			Accessible *tmp_child;
			tmp_child = child_object;
			child_object = Accessible_getChildAtIndex (child_object, 0);
			Accessible_unref (tmp_child);
		}
	} while (child_count > 0);

	cctxt->gui_handle->handle = child_object;
	if (Accessible_isComponent (child_object)) {
		child_type = get_object_type (child_object);
		g_print ("Child type is %d ", child_type);
		if (child_type == SPI_ROLE_TABLE_CELL) {
			if (Accessible_isText (child_object)) {	
				g_print ("Table cell is a text object\n");
				l = l->next;
				cctxt->req->arg_list = g_slist_prepend (cctxt->req->arg_list,
									l->data);
				error = text_main (cctxt, LDTP_CMD_VERIFYPARTIALMATCH);
				cctxt->req->arg_list = g_slist_remove (cctxt->req->arg_list,
								       l->data);
				Accessible_unref (table);
				Accessible_unref (child_object);
				cctxt->gui_handle->handle = object;
				return error;				
			}
		}
		else if (child_type == SPI_ROLE_TEXT) {
			g_print ("Table cell is text box\n");
			l = l->next;
			cctxt->req->arg_list = g_slist_prepend (cctxt->req->arg_list,
								l->data);
			error = text_main (cctxt, LDTP_CMD_VERIFYPARTIALMATCH);
			cctxt->req->arg_list = g_slist_remove (cctxt->req->arg_list,
							       l->data);
			Accessible_unref (table);
			Accessible_unref (child_object);
			cctxt->gui_handle->handle = object;
			return error;
		}
	} //if
	Accessible_unref (table);
	Accessible_unref (child_object);
	cctxt->gui_handle->handle = object;
	error =  (LDTP_ERROR_VERIFY_TABLE_CELL_PARTIAL_MATCH_FAILED);
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
	return error;
}

static Accessible*
get_row (Accessible *object, char *row_text, FILE *log_fp)
{
	long i, j;
	long n_rows, n_cols;
	Accessible *cell;
	AccessibleTable *table;
	char msg[256];

	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);
	g_print ("Number of rows: %ld\tColumn: %ld\n", n_rows, n_cols);

	for (i = 0; i < n_rows; i++) {
		for (j = 0; j < n_cols; j++) {
			long child_count;
			cell = AccessibleTable_getAccessibleAt (table, i, j);
			child_count = Accessible_getChildCount (cell);
			g_print ("Child count: %ld\n", child_count);
			if (child_count > 0) {
				long k;
				char *name;
				Accessible *child;
				for (k = 0; k < child_count; k++) {
					child = Accessible_getChildAtIndex (cell, k);
					name = Accessible_getName (child);
					g_print ("Table cell name: %s\n", name);
					if (g_utf8_collate (name, row_text) == 0) {
						SPI_freeString (name);
						Accessible_unref (cell);
						Accessible_unref (table);
						return child;
					} // if
					SPI_freeString (name);
					Accessible_unref (child);
				} // for
			} else {
				char *name;
				name = Accessible_getName (cell);
				g_print ("Table cell name: %s\n", name);
				if (g_utf8_collate (name, row_text) == 0) {
					SPI_freeString (name);
					Accessible_unref (table);
					return cell;
				} // if
				SPI_freeString (name);
			}
			Accessible_unref (cell);
		} // for j
	} // for i

	Accessible_unref (table);
	g_sprintf (msg, "Unable to find %s in table", row_text);
	log_msg (LDTP_LOG_CAUSE, msg, log_fp);
	return NULL;
}

static LDTPErrorCode
select_row (Accessible *object, GSList *l, FILE *log_fp)
{
	long i, j;
	gboolean flag = FALSE;
	LDTPErrorCode error;
	long n_rows, n_cols;
	Accessible *cell;
	AccessibleTable *table;
	char *name = NULL;
	char *row_text = l->data;
	long n_matches = 0;

	l = l->next;
	if (l && l->data)
		n_matches = atol (l->data);
    
	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);
	g_print ("Number of rows: %ld\tColumn: %ld\tMatch index: %ld\n", n_rows, n_cols, n_matches);

	for (i = 0; i < n_rows; i++) {
		for (j = 0; j < n_cols; j++) {
			long child_count;
			cell = AccessibleTable_getAccessibleAt (table, i, j);
			child_count = Accessible_getChildCount (cell);
			if (child_count > 0) {
				long k;
				Accessible *child;
				for (k = 0; k < child_count; k++) {
					child = Accessible_getChildAtIndex (cell, k);
					name = Accessible_getName (child);
					if (name)
						g_print ("Table cell Name: %s\n", name);
					if (g_utf8_collate (name, row_text) == 0) {
						if (n_matches != 0) { 
							n_matches--;
							Accessible_unref (child);
							SPI_freeString(name);
							continue;
						}
						if (Accessible_isComponent (cell)) {
							AccessibleComponent *accessible_component;
							accessible_component = Accessible_getComponent (cell);
							if (!AccessibleComponent_grabFocus (accessible_component)) {
								Accessible_unref (accessible_component);
								Accessible_unref (child);
								Accessible_unref (cell);
								error =  (LDTP_ERROR_UNABLE_TO_GRAB_FOCUS);
								log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
								SPI_freeString(name);
								return error;
							}
							Accessible_unref (accessible_component);
						}
						SPI_freeString (name);
						Accessible_unref (child);
						Accessible_unref (cell);
						flag = TRUE;
						break;
					} // if
					SPI_freeString (name);
					Accessible_unref (child);
				} // for
			} // if
			else {
				name = Accessible_getName (cell);
				if (g_utf8_collate (name, row_text) == 0) {
					if (n_matches != 0) { 
						n_matches--;
						Accessible_unref (cell);
						SPI_freeString(name);
						continue;
					}

					if (Accessible_isComponent (cell)) {
						AccessibleComponent *accessible_component;
						accessible_component = Accessible_getComponent (cell);
						if (!AccessibleComponent_grabFocus (accessible_component)) {
							Accessible_unref (accessible_component);
							Accessible_unref (cell);
							error =  (LDTP_ERROR_UNABLE_TO_GRAB_FOCUS);
							log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
							return error;
						}
						Accessible_unref (accessible_component);
					}
					SPI_freeString (name);
					Accessible_unref (cell);
					flag = TRUE;
					break;
				}
			}
			Accessible_unref (cell);
		} // for j
		if (flag == TRUE)
			break;
	} // for i
  
	Accessible_unref (table);
	if (flag == FALSE) {
		error =  (LDTP_ERROR_UNABLE_TO_SELECT_ROW);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
	return  (LDTP_ERROR_SUCCESS);
}

static LDTPErrorCode
select_row_partial_match (Accessible *object, char *row_text, FILE *log_fp)
{
	long i, j;
	gboolean flag = FALSE;
	long n_rows, n_cols;
	LDTPErrorCode error;
	Accessible *cell;
	AccessibleTable *table;

	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);
	g_print ("Number of rows: %ld\tColumn: %ld\n", n_rows, n_cols);

	for (i = 0; i < n_rows; i++) {
		for (j = 0; j < n_cols; j++) {
			long k;
			char *name;
			long child_count;
			Accessible *child;

			cell = AccessibleTable_getAccessibleAt (table, i, j);
			child_count = Accessible_getChildCount (cell);
			for (k = 0; k < child_count; k++) {
				child = Accessible_getChildAtIndex (cell, k);
				name = Accessible_getName (child);
				/*
				  FIXME: Use g_utf8_*
				*/
				if (strstr (name, row_text) != NULL) {
					SPI_freeString (name);
					if (Accessible_isComponent (cell)) {
						SPIBoolean flag;
						AccessibleComponent *accessible_component;
						accessible_component = Accessible_getComponent (cell);
						flag = AccessibleComponent_grabFocus (accessible_component);
						Accessible_unref (accessible_component);
						if (flag == FALSE) {
							Accessible_unref (child);
							Accessible_unref (cell);
							error =  (LDTP_ERROR_UNABLE_TO_GRAB_FOCUS);
							log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
							return error;
						}
					}
					Accessible_unref (child);
					Accessible_unref (cell);
					flag = TRUE;
					break;
				} // if
				else
					SPI_freeString (name);
				Accessible_unref (child);
			} // for
			Accessible_unref (cell);
		} // for j
		if (flag == TRUE)
			break;
	} // for i

	Accessible_unref (table);
	if (flag == FALSE) {
		char msg[256];
		g_sprintf (msg, "Unable to find %s in table", row_text);
		log_msg (LDTP_LOG_CAUSE, msg, log_fp);
		error =  (LDTP_ERROR_UNABLE_TO_SELECT_ROW);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
	return  (LDTP_ERROR_SUCCESS);
}

static LDTPErrorCode
get_current_row_index (Accessible *object, GSList **l, FILE *log_fp)
{
	long i, j;
	long n_rows, n_cols;
	char msg[256];
	char *name;
	char *row_text = (*l)->data;
	LDTPErrorCode error;
	Accessible *cell;
	AccessibleTable *table;

	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);
	g_print ("Number of rows: %ld\tColumn: %ld\n", n_rows, n_cols);

	for (i = 0; i < n_rows; i++) {
		for (j = 0; j < n_cols; j++) {
			long child_count;
			cell = AccessibleTable_getAccessibleAt (table, i, j);
			child_count = Accessible_getChildCount (cell);
			if (child_count > 0) {
				long k;
				Accessible *child;
				for (k = 0; k < child_count; k++) {
					child = Accessible_getChildAtIndex (cell, k);
					name = Accessible_getName (child);
					if (g_utf8_collate (name, row_text) == 0) {
						SPI_freeString (name);
						Accessible_unref (child);
						Accessible_unref (cell);
						Accessible_unref (table);
						*l = g_slist_prepend (*l, g_strdup_printf ("%ld", i));
						return  (LDTP_ERROR_SUCCESS);
					} // if
					SPI_freeString (name);
					Accessible_unref (child);
				} // for
			} // if
			else {
				name = Accessible_getName (cell);
				g_print ("Name = %s, Row Text = %s\n", name, row_text);
				if (g_utf8_collate (name, row_text) == 0) {
					SPI_freeString (name);
					Accessible_unref (cell);
					Accessible_unref (table);
					*l = g_slist_prepend (*l, g_strdup_printf ("%ld", i));
					return  (LDTP_ERROR_SUCCESS);
				}
				SPI_freeString (name);
			}
			Accessible_unref (cell);
		} // for j
	} // for i

	Accessible_unref (table);
	g_sprintf (msg, "Unable to find %s in table", row_text);
	log_msg (LDTP_LOG_CAUSE, msg, log_fp);
	error =  (LDTP_ERROR_UNABLE_TO_GET_ROW_INDEX);
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

/*
  Double click
*/
static LDTPErrorCode
double_click (Accessible *object, FILE *log_fp)
{
	long x, y, height, width;
	SPIBoolean flag = FALSE;
	LDTPErrorCode error;
	AccessibleComponent *component;

	if (Accessible_isComponent (object))
		component = Accessible_getComponent (object);
	else {
		Accessible_unref (object);
		error =  (LDTP_ERROR_OBJ_NOT_COMPONENT_TYPE);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
	AccessibleComponent_getExtents (component, &x, &y, &width, &height, SPI_COORD_TYPE_SCREEN);
	/* NOTE: Some widgets doesnot sense the double click event if they are performed exactly on the 
	 * border. In order to take care of those situtations also we make adjustments in the x and y value
	 * to fire the event at the center of the object.
	 */
	x = x + width / 2;
	y = y + height / 2;
	flag = AccessibleComponent_grabFocus (component);
	Accessible_unref (component);
	if (!flag) {
		error =  (LDTP_ERROR_UNABLE_TO_GRAB_FOCUS);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
	flag = SPI_generateMouseEvent (x, y, "b1d");
	if (flag)
		return  (LDTP_ERROR_SUCCESS);
	else {
		error =  (LDTP_ERROR_DOUBLE_CLICK_FAILED);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
}

/*
  Double click row
*/
static LDTPErrorCode
double_click_row (Accessible *object, char *row_text, FILE *log_fp)
{
	long x, y, height, width;
	SPIBoolean flag = FALSE;
	LDTPErrorCode error;
	Accessible *cell;
	AccessibleComponent *component;

	cell = get_row (object, row_text, log_fp);
	if (cell) {
		if (Accessible_isComponent (cell))
			component = Accessible_getComponent (cell);
		else {
			Accessible_unref (cell);
			error =  (LDTP_ERROR_OBJ_NOT_COMPONENT_TYPE);
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
			return error;
		}
		AccessibleComponent_getExtents (component, &x, &y, &width, &height, SPI_COORD_TYPE_SCREEN);
		/* NOTE: Some widgets doesnot sense the double click event if they are performed exactly on the 
		 * border. In order to take care of those situtations also we make adjustments in the x and y value
		 * to fire the event at the center of the object.
		 */
		x = x + width / 2;
		y = y + height / 2;
		flag = AccessibleComponent_grabFocus (component);
		Accessible_unref (cell);
		Accessible_unref (component);
		if (!flag) {
			error =  (LDTP_ERROR_UNABLE_TO_GRAB_FOCUS);
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
			return error;
		}
		flag = SPI_generateMouseEvent (x, y, "b1d");
		if (flag)
			return  (LDTP_ERROR_SUCCESS);
		else {
			error =  (LDTP_ERROR_DOUBLE_CLICK_FAILED);
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
			return error;
		}
	}
	error =  (LDTP_ERROR_UNABLE_TO_GET_CELL_HANDLE_FAILED);
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static Accessible*
get_row_index (Accessible *object, long index, FILE *log_fp)
{
	long n_rows, n_cols;
	LDTPErrorCode error;
	Accessible *cell;
	AccessibleTable *table;

	table = Accessible_getTable (object);
	if (!table)
		return NULL;
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);
	g_print ("Tree table - number of rows:%ld\tColumn: %ld\n", n_rows, n_cols);

	if (index < 0 || index >= n_rows) {
		char msg [256];
		g_sprintf (msg, "Row %ld doesnot exist in table", index);
		log_msg (LDTP_LOG_CAUSE, msg, log_fp);
		Accessible_unref (table);
		error =  (LDTP_ERROR_ROW_DOES_NOT_EXIST);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return NULL;
	}

	/*
	  FIXME: Column value should be dynamic instead of static value
	*/
	cell = AccessibleTable_getAccessibleAt (table, index, 0);
	Accessible_unref (table);
	return cell;
}

/*
  Double click row
*/
static LDTPErrorCode
double_click_row_index (Accessible *object, long index, FILE *log_fp)
{
	long x, y, height, width;
	SPIBoolean flag = FALSE;
	LDTPErrorCode error;
	Accessible *cell;

	cell = get_row_index (object, index, log_fp);
	if (cell) {
		AccessibleComponent *component;
		if (Accessible_isComponent (cell))
			component = Accessible_getComponent (cell);
		else {
			Accessible_unref (cell);
			error =  (LDTP_ERROR_OBJ_NOT_COMPONENT_TYPE);
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
			return error;
		}
		AccessibleComponent_getExtents (component, &x, &y, &width, &height, SPI_COORD_TYPE_SCREEN);
		/* NOTE: Some widgets doesnot sense the double click event if they are performed exactly on the 
		 * border. In order to take care of those situtations also we make adjustments in the x and y value
		 * to fire the event at the center of the object.
		 */
		x = x + width / 2;
		y = y + height / 2;
		flag = AccessibleComponent_grabFocus (component);
		Accessible_unref (cell);
		Accessible_unref (component);
		if (!flag) {
			error =  (LDTP_ERROR_UNABLE_TO_GRAB_FOCUS);
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
			return error;
		}
		flag = SPI_generateMouseEvent (x, y, "b1d");
		if (flag)
			return  (LDTP_ERROR_SUCCESS);
		else {
			error =  (LDTP_ERROR_DOUBLE_CLICK_FAILED);
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
			return error;
		}
	}
	error = LDTP_ERROR_UNABLE_TO_GET_CELL_HANDLE_FAILED;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static LDTPErrorCode
select_row_index (Accessible *object, long row, FILE *log_fp)
{
	LDTPErrorCode error;
	Accessible *cell = get_row_index (object, row, log_fp);
	if (cell && Accessible_isComponent (cell)) {
		SPIBoolean flag;
		AccessibleComponent *accessible_component;

		accessible_component = Accessible_getComponent (cell);
		flag = AccessibleComponent_grabFocus (accessible_component);

		Accessible_unref (accessible_component);
		Accessible_unref (cell);
		if (flag)
			return  (LDTP_ERROR_SUCCESS);
		error =  (LDTP_ERROR_UNABLE_TO_GRAB_FOCUS);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
	else {
		if (cell)
			Accessible_unref (cell);
		log_msg (LDTP_LOG_CAUSE, "Object is not accessible component", log_fp);
		error =  (LDTP_ERROR_OBJ_NOT_COMPONENT_TYPE);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
}

static LDTPErrorCode
select_last_row (Accessible *object, FILE *log_fp)
{
	long n_rows;
	LDTPErrorCode error;
	Accessible *cell;
	AccessibleTable *table;

	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
  
	g_print ("Tree table - number of rows:%ld\n", n_rows);
 
	cell = AccessibleTable_getAccessibleAt (table, n_rows-1, 0);
	if (Accessible_isComponent (cell)) {
		SPIBoolean flag = FALSE;
		AccessibleComponent *accessible_component;

		accessible_component = Accessible_getComponent (cell);
		AccessibleComponent_grabFocus (accessible_component);

		Accessible_unref (accessible_component);
		Accessible_unref (cell);
		Accessible_unref (table);
		if (flag == TRUE)
			return  (LDTP_ERROR_SUCCESS);
		else {
			error =  (LDTP_ERROR_UNABLE_TO_GRAB_FOCUS);
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
			return error;
		}
	}
	else {
		Accessible_unref (cell);
		Accessible_unref (table);
		error =  (LDTP_ERROR_OBJ_NOT_COMPONENT_TYPE);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
}

static LDTPErrorCode
right_click (Accessible *object, GSList *l, FILE *log_fp)
{
	long i, j;
	LDTPErrorCode error;
	gboolean flag = FALSE;
	long n_rows, n_cols;
	Accessible *cell;
	AccessibleTable *table;
	char *name = NULL;
	char *row_text = NULL;
	long n_matches = 0, matches = 0;
	long x, y, height, width;

	if (l && l->data)
		row_text = l->data;
	else {
		error = LDTP_ERROR_RIGHT_CLICK_FAILED;
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
	l = l->next;
	if (l && l->data)
		n_matches = atol (l->data);
    
	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);
	g_print ("Number of rows: %ld\tColumn: %ld\n", n_rows, n_cols);

	for (i = 0; i < n_rows; i++) {
		for (j = 0; j < n_cols; j++) {
			long child_count;
			cell = AccessibleTable_getAccessibleAt (table, i, j);
			child_count = Accessible_getChildCount (cell);
			g_print ("i = %ld, j = %ld, child_count: %ld\n", i, j, child_count);
			if (child_count > 0) {
				long i;
				Accessible *child;
				for (i = 0; i < child_count; i++) {
					child = Accessible_getChildAtIndex (cell, i);
					name = Accessible_getName (child);
					if (ldtp_debug && name && row_text)
						g_print ("Name: *%s* - Row: *%s*\n", name, row_text);
					if (g_utf8_collate (name, row_text) == 0) {
						g_print ("Matches: %ld - n_matches: %ld\n", matches, n_matches);
						if (n_matches != matches) {
							matches++;
							Accessible_unref (child);
							g_print ("Continue\n");
							continue;
						} 

						if (Accessible_isComponent (cell)) {
							AccessibleComponent *accessible_component;
							accessible_component = Accessible_getComponent (cell);
							AccessibleComponent_getExtents (accessible_component,
											&x, &y, &height, &width,
											SPI_COORD_TYPE_SCREEN);
							AccessibleComponent_grabFocus (accessible_component);
							flag = TRUE;
							Accessible_unref (accessible_component);
						}
						SPI_freeString (name);
						Accessible_unref (child);
						Accessible_unref (cell);
						break;
					} // if
					SPI_freeString (name);
					Accessible_unref (child);
				} // for
			} else {
				name = Accessible_getName (cell);
				g_print ("name: %s row_text: %s\n", name, row_text);
				if (g_utf8_collate (name, row_text) == 0) {
					if (Accessible_isComponent (cell)) {
						AccessibleComponent *accessible_component;
						accessible_component = Accessible_getComponent (cell);
						/*
						AccessibleComponent_getExtents (accessible_component,
										&x, &y, &height, &width,
										SPI_COORD_TYPE_SCREEN);
						*/
						AccessibleComponent_getPosition (accessible_component,
										&x, &y,
										SPI_COORD_TYPE_SCREEN);
						AccessibleComponent_getSize (accessible_component,
										 &height, &width);
						AccessibleComponent_grabFocus (accessible_component);
						flag = TRUE;
						Accessible_unref (accessible_component);
					}
					SPI_freeString (name);
					Accessible_unref (cell);
					break;
				}
			}
			Accessible_unref (cell);
		} // for j
		if (flag == TRUE)
			break;
	} // for i
	Accessible_unref (table);
	if (flag) {
		g_print ("x=%ld, y=%ld, width=%ld, height=%ld\n", x, y, width, height);
		/*
		if (SPI_generateMouseEvent (x + width / 2, y + height / 2, "b3c"))
			return  (LDTP_ERROR_SUCCESS);
		*/
		if (SPI_generateMouseEvent (x, y, "b3c"))
			return  (LDTP_ERROR_SUCCESS);
	}
	error = LDTP_ERROR_RIGHT_CLICK_FAILED;
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static LDTPErrorCode
get_row_count (Accessible *object, GSList **l)
{
	long n_rows;
	AccessibleTable *table;

	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
  
	g_print ("table - number of rows:%ld\n", n_rows);

	Accessible_unref(table);
	*l = g_slist_prepend (*l, g_strdup_printf ("%ld", n_rows));
	return  (LDTP_ERROR_SUCCESS);
}

static LDTPErrorCode
does_row_exists (Accessible *object, char *row_text, FILE *log_fp)
{
	long n_rows, n_cols;
	AccessibleTable *table;
	LDTPErrorCode error;
	Accessible *cell;
	gboolean flag = FALSE;

	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);
  
	if (!row_text) {
		if (n_rows != 0) {
			log_msg (LDTP_LOG_INFO, "Rows available in Table cell", log_fp);
			Accessible_unref (table);
			return  (LDTP_ERROR_SUCCESS);
		}
		Accessible_unref (table);
		error =  (LDTP_ERROR_ROW_DOES_NOT_EXIST);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
		return error;
	}
	else {
		long i, j;
		for (i = 0; i < n_rows; i++) {
			for (j = 0; j < n_cols; j++) {
				long child_count;
				cell = AccessibleTable_getAccessibleAt (table, i, j);
				child_count = Accessible_getChildCount (cell);
				long k;
				char *name;
				Accessible *child;
				if (child_count > 0) {
					for (k = 0; k < child_count; k++) {
						child = Accessible_getChildAtIndex (cell, k);
						name = Accessible_getName (child);
						if (name && g_utf8_collate (name, row_text) == 0) {
							SPI_freeString (name);
							Accessible_unref (child);
							flag = TRUE;
							break;
						} // if
						SPI_freeString (name);
						Accessible_unref (child);
					} // for
					Accessible_unref (cell);
					if (flag == TRUE)
						break;
				}
				else {
					name = Accessible_getName (cell);
					if (name && g_utf8_collate (name, row_text) == 0) {
						SPI_freeString (name);
						flag = TRUE;
					} // if
					Accessible_unref (cell);
				}
			} // for j
			if (flag == TRUE)
				break;
		} // for i
		Accessible_unref (table);
		if (flag == TRUE)
			return  (LDTP_ERROR_SUCCESS);
		else {
			error =  (LDTP_ERROR_ROW_DOES_NOT_EXIST);
			log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
			return error;
		}
	}
}

/*
  FIXME: Logic should be same as get_cell_value
*/

static LDTPErrorCode
verify_cell_value (LDTPClientContext* cctxt)
{
	char *name;
	long row = 0, column = 0, n_rows, n_cols, n_actions;
	long child_count;
	LDTPErrorCode error;
	Accessible *child_object;
	AccessibleAction *action;
	AccessibleTable *table;
	GSList *l = cctxt->req->arg_list;
	Accessible *object = cctxt->gui_handle->handle;

	if (l && l->data) {
		row = atol (l->data);
		l = l->next;
		if (l && l->data) {
			column = atol (l->data);
		}
	}

	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);
 
	if (n_rows < row) {
		Accessible_unref (table);
		error =  (LDTP_ERROR_ACTUAL_ROW_COUNT_LESS_THAN_GIVEN_ROW_COUNT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
		return error;
	}
	if (n_cols < column) {
		Accessible_unref (table);
		error =  (LDTP_ERROR_ACTUAL_COLUMN_COUNT_LESS_THAN_GIVEN_COLUMN_COUNT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
		return error;
	}  

	g_print ("Tree table - number of rows:%ld\tColumn: %ld\n", n_rows, n_cols);
	child_object = AccessibleTable_getAccessibleAt (table, row, column);

	do { 
		child_count = Accessible_getChildCount (child_object);
		if (child_count != 0)
			child_object = Accessible_getChildAtIndex (child_object, 0);
	} while (child_count > 0);

	cctxt->gui_handle->handle = child_object;
	if (Accessible_isComponent (child_object)) {
		int child_type;
		child_type = get_object_type (child_object);
		if (child_type == SPI_ROLE_TABLE_CELL) {
			long j;
			action = Accessible_getAction (child_object);
			n_actions = AccessibleAction_getNActions (action);
			for (j = 0; j < n_actions; j++) {
				/*
				  FIXME: Use Accessible_getRole
				*/
				name = AccessibleAction_getName (action, j);
				/* Handled at present only Toggle & Text type of cell */
				if (g_ascii_strcasecmp (name, "toggle") == 0) {
					GSList *l = g_slist_last (cctxt->req->arg_list);

					cctxt->req->arg_list = g_slist_prepend (cctxt->req->arg_list, l);
					g_print ("Table cell is of type 'Toggle'\n");
					error = toggle_button_main (cctxt, LDTP_CMD_VERIFYTOGGLED);
					cctxt->req->arg_list = g_slist_remove (cctxt->req->arg_list, l->data);

					SPI_freeString (name);
					Accessible_unref (action);
					Accessible_unref (table);
					Accessible_unref (child_object);
					cctxt->gui_handle->handle = object;
					return error;
				}
				if (g_ascii_strcasecmp (name, "edit") == 0) {
					GSList *l = g_slist_last (cctxt->req->arg_list);

					cctxt->req->arg_list = g_slist_prepend (cctxt->req->arg_list, l);
					g_print ("Table cell is text box\n");
					error = text_main (cctxt, LDTP_CMD_VERIFYSETTEXT);
					cctxt->req->arg_list = g_slist_remove (cctxt->req->arg_list, l->data);
					SPI_freeString (name);
					Accessible_unref (action);
					Accessible_unref (table);
					Accessible_unref (child_object);
					cctxt->gui_handle->handle = object;
					return error;
				}
				SPI_freeString (name);
			} //for
			Accessible_unref (action);
		}  // end of table cell
		else if (child_type == SPI_ROLE_TEXT) {
			GSList *l = g_slist_last (cctxt->req->arg_list);

			cctxt->req->arg_list = g_slist_prepend (cctxt->req->arg_list, l);
			g_print ("Table cell is text box\n");
			error = text_main (cctxt, LDTP_CMD_VERIFYSETTEXT);
			cctxt->req->arg_list = g_slist_remove (cctxt->req->arg_list, l->data);
			Accessible_unref (table);
			Accessible_unref (child_object);
			cctxt->gui_handle->handle = object;
			return error;
		}
		else if (child_type == SPI_ROLE_TOGGLE_BUTTON) {
			GSList *l = g_slist_last (cctxt->req->arg_list);

			cctxt->req->arg_list = g_slist_prepend (cctxt->req->arg_list, l);
			g_print ("Table cell is of type Toggle\n");
			error = toggle_button_main (cctxt, LDTP_CMD_VERIFYTOGGLED);
			cctxt->req->arg_list = g_slist_remove (cctxt->req->arg_list, l->data);
			Accessible_unref (table);
			Accessible_unref (child_object);
			cctxt->gui_handle->handle = object;
			return error;
		}
	} //if
	Accessible_unref (table);
	Accessible_unref (child_object);
	cctxt->gui_handle->handle = object;
	error =  (LDTP_ERROR_VERIFY_TABLE_CELL_FAILED);
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
	return error;
}  

/* This function has to be modified like verify_cell_value */
static LDTPErrorCode
set_cell_value (LDTPClientContext* cctxt)
{
	char *name;
	long row = 0, column = 0;
	long n_rows, n_cols, n_actions;
	LDTPErrorCode error;
	Accessible *cell;
	AccessibleAction *action;
	AccessibleTable *table;
	GSList *l = cctxt->req->arg_list;
	Accessible *object = cctxt->gui_handle->handle;
  
	if (l && l->data) {
		row = atol (l->data);
		l = l->next;
		if (l && l->data) {
			column = atol (l->data);
		}
	}

	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);

	if (n_rows < row) {
		Accessible_unref (table);
		error =  (LDTP_ERROR_ACTUAL_ROW_COUNT_LESS_THAN_GIVEN_ROW_COUNT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
		return error;
	}
	if (n_cols < column) {
		Accessible_unref (table);
		error =  (LDTP_ERROR_ACTUAL_COLUMN_COUNT_LESS_THAN_GIVEN_COLUMN_COUNT);
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
		return error;
	}  

	g_print ("Tree table - number of rows:%ld\tColumn: %ld\n", n_rows, n_cols);
	cell = AccessibleTable_getAccessibleAt (table, row, column);
	cctxt->gui_handle->handle = cell;
	if (Accessible_isComponent (cell)) {
		long j;
		action = Accessible_getAction (cell);
		n_actions = AccessibleAction_getNActions (action);
		for (j = 0; j < n_actions; j++) {
			name = AccessibleAction_getName (action, j);
			/* Handled at present only Toggle & Text type of cell */
			if (g_ascii_strcasecmp (name, "toggle") == 0) {
				GSList *l = g_slist_last (cctxt->req->arg_list);

				cctxt->req->arg_list = g_slist_prepend (cctxt->req->arg_list, l);
				g_print ("Table cell is of type 'Toggle'\n");
				error = toggle_button_main (cctxt, LDTP_CMD_CLICK); 
				cctxt->req->arg_list = g_slist_remove (cctxt->req->arg_list, l->data);
				SPI_freeString (name);
				Accessible_unref (action);
				Accessible_unref (cell);
				Accessible_unref (table);
				cctxt->gui_handle->handle = object;
				return error;
			}
			if (g_ascii_strcasecmp (name, "edit") == 0) {
				GSList *l = g_slist_last (cctxt->req->arg_list);

				cctxt->req->arg_list = g_slist_prepend (cctxt->req->arg_list, l);
				g_print ("Table cell is text box\n");
				error = text_main (cctxt, LDTP_CMD_SETTEXTVALUE);
				cctxt->req->arg_list = g_slist_remove (cctxt->req->arg_list, l->data);
				SPI_freeString (name);
				Accessible_unref (action);
				Accessible_unref (cell);
				Accessible_unref (table);
				cctxt->gui_handle->handle = object;
				return error;
			}
			SPI_freeString (name);
		} //for
		Accessible_unref (action);
	} //if
	Accessible_unref (cell);
	Accessible_unref (table);
	cctxt->gui_handle->handle = object;
	error =  (LDTP_ERROR_SET_TABLE_CELL_FAILED);
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), cctxt->log_fp);
	return error;
}  

static LDTPErrorCode
do_action (Accessible *object, long row, long col, char* command, FILE *log_fp)
{
	long j;
	long n_rows, n_cols;
	char msg[256];
	char *name = NULL;
	LDTPErrorCode error;
	Accessible *cell;
	AccessibleTable *table;
	long actionCount;

	table = Accessible_getTable (object);
	n_rows = AccessibleTable_getNRows (table);
	n_cols = AccessibleTable_getNColumns (table);
	g_print ("Tree table - number of rows:%ld\tColumn: %ld\n", n_rows, n_cols);

	cell = AccessibleTable_getAccessibleAt (table, row, col);
	AccessibleTable_unref (table);

	if (Accessible_isComponent (cell)) {
		long count = -1;
		AccessibleAction *childAction = NULL;

		count = Accessible_getChildCount (cell);
		if (count > 0) {
			long i;
			for (i = 0; i < count; i++) {
				Accessible* child;
				child = Accessible_getChildAtIndex (cell, i);
				childAction = Accessible_getAction (child);
				Accessible_unref (child);
				actionCount = AccessibleAction_getNActions (childAction);

				for (j = 0; j < actionCount; j++) {	
					name = AccessibleAction_getName (childAction, j);
					if (g_utf8_collate (name, command) == 0) {
						Accessible_unref (cell);
						if (AccessibleAction_doAction (childAction, j)) {
							SPI_freeString (name);
							AccessibleAction_unref (childAction);
							return  (LDTP_ERROR_SUCCESS);
						}
						else {
							AccessibleAction_unref (childAction);
							g_sprintf (msg, "Failed to perform %s action on object!!", name);
							log_msg (LDTP_LOG_CAUSE, msg, log_fp);
							SPI_freeString (name);
							error =  (LDTP_ERROR_UNABLE_TO_PERFORM_ACTION);
							log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
							return error;
						}
					}
					SPI_freeString (name);
				}
				Accessible_unref (childAction);
			}
		} 
		else {
			childAction = Accessible_getAction (cell);
			actionCount = AccessibleAction_getNActions (childAction);
			for (j = 0; j < actionCount; j++) {	
				name = AccessibleAction_getName (childAction, j);
				if (g_utf8_collate (name, command) == 0) {
					Accessible_unref (cell);					
					if (AccessibleAction_doAction (childAction, j)) {
						SPI_freeString (name);
						Accessible_unref (childAction);
						return  (LDTP_ERROR_SUCCESS);
					}
					else {
						AccessibleAction_unref (childAction);
						g_sprintf (msg, "Failed to perform %s action on object!!", name);
						log_msg (LDTP_LOG_CAUSE, msg, log_fp);
						SPI_freeString (name);
						error =  (LDTP_ERROR_UNABLE_TO_PERFORM_ACTION);
						log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
						return error;
					}
				}
				SPI_freeString (name);
			}
			Accessible_unref (childAction);
		}
	}
	Accessible_unref (cell);
	error =  (LDTP_ERROR_UNABLE_TO_PERFORM_ACTION);
	log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (error), log_fp);
	return error;
}

static gboolean
get_status (Accessible *object, int row, int col)
{
	gboolean ret = FALSE;
	Accessible *cell;
	AccessibleTable *table;
	AccessibleStateSet* stateset;

	table = Accessible_getTable (object);

	cell = AccessibleTable_getAccessibleAt (table, row, col);
	AccessibleTable_unref (table);

	stateset = Accessible_getStateSet (cell);

	if (AccessibleStateSet_contains (stateset, SPI_STATE_CHECKED))
		ret = TRUE; // Checked state
	else
		ret = FALSE; // Unchecked state

	Accessible_unref (cell);
	return ret;
}

LDTPErrorCode
tree_table_main (LDTPClientContext* cctxt, int command)
{
	LDTPErrorCode error;
	switch (command) {
	case LDTP_CMD_SELECTROWINDEX: {
		long row = 0;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			row = atol (cctxt->req->arg_list->data);
		g_print ("Select index: %ld\n", row);
		error = select_row_index (cctxt->gui_handle->handle, row, cctxt->log_fp);
		break;
	}
	case LDTP_CMD_SELECTROW:
		error = select_row (cctxt->gui_handle->handle, cctxt->req->arg_list, cctxt->log_fp);
		break;
	case LDTP_CMD_SELECTROWPARTIALMATCH:
		error = select_row_partial_match (cctxt->gui_handle->handle,
						  cctxt->req->arg_list->data, cctxt->log_fp);
		break;
	case LDTP_CMD_DOUBLECLICKROW:
		error = double_click_row (cctxt->gui_handle->handle,
					  cctxt->req->arg_list->data, cctxt->log_fp);
		break;
	case LDTP_CMD_DOUBLECLICKROWINDEX: {
		long row_index = 0;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			row_index = atol (cctxt->req->arg_list->data);
		g_print ("Select index: %ld\n", row_index);
		error = double_click_row_index (cctxt->gui_handle->handle,
						row_index, cctxt->log_fp);
		break;
	}
	case LDTP_CMD_DOUBLECLICK:
		error = double_click (cctxt->gui_handle->handle,
				      cctxt->log_fp);
		break;
	case LDTP_CMD_GETTREETABLEROWINDEX:
		error = get_current_row_index (cctxt->gui_handle->handle, &cctxt->req->arg_list, cctxt->log_fp);
		if (error == LDTP_ERROR_SUCCESS) {
			cctxt->resp->data = cctxt->req->arg_list->data;
			cctxt->resp->data_len = g_utf8_strlen (cctxt->resp->data, -1);
			cctxt->req->arg_list = g_slist_remove (cctxt->req->arg_list, cctxt->req->arg_list->data);
		}
		break;
	case LDTP_CMD_SELECTLASTROW:
		error = select_last_row (cctxt->gui_handle->handle, cctxt->log_fp);
		break;
	case LDTP_CMD_RIGHTCLICK:
		error = right_click (cctxt->gui_handle->handle, cctxt->req->arg_list, cctxt->log_fp);
		break;
	case LDTP_CMD_VERIFYTABLECELL:
		error = verify_cell_value (cctxt);
		break;
	case LDTP_CMD_GETROWCOUNT:
		error = get_row_count (cctxt->gui_handle->handle, &cctxt->req->arg_list);
		if (error == LDTP_ERROR_SUCCESS) {
			cctxt->resp->data = cctxt->req->arg_list->data;
			cctxt->resp->data_len = g_utf8_strlen (cctxt->resp->data, -1);
			cctxt->req->arg_list = g_slist_remove (cctxt->req->arg_list, cctxt->req->arg_list->data);
		}
		break;
	case LDTP_CMD_SETCELLVALUE:
		error = set_cell_value (cctxt);
		break;
	case LDTP_CMD_DOESROWEXIST:
		error = does_row_exists (cctxt->gui_handle->handle,
					 cctxt->req->arg_list->data, cctxt->log_fp);
		break;
	case LDTP_CMD_VERIFYPARTIALTABLECELL:
		error = verify_partial_cell_value (cctxt);
		break;
	case LDTP_CMD_GETCELLVALUE:
		error = get_cell_value (cctxt);
		/*
		  NOTE: We no need to copy data to cctxt->resp->data, as the get_cell_value
		  function will already have the required values on success
		*/
		break;
	case LDTP_CMD_CHECKROW: {
		long row = 0;
		long col = 0;
		GSList *l = cctxt->req->arg_list;
		if (l && l->data) {
			row = atol (l->data);
			l = l->next;
			if (l && l->data) {
				col = atol (l->data);
			}
		}
		if (get_status (cctxt->gui_handle->handle, row, col) == FALSE)
			error = do_action (cctxt->gui_handle->handle,
					   row, col, "toggle", cctxt->log_fp);
		else {
			log_msg (LDTP_LOG_WARNING, "Row already checked", cctxt->log_fp);
			error =  (LDTP_ERROR_SUCCESS);
		}
		break;
	}
	case LDTP_CMD_UNCHECKROW: {
		long row = 0;
		long col = 0;
		GSList *l = cctxt->req->arg_list;
		if (l && l->data) {
			row = atol (l->data);
			l = l->next;
			if (l && l->data) {
				col = atol (l->data);
			}
		}
		if (get_status (cctxt->gui_handle->handle, row, col) == TRUE)
			error = do_action (cctxt->gui_handle->handle,
					   row, col, "toggle", cctxt->log_fp);
		else {
			log_msg (LDTP_LOG_WARNING, "Row already unchecked", cctxt->log_fp);
			error =  (LDTP_ERROR_SUCCESS);
		}
		break;
	}
	case LDTP_CMD_VERIFYCHECKROW: {
		long row = 0;
		long col = 0;
		GSList *l = cctxt->req->arg_list;
		if (l && l->data) {
			row = atol (l->data);
			l = l->next;
			if (l && l->data) {
				col = atol (l->data);
			}
		}
		if (get_status (cctxt->gui_handle->handle, row, col) == TRUE)
			error =  (LDTP_ERROR_STATE_CHECKED);
		else
			error =  (LDTP_ERROR_STATE_UNCHECKED);
		break;
	}
	case LDTP_CMD_VERIFYUNCHECKROW: {
		long row = 0;
		long col = 0;
		GSList *l = cctxt->req->arg_list;
		if (l && l->data) {
			row = atol (l->data);
			l = l->next;
			if (l && l->data) {
				col = atol (l->data);
			}
		}
		if (get_status (cctxt->gui_handle->handle, row, col) == TRUE)
			error =  (LDTP_ERROR_STATE_CHECKED);
		else
			error =  (LDTP_ERROR_STATE_UNCHECKED);
		break;
	}
	case LDTP_CMD_EXPANDTABLECELL: {
		long row = 0;
		long col = 0;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			row = atol (cctxt->req->arg_list->data);
		/* FIXME:
		 * currently expand table cell performs 
		 * action on the first column only 
		 */
		error = do_action (cctxt->gui_handle->handle,
				   row, col, "expand or contract",
				   cctxt->log_fp);
		break;
	}
	case LDTP_CMD_SORTCOLUMNINDEX: {
		long col = 0;
		if (cctxt->req->arg_list && cctxt->req->arg_list->data)
			col = atol (cctxt->req->arg_list->data);
		g_print ("Sort column: %ld\n", col);
		error = sort_column_index (cctxt->gui_handle->handle, col, cctxt->log_fp);
		break;
	}	
	case LDTP_CMD_SORTCOLUMN:
		error = sort_column (cctxt->gui_handle->handle,
				     cctxt->req->arg_list->data, cctxt->log_fp);
		break;
	case LDTP_CMD_KBDENTER:
		error = device_main (cctxt, command);
		break;
	default:
		error = LDTP_ERROR_COMMAND_NOT_IMPLEMENTED;
	}
	return error;
}
