/* Simple I/O Function header file
 *
 * Copyright (C) 2011 Andres Mejia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef LAME_PORTABLEIO_H
#define LAME_PORTABLEIO_H

#include <stdio.h>
#include <stdint.h>
#include <math.h>

#include "config.h"

static inline int ReadByte(FILE * fp)
{
  int val = (getc(fp)) & 0xFF;
  return (val & 0x80) ? (val - 0x100) & 0xFF : val;
}

static inline int Read16BitsLowHigh(FILE * fp)
{
  int val = ReadByte(fp);
  val |= ReadByte(fp) << 8;
  return (val & 0x8000) ? (val - 0x10000) & 0xFFFF : val;
}

static inline int Read16BitsHighLow(FILE * fp)
{
  int val = (ReadByte(fp) << 8) | ReadByte(fp);
  return (val & 0x8000) ? (val - 0x10000) & 0xFFFF : val;
}

static inline void Write8Bits(FILE * fp, int i)
{
  putc(i & 0xFF, fp);
}

static inline void Write16BitsLowHigh(FILE * fp, int i)
{
  Write8Bits(fp, i & 0xFF);
  Write8Bits(fp, (i >> 8) & 0xFF);
}

static inline void Write16BitsHighLow(FILE * fp, int i)
{
  Write8Bits(fp, (i >> 8) & 0xFF);
  Write8Bits(fp, i & 0xFF);
}

static inline int Read24BitsHighLow(FILE * fp)
{
  int val = (ReadByte(fp) << 16) | (ReadByte(fp) << 8) | ReadByte(fp);
  return (val & 0x800000) ? (val - 0x1000000) & 0xFFFFFF : val;
}

static inline int Read32Bits(FILE * fp)
{
  int val = Read16BitsLowHigh(fp);
  val |= Read16BitsLowHigh(fp) << 16;
  return val;
}

static inline int Read32BitsHighLow(FILE * fp)
{
  return (Read16BitsHighLow(fp) << 16) | Read16BitsHighLow(fp);
}

static inline void Write32Bits(FILE * fp, int i)
{
  Write16BitsLowHigh(fp, i & 0xFFFF);
  Write16BitsLowHigh(fp, (i >> 16) & 0xFFFF);
}

static inline void Write32BitsHighLow(FILE * fp, int i)
{
  Write16BitsHighLow(fp, (i >> 16) & 0xFFFF);
  Write16BitsHighLow(fp, i & 0xFFFF);
}

static inline void ReadBytes(FILE * fp, char *p, int n)
{
  while (!feof(fp) && (n-- > 0))
    *p++ = ReadByte(fp);
  while (n-- > 0)
    *p++ = 0;
}

static inline void ReadBytesSwapped(FILE * fp, char *p, int n)
{
  char *tmp = p;
  while (!feof(fp) && (n-- > 0))
    *tmp++ = ReadByte(fp);
  while (n-- > 0)
    *tmp++ = 0;
  tmp--;
  while (p < tmp)
  {
    n = *p;
    *p++ = *tmp;
    *tmp-- = n;
  }
}

static inline void WriteBytes(FILE * fp, char *p, int n)
{
  while (n-- > 0)
    Write8Bits(fp, *p++);
}

static inline void WriteBytesSwapped(FILE * fp, char *p, int n)
{
  p += n;
  while (n-- > 0)
    Write8Bits(fp, *p--);
}

static inline long double BytesToLongDouble(void *d)
{
  long double val = *((long double *)d);
  return val;
}

static inline double ReadIeeeExtendedHighLow(FILE * fp)
{
  char bytes[10];
#if (defined(__GNUC__) && (defined(__i386__) || defined(__x86_64__)))
#ifdef WORDS_BIGENDIAN
  ReadBytes(fp, bytes, sizeof(bytes));
#else
  ReadBytesSwapped(fp, bytes, sizeof(bytes));
#endif
  /* Take advantage of 80-bit precision of long double from GNU C compiler */
  return BytesToLongDouble(&bytes);
#else
  ReadBytes(fp, &bytes, sizeof(bytes));
  double val;
  uint32_t mantissa_high, mantissa_low;
  int16_t exponent;
  exponent = ((bytes[0] & 0x7F) << 8) | bytes[1];
  mantissa_high = (bytes[2] << 24) |
                  (bytes[3] << 16) |
                  (bytes[4] << 8)  |
                   bytes[5];
  mantissa_low = (bytes[6] << 24) |
                 (bytes[7] << 16) |
                 (bytes[8] << 8)  |
                  bytes[9];
  if (exponent == 0 && mantissa_high == 0 && mantissa_low == 0)
    val = 0;
  else
  {
    /* TODO: Perhaps this should also detect NaN */
    if (exponent == 0x7FFF)
      val = HUGE_VAL;
    else
    {
      exponent -= 0x3FFF; /* Bias for long double precision */
      val = ldexp(mantissa_high, (exponent -= 31));
      val += ldexp(mantissa_low, (exponent -= 32));
    }
  }
  return (bytes[0] & 0x80) ? -val : val;
#endif
}

#define Read32BitsLowHigh(f)  Read32Bits(f)
#define Write32BitsLowHigh(f, i) Write32Bits(f, i)

#endif
