/***************************************************************************
 *   Copyright (C) 2003 by Stephen Allewell                                *
 *   stephen@mirramar.fsnet.co.uk                                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <qimage.h>
#include <qpixmap.h>
#include <qpainter.h>
#include <qcursor.h>
#include <qpen.h>
#include <qpopupmenu.h>
#include <qpaintdevicemetrics.h>
#include <qclipboard.h>
#include <qmime.h>
#include <qspinbox.h>
#include <qevent.h>
#include <kapplication.h>
#include <kdebug.h>
#include <kruler.h>
#include <kprinter.h>
#include <klocale.h>
#include <kaction.h>
#include <kiconloader.h>
#include "editview.h"
#include "kxstitchview.h"
#include "preview.h"
#include "paletteview.h"
#include "kxstitchdoc.h"
#include "patterncanvas.h"
#include "flosspalette.h"
#include "kxstitch.h"
#include "kxstitchdoc.h"
#include "configuration.h"
#include "patternpropertiesdialog.h"
#include "patternlibrarydialog.h"
#include "texttooldialog.h"
#include "extendpattern.h"
#include "kxstitchmimesource.h"
#include "flosspalette.h"
#include <iostream>
#include <math.h>
#include <stdio.h>

extern "C" double round(double x);

#define TIMERSPEED 50

EditView::EditView(KXStitchDoc* doc,QWidget* parent)
  : QScrollView(parent),
    m_doc(doc),
    m_pasteData(0),
    m_scaleFormat(Stitches),
    m_updatingDisplay(false),
    m_points(0)
{
  KActionCollection* actionCollection = ((KXStitchApp*)topLevelWidget())->actionCollection();
  KIconLoader *loader = KGlobal::iconLoader();
  KRadioAction* radioAction;

  m_editUndo = KStdAction::undo(this, SLOT(slotEditUndo()), actionCollection);
  m_editUndo->setEnabled(false);
  m_editRedo = KStdAction::redo(this, SLOT(slotEditRedo()), actionCollection);
  m_editRedo->setEnabled(false);
  m_editCut = KStdAction::cut(this, SLOT(slotEditCut()), actionCollection);
  m_editCopy = KStdAction::copy(this, SLOT(slotEditCopy()), actionCollection);
  m_editPaste = KStdAction::paste(this, SLOT(slotEditPaste()), actionCollection);
  m_editMirrorRotateCopies = new KToggleAction(i18n("Mirror/rotate makes copies"), 0, 0, 0, actionCollection, "mirrorrotatecopies");
  m_editRotate90 = new KAction(i18n("90 degrees"), 0, this, SLOT(slotEditRotate90()), actionCollection, "rotate90");
  m_editRotate180 = new KAction(i18n("180 degrees"), 0, this, SLOT(slotEditRotate180()), actionCollection, "rotate180");
  m_editRotate270 = new KAction(i18n("270 degrees"), 0, this, SLOT(slotEditRotate270()), actionCollection, "rotate270");
  m_editMirrorVertical = new KAction(i18n("Vertically"), 0, this, SLOT(slotEditMirrorVertical()), actionCollection, "mirrorvertical");
  m_editMirrorHorizontal = new KAction(i18n("Horizontally"), 0, this, SLOT(slotEditMirrorHorizontal()), actionCollection, "mirrorhorizontal");
  new KAction(i18n("Insert text..."), QIconSet(loader->loadIcon("text",KIcon::Toolbar)), 0,this,SLOT(slotTextTool()),actionCollection,"Tool_text");
  new KAction(i18n("Insert library pattern..."),0,this,SLOT(slotLibraryPattern()),actionCollection,"librarypattern");
  m_createLibraryPattern = new KAction(i18n("Create library pattern..."),0,this,SLOT(slotLibraryCreateLibraryPattern()),actionCollection,"createlibrarypattern");
  new KAction(i18n("Export library patterns..."),0,this,SLOT(slotLibraryExport()),actionCollection,"exportlibrary");
  new KAction(i18n("Import library patterns..."),0,this,SLOT(slotLibraryImport()),actionCollection,"importlibrary");
  KStdAction::fitToPage(this, SLOT(slotFitPage()), actionCollection);
  KStdAction::fitToWidth(this, SLOT(slotFitWidth()), actionCollection);
  KStdAction::fitToHeight(this, SLOT(slotFitHeight()), actionCollection);
  m_zoomIn = KStdAction::zoomIn(this, SLOT(slotZoomIn()), actionCollection);
  m_zoomOut = KStdAction::zoomOut(this, SLOT(slotZoomOut()), actionCollection);
  new KAction(i18n("Extend pattern..."), QIconSet(loader->loadIcon("extpattern",KIcon::User)), 0, this, SLOT(slotPatternExtend()), actionCollection, "extendpattern");
  new KAction(i18n("Center pattern"), QIconSet(loader->loadIcon("centerpattern",KIcon::User)), 0, this, SLOT(slotPatternCenter()), actionCollection, "centerpattern");
  new KAction(i18n("Crop canvas to pattern"), 0, this, SLOT(slotCropCanvasToPattern()), actionCollection, "croptopattern");
  m_cropToSelection = new KAction(i18n("Crop canvas to selection"), 0, this, SLOT(slotCropCanvasToSelection()), actionCollection, "croptoselection");
  m_backgroundToSelection = new KAction(i18n("Fit background to selection"), 0, this, SLOT(slotFitBackgroundToSelection()), actionCollection, "fitbackgroundselection");

  radioAction = new KRadioAction(i18n("Full Stitch"), QIconSet(loader->loadIcon("full",KIcon::User)), 0, this, SLOT(slotStitchFull()), actionCollection, "Full");
  radioAction->setExclusiveGroup("Stitches");
  radioAction->setChecked(true);
  radioAction = new KRadioAction(i18n("Left to Right Half Stitch"), QIconSet(loader->loadIcon("tlbrhf",KIcon::User)), 0, this, SLOT(slotStitchTBHalf()), actionCollection, "TBHalf");
  radioAction->setExclusiveGroup("Stitches");
  radioAction = new KRadioAction(i18n("Right to Left Half Stitch"), QIconSet(loader->loadIcon("bltrhf",KIcon::User)), 0, this, SLOT(slotStitchBTHalf()), actionCollection, "BTHalf");
  radioAction->setExclusiveGroup("Stitches");
  radioAction = new KRadioAction(i18n("Bottom Left Quarter Stitch"), QIconSet(loader->loadIcon("blq",KIcon::User)), 0, this, SLOT(slotStitchBLQtr()), actionCollection, "BLQtr");
  radioAction->setExclusiveGroup("Stitches");
  radioAction = new KRadioAction(i18n("Top Left Quarter Stitch"), QIconSet(loader->loadIcon("tlq",KIcon::User)), 0, this, SLOT(slotStitchTLQtr()), actionCollection, "TLQtr");
  radioAction->setExclusiveGroup("Stitches");
  radioAction = new KRadioAction(i18n("Top Right Quarter Stitch"), QIconSet(loader->loadIcon("trq",KIcon::User)), 0, this, SLOT(slotStitchTRQtr()), actionCollection, "TRQtr");
  radioAction->setExclusiveGroup("Stitches");
  radioAction = new KRadioAction(i18n("Bottom Right Quarter Stitch"), QIconSet(loader->loadIcon("brq",KIcon::User)), 0, this, SLOT(slotStitchBRQtr()), actionCollection, "BRQtr");
  radioAction->setExclusiveGroup("Stitches");
  radioAction = new KRadioAction(i18n("Bottom Left 3/4 Stitch"), QIconSet(loader->loadIcon("bl3q",KIcon::User)), 0, this, SLOT(slotStitchBL3Qtr()), actionCollection, "BL3Qtr");
  radioAction->setExclusiveGroup("Stitches");
  radioAction = new KRadioAction(i18n("Top Left 3/4 Stitch"), QIconSet(loader->loadIcon("tl3q",KIcon::User)), 0, this, SLOT(slotStitchTL3Qtr()), actionCollection, "TL3Qtr");
  radioAction->setExclusiveGroup("Stitches");
  radioAction = new KRadioAction(i18n("Top Right 3/4 Stitch"), QIconSet(loader->loadIcon("tr3q",KIcon::User)), 0, this, SLOT(slotStitchTR3Qtr()), actionCollection, "TR3Qtr");
  radioAction->setExclusiveGroup("Stitches");
  radioAction = new KRadioAction(i18n("Bottom Right 3/4 Stitch"), QIconSet(loader->loadIcon("br3q",KIcon::User)), 0, this, SLOT(slotStitchBR3Qtr()), actionCollection, "BR3Qtr");
  radioAction->setExclusiveGroup("Stitches");
  radioAction = new KRadioAction(i18n("French Knot"), QIconSet(loader->loadIcon("frknot",KIcon::User)), 0, this, SLOT(slotStitchFRKnot()), actionCollection, "FRKnot");
  radioAction->setExclusiveGroup("Stitches");

  radioAction = new KRadioAction(i18n("Paint Mode"), QIconSet(loader->loadIcon("paintbrush",KIcon::User)), 0, this, SLOT(slotToolPaint()), actionCollection, "Tool_paint");
  radioAction->setExclusiveGroup("tools");
  radioAction->setChecked(true);
  radioAction = new KRadioAction(i18n("Draw Mode"), QIconSet(loader->loadIcon("pencil",KIcon::User)), 0, this, SLOT(slotToolDraw()), actionCollection, "Tool_draw");
  radioAction->setExclusiveGroup("tools");
  radioAction = new KRadioAction(i18n("Backstitching"), QIconSet(loader->loadIcon("backstitch",KIcon::User)), 0, this, SLOT(slotToolBackStitch()), actionCollection, "Tool_backstitch");
  radioAction->setExclusiveGroup("tools");
  radioAction = new KRadioAction(i18n("Erase stitches"), QIconSet(loader->loadIcon("eraser",KIcon::User)), 0, this, SLOT(slotToolErase()), actionCollection, "Tool_erase");
  radioAction->setExclusiveGroup("tools");
  radioAction = new KRadioAction(i18n("Outline Rectangle"), QIconSet(loader->loadIcon("o_rect",KIcon::User)), 0, this, SLOT(slotRectangle()), actionCollection, "Tool_rectangle");
  radioAction->setExclusiveGroup("tools");
  radioAction = new KRadioAction(i18n("Filled Rectangle"), QIconSet(loader->loadIcon("f_rect",KIcon::User)), 0, this, SLOT(slotFillRectangle()), actionCollection, "Tool_fill_rectangle");
  radioAction->setExclusiveGroup("tools");
  radioAction = new KRadioAction(i18n("Outline Ellipse"), QIconSet(loader->loadIcon("o_ellipse",KIcon::User)), 0, this, SLOT(slotEllipse()), actionCollection, "Tool_ellipse");
  radioAction->setExclusiveGroup("tools");
  radioAction = new KRadioAction(i18n("Filled Ellipse"), QIconSet(loader->loadIcon("f_ellipse",KIcon::User)), 0, this, SLOT(slotFillEllipse()), actionCollection, "Tool_fill_ellipse");
  radioAction->setExclusiveGroup("tools");
  radioAction = new KRadioAction(i18n("Filled Polyline"), QIconSet(loader->loadIcon("polyline",KIcon::User)), 0, this, SLOT(slotFillPolyline()), actionCollection, "Tool_fill_polyline");
  radioAction->setExclusiveGroup("tools");
  radioAction = new KRadioAction(i18n("Select Rectangular Area"), QIconSet(loader->loadIcon("s_rect",KIcon::User)), 0, this, SLOT(slotSelectRectangle()), actionCollection, "Tool_select_rectangle");
  radioAction->setExclusiveGroup("tools");
  radioAction = new KRadioAction(i18n("Pick color from image"), QIconSet(loader->loadIcon("colorpicker",KIcon::User)), 0, this, SLOT(slotColorPicker()), actionCollection, "colorpicker");
  radioAction->setExclusiveGroup("tools");

  radioAction = new KRadioAction(i18n("Stitches"), 0, this, SLOT(slotScaleFormatStitches()), actionCollection, "stitches");
  radioAction->setExclusiveGroup("ScaleFormat");
  radioAction = new KRadioAction(i18n("CM"), 0, this, SLOT(slotScaleFormatCM()), actionCollection, "cm");
  radioAction->setExclusiveGroup("ScaleFormat");
  radioAction = new KRadioAction(i18n("Inches"), 0, this, SLOT(slotScaleFormatInches()), actionCollection, "inches");
  radioAction->setExclusiveGroup("ScaleFormat");

  m_showBackground = new KToggleAction(i18n("Show background"), 0, this, SLOT(slotShowBackground()), actionCollection, "showbackground");
  m_showBackground->setChecked(false);
  m_showStitches = new KToggleAction(i18n("Show stitches"), 0, this, SLOT(slotShowStitches()), actionCollection, "showstitches");
  m_showBackstitches = new KToggleAction(i18n("Show backstitches"), 0, this, SLOT(slotShowBackStitches()), actionCollection, "showbackstitches");
  m_showGrid = new KToggleAction(i18n("Show grid lines"), 0, this, SLOT(slotShowGrid()), actionCollection, "showgrid");
  m_showScales = new KToggleAction(i18n("Show scales"), 0, this, SLOT(slotShowScales()), actionCollection, "showscales");

  radioAction = new KRadioAction(i18n("Regular stitches"), 0, this, SLOT(slotRegular()), actionCollection, "regular");
  radioAction->setExclusiveGroup("StitchView");
  radioAction = new KRadioAction(i18n("Black and white symbols"), 0, this, SLOT(slotBWSymbols()), actionCollection, "bwsymbols");
  radioAction->setExclusiveGroup("StitchView");
  radioAction = new KRadioAction(i18n("Colored Symbols"), 0, this, SLOT(slotColorSymbols()), actionCollection, "colorsymbols");
  radioAction->setExclusiveGroup("StitchView");
  radioAction = new KRadioAction(i18n("Colored blocks"), 0, this, SLOT(slotColorBlocks()), actionCollection, "colorblocks");
  radioAction->setExclusiveGroup("StitchView");
  radioAction = new KRadioAction(i18n("Colored blocks with symbols"), 0, this, SLOT(slotColorBlocksSymbols()), actionCollection, "colorblocksymbols");
  radioAction->setExclusiveGroup("StitchView");
  radioAction = new KRadioAction(i18n("Color Highlight"), 0, this, SLOT(slotColorHighlight()), actionCollection, "colorhighlight");
  radioAction->setExclusiveGroup("StitchView");

  new KToggleAction(i18n("Stitch Mask"), 0, this, SLOT(slotStitchMask()), actionCollection, "StitchMask");
  new KToggleAction(i18n("Color Mask"), 0, this, SLOT(slotColorMask()), actionCollection, "ColorMask");
  new KToggleAction(i18n("Exclude Backstitches"), 0, this, SLOT(slotExcludeBackstitches()), actionCollection, "ExcludeBackstitches");
  new KToggleAction(i18n("Exclude Knots"), 0, this, SLOT(slotExcludeKnots()), actionCollection, "ExcludeKnots");

  configure(true);

  setHScrollBarMode(AlwaysOn);
  setVScrollBarMode(AlwaysOn);
  setBackgroundMode(PaletteBase);
  viewport()->setBackgroundMode(NoBackground);
  viewport()->setFocusPolicy(QWidget::StrongFocus);
  viewport()->setMouseTracking(true);
  m_currentStitchType = Stitch::Full;
  m_stitchMask=false;
  m_colorMask=false;
  m_excludeBackstitches=false;
  m_excludeKnots=false;

  slotClipboardDataChanged();
  connect(kapp->clipboard(),SIGNAL(dataChanged()),this,SLOT(slotClipboardDataChanged()));
  connect(this,SIGNAL(selectionMade(bool)),m_editCut,SLOT(setEnabled(bool)));
  connect(this,SIGNAL(selectionMade(bool)),m_editCopy,SLOT(setEnabled(bool)));
  connect(this,SIGNAL(selectionMade(bool)),m_cropToSelection,SLOT(setEnabled(bool)));
  connect(this,SIGNAL(selectionMade(bool)),m_backgroundToSelection,SLOT(setEnabled(bool)));
  connect(this,SIGNAL(selectionMade(bool)),m_createLibraryPattern,SLOT(setEnabled(bool)));
  connect(this,SIGNAL(selectionMade(bool)),m_editRotate90,SLOT(setEnabled(bool)));
  connect(this,SIGNAL(selectionMade(bool)),m_editRotate180,SLOT(setEnabled(bool)));
  connect(this,SIGNAL(selectionMade(bool)),m_editRotate270,SLOT(setEnabled(bool)));
  connect(this,SIGNAL(selectionMade(bool)),m_editMirrorVertical,SLOT(setEnabled(bool)));
  connect(this,SIGNAL(selectionMade(bool)),m_editMirrorHorizontal,SLOT(setEnabled(bool)));
  connect(this,SIGNAL(contentsMoving(int,int)),this,SLOT(slotScrolling(int,int)));
  connect(&m_selectionTimer,SIGNAL(timeout()),this,SLOT(slotSelectionTimedOut()));
  emit selectionMade(false);

  connect(doc,SIGNAL(undoStackNotEmpty(bool)),m_editUndo,SLOT(setEnabled(bool)));
  connect(doc,SIGNAL(redoStackNotEmpty(bool)),m_editRedo,SLOT(setEnabled(bool)));
}

void EditView::setBackgroundImage(QImage image)
{
  m_background = image;
  int width=m_doc->canvas()->patternWidth();
  int height=m_doc->canvas()->patternHeight();
  m_requestedSize = QRect(0,0,width,height); // in cells
  QSize finalSize = m_requestedSize.size()*m_cellSize;
  if (!image.isNull())
  {
    m_backgroundScaled.convertFromImage(image.smoothScale(finalSize));
    m_backgroundImage = true;
    m_showBackground->setChecked(true);
  }
  m_doc->slotResetAllViews();
}

void EditView::clearBackgroundImage()
{
  m_background.reset();
  m_backgroundScaled.resize(0,0);
  m_backgroundImage = false;
  m_showBackground->setChecked(false);
  m_doc->slotResetAllViews();
}

void EditView::slotPatternExtend()
{
  ExtendPattern *dlg = new ExtendPattern(this, 0, TRUE);
  if (dlg->exec() == QDialog::Accepted)
  {
    m_doc->extendPattern(dlg->leftMargin->value(), dlg->topMargin->value(), dlg->rightMargin->value(), dlg->bottomMargin->value());
    m_doc->slotResetAllViews();
  }
}

void EditView::slotPatternCenter()
{
  m_doc->centerPattern();
  m_doc->slotResetAllViews();
}

void EditView::slotCropCanvasToPattern()
{
  m_doc->cropCanvasToPattern();
  m_doc->slotResetAllViews();
}

void EditView::slotCropCanvasToSelection()
{
  m_doc->cropCanvasToRect(selectionArea());
  clearSelection();
  m_doc->slotResetAllViews();
}

void EditView::slotFitBackgroundToSelection()
{
  m_requestedSize = selectionArea();
  clearSelection();
  QSize finalSize = m_requestedSize.size() * m_cellSize;
  if (!m_background.isNull() && finalSize != m_backgroundScaled.size())
  {
    m_backgroundScaled.convertFromImage(m_background.smoothScale(finalSize));
  }
  updateContents();
}

void EditView::slotEditUndo()
{
  m_doc->undo();
}

void EditView::slotEditRedo()
{
  m_doc->redo();
}

void EditView::slotEditCut()
{
  kapp->clipboard()->setData(new KXStitchMimeSource(m_doc->copySelection(selectionArea(),m_stitchMask?m_currentStitchType:Stitch::Delete,m_colorMask,m_excludeBackstitches,m_excludeKnots,true)));
  clearSelection();
}

void EditView::slotEditCopy()
{
  kapp->clipboard()->setData(new KXStitchMimeSource(m_doc->copySelection(selectionArea(),m_stitchMask?m_currentStitchType:Stitch::Delete,m_colorMask,m_excludeBackstitches,m_excludeKnots,false)));
  clearSelection();
}

void EditView::slotEditPaste()
{
  m_pasteData = kapp->clipboard()->data();
  m_pasteDataIsLibrary = false;
  setMode(EditView::Paste);
}

void EditView::editRotate(int angle)
{
  QByteArray selection = m_doc->copySelection(selectionArea(),m_stitchMask?m_currentStitchType:Stitch::Delete,m_colorMask,m_excludeBackstitches,m_excludeKnots,!m_editMirrorRotateCopies->isChecked());
  clearSelection();

  QDataStream stream(selection, IO_ReadOnly);
  QString scheme;
  Q_INT32 width;
  Q_INT32 height;
  Q_INT8  count;
  Q_INT8  type;
  Q_INT32 backstitches;
  Q_INT32 knots;
  QPoint  start;
  QPoint  end;
  QColor  color;
  FlossPalette palette;
  PatternCanvas canvas;

  struct TRANSLATION
  {
    Stitch::Type original;
    Stitch::Type replace;
  };

   struct TRANSLATION rotate90Map[] =
  {
    { Stitch::TLQtr, Stitch::BLQtr },
    { Stitch::TRQtr, Stitch::TLQtr },
    { Stitch::BLQtr, Stitch::BRQtr },
    { Stitch::BRQtr, Stitch::TRQtr },
    { Stitch::TL3Qtr, Stitch::BL3Qtr },
    { Stitch::TR3Qtr, Stitch::TL3Qtr },
    { Stitch::BL3Qtr, Stitch::BR3Qtr },
    { Stitch::BR3Qtr, Stitch::TR3Qtr },
    { Stitch::BTHalf, Stitch::TBHalf },
    { Stitch::TBHalf, Stitch::BTHalf },
    { Stitch::Full, Stitch::Full }
  };

   struct TRANSLATION rotate180Map[] =
  {
    { Stitch::TLQtr, Stitch::BRQtr },
    { Stitch::TRQtr, Stitch::BLQtr },
    { Stitch::BLQtr, Stitch::TRQtr },
    { Stitch::BRQtr, Stitch::TLQtr },
    { Stitch::TL3Qtr, Stitch::BR3Qtr },
    { Stitch::TR3Qtr, Stitch::BL3Qtr },
    { Stitch::BL3Qtr, Stitch::TR3Qtr },
    { Stitch::BR3Qtr, Stitch::TL3Qtr },
    { Stitch::BTHalf, Stitch::TBHalf },
    { Stitch::TBHalf, Stitch::BTHalf },
    { Stitch::Full, Stitch::Full }
  };

   struct TRANSLATION rotate270Map[] =
  {
    { Stitch::TLQtr, Stitch::TRQtr },
    { Stitch::TRQtr, Stitch::BRQtr },
    { Stitch::BLQtr, Stitch::TLQtr },
    { Stitch::BRQtr, Stitch::BLQtr },
    { Stitch::TL3Qtr, Stitch::TR3Qtr },
    { Stitch::TR3Qtr, Stitch::BR3Qtr },
    { Stitch::BL3Qtr, Stitch::TL3Qtr },
    { Stitch::BR3Qtr, Stitch::BL3Qtr },
    { Stitch::BTHalf, Stitch::TBHalf },
    { Stitch::TBHalf, Stitch::BTHalf },
    { Stitch::Full, Stitch::Full }
  };

  stream >> scheme;
  palette.setScheme(scheme);
  stream >> width >> height;
  if (angle != 180)
  {
    Q_INT32 tmp = height;
    height = width;
    width = tmp;
  }
  canvas.resize(width, height);

  if (angle == 90)
  {
    for (int dx = 0 ; dx < width ; dx++)
    {
      for (int dy = height ; dy ; )
      {
        dy--;
        stream >> count;
        while (count--)
        {
          stream >> type >> color;
          for (int i = 0 ; i < 11 ; i++)
          {
            if (rotate90Map[i].original == type)
            {
              type = rotate90Map[i].replace;
              break;
            }
          }
          canvas.addStitch(QPoint(dx,dy), (Stitch::Type)type, palette.addColor(color));
        }
      }
    }
  }
  else if (angle == 180)
  {
    for (int dy = height ; dy ; )
    {
      dy--;
      for (int dx = width ; dx ; )
      {
        dx--;
        stream >> count;
        while (count--)
        {
          stream >> type >> color;
          for (int i = 0 ; i < 11 ; i++)
          {
            if (rotate180Map[i].original == type)
            {
              type = rotate180Map[i].replace;
              break;
            }
          }
          canvas.addStitch(QPoint(dx,dy), (Stitch::Type)type, palette.addColor(color));
        }
      }
    }
  }
  else
  {
    // angle is 270
    for (int dx = width ; dx ; )
    {
      dx--;
      for (int dy = 0 ; dy < height ; dy++)
      {
        stream >> count;
        while (count--)
        {
          stream >> type >> color;
          for (int i = 0 ; i < 11 ; i++)
          {
            if (rotate270Map[i].original == type)
            {
              type = rotate270Map[i].replace;
              break;
            }
          }
          canvas.addStitch(QPoint(dx,dy), (Stitch::Type)type, palette.addColor(color));
        }
      }
    }
  }
  stream >> backstitches;
  int bl = height*2;
  int tr = width*2;
  for (int backstitch = 0 ; backstitch < backstitches ; backstitch++)
  {
    stream >> start >> end >> color;
    if (angle == 90)
      canvas.addBackstitch(QPoint(start.y(), bl-start.x()), QPoint(end.y(), bl-end.x()), palette.addColor(color));
    else if (angle == 180)
      canvas.addBackstitch(QPoint(tr-start.x(), bl-start.y()), QPoint(tr-end.x(), bl-end.y()), palette.addColor(color));
    else
      canvas.addBackstitch(QPoint(tr-start.y(), start.x()), QPoint(tr-end.y(), end.x()), palette.addColor(color));
  }
  stream >> knots;
  for (int knot = 0 ; knot < knots ; knot++)
  {
    stream >> start >> color;
    if (angle == 90)
      canvas.addFrenchKnot(QPoint(start.y(), bl-start.x()), palette.addColor(color));
    else if (angle == 180)
      canvas.addFrenchKnot(QPoint(tr-start.x(), bl-start.y()), palette.addColor(color));
    else
      canvas.addFrenchKnot(QPoint(tr-start.y(), start.x()), palette.addColor(color));
  }

  QByteArray rotated;
  QDataStream rotateStream(rotated, IO_WriteOnly);
  rotateStream << scheme;
  rotateStream << (Q_INT32)width << (Q_INT32)height;
  for (int y = 0 ; y < height ; y++)
  {
    for (int x = 0 ; x < width ; x++)
    {
      Stitch::Queue *queue = canvas.stitchAt(QPoint(x,y));
      if (queue)
      {
        rotateStream << (Q_INT8)queue->count();
        for (int stitches = queue->count() ; stitches ; stitches--)
        {
          Stitch *s = queue->dequeue();
          Floss* f = palette.flossAt(s->floss);
          rotateStream << (Q_INT8)s->type << f->color;
          delete s;
        }
      }
      else
        rotateStream << (Q_INT8)0;
    }
  }
  rotateStream << (Q_INT32)backstitches;
  QPtrListIterator<BackStitch>* bs = canvas.backstitches();
  for (BackStitch* b = 0 ; (b = bs->current()) ; ++(*bs))
  {
    rotateStream << b->start << b->end << (palette.flossAt(b->floss))->color;
  }
  rotateStream << (Q_INT32)knots;
  QPtrListIterator<Knot>* kn = canvas.knots();
  for (Knot* k = 0 ; (k = kn->current()) ; ++(*kn))
  {
    rotateStream << k->pos << (palette.flossAt(k->floss))->color;
  }

  m_pasteData = new KXStitchMimeSource(rotated);
  m_pasteDataIsLibrary = true;
  setMode(EditView::Paste);
}

void EditView::slotEditRotate90()
{
  editRotate(90);
}

void EditView::slotEditRotate180()
{
  editRotate(180);
}

void EditView::slotEditRotate270()
{
  editRotate(270);
}

void EditView::editMirror(bool vertical)
{
  QByteArray selection = m_doc->copySelection(selectionArea(),m_stitchMask?m_currentStitchType:Stitch::Delete,m_colorMask,m_excludeBackstitches,m_excludeKnots,!m_editMirrorRotateCopies->isChecked());
  clearSelection();

  QDataStream stream(selection, IO_ReadOnly);
  QString scheme;
  Q_INT32 width;
  Q_INT32 height;
  Q_INT8  count;
  Q_INT8  type;
  Q_INT32 backstitches;
  Q_INT32 knots;
  QPoint  start;
  QPoint  end;
  QColor  color;
  FlossPalette palette;
  PatternCanvas canvas;

  struct TRANSLATION
  {
    Stitch::Type original;
    Stitch::Type replace;
  };

   struct TRANSLATION horizontalMap[] =
  {
    { Stitch::TLQtr, Stitch::TRQtr },
    { Stitch::TRQtr, Stitch::TLQtr },
    { Stitch::BLQtr, Stitch::BRQtr },
    { Stitch::BRQtr, Stitch::BLQtr },
    { Stitch::TL3Qtr, Stitch::TR3Qtr },
    { Stitch::TR3Qtr, Stitch::TL3Qtr },
    { Stitch::BL3Qtr, Stitch::BR3Qtr },
    { Stitch::BR3Qtr, Stitch::BL3Qtr },
    { Stitch::BTHalf, Stitch::TBHalf },
    { Stitch::TBHalf, Stitch::BTHalf },
    { Stitch::Full, Stitch::Full }
  };

   struct TRANSLATION verticalMap[] =
  {
    { Stitch::TLQtr, Stitch::BLQtr },
    { Stitch::TRQtr, Stitch::BRQtr },
    { Stitch::BLQtr, Stitch::TLQtr },
    { Stitch::BRQtr, Stitch::TRQtr },
    { Stitch::TL3Qtr, Stitch::BL3Qtr },
    { Stitch::TR3Qtr, Stitch::BR3Qtr },
    { Stitch::BL3Qtr, Stitch::TL3Qtr },
    { Stitch::BR3Qtr, Stitch::TR3Qtr },
    { Stitch::BTHalf, Stitch::TBHalf },
    { Stitch::TBHalf, Stitch::BTHalf },
    { Stitch::Full, Stitch::Full }
  };

  stream >> scheme;
  palette.setScheme(scheme);
  stream >> width >> height;
  canvas.resize(width, height);

  for (int dy = 0 ; dy < height ; dy++)
  {
    for (int dx = 0; dx < width ; dx++)
    {
      stream >> count;
      while (count--)
      {
        stream >> type >> color;
        if (vertical)
        {
          for (int i = 0 ; i < 11 ; i++)
          {
            if (verticalMap[i].original == type)
            {
              type = verticalMap[i].replace;
              break;
            }
          }
          canvas.addStitch(QPoint(dx,height-1-dy), (Stitch::Type)type, palette.addColor(color));
        }
        else // its horizontal
        {
          for (int i = 0 ; i < 11 ; i++)
          {
            if (horizontalMap[i].original == type)
            {
              type = horizontalMap[i].replace;
              break;
            }
          }
          canvas.addStitch(QPoint(width-1-dx,dy), (Stitch::Type)type, palette.addColor(color));
        }
      }
    }
  }
  stream >> backstitches;
  int bl = height*2;
  int tr = width*2;
  for (int backstitch = 0 ; backstitch < backstitches ; backstitch++)
  {
    stream >> start >> end >> color;
    if (vertical)
      canvas.addBackstitch(QPoint(start.x(), bl-start.y()), QPoint(end.x(), bl-end.y()), palette.addColor(color));
    else
      canvas.addBackstitch(QPoint(tr-start.x(), start.y()), QPoint(tr-end.x(), end.y()), palette.addColor(color));
  }
  stream >> knots;
  for (int knot = 0 ; knot < knots ; knot++)
  {
    stream >> start >> color;
    if (vertical)
      canvas.addFrenchKnot(QPoint(start.x(), bl-start.y()), palette.addColor(color));
    else
      canvas.addFrenchKnot(QPoint(tr-start.x(), start.y()), palette.addColor(color));
  }

  QByteArray mirrored;
  QDataStream mirrorStream(mirrored, IO_WriteOnly);
  mirrorStream << scheme;
  mirrorStream << (Q_INT32)width << (Q_INT32)height;
  for (int y = 0 ; y < height ; y++)
  {
    for (int x = 0 ; x < width ; x++)
    {
      Stitch::Queue *queue = canvas.stitchAt(QPoint(x,y));
      if (queue)
      {
        mirrorStream << (Q_INT8)queue->count();
        for (int stitches = queue->count() ; stitches ; stitches--)
        {
          Stitch *s = queue->dequeue();
          Floss* f = palette.flossAt(s->floss);
          mirrorStream << (Q_INT8)s->type << f->color;
          delete s;
        }
      }
      else
        mirrorStream << (Q_INT8)0;
    }
  }
  QPtrListIterator<BackStitch>* bs = canvas.backstitches();
  mirrorStream << (Q_INT32)(bs->count());
  for (BackStitch* b = 0 ; (b = bs->current()) ; ++(*bs))
    mirrorStream << b->start << b->end << (palette.flossAt(b->floss))->color;
  QPtrListIterator<Knot>* kn = canvas.knots();
  mirrorStream << (Q_INT32)(kn->count());
  for (Knot* k = 0 ; (k = kn->current()) ; ++(*kn))
    mirrorStream << k->pos << (palette.flossAt(k->floss))->color;
  m_pasteData = new KXStitchMimeSource(mirrored);
  m_pasteDataIsLibrary = true;
  setMode(EditView::Paste);
}

void EditView::slotEditMirrorVertical()
{
  editMirror(true);
}

void EditView::slotEditMirrorHorizontal()
{
  editMirror(false);
}

void EditView::slotTextTool()
{
  if (m_doc->palette()->currentColor() == -1)
    return;
  TextToolDialog *dialog = new TextToolDialog(this);
  if (dialog->exec() == QDialog::Accepted)
  {
    QByteArray data = dialog->pattern();
    QByteArray a;
    QDataStream s(a,IO_WriteOnly);
    FlossPalette *palette = m_doc->palette();
    Floss* f = palette->flossAt(palette->currentColor());
    s << palette->getScheme();
    int w = dialog->boundingWidth();
    s << (Q_INT32)w;
    int h = dialog->boundingHeight();
    s << (Q_INT32)h;
    for (int y = 0 ; y < h ; y++)
    {
      for (int x = 0 ; x < w ; x++)
      {
        if (data[y*w+x])
          s << (Q_INT8)1 << (Q_INT8)Stitch::Full << f->color;
        else
          s << (Q_INT8)0;
      }
    }
    s << (Q_INT32)0;
    s << (Q_INT32)0;
    m_pasteData = new KXStitchMimeSource(a);
    m_pasteDataIsLibrary = true;
    setMode(EditView::Paste);
  }
  delete dialog;
}

void EditView::slotLibraryCreateLibraryPattern()
{
  QByteArray d = m_doc->copySelection(selectionArea(),m_stitchMask?m_currentStitchType:Stitch::Delete,m_colorMask,m_excludeBackstitches,m_excludeKnots,false);
  PatternLibraryDialog* dialog = new PatternLibraryDialog(this, PatternLibraryDialog::Create, new KXStitchMimeSource(d));
  dialog->exec();
  clearSelection();
}

void EditView::slotLibraryPattern()
{
  PatternLibraryDialog* dialog = new PatternLibraryDialog(this, PatternLibraryDialog::Insert);
  if (dialog->exec() == QDialog::Accepted)
  {
    m_pasteData = dialog->pattern();
    m_pasteDataIsLibrary = true;
    setMode(EditView::Paste);
  }
  delete dialog;
}

void EditView::slotLibraryExport()
{
  kdDebug() << "EditView::slotLibraryExport()" << endl;
  PatternLibraryDialog* dialog = new PatternLibraryDialog(this,PatternLibraryDialog::Export);
  dialog->exec();
  delete dialog;
}

void EditView::slotLibraryImport()
{
  kdDebug() << "EditView::slotLibraryImport()" << endl;
  // initially create the dialog for import, after import the mode is changed to allow selecting and insert
  PatternLibraryDialog* dialog = new PatternLibraryDialog(this,PatternLibraryDialog::Import);
  if (dialog->exec() == QDialog::Accepted)
  {
    m_pasteData = dialog->pattern();
    m_pasteDataIsLibrary = true;
    setMode(EditView::Paste);
  }
  delete dialog;
}

void EditView::slotClipboardDataChanged()
{
  QMimeSource* source = kapp->clipboard()->data();
  m_editPaste->setEnabled(source && source->provides("application/kxstitch"));
}

void EditView::resizeGrid()
{
  int width=m_doc->canvas()->patternWidth()*m_cellSize;
  int height=m_doc->canvas()->patternHeight()*m_cellSize;
  resizeContents(width, height);
  if (!m_background.isNull())
  {
    QSize finalSize = m_requestedSize.size() * m_cellSize;
    m_backgroundScaled.convertFromImage(m_background.smoothScale(finalSize));
  }
  update();
  updateContents();
}

void EditView::configure(bool init)
{
  KActionCollection* actionCollection = ((KXStitchApp*)topLevelWidget())->actionCollection();
  QString stitchesAs;
  KAction* action;

  if (init)
  {
    m_stitches = KXSConfig().Editor_ShowStitches;
    m_showStitches->setChecked(m_stitches);
    stitchesAs = KXSConfig().Editor_StitchesAs;
    action = actionCollection->action(stitchesAs);
    if (!action)
      action = actionCollection->action("regular");
    action->activate();
    m_backstitches = KXSConfig().Editor_ShowBackstitches;
    m_showBackstitches->setChecked(m_backstitches);
    stitchesAs = KXSConfig().Editor_BackstitchesAs;
    m_bsvmode = BWLines;
    m_grid = KXSConfig().Editor_ShowGrid;
    m_showGrid->setChecked(m_grid);
    m_cellSize = KXSConfig().Editor_CellSize;
    m_scales = KXSConfig().Editor_ShowScales;
    QString scaleFormat = KXSConfig().Editor_ScaleFormat;
    action = actionCollection->action(scaleFormat);
    if (!action)
      action = actionCollection->action("stitches");
    action->activate();
    m_mode = Paint;
  }

  m_cellGrouping = KXSConfig().Editor_CellGrouping;
  m_gridMajor = KXSConfig().Editor_GridMajorColor;
  m_gridMinor = KXSConfig().Editor_GridMinorColor;

  m_showScales->setChecked(m_scales);
  m_scaleSize = KXSConfig().Editor_ScaleSize;
  if (m_scales)
    setMargins(m_scaleSize,m_scaleSize,0,0);
  else
  {
    setMargins(0,0,0,0);
    emit scaleFormat("");
  }

  resizeGrid();
}

QRect EditView::selectionArea()
{
  int left = m_selectedArea.left();
  left = std::max(left,0);
  int top = m_selectedArea.top();
  top = std::max(top,0);
  int right = m_selectedArea.right();
  right = std::min(right,contentsWidth());
  int bottom = m_selectedArea.bottom();
  bottom = std::min(bottom,contentsHeight());
  int width = (right-left+1)/m_cellSize;
  int height = (bottom-top+1)/m_cellSize;
  left /= m_cellSize;
  top /= m_cellSize;
  return QRect(left,top,width,height);
}

int EditView::print(KPrinter* printer, QPainter* painter, int width, int height, int pages)
{
/** Options to be checked from the printer dialog page
    kde-kxstitch-instructions       TRUE if instructions to be printed
    kde-kxstitch-fitsingle          TRUE if pattern to be printed on a single page
    kde-kxstitch-selectedarea       TRUE if printing restricted to a selected area ; Not currently supported
    kde-kxstitch-gridlines          TRUE if gridlines to be printed
    kde-kxstitch-squaresinch        number of stitches to be printed per inch
    kde-kxstitch-spillover          minimum number of stitches that can spill over to the next page
    kde-kxstitch-stitches           how stitches are to be printed
    kde-kxstitch-backstitches       how backstitches are to be printed
  */
  QPaintDeviceMetrics pdm(printer);
  QFontMetrics fm = painter->fontMetrics();
  int dy = fm.lineSpacing();

  if (printer->option("kde-kxstitch-instructions") == "true")
  {
    QFont f = painter->font();
    QFont bf = f;
    bf.setBold(true);
    painter->setFont(bf);
    int y = dy;
    painter->drawText(0,y,i18n("Instructions"));
    y = y+2*dy;
    painter->setFont(f);
    QString instructions = m_doc->instructions();
    if (instructions.isEmpty())
      painter->drawText(0,y,i18n("No additional instructions have been specified."));
    else
      painter->drawText(0,y-dy,width,height-y-dy,Qt::WordBreak,instructions);
    /** TODO
        There is the potential for the instructions to span more than one page
        This should be checked for and page breaks added accordingly.
      */
    painter->drawText(0,height-dy,width,dy,Qt::AlignHCenter,QString(i18n("Page %1")).arg(++pages));
    printer->newPage();
  }

  int patternWidth = m_doc->canvas()->patternWidth();
  int patternHeight = m_doc->canvas()->patternHeight();
  QSize keyplan(0,0);
  if (printer->option("kde-kxstitch-fitsingle") == "false")
  {
    int pw = patternWidth;
    int ph = patternHeight;
    keyplan = QSize(50*pw/ph,50);
  }

  QString title = m_doc->title();

  int reservedHeader = 2*fm.lineSpacing();
  if (title.isEmpty())
    reservedHeader = 0;
  int reservedFooter = 2*fm.lineSpacing();
  QRect header(0,0,width,reservedHeader+keyplan.height());
  QRect footer(0,height-reservedFooter,width,reservedFooter);
  QRect printable(0,header.height()+10,width,height-header.height()-footer.height());

  int pageCellsWide;
  int pageCellsTall;
  int printPagesWide;
  int printPagesTall;
  int wrapCellsWide;
  int wrapCellsTall;
  int printCellSize;

  int cellsWrapMin = printer->option("kde-kxstitch-spillover").toInt();
  bool pageLayoutOK;

  if (printer->option("kde-kxstitch-fitsingle") == "true")
  {
    printCellSize = printable.width() / patternWidth;
    while (patternHeight*printCellSize > printable.height())
      printCellSize--; // reduce the size of the cell until it fits in one page
    printPagesWide = printPagesTall = 1;
    pageCellsWide = patternWidth;
    pageCellsTall = patternHeight;
  }
  else
  {
    printCellSize = pdm.logicalDpiX() / (printer->option("kde-kxstitch-squaresinch")).toInt(); // dots per inch divided by cells per inch
    do
    {
      pageLayoutOK = true;
      pageCellsWide = printable.width() / printCellSize;
      pageCellsTall = printable.height() / printCellSize;
      printPagesWide = patternWidth / pageCellsWide;
      printPagesTall = patternHeight / pageCellsTall;
      wrapCellsWide = patternWidth % pageCellsWide;
      wrapCellsTall = patternHeight % pageCellsTall;
      // do some checks here to see if there are odd lines of the pattern below a define minimum going
      // to be printed on seperate sheets, then modify printCellWidth to bring them onto the previous page
      if (wrapCellsWide && (wrapCellsWide < cellsWrapMin) && (printPagesWide > 0)) pageLayoutOK = false;
      if (wrapCellsTall && (wrapCellsTall < cellsWrapMin) && (printPagesTall > 0)) pageLayoutOK = false;
      if (!pageLayoutOK) printCellSize--;
    } while (!pageLayoutOK);
    if (wrapCellsWide) printPagesWide++;
    if (wrapCellsTall) printPagesTall++;
  }

  int totalPages = printPagesWide * printPagesTall;
  if (totalPages == 1)
    keyplan = QSize();

  for (int y = 0 ; y < patternHeight ; y+=pageCellsTall)
  {
    for (int x = 0 ; x < patternWidth ; x+=pageCellsWide)
    {
      painter->save();
      // print header
      if (!title.isEmpty())
        painter->drawText(header,Qt::AlignVCenter,QString(i18n("Title:%1")).arg(title));
      // print keyplan
      if (!keyplan.isEmpty())
      {
        QRect r(QPoint(width-keyplan.width()-2,0),keyplan);
        QRect b = r;
        b.moveBy(2,2);
        painter->fillRect(b,QBrush(Qt::black));
        painter->fillRect(r,QBrush(Qt::white));
        painter->drawRect(r);
        QRect currentPage(r.left()+(50*x/patternHeight),r.top()+(50*y/patternHeight),std::min((50*pageCellsWide/patternHeight),(50*(patternWidth-x)/patternHeight)),std::min((50*pageCellsTall/patternHeight),(50*(patternHeight-y)/patternHeight)));
        painter->fillRect(currentPage,QBrush(QColor(lightGray)));
        painter->drawRect(currentPage);
      }
      printContents(printer, painter, x, y, printable, std::min(pageCellsWide,(patternWidth-x)), std::min(pageCellsTall,(patternHeight-y)), printCellSize);
      // print footer
      painter->drawText(0,height-dy,width,dy,Qt::AlignHCenter,QString(i18n("Page %1")).arg(++pages));
      if (--totalPages)
        printer->newPage();
      painter->restore();
    }
  }
  return pages;
}

void EditView::slotSetCenter(QPoint pos)
{
  pos = cellToRect(pos).center();
  center(pos.x(), pos.y());
  emit changedVisibleCells(visibleCells());
}

void EditView::keyPressEvent(QKeyEvent* event)
{
  if (m_mode == FillPolyline && m_points)
  {
    switch (event->key())
    {
      case Qt::Key_Backspace: // delete the last selected vertex
        m_points--;
        event->accept();
        break;

      case Qt::Key_Escape: // cancel the operation
        m_points = 0;
        event->accept();
        break;

      default:
        event->ignore();
        break;
    }
    updateContents();
  }
  else
    event->ignore();
}

void EditView::paintEvent(QPaintEvent *event)
{
  PatternCanvas* pCanvas = m_doc->canvas();
  if (m_scales && pCanvas->patternWidth() != 0 && pCanvas->patternHeight() !=0)
  {
    int x = pCanvas->patternWidth()*m_cellSize;
    int y = pCanvas->patternHeight()*m_cellSize;

    QPixmap buffer;
    QPainter bufferPainter;
    int fw = frameWidth();
    int lm = leftMargin();
    int tm = topMargin();
    int vw = visibleWidth();
    int vh = visibleHeight();
    int cx = contentsX();
    int cy = contentsY();
    int majorTick = 0;
    int minorTick = 0;
    int subTick = 0;
    int textValueIncrement = 0;
    double clothCount = m_doc->m_propertiesDialog->clothCount();
    if (m_doc->m_propertiesDialog->clothCountUnits() == "cm")
      clothCount*=2.54;
    m_cellSize *= 100;
    x *= 100;
    y *= 100;
    switch (m_scaleFormat)
    {
      case Stitches:
        subTick = minorTick = m_cellSize;
        majorTick = minorTick*m_cellGrouping;
        textValueIncrement = m_cellGrouping;
        break;
      case CM:
        subTick = (int)(m_cellSize*clothCount*100/2540);
        minorTick = subTick*5;
        majorTick = subTick*10;
        textValueIncrement = 1;
        break;
      case Inches:
        majorTick = (int)(m_cellSize*clothCount);
        minorTick = majorTick/4;
        subTick = majorTick/16;
        textValueIncrement = 1;
        break;
    }
    QRect hscale(fw,fw,width()-(2*fw),m_scaleSize);
    QRect vscale(fw,fw,m_scaleSize,height()-(2*fw));
    hscale &= event->rect();
    if (hscale.isValid())
    {
      // update the horizontal scale
      buffer.resize(hscale.width(),hscale.height());
      buffer.fill();
      bufferPainter.begin(&buffer);
      bufferPainter.translate(-hscale.x()-cx,-hscale.y());
      bufferPainter.drawLine(lm+cx,tm,lm+contentsWidth(),tm);
      for (int dx = 0 ; dx < x+1 ; dx++)
      {
        int tickLength = 0;
        if ((dx >= cx*100) && (dx <= (cx+vw)*100))
        {
          if ((dx % subTick) == 0) tickLength = 2;
          if ((dx % minorTick) == 0) tickLength = 4;
          if ((dx % majorTick) == 0) tickLength = 6;
          if (tickLength)
          {
            bufferPainter.drawLine(fw+lm+(dx/100),tm-tickLength,fw+lm+(dx/100),tm);
            if ((dx % majorTick) == 0)
            {
              bufferPainter.drawText(fw+lm+(dx/100)-10,fw,20,tm-10,AlignCenter,QString("%1").arg(dx/majorTick*textValueIncrement));
            }
          }
        }
      }
      int xc = (x/200)+lm+fw;
      if (xc >= 0)
      {
        bufferPainter.moveTo(xc,tm);
        bufferPainter.lineTo(xc-3,tm-6);
        bufferPainter.lineTo(xc+3,tm-6);
        bufferPainter.lineTo(xc,tm);
      }
      bufferPainter.end();
      bitBlt(this,hscale.x(),hscale.y(),&buffer,0,0,buffer.width(),buffer.height());
    }
    vscale &= event->rect();
    if (vscale.isValid())
    {
      // update the vertical scale
      buffer.resize(vscale.width(),vscale.height());
      buffer.fill();
      bufferPainter.begin(&buffer);
      bufferPainter.translate(-hscale.x(),-hscale.y()-cy);
      bufferPainter.drawLine(lm,tm+cy,lm,tm+contentsHeight());
      for (int dy = 0 ; dy < y+1 ; dy++)
      {
        int tickLength = 0;
        if ((dy >= cy*100) && (dy <= (cy+vh)*100))
        {
          if ((dy % subTick) == 0) tickLength = 2;
          if ((dy % minorTick) == 0) tickLength = 4;
          if ((dy % majorTick) == 0) tickLength = 6;
          if (tickLength)
          {
            bufferPainter.drawLine(lm-tickLength,fw+tm+(dy/100),lm,fw+tm+(dy/100));
            if ((dy % majorTick) == 0)
            {
              bufferPainter.drawText(fw,fw+tm+(dy/100)-10,lm-12,20,AlignRight|AlignVCenter,QString("%1").arg(dy/majorTick*textValueIncrement));
            }
          }
        }
      }
      int yc = (y/200)+tm+fw;
      if (yc >= 0)
      {
        bufferPainter.moveTo(lm,yc);
        bufferPainter.lineTo(lm-6, yc-3);
        bufferPainter.lineTo(lm-6, yc+3);
        bufferPainter.lineTo(lm,yc);
      }
      bufferPainter.end();
      bitBlt(this,hscale.x(),hscale.y(),&buffer,0,0,buffer.width(),buffer.height(),AndROP);
    }
    m_cellSize /= 100;
    x /= 100;
    y /= 100;
  }
  QScrollView::paintEvent(event);
}

void EditView::drawContents(QPainter *p, int clipx, int clipy, int clipw, int cliph)
{
  m_updatingDisplay = true;
  QRect clip(clipx,clipy,clipw,cliph);

  PatternCanvas *pCanvas = m_doc->canvas();

  if (pCanvas->patternWidth() == 0 || pCanvas->patternHeight() == 0) return;
  QPoint s = contentsToCell(QPoint(clipx,clipy));
  QPoint e = contentsToCell(QPoint(clipx+clipw, clipy+cliph));

  int x = pCanvas->patternWidth()*m_cellSize;
  int y = pCanvas->patternHeight()*m_cellSize;

  QPixmap* buffer = new QPixmap(clipw,cliph);
  buffer->fill(m_doc->m_propertiesDialog->fabricColor());
  QPainter* doubleBuffer = new QPainter(buffer);
  doubleBuffer->translate(-clipx,-clipy);
  if (m_backgroundImage)
  {
    QPoint backgroundPos = m_requestedSize.topLeft() * m_cellSize;
    doubleBuffer->drawPixmap(backgroundPos,m_backgroundScaled);
  }
  if (m_grid)
  {
    QPen op = doubleBuffer->pen();
    QPen tp = op;
    op.setColor(m_gridMajor);
    tp.setColor(m_gridMinor);
    for (int cx = 0 ; cx < x+1 ; cx+=m_cellSize)
    {
      if (cx >= clipx && cx <= clipx+clipw)
      {
        if (cx % (m_cellGrouping*m_cellSize))
        {
          doubleBuffer->setPen(tp);
          doubleBuffer->drawLine(cx,0,cx,y);
          doubleBuffer->setPen(op);
        }
        else
          doubleBuffer->drawLine(cx,0,cx,y);
      }
    }

    for (int cy = 0 ; cy < y+1 ; cy+=m_cellSize)
    {
      if (cy >= clipy && cy <= clipy+cliph)
      {
        if (cy % (m_cellGrouping*m_cellSize))
        {
          doubleBuffer->setPen(tp);
          doubleBuffer->drawLine(0,cy,x,cy);
          doubleBuffer->setPen(op);
        }
        else
          doubleBuffer->drawLine(0,cy,x,cy);
      }
    }
  }

  FlossPalette *pPalette = m_doc->palette();

  if (m_stitches)
  {
    if (s.x()>0)
      s.rx()--;
    if (s.y()>0)
      s.ry()--;

    if (e.x()>pCanvas->patternWidth())
      e.rx()=pCanvas->patternWidth();
    else
      e.rx()++;

    if (e.y()>pCanvas->patternHeight())
      e.ry()=pCanvas->patternHeight();
    else
      e.ry()++;

    QFont fnt = doubleBuffer->font();
    QFont sym = doubleBuffer->font();
    QFontMetrics fntMetrics(sym);

    QPaintDeviceMetrics pdm(this);
    int dpix = pdm.logicalDpiX();
    sym.setFamily(KXSConfig().Editor_SymbolFont);
    sym.setPointSize(72*m_cellSize/dpix);
    doubleBuffer->setFont(sym);
    initialiseRenderCellsFonts(doubleBuffer,m_cellSize);
    for (int yCell = s.y() ; yCell < e.y() ; yCell++)
    {
      for (int xCell = s.x() ; xCell < e.x() ; xCell++)
      {
        QPoint cell(xCell, yCell);
        QRect cellRect(cellToRect(cell));
        if (clip.intersects(cellRect))
        {
          doubleBuffer->resetXForm();
          doubleBuffer->translate(cellRect.left()-clipx,cellRect.top()-clipy);
          renderCell(doubleBuffer,pCanvas->stitchAt(cell),m_vmode);
        }
      }
    }
    doubleBuffer->resetXForm();
    doubleBuffer->translate(-clipx,-clipy);
    sym.setPointSize(sym.pointSize()/2);
    doubleBuffer->setFont(sym);
    QPtrListIterator<Knot> *itK = pCanvas->knots();
    for (Knot *k = 0 ; (k = itK->current()) ; ++(*itK))
    {
      QChar s(pPalette->symbolAt(k->floss));
      QColor c(pPalette->flossAt(k->floss)->color);
      QRect r(snapToContents(k->pos)-QPoint(m_cellSize/3,m_cellSize/3),QSize(m_cellSize*2/3,m_cellSize*2/3));
      switch(m_vmode)
      {
        case Regular:
        case ColorBlocks:
        case ColorBlocksSymbols:
          doubleBuffer->setBrush(QBrush(c));
          doubleBuffer->setPen(QPen(c));
          doubleBuffer->drawEllipse(r);
          if (m_vmode==ColorBlocksSymbols)
          {
            if(qGray(c.rgb()) < 128)
              doubleBuffer->setPen(white);
            else
              doubleBuffer->setPen(black);
            doubleBuffer->drawText(r, AlignCenter, s, 1);
          }
          break;
        case BWSymbols:
          doubleBuffer->drawText(r, AlignCenter, s, 1);
          doubleBuffer->drawEllipse(r);
          break;
        case ColorSymbols:
          doubleBuffer->setPen(c);
          doubleBuffer->drawText(r, AlignCenter, s, 1);
          doubleBuffer->drawEllipse(r);
          break;
        case ColorHighlight:
          if (k->floss == pPalette->currentColor())
          {
            doubleBuffer->setBrush(QBrush(c));
            doubleBuffer->setPen(c);
          }
          else
          {
            doubleBuffer->setPen(QColor(lightGray));
            doubleBuffer->setBrush(QBrush(QColor(lightGray)));
          }
          doubleBuffer->drawEllipse(r);
          break;
        default:
          // no other values are relevant
          break;
      }
    }
  }

  if (m_backstitches)
  { // TODO modify this code to display backstitches in users preferred style
    // draw backstitching
    QPtrListIterator<BackStitch> *it = pCanvas->backstitches();
    BackStitch *pBackstitch;
    for ( ; (pBackstitch = it->current()) ; ++(*it) )
    {
      doubleBuffer->setPen(QPen(pPalette->flossAt(pBackstitch->floss)->color,3));
      if (m_vmode == ColorHighlight && pBackstitch->floss != pPalette->currentColor())
      {
        doubleBuffer->setPen(QPen(QColor(lightGray)));
      }
      doubleBuffer->drawLine(snapToContents(pBackstitch->start), snapToContents(pBackstitch->end));
    }
    delete it;
  }
  if (m_points)
  {
    doubleBuffer->setBrush(Qt::NoBrush);
    doubleBuffer->setPen(Qt::red);    // perhaps make this user defined
    doubleBuffer->drawEllipse(cellToRect(m_pointArray[0]));
    doubleBuffer->setPen(Qt::green);  // perhaps make this user defined
    for (int i = 1 ; i < m_points ; i++)
    {
      doubleBuffer->drawEllipse(cellToRect(m_pointArray[i]));
      doubleBuffer->drawLine((cellToRect(m_pointArray[i-1]).center()),(cellToRect(m_pointArray[i]).center()));
    }
  }
  doubleBuffer->end();
  p->drawPixmap(clipx,clipy,*buffer);
  delete doubleBuffer;
  delete buffer;

  m_updatingDisplay = false;

  // update the preview location marker
  emit changedVisibleCells(visibleCells());
}

void EditView::printContents(KPrinter *printer, QPainter *painter, int x, int y, QRect printable, int pageCellsWide, int pageCellsTall, int printCellSize)
{
  PatternCanvas *pCanvas = m_doc->canvas();
  QPen op = painter->pen();

  QPaintDeviceMetrics pdm(printer);
  int dpix = pdm.logicalDpiX();
  QFont fnt = painter->font();
  QFont sym = painter->font();
  sym.setFamily(KXSConfig().Editor_SymbolFont);
  sym.setPointSize(72*printCellSize/dpix);
  painter->setFont(sym);
  initialiseRenderCellsFonts(painter,printCellSize);

  if (printer->option("kde-kxstitch-gridlines") == "true")
  {
    // print the grid lines
    QPen tp = op;
    tp.setWidth(1);
    // vertical lines
    for (int cx = x, dx = printable.left() ; cx < x+pageCellsWide+1 ; cx++, dx+=printCellSize)
    {
      if (cx % m_cellGrouping)
      {
        painter->setPen(op);
        painter->drawLine(dx,printable.top(),dx,printable.top()+pageCellsTall*printCellSize);
      }
      else
      {
        painter->setPen(tp);
        painter->drawLine(dx,printable.top(),dx,printable.top()+pageCellsTall*printCellSize);
      }
    }
    // horizontal lines
    for (int cy = y, dy = printable.top() ; cy < y+pageCellsTall+1 ; cy++, dy+=printCellSize)
    {
      if (cy % m_cellGrouping)
      {
        painter->setPen(op);
        painter->drawLine(printable.left(),dy,printable.left()+pageCellsWide*printCellSize,dy);
      }
      else
      {
        painter->setPen(tp);
        painter->drawLine(printable.left(),dy,printable.left()+pageCellsWide*printCellSize,dy);
      }
    }
    painter->setPen(op);
  }
  // check for how to print the stitches
  QString stitchesAs = printer->option("kde-kxstitch-stitches");
  enum ViewMode sViewMode = BWSymbols;
  if (stitchesAs == i18n("Color Symbols")) sViewMode = ColorSymbols;
  if (stitchesAs == i18n("Colored Blocks")) sViewMode = ColorBlocks;
  if (stitchesAs == i18n("Colored Blocks & Symbols")) sViewMode = ColorBlocksSymbols;
  if (stitchesAs != i18n("None"))
  {
    painter->save();
    for (int dy = 0 ; dy < pageCellsTall ; dy++)
    {
      for (int dx = 0 ; dx < pageCellsWide ; dx++)
      {
        painter->translate(printable.left()+(dx*printCellSize),printable.top()+(dy*printCellSize));
        renderCell(painter,pCanvas->stitchAt(QPoint(x+dx,y+dy)),sViewMode);// render the cell
        painter->resetXForm();
      }
    }
    painter->restore();
  }

  FlossPalette *pPalette = m_doc->palette();

  QPtrListIterator<Knot> *itK = pCanvas->knots();
  for (Knot *k = 0 ; (k = itK->current()) ; ++(*itK))
  {
    QChar s(pPalette->symbolAt(k->floss));
    QColor c(pPalette->flossAt(k->floss)->color);
    QRect r(k->pos*printCellSize/2-QPoint(printCellSize*x,printCellSize*y)+printable.topLeft()-QPoint(printCellSize/3,printCellSize/3),QSize(printCellSize*2/3,printCellSize*2/3));
    switch(sViewMode)
    {
      case Regular:
      case ColorBlocks:
      case ColorBlocksSymbols:
        painter->setBrush(QBrush(c));
        painter->setPen(QPen(c));
        painter->drawEllipse(r);
        if (sViewMode==ColorBlocksSymbols)
        {
          if(qGray(c.rgb()) < 128)
            painter->setPen(white);
          else
            painter->setPen(black);
          painter->drawText(r, AlignCenter, s, 1);
        }
        break;
      case BWSymbols:
        painter->drawText(r, AlignCenter, s, 1);
        painter->drawEllipse(r);
        break;
      case ColorSymbols:
        painter->setPen(c);
        painter->drawText(r, AlignCenter, s, 1);
        painter->drawEllipse(r);
        break;
      default:
        // no other values are relevant
        break;
    }
  }

  painter->setFont(fnt);

  /** print the backstitches */
  QString backstitchesAs = printer->option("kde-kxstitch-backstitches");
  if (backstitchesAs != i18n("None"))
  {
    enum ViewMode bsViewMode = BWLines;
    if (backstitchesAs == i18n("Color Lines")) bsViewMode = ColorLines;
    QPtrListIterator<BackStitch> *it = pCanvas->backstitches();
    BackStitch *pBackstitch;
    for ( ; (pBackstitch = it->current()) ; ++(*it) )
    {
      QRect clipping(printable.left(),printable.top(),pageCellsWide*printCellSize,pageCellsTall*printCellSize);
      painter->setClipRegion(QRegion(clipping));
      if (bsViewMode == BWLines)
        painter->setPen(QPen(QColor(black),2));
      else
        painter->setPen(QPen(pPalette->flossAt(pBackstitch->floss)->color,2));
      QPoint bsStart = pBackstitch->start;
      QPoint bsEnd = pBackstitch->end;
      bsStart.rx() = bsStart.x()*printCellSize/2;
      bsStart.ry() = bsStart.y()*printCellSize/2;
      bsEnd.rx() = bsEnd.x()*printCellSize/2;
      bsEnd.ry() = bsEnd.y()*printCellSize/2;
      bsStart -= QPoint(printCellSize*x,printCellSize*y);
      bsEnd -= QPoint(printCellSize*x,printCellSize*y);
      bsStart += printable.topLeft();
      bsEnd += printable.topLeft();
      painter->drawLine(bsStart, bsEnd);
      painter->setClipping(false);
    }
    delete it;
  }
}

void EditView::contextMenuEvent(QContextMenuEvent* e)
{
  KXStitchApp *app = (KXStitchApp *)topLevelWidget();
  QPopupMenu *context = (QPopupMenu *)app->guiFactory()->container("EditorPopup",app);
  context->popup(e->globalPos());
  e->accept();
}

void EditView::contentsMousePressEvent(QMouseEvent *e)
{
  if (!(e->button() & LeftButton))
  {
    return;
  }
  if (m_mode != FillPolyline || m_points == 0)
    m_doc->createSavePoint();
  QRect rect;
  QPoint p = e->pos();
  if (m_selectionTimer.isActive())
  {
    m_selectionTimer.stop();
    updateContents();
    emit selectionMade(false);
  }
  m_start = m_tracking = m_end = contentsToCell(p);
  switch (m_mode)
  {
    case Paint:
      if (m_currentStitchType == Stitch::FRKnot)
      {
        if (QRect(0,0,contentsWidth(),contentsHeight()).contains(p))
        {
          m_start = m_tracking = m_end = contentsToSnap(p);
          m_doc->addFrenchKnot(m_start);
          rect = QRect(snapToContents(m_start)-(QPoint(m_cellSize,m_cellSize)/2),QSize(m_cellSize,m_cellSize));
        }
      }
      else
      {
        m_doc->addStitch(m_start,m_currentStitchType);
        rect = cellToRect(m_start);
      }
      updateContents(rect);
      break;
    case Draw: // no additional code for this tool
      break;
    case Erase:
      if (e->state() & ControlButton)
      {
        // erase a backstitch
        m_start = m_tracking = m_end = contentsToSnap(p);
      }
      else
      {
        if (e->state() & ShiftButton)
        {
          // delete french knots
          m_start = m_tracking = m_end = contentsToSnap(p);
          m_doc->deleteFrenchKnot(m_start, m_colorMask);
          rect = QRect(snapToContents(m_start)-(QPoint(m_cellSize,m_cellSize)/2),QSize(m_cellSize,m_cellSize));
        }
        else
        {
          m_doc->deleteStitch(m_start,m_stitchMask?m_currentStitchType:Stitch::Delete,m_colorMask);
          rect = cellToRect(m_start);
        }
        updateContents(rect);
      }
      break;
    case Backstitch:
      if (QRect(0,0,contentsWidth(),contentsHeight()).contains(p))
      {
        m_start = contentsToSnap(p);
        m_end = m_start;
      }
      break;
    case Rectangle: // no additional code for this tool
      break;
    case FillRectangle: // no additional code for this tool
      break;
    case Ellipse: // no additional code for this tool
      break;
    case FillEllipse: // no additional code for this tool
      break;
    case FillPolyline:
      if (m_points == m_pointArray.size())
      {
        m_pointArray.resize(m_pointArray.size()+20);
      }
      m_pointArray.setPoint(m_points++, m_start.x(), m_start.y());
      break;
    case SelectRectangle:
      m_selectedArea = QRect();
      m_selectedAreaOffset = QPoint(0,0);
      break;
    case ColorPicker: // no additional code for this tool
      break;
    case Paste:
      QByteArray array = m_pasteData->encodedData("application/kxstitch");
      QDataStream stream(array, IO_ReadOnly);
      QString scheme;
      stream >> scheme;
      Q_INT32 width, height;
      stream >> width >> height;
      m_pasteImage = new QPixmap(width*m_cellSize, height*m_cellSize);
      m_pasteImage->fill();
      QPainter painter(m_pasteImage);
      int halfCellSize = m_cellSize/2;
      for (int dy = 0 ; dy < height*m_cellSize ; dy+=m_cellSize)
      {
        for (int dx = 0 ; dx < width*m_cellSize ; dx+=m_cellSize)
        {
          Q_INT8 count;
          stream >> count;
          while (count--)
          {
            Q_INT8 type;
            QColor color;
            stream >> type >> color;
            painter.setPen(color);
            if (type & Stitch::TLQtr)
              painter.drawLine(dx,dy,dx+halfCellSize,dy+halfCellSize);
            if (type & Stitch::TRQtr)
              painter.drawLine(dx+halfCellSize,dy+halfCellSize,dx+m_cellSize,dy);
            if (type & Stitch::BLQtr)
              painter.drawLine(dx,dy+m_cellSize,dx+halfCellSize,dy+halfCellSize);
            if (type & Stitch::BRQtr)
              painter.drawLine(dx+halfCellSize,dy+halfCellSize,dx+m_cellSize,dy+m_cellSize);
          }
        }
      }
      int backstitches;
      stream >> backstitches;
      while (backstitches--)
      {
        QPoint start, end;
        QColor color;
        stream >> start >> end >> color;
        painter.setPen(color);
        painter.drawLine(start*halfCellSize,end*halfCellSize);
      }
      int knots;
      stream >> knots;
      while (knots--)
      {
        QPoint start;
        QColor color;
        stream >> start >> color;
        painter.setBrush(color);
        painter.setPen(color);
        QPoint topLeft = QPoint(start*halfCellSize)-QPoint(halfCellSize/2,halfCellSize/2);
        QPoint bottomRight = QPoint(start*halfCellSize)+QPoint(halfCellSize/2,halfCellSize/2);
        painter.drawPie(QRect(topLeft,bottomRight),0,5760);
      }
      painter.end();
      bitBlt(viewport(),contentsToViewport(cellToRect(m_start).topLeft()),m_pasteImage,QRect(0,0,m_pasteImage->width(),m_pasteImage->height()),NotXorROP);
      break;
  }
  e->accept();
}

void EditView::contentsMouseMoveEvent(QMouseEvent *e)
{
  QRect rect;
  QPainter painter;
  QPoint p = e->pos();
  m_tracking = contentsToCell(p);
  if (e->state() == Qt::NoButton)
  {
    // this is a mouse move event only, no command is currently in operation
    emit locationOrSize(m_tracking);
    return;
  }
  switch (m_mode)
  {
    case Paint:
      if (m_currentStitchType == Stitch::FRKnot)
      {
        if (QRect(0,0,contentsWidth(),contentsHeight()).contains(p))
        {
          m_tracking = contentsToSnap(p);
          if (m_tracking != m_start)
          {
            m_start = m_tracking;
            m_doc->addFrenchKnot(m_start);
            rect = QRect(snapToContents(m_start)-(QPoint(m_cellSize,m_cellSize)/2),QSize(m_cellSize,m_cellSize));
          }
        }
      }
      else
      {
        if (m_tracking != m_start)
        {
          m_start = m_tracking;
          m_doc->addStitch(m_start, m_currentStitchType);
          rect = cellToRect(m_start);
        }
      }
      updateContents(rect);
      break;
    case Draw:
      if (QRect(0,0,contentsWidth(),contentsHeight()).contains(p))
      {
        drawBoundingLine(contentsToViewport(cellToRect(m_start).center()),contentsToViewport(cellToRect(m_end).center()));
        drawBoundingLine(contentsToViewport(cellToRect(m_start).center()),contentsToViewport(cellToRect(m_tracking).center()));
        ensureVisible(p.x(),p.y());
        m_end = m_tracking;
        rect = QRect(m_start,m_end).normalize();
        emit locationOrSize(QPoint(rect.width(),rect.height()));
      }
      break;
    case Erase:
      if (e->state() & ControlButton)
      {
        // erasing a backstitch
        // don't need to do anything else here
      }
      else
      {
        if (e->state() & ShiftButton)
        {
          // delete french knots
          m_tracking = contentsToSnap(p);
          if (m_tracking != m_start)
          {
            m_start = m_tracking;
            m_doc->deleteFrenchKnot(m_start, m_colorMask);
            rect = QRect(snapToContents(m_start)-(QPoint(m_cellSize,m_cellSize)/2),QSize(m_cellSize,m_cellSize));
          }
        }
        else
        {
          if (m_tracking != m_start)
          {
            m_start = m_tracking;
            m_doc->deleteStitch(m_start,m_stitchMask?m_currentStitchType:Stitch::Delete,m_colorMask);
            rect = cellToRect(m_start);
          }
        }
        updateContents(rect);
      }
      break;
    case Backstitch:
      if (QRect(0,0,contentsWidth(),contentsHeight()).contains(p))
      {
        m_tracking = contentsToSnap(p);
        drawBoundingLine(snapToViewport(m_start),snapToViewport(m_end));
        drawBoundingLine(snapToViewport(m_start),snapToViewport(m_tracking));
        m_end = m_tracking;
      }
      break;
    case Rectangle:
    case FillRectangle: // Rectangle, FillRectangle and SelectRectangle have the same code base for mouseMoveEvent
      if (m_tracking != m_end)
      {
        drawBoundingRect(m_start, m_end);
        ensureVisible(p.x(),p.y());
        drawBoundingRect(m_start, m_tracking);
        m_end = m_tracking;
        rect = QRect(m_start,m_end).normalize();
        emit locationOrSize(QPoint(rect.width(),rect.height()));
      }
      break;
    case SelectRectangle:
      drawSelectionRectangle();
      m_end = m_tracking;
      ensureVisible(p.x(),p.y());
      drawSelectionRectangle();
      rect = QRect(m_start,m_end).normalize();
      emit locationOrSize(QPoint(rect.width(),rect.height()));
      break;
    case Ellipse:
    case FillEllipse:
      if (m_tracking != m_end)
      {
        drawBoundingEllipse(m_start, m_end);
        ensureVisible(p.x(),p.y());
        drawBoundingEllipse(m_start, m_tracking);
        m_end = m_tracking;
        rect = QRect(m_start,m_end).normalize();
        emit locationOrSize(QPoint(rect.width(),rect.height()));
      }
      break;
    case FillPolyline:
      if (m_tracking != m_start)
      {
        if (m_points == m_pointArray.size())
        {
          m_pointArray.resize(m_pointArray.size()+20);
        }
        m_pointArray.setPoint(m_points++, m_tracking.x(), m_tracking.y());
        drawBoundingLine(contentsToViewport(cellToRect(m_start).center()),contentsToViewport(cellToRect(m_tracking).center()));
        m_start = m_tracking;
      }
      break;
    case Paste:
      if ( m_pasteImage )
      {
        repaintContents(QRect(cellToRect(m_end).topLeft(),QSize(m_pasteImage->size())));
        bitBlt(viewport(),contentsToViewport(cellToRect(m_tracking).topLeft()),m_pasteImage,QRect(0,0,m_pasteImage->width(),m_pasteImage->height()),NotXorROP);
      }
      m_end = m_tracking;
      break;
  }
}

void EditView::contentsMouseReleaseEvent(QMouseEvent *e)
{
  QPixmap dm(m_doc->canvas()->patternWidth(),m_doc->canvas()->patternHeight(),1);
  QRect rect;
  QPainter painter;
  QPoint p = e->pos();
  FlossPalette *pPalette;
  QPtrListIterator<BackStitch> *it;
  QRect boundingRect;

  dm.fill();
  switch (e->button())
  {
    case LeftButton:
      switch (m_mode)
      {
        case Paint:
          if (m_currentStitchType == Stitch::FRKnot)
          {
            if (QRect(0,0,contentsWidth(),contentsHeight()).contains(p))
            {
              m_end = contentsToSnap(p);
              if (m_end != m_start)
              {
                m_doc->addFrenchKnot(m_end);
                rect = QRect(snapToContents(m_end)-(QPoint(m_cellSize,m_cellSize)/2),QSize(m_cellSize,m_cellSize));
              }
            }
          }
          else
          {
            m_end = contentsToCell(p);
            if (m_end != m_start)
            {
              m_doc->addStitch(m_end, m_currentStitchType);
              rect = cellToRect(m_end);
            }
          }
          updateContents(rect);
          break;
        case Draw:
          drawBoundingLine(contentsToViewport(cellToRect(m_start).center()),contentsToViewport(cellToRect(m_end).center()));
          if (m_end != m_start)
          {
            painter.begin(&dm);
            painter.setPen(QPen(black,1));
            painter.drawLine(m_start,m_end);
            painter.drawPoint(m_start);
            painter.drawPoint(m_end);
            painter.end();
            m_pointArray = processPixmap(dm);
            m_doc->addStitches(m_pointArray,m_currentStitchType);
          }
          emit locationOrSize(QPoint(-1,-1));
          updateContents();
          break;
        case Erase:
          if (e->state() & ControlButton)
          {
            // erase a backstitch
            m_end = contentsToSnap(p);
            it = m_doc->canvas()->backstitches();
            BackStitch *bs;
            for ( ; (bs = it->current()) ; ++(*it))
            {
              if (bs->contains(m_start) && bs->contains(m_end))
              {
                m_start = snapToContents(bs->start);
                m_end = snapToContents(bs->end);
                // found a backstitch that starts and ends where indicated
                m_doc->deleteBackstitch(bs); // this should also update the iterator
                updateContents(QRect(QPoint((std::min(m_start.x(),m_end.x()))-2,(std::min(m_start.y(),m_end.y()))-2),QPoint((std::max(m_start.x(),m_end.x()))+2,(std::max(m_start.y(),m_end.y()))+2)));
                break;
              }
            }
          }
          else
          {
            if (e->state() & ShiftButton)
            {
              // delete french knots
              m_end = contentsToSnap(p);
              if (m_end != m_start)
              {
                m_doc->deleteFrenchKnot(m_end, m_colorMask);
                rect = QRect(snapToContents(m_end)-(QPoint(m_cellSize,m_cellSize)/2),QSize(m_cellSize,m_cellSize));
              }
            }
            else
            {
              // delete ordinary stitches
              m_end = contentsToCell(p);
              if (m_end != m_start)
              {
                m_doc->deleteStitch(m_end, m_stitchMask?m_currentStitchType:Stitch::Delete, m_colorMask);
                rect = cellToRect(m_end);
              }
            }
          }
          updateContents(rect);
          break;
        case Backstitch:
          drawBoundingLine(snapToViewport(m_start), snapToViewport(m_end));
          if (m_end != m_start)
          {
            painter.begin(this->viewport());
            painter.setRasterOp(CopyROP);
            pPalette = m_doc->palette();
            int color = pPalette->currentColor();
            if (color != -1)
            {
              painter.setPen(QPen(pPalette->flossAt(color)->color,3));
              painter.drawLine(snapToViewport(m_start), snapToViewport(m_end));
              m_doc->addBackstitch(m_start, m_end);
            }
            painter.end();
          }
          updateContents();
          break;
        case Rectangle:
          drawBoundingRect(m_start, m_end);
          painter.begin(&dm);
          painter.setPen(QPen(black,1));
          painter.drawRect(QRect(QPoint(std::min(m_start.x(),m_end.x()),std::min(m_start.y(),m_end.y())),QPoint(std::max(m_start.x(),m_end.x()),std::max(m_start.y(),m_end.y()))));
          painter.end();
          m_doc->addStitches(processPixmap(dm),m_currentStitchType);
          emit locationOrSize(QPoint(-1,-1));
          updateContents();
          break;
        case FillRectangle:
          drawBoundingRect(m_start, m_end);
          painter.begin(&dm);
          painter.setPen(QPen(black,1));
          painter.setBrush(black);
          painter.drawRect(QRect(QPoint(std::min(m_start.x(),m_end.x()),std::min(m_start.y(),m_end.y())),QPoint(std::max(m_start.x(),m_end.x()),std::max(m_start.y(),m_end.y()))));
          painter.end();
          m_doc->addStitches(processPixmap(dm),m_currentStitchType);
          emit locationOrSize(QPoint(-1,-1));
          updateContents();
          break;
        case Ellipse:
          drawBoundingEllipse(m_start, m_end);
          painter.begin(&dm);
          painter.setPen(QPen(black,1));
          painter.drawEllipse(QRect(QPoint(std::min(m_start.x(),m_end.x()),std::min(m_start.y(),m_end.y())),QPoint(std::max(m_start.x(),m_end.x()),std::max(m_start.y(),m_end.y()))));
          painter.end();
          m_doc->addStitches(processPixmap(dm),m_currentStitchType);
          emit locationOrSize(QPoint(-1,-1));
          updateContents();
          break;
        case FillEllipse:
          drawBoundingEllipse(m_start, m_end);
          painter.begin(&dm);
          painter.setPen(QPen(black,1));
          painter.setBrush(black);
          painter.drawEllipse(QRect(QPoint(std::min(m_start.x(),m_end.x()),std::min(m_start.y(),m_end.y())),QPoint(std::max(m_start.x(),m_end.x()),std::max(m_start.y(),m_end.y()))));
          painter.end();
          m_doc->addStitches(processPixmap(dm),m_currentStitchType);
          emit locationOrSize(QPoint(-1,-1));
          updateContents();
          break;
        case FillPolyline:
          m_end = contentsToCell(p);
          if ((m_end == m_pointArray.point(0)) && (m_points > 1))
          {
            m_pointArray.truncate(m_points);
            painter.begin(&dm);
            painter.setPen(QPen(black,1));
            painter.setBrush(black);
            painter.drawPolygon(m_pointArray);
            painter.end();
            m_doc->addStitches(processPixmap(dm),m_currentStitchType);
            updateContents(cellsToRect(m_pointArray.boundingRect()));
            m_pointArray.resize(0);
            m_points = 0;
          }
          updateContents();
          break;
        case SelectRectangle:
          drawSelectionRectangle();
          m_end = m_tracking;
          if (m_start != m_end)
          {
            emit selectionMade(true);
            drawSelectionRectangle();
            m_selectionTimer.start(TIMERSPEED);
          }
          else
            m_selectedArea = QRect();
          emit locationOrSize(QPoint(-1,-1));
          break;
        case Paste:
          repaintContents(QRect(cellToRect(m_end).topLeft(),QSize(m_pasteImage->size())));
          m_end = contentsToCell(p);
          m_doc->pasteSelection(m_end,m_pasteData->encodedData("application/kxstitch"),e->state() & ControlButton);
          setMode(m_oldMode);
          updateContents(QRect(cellToRect(m_tracking).topLeft(),QSize(m_pasteImage->size())));
          delete m_pasteImage;
          m_pasteImage = 0;
          if (m_pasteDataIsLibrary)
          {
            delete m_pasteData;
            m_pasteData = 0;
          }
          break;
        case ColorPicker:
          QPoint cell = contentsToCell(p);
          int colorIndex = m_doc->findColor(cell,m_stitchMask?m_currentStitchType:Stitch::Delete);
          if (colorIndex != -1)
          {
            ((KXStitchView *)parent())->m_palette->setCurrentColor(colorIndex);
            ((KXStitchView *)parent())->m_palette->repaint(false);
          }
          break;
      }
      break;

    case MidButton:
      break;

    case RightButton: // some popup menu functionality
      break;
  }
}

void EditView::clearSelection()
{
  m_selectionTimer.stop();
  m_selectedArea = QRect();
  emit selectionMade(false);
  updateContents();
}

void EditView::slotSelectionTimedOut()
{
  drawSelectionRectangle();
  m_selectedAreaOffset.rx()++;
  if (m_selectedAreaOffset.x() == 10)
    m_selectedAreaOffset = QPoint(0,0);
  drawSelectionRectangle();
}

void EditView::drawSelectionRectangle()
{
  QPainter p;
  QPointArray a(6);
  if (m_start != m_end && !m_updatingDisplay)
  {
    m_selectedArea = cellToRect(m_start);
    m_selectedArea |= cellToRect(m_end);
    a.setPoint(0,contentsToViewport(m_selectedArea.topLeft()+m_selectedAreaOffset));
    a.setPoint(1,contentsToViewport(m_selectedArea.topRight()));
    a.setPoint(2,contentsToViewport(m_selectedArea.bottomRight()));
    a.setPoint(3,contentsToViewport(m_selectedArea.bottomLeft()));
    a.setPoint(4,contentsToViewport(m_selectedArea.topLeft()));
    a.setPoint(5,contentsToViewport(m_selectedArea.topLeft()+m_selectedAreaOffset));
    p.begin(viewport());
    p.setRasterOp(XorROP);
    p.setPen(QPen(white,3,DashLine));
    p.drawPolyline(a);
    p.end();
  }
}

QPoint EditView::visibleCenterCell()
{
  int cx = contentsX()+(visibleWidth()/2);
  int cy = contentsY()+(visibleHeight()/2);
  return contentsToCell(QPoint(cx,cy));
}

QRect EditView::visibleCells()
{
  QRect cells;
  int width=m_doc->canvas()->patternWidth();
  int height=m_doc->canvas()->patternHeight();
  QPoint tl = contentsToCell(QPoint(contentsX(), contentsY()));
  QPoint br = contentsToCell(QPoint(contentsX()+visibleWidth(), contentsY()+visibleHeight()));
  cells.setCoords(tl.x(), tl.y(), std::min(br.x(), width), std::min(br.y(), height));
  return cells;
}

QPoint EditView::contentsToCell(QPoint contents)
{
  int xstitches = m_doc->canvas()->patternWidth();
  int ystitches = m_doc->canvas()->patternHeight();
  int xcontents = contentsWidth();
  int ycontents = contentsHeight();
  int xdig = contents.x();
  int ydig = contents.y();
  int xcell = xdig*xstitches/xcontents;
  int ycell = ydig*ystitches/ycontents;
  return QPoint(xcell, ycell);
}

QPoint EditView::contentsToSnap(QPoint contents)
{
  int xSnaps = (m_doc->canvas()->patternWidth())*2;
  int ySnaps = (m_doc->canvas()->patternHeight())*2;

  int snapX = (int)(round((double)(xSnaps*contents.x())/contentsWidth()));
  int snapY = (int)(round((double)(ySnaps*contents.y())/contentsHeight()));

  return QPoint(snapX,snapY);
}

QPoint EditView::snapToViewport(QPoint snap)
{
  return contentsToViewport(snapToContents(snap));
}

QPoint EditView::snapToContents(QPoint snap)
{
  int contentsX = m_cellSize*snap.x()/2;
  int contentsY = m_cellSize*snap.y()/2;

  return QPoint(contentsX,contentsY);
}

QRect EditView::cellToRect(QPoint cell)
{
  int x = cell.x()*m_cellSize;
  int y = cell.y()*m_cellSize;
  return QRect(x,y,m_cellSize,m_cellSize);
}

QRect EditView::cellsToRect(QRect cells)
{
  int x = cells.left()*m_cellSize;
  int y = cells.top()*m_cellSize;
  return QRect(x,y,m_cellSize*cells.width(),m_cellSize*cells.height());
}

QRect EditView::rectToViewport(QRect rect)
{
  return QRect(contentsToViewport(rect.topLeft()),contentsToViewport(rect.bottomRight()));
}

QPointArray EditView::processPixmap(QPixmap& pix)
{
  QImage im;
  QPointArray pointArray(pix.width()*pix.height());
  im = pix.convertToImage();
  int i = 0;
  for (int y = 0 ; y < im.height() ; y++)
  {
    for (int x = 0 ; x < im.width() ; x++)
    {
      if (im.pixelIndex(x,y) == 0)
      {
        pointArray.setPoint(i++,x,y);
      }
    }
  }
  pointArray.truncate(i);
  return pointArray;
}

void EditView::slotZoomIn()
{
  bool startTimer;
  if (startTimer = m_selectionTimer.isActive())
  {
    m_selectionTimer.stop();
    drawSelectionRectangle();
    m_selectedAreaOffset = QPoint(0,0);
  }
  QPoint zoomCenter = visibleCenterCell();
  viewport()->setUpdatesEnabled(false);
  m_cellSize+=4;
  resizeGrid();
  slotSetCenter(zoomCenter);
  viewport()->setUpdatesEnabled(true);
  update();
  viewport()->repaint(true);
  emit changedVisibleCells(visibleCells());
  m_zoomOut->setEnabled(m_cellSize > 4);
  m_zoomIn->setEnabled(m_cellSize < 64);
  if (startTimer)
  {
    drawSelectionRectangle();
    m_selectionTimer.start(TIMERSPEED);
  }
}

void EditView::slotZoomOut()
{
  bool startTimer;
  if (startTimer = m_selectionTimer.isActive())
  {
    m_selectionTimer.stop();
    drawSelectionRectangle();
    m_selectedAreaOffset = QPoint(0,0);
  }
  QPoint zoomCenter = visibleCenterCell();
  viewport()->setUpdatesEnabled(false);
  m_cellSize-=4;
  resizeGrid();
  slotSetCenter(zoomCenter);
  viewport()->setUpdatesEnabled(true);
  update();
  viewport()->repaint(true);
  emit changedVisibleCells(visibleCells());
  m_zoomOut->setEnabled(m_cellSize > 4);
  m_zoomIn->setEnabled(m_cellSize < 64);
  if (startTimer)
  {
    drawSelectionRectangle();
    m_selectionTimer.start(TIMERSPEED);
  }
}

void EditView::slotStitchFull()
{
  m_currentStitchType = Stitch::Full;
}

void EditView::slotStitchTBHalf()
{
  m_currentStitchType = Stitch::TBHalf;
}

void EditView::slotStitchBTHalf()
{
  m_currentStitchType = Stitch::BTHalf;
}

void EditView::slotStitchBLQtr()
{
  m_currentStitchType = Stitch::BLQtr;
}

void EditView::slotStitchTLQtr()
{
  m_currentStitchType = Stitch::TLQtr;
}

void EditView::slotStitchTRQtr()
{
  m_currentStitchType = Stitch::TRQtr;
}

void EditView::slotStitchBRQtr()
{
  m_currentStitchType = Stitch::BRQtr;
}

void EditView::slotStitchBL3Qtr()
{
  m_currentStitchType = Stitch::BL3Qtr;
}

void EditView::slotStitchTL3Qtr()
{
  m_currentStitchType = Stitch::TL3Qtr;
}

void EditView::slotStitchTR3Qtr()
{
  m_currentStitchType = Stitch::TR3Qtr;
}

void EditView::slotStitchBR3Qtr()
{
  m_currentStitchType = Stitch::BR3Qtr;
}

void EditView::slotStitchFRKnot()
{
  m_currentStitchType = Stitch::FRKnot;
}

void EditView::slotToolPaint()
{
  setMode(Paint);
}

void EditView::slotToolDraw()
{
  setMode(Draw);
}

void EditView::slotToolBackStitch()
{
  setMode(Backstitch);
}

void EditView::slotToolErase()
{
  setMode(Erase);
}

void EditView::slotRectangle()
{
  setMode(Rectangle);
}

void EditView::slotFillRectangle()
{
  setMode(FillRectangle);
}

void EditView::slotEllipse()
{
  setMode(Ellipse);
}

void EditView::slotFillEllipse()
{
  setMode(FillEllipse);
}

void EditView::slotFillPolyline()
{
  setMode(FillPolyline);
}

void EditView::slotSelectRectangle()
{
  setMode(SelectRectangle);
}

void EditView::slotColorPicker()
{
  setMode(ColorPicker);
}

void EditView::slotRegular()
{
  setViewMode(Regular);
}

void EditView::slotBWSymbols()
{
  setViewMode(BWSymbols);
}

void EditView::slotColorSymbols()
{
  setViewMode(ColorSymbols);
}

void EditView::slotColorBlocks()
{
  setViewMode(ColorBlocks);
}

void EditView::slotColorBlocksSymbols()
{
  setViewMode(ColorBlocksSymbols);
}

void EditView::slotColorHighlight()
{
  setViewMode(ColorHighlight);
}

void EditView::slotStitchMask()
{
  m_stitchMask = !m_stitchMask;
  emit stitchMask(m_stitchMask);
}

void EditView::slotColorMask()
{
  m_colorMask = !m_colorMask;
  emit colorMask(m_colorMask);
}

void EditView::slotExcludeBackstitches()
{
  m_excludeBackstitches = !m_excludeBackstitches;
  emit excludeBackstitches(m_excludeBackstitches);
}

void EditView::slotExcludeKnots()
{
  m_excludeKnots = !m_excludeKnots;
  emit excludeKnots(m_excludeKnots);
}

void EditView::slotFitWidth()
{
  m_cellSize = visibleWidth()/m_doc->canvas()->patternWidth();
  resizeGrid();
}

void EditView::slotFitHeight()
{
  m_cellSize = visibleHeight()/m_doc->canvas()->patternHeight();
  resizeGrid();
}

void EditView::slotFitPage()
{
  m_cellSize = std::min((visibleWidth()/m_doc->canvas()->patternWidth()),(visibleHeight()/m_doc->canvas()->patternHeight()));
  resizeGrid();
}

void EditView::slotScaleFormatStitches()
{
  m_scaleFormat = Stitches;
  emit scaleFormat("ST");
  update();
}

void EditView::slotScaleFormatCM()
{
  m_scaleFormat = CM;
  emit scaleFormat("CM");
  update();
}

void EditView::slotScaleFormatInches()
{
  m_scaleFormat = Inches;
  emit scaleFormat("IN");
  update();
}

void EditView::slotShowBackground()
{
  m_backgroundImage = !m_backgroundImage;
  updateContents();
}

void EditView::slotShowStitches()
{
  m_stitches = !m_stitches;
  updateContents();
}

void EditView::slotShowBackStitches()
{
  m_backstitches = !m_backstitches;
  updateContents();
}

void EditView::slotShowGrid()
{
  m_grid = !m_grid;
  updateContents();
}

void EditView::slotShowScales()
{
  m_scales = !m_scales;
  int marginSize = (m_scales)?m_scaleSize:0;
  setMargins(marginSize,marginSize,0,0);
  if (m_scales)
  {
    switch (m_scaleFormat)
    {
      case Stitches:
        emit scaleFormat("ST");
        break;
      case Inches:
        emit scaleFormat("IN");
        break;
      case CM:
        emit scaleFormat("CM");
        break;
    }
  }
  else
    emit scaleFormat("");
}

void EditView::hideSelectionRect()
{
  m_selectionTimer.stop();
  if (m_selectedRectangleVisible)
    drawSelectionRectangle();
}

void EditView::showSelectionRect()
{
  if (m_selectedArea.isValid())
    m_selectionTimer.start(TIMERSPEED);
}

void EditView::slotScrolling(int, int)
{
  if (m_scales)
  {
    update(frameWidth(),frameWidth(),width(),m_scaleSize);
    update(frameWidth(),frameWidth(),m_scaleSize,height());
  }
}

void EditView::setMode(ToolMode m)
{
  m_oldMode = m_mode;
  m_mode = m;
  if (m_oldMode == SelectRectangle && m_selectionTimer.isActive())
  {
    m_selectionTimer.stop();
    drawSelectionRectangle();
    m_selectedArea = QRect();
    emit selectionMade(false);
  }
  if (m_mode == FillPolyline)
  {
    m_pointArray.resize(0);
    m_points = 0;
  }
  if (m_mode == Paste)
  {
    m_pasteImage = NULL;
  }
}

void EditView::setViewMode(ViewMode m)
{
  m_vmode = m;
  updateContents();
}

void EditView::setBSViewMode(ViewMode m)
{
  m_bsvmode = m;
  updateContents();
}

void EditView::drawBoundingRect(QPoint s, QPoint e)
{
  QPainter painter(this->viewport());
  painter.setRasterOp(XorROP);
  painter.setPen(QPen(white, m_cellSize));
  QRect rect = QRect(cellToRect(s).center(),cellToRect(e).center());
  painter.drawRect(QRect(contentsToViewport(rect.topLeft()), contentsToViewport(rect.bottomRight())));
}

void EditView::drawBoundingEllipse(QPoint s, QPoint e)
{
  QPainter painter(this->viewport());
  painter.setRasterOp(XorROP);
  painter.setPen(QPen(white, m_cellSize));
  QRect rect = QRect(cellToRect(s).center(),cellToRect(e).center());
  painter.drawEllipse(QRect(contentsToViewport(rect.topLeft()), contentsToViewport(rect.bottomRight())));
}

void EditView::drawBoundingLine(QPoint s, QPoint e)
{
  QPainter painter(this->viewport());
  painter.setRasterOp(XorROP);
  painter.setPen(QPen(white, 1));
  painter.drawLine(s, e);
}

void EditView::initialiseRenderCellsFonts(QPainter *p, int size)
{
  int half          = size/2;
  int third         = size/3;

  m_renderCell      = QRect(0,0,size,size);
  m_renderTLCell    = QRect(0,0,half,half);
  m_renderTL3Cell   = QRect(0,0,size-third,size-third);
  m_renderTRCell    = QRect(half,0,half,half);
  m_renderTR3Cell   = QRect(third,0,size-third,size-third);
  m_renderBLCell    = QRect(0,half,half,half);
  m_renderBL3Cell   = QRect(third,0,size-third,size-third);
  m_renderBRCell    = QRect(half,half,half,half);
  m_renderBR3Cell   = QRect(third,third,size-third,size-third);

  m_renderTLQ[0]    = 0;
  m_renderTLQ[1]    = 0;
  m_renderTLQ[2]    = half;
  m_renderTLQ[3]    = 0;
  m_renderTLQ[4]    = 0;
  m_renderTLQ[5]    = half;

  m_renderTRQ[0]    = half;
  m_renderTRQ[1]    = 0;
  m_renderTRQ[2]    = size;
  m_renderTRQ[3]    = half;
  m_renderTRQ[4]    = size;
  m_renderTRQ[5]    = half;

  m_renderBLQ[0]    = 0;
  m_renderBLQ[1]    = half;
  m_renderBLQ[2]    = half;
  m_renderBLQ[3]    = size;
  m_renderBLQ[4]    = 0;
  m_renderBLQ[5]    = size;

  m_renderBRQ[0]    = size;
  m_renderBRQ[1]    = half;
  m_renderBRQ[2]    = size;
  m_renderBRQ[3]    = size;
  m_renderBRQ[4]    = half;
  m_renderBRQ[5]    = size;

  m_renderBLTRH[0]  = 0;
  m_renderBLTRH[1]  = size;
  m_renderBLTRH[2]  = 0;
  m_renderBLTRH[3]  = half;
  m_renderBLTRH[4]  = half;
  m_renderBLTRH[5]  = 0;
  m_renderBLTRH[6]  = size;
  m_renderBLTRH[7]  = half;
  m_renderBLTRH[8]  = size;
  m_renderBLTRH[9]  = half;
  m_renderBLTRH[10] = half;
  m_renderBLTRH[11] = size;

  m_renderTLBRH[0]  = 0;
  m_renderTLBRH[1]  = 0;
  m_renderTLBRH[2]  = half;
  m_renderTLBRH[3]  = 0;
  m_renderTLBRH[4]  = size;
  m_renderTLBRH[5]  = half;
  m_renderTLBRH[6]  = size;
  m_renderTLBRH[7]  = size;
  m_renderTLBRH[8]  = half;
  m_renderTLBRH[9]  = size;
  m_renderTLBRH[10] = 0;
  m_renderTLBRH[11] = half;

  m_renderTL3Q[0]   = 0;
  m_renderTL3Q[1]   = 0;
  m_renderTL3Q[2]   = size;
  m_renderTL3Q[3]   = 0;
  m_renderTL3Q[4]   = size;
  m_renderTL3Q[5]   = half;
  m_renderTL3Q[6]   = half;
  m_renderTL3Q[7]   = size;
  m_renderTL3Q[8]   = 0;
  m_renderTL3Q[9]   = size;

  m_renderTR3Q[0]   = 0;
  m_renderTR3Q[1]   = 0;
  m_renderTR3Q[2]   = size;
  m_renderTR3Q[3]   = 0;
  m_renderTR3Q[4]   = size;
  m_renderTR3Q[5]   = size;
  m_renderTR3Q[6]   = half;
  m_renderTR3Q[7]   = size;
  m_renderTR3Q[8]   = 0;
  m_renderTR3Q[9]   = half;

  m_renderBL3Q[0]   = 0;
  m_renderBL3Q[1]   = 0;
  m_renderBL3Q[2]   = half;
  m_renderBL3Q[3]   = 0;
  m_renderBL3Q[4]   = size;
  m_renderBL3Q[5]   = half;
  m_renderBL3Q[6]   = size;
  m_renderBL3Q[7]   = size;
  m_renderBL3Q[8]   = 0;
  m_renderBL3Q[9]   = size;

  m_renderBR3Q[0]   = half;
  m_renderBR3Q[1]   = 0;
  m_renderBR3Q[2]   = size;
  m_renderBR3Q[3]   = 0;
  m_renderBR3Q[4]   = size;
  m_renderBR3Q[5]   = size;
  m_renderBR3Q[6]   = 0;
  m_renderBR3Q[7]   = size;
  m_renderBR3Q[8]   = 0;
  m_renderBR3Q[9]   = half;

  m_renderFont      = p->font();
  m_renderQtrFont   = m_renderFont;
  m_render3QtrFont  = m_renderFont;

  m_renderQtrFont.setPointSize(m_renderFont.pointSize()/2);
  m_render3QtrFont.setPointSize(m_renderFont.pointSize()*3/4);
}

void EditView::renderCell(QPainter *p, Stitch::Queue *pStitches,ViewMode viewMode)
{
  FlossPalette *pPalette = m_doc->palette();

  int x = m_renderCell.left(),
      y = m_renderCell.top(),
      x1 = m_renderCell.right(),
      y1 = m_renderCell.bottom();
  QPoint c = m_renderCell.center();
  int cx = c.x(),
      cy = c.y();

  if (viewMode==BWSymbols)
    p->setPen(black);
  if (pStitches)
  {
    int nStitches = pStitches->count();
    while (nStitches--)
    {
      Stitch *pStitch = pStitches->dequeue();
      Stitch::Type cst = pStitch->type;
      QColor c(pPalette->flossAt(pStitch->floss)->color);
      if (viewMode == ColorBlocks || viewMode == ColorBlocksSymbols)
        p->setBrush(c);
      QChar s(pPalette->symbolAt(pStitch->floss));
      if (viewMode!=BWSymbols) p->setPen(QPen(c, 2)); // this will be set from the stitch color index
      if (viewMode == ColorHighlight)
      {
        if (pPalette->currentColor() != pStitch->floss)
        {
          p->setPen(QColor(lightGray));
        }
      }
      switch (cst)
      {
        case Stitch::TLQtr:
          p->setFont(m_renderQtrFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(0,0,cx,cy);
              break;
            case BWSymbols:
              p->drawText(m_renderTLCell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderTLCell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->drawPolygon(QPointArray(3,m_renderTLQ));
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderTLCell, AlignCenter, s, 1);
              }
              break;
          }
          break;
        case Stitch::TRQtr:
          p->setFont(m_renderQtrFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(x1,y,cx,cy);
              break;
            case BWSymbols:
              p->drawText(m_renderTRCell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderTRCell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->drawPolygon(QPointArray(3,m_renderTRQ));
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderTRCell, AlignCenter, s, 1);
              }
              break;
          }
          break;
        case Stitch::BLQtr:
          p->setFont(m_renderQtrFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(x,y1,cx,cy);
              break;
            case BWSymbols:
              p->drawText(m_renderBLCell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderBLCell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->drawPolygon(QPointArray(3,m_renderBLQ));
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderBLCell, AlignCenter, s, 1);
              }
              break;
          }
          break;
        case Stitch::BRQtr:
          p->setFont(m_renderQtrFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(x1,y1,cx,cy);
              break;
            case BWSymbols:
              p->drawText(m_renderBRCell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderBRCell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->drawPolygon(QPointArray(3,m_renderBRQ));
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderBRCell, AlignCenter, s, 1);
              }
              break;
          }
          break;
        case Stitch::BTHalf:
          p->setFont(m_render3QtrFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(x1,y,x,y1);
              break;
            case BWSymbols:
              p->drawText(m_renderCell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderCell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->drawPolygon(QPointArray(6,m_renderBLTRH));
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderCell, AlignCenter, s, 1);
              }
              break;
          }
          break;
        case Stitch::TBHalf:
          p->setFont(m_render3QtrFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(x,y,x1,y1);
              break;
            case BWSymbols:
              p->drawText(m_renderCell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderCell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->drawPolygon(QPointArray(6,m_renderTLBRH));
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderCell, AlignCenter, s, 1);
              }
              break;
          }
          break;
        case Stitch::TL3Qtr:
          p->setFont(m_render3QtrFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(x1,y,x,y1);
              p->drawLine(x,y,cx,cy);
              break;
            case BWSymbols:
              p->drawText(m_renderTL3Cell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderTL3Cell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->drawPolygon(QPointArray(5,m_renderTL3Q));
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderTL3Cell, AlignCenter, s, 1);
              }
              break;
          }
          break;
        case Stitch::TR3Qtr:
          p->setFont(m_render3QtrFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(x,y,x1,y1);
              p->drawLine(x1,y,cx,cy);
              break;
            case BWSymbols:
              p->drawText(m_renderTR3Cell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderTR3Cell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->drawPolygon(QPointArray(5,m_renderTR3Q));
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderTR3Cell, AlignCenter, s, 1);
              }
              break;
          }
          break;
        case Stitch::BL3Qtr:
          p->setFont(m_render3QtrFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(x,y,x1,y1);
              p->drawLine(x,y1,cx,cy);
              break;
            case BWSymbols:
              p->drawText(m_renderBL3Cell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderBL3Cell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->drawPolygon(QPointArray(5,m_renderBL3Q));
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderBL3Cell, AlignCenter, s, 1);
              }
              break;
          }
          break;
        case Stitch::BR3Qtr:
          p->setFont(m_render3QtrFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(x1,y,x,y1);
              p->drawLine(x1,y1,cx,cy);
              break;
            case BWSymbols:
              p->drawText(m_renderBR3Cell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderBR3Cell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->drawPolygon(QPointArray(5,m_renderBR3Q));
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderBR3Cell, AlignCenter, s, 1);
              }
              break;
          }
          break;
        case Stitch::Full:
          p->setFont(m_renderFont);
          switch(viewMode)
          {
            case Regular :
	    case ColorHighlight :
              p->drawLine(x,y,x1,y1);
              p->drawLine(x1,y,x,y1);
              break;
            case BWSymbols:
              p->drawText(m_renderCell, AlignCenter, s, 1);
              break;
            case ColorSymbols:
              p->drawText(m_renderCell, AlignCenter, s, 1);
              break;
            case ColorBlocks:
            case ColorBlocksSymbols:
              p->fillRect(m_renderCell,c);
              if (viewMode==ColorBlocksSymbols)
              {
                if(qGray(c.rgb()) < 128)
                  p->setPen(white);
                else
                  p->setPen(black);
                p->drawText(m_renderCell, AlignCenter, s, 1);
              }
              break;
            default :
              // Shouldn't happen
              break;
          }
          break;
      }
      pStitches->enqueue(pStitch);
    }
  }
}

