/***************************************************************************
                          waterfall.cpp  -  description
                             -------------------
    begin                : Tue Jul 25 2000
    copyright            : (C) 2000 by Luc Langehegermann
    email                : lx2gt@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "waterfall.h"
#include <stdio.h>
#include <stdlib.h>
#include <qtooltip.h>
#include <qbitmap.h>
#include <qpixmap.h>
#include <qtimer.h>
#include <qcursor.h>
#include <math.h>
#include <kwin.h>
#include <klocale.h>
#include "server/server.h"
#include "arrow.xpm"
#include "globals.h"



#define DIST 2
#define FREQ_HEIGHT 22
#define IMD_SECURITY 5
#define FREQ_INCREMENT 5 // 5 Hz increment for fine freq control

#define SIGDIFF (SAMPLES/512)
#define IMDDIFF ((SAMPLES/512)*3)

waterfall::waterfall(QWidget *parent) : QWidget(parent)
{

	int i;
	setBackgroundMode (NoBackground);
	// Create the 3 Aux windows
	for (i=0;i<3;++i) {
		aux[i] = new auxWindow (i+COMM_RXCH+1);
	}
	for (i=0;i<4;++i)
		IMDcount[i]=0;

// xmax and ymax initial values
  xmax=width();
  ymax=height();

// min & max freq set for common tx bandwidth
  minfreq=201;
  maxfreq=3000;


	reInit();

	setCursor (QCursor (QPixmap(arrow),6,0));
	commControl (COMM_FFTCH, COMM_FFTN, SAMPLES);
  commControl (COMM_FFTCH, COMM_FFTOVERLAP, SAMPLES/4);
  activechannel=COMM_RXCH;

// display limits in sample space
  from = (minfreq * SAMPLES)/8000.0;
  to = (maxfreq * SAMPLES)/8000.0;

// initialize all lastpos values in sample space
  for (i=0; i<4; i++) lastpos[i] =-1;

// dynamic tooltip for waterfall
  ktip = new KpskTip(this);


// basic display timer for waterfall
	timer = new QTimer(this);
	QObject::connect( timer, SIGNAL(timeout()), SLOT(slotProcessFFTData()));
	timer->start(10, false);
}

waterfall::~waterfall()
{
delete ktip;  // delete tooltip, Qtooltip is not subclass of QObject
}


void waterfall::resizeEvent (QResizeEvent* )
{

  xmax = width();
  ymax = height();
// create lookup tables for x coordinates to samples, and back
  translate();

// create display buffer
  buffer.resize (xmax,ymax);
	buffer.fill (Qt::black);

  resize (xmax, ymax);

// causes new lineal etc
  reInit();

}

void waterfall::paintEvent (QPaintEvent*)
{
	int i;
	bitBlt (this, 0, 0, &lineal);
	// Make a copy of the water, draw the markers on it and draw that pixmap
	QPixmap* tmpbuffer = new QPixmap (buffer);

  // debugging draw vertical line

  QPainter painter(tmpbuffer);
//  painter.setPen(Qt::red);
//  painter.drawLine(stranslate[lastpos[0]], 0, stranslate[lastpos[0]], 99);

  for (i=0;i<4;++i) {	
		if (lastpos[i] > 0 ) {
			if (activechannel-COMM_RXCH == i){

/* stranslate converts samples space, lastpos[] to x coordinates
-4 offset centers marker in signal */
       painter.drawPixmap(stranslate[lastpos[i]]-4, DIST, activemarker);
       }
			else
			 painter.drawPixmap(stranslate[lastpos[i]]-4, DIST, marker);
		}
  }
  painter.end();
  bitBlt (this, 0, FREQ_HEIGHT, tmpbuffer);
	delete tmpbuffer;
}

void waterfall::translate(void)
{
  from = double ((minfreq * SAMPLES)/8000);
  to   = double ((maxfreq * SAMPLES)/8000);
  int i;
  // convert x coordinates to samples
  for (i=1; i<width(); i++){
    double a = (maxfreq - minfreq) *i *to/width();
    xtranslate[i] = (unsigned int) nearbyint( (a + minfreq * to)/maxfreq);
  }
  // convert samples to x coordinates
  for (i=1; i<SAMPLES; i++){
  stranslate[i] = (int)nearbyint( (((i * maxfreq)-(minfreq*to))*width()/((maxfreq-minfreq)*to)));
  }
}

void waterfall::slotProcessFFTData()
{
  bool overload;
  QPainter painter;
  int l, x, z;
  float f, ratio, imdtotal, maxfftval;
  short value;
  float fftval[SAMPLES];
  int IMDtemp[4][4];

  l = commGetData (COMM_FFTCH, (char *) fftval, sizeof (fftval));
  if (l == 0) return;
  // find out if we might have too much input in the sound card
  overload=false; maxfftval=0.0;
  for (x=2; x<SAMPLES/2; ++x) {
    if (50*log10(fftval[x]+1) > 105.0) overload=true;
}

  bitBlt (&buffer, 0, 1, &buffer, 0, 0);
  painter.begin (&buffer);
  	
  for (x=2; x<xmax;x++) {
    z = xtranslate[x]; //convert x coordinates to sample space

// value of signal adjusted by display sensitivity
    value = (short) (50 * log10 (fftval[z]+1));
    if (value > 99) value = 99;  // range from 0 to 99
    if (!overload) 
      painter.setPen (watercolor[value]); // assign color
    else
      painter.setPen (Qt::red);
    painter.drawPoint (x, 0); // plot point on display buffer
  }

  /* Calculate the positions */

  f = (kpsk->rxinfo.freq / 100);

  lastpos[0] = (int) nearbyint((f*SAMPLES)/8000.0); //convert freq to sample space

	/* Calculate the IMDs */

// main rx display
	if (fftval[lastpos[0]]+IMD_SECURITY < fftval[lastpos[0]+SIGDIFF] && fftval[lastpos[0]]+IMD_SECURITY < fftval[lastpos[0]+SIGDIFF]) {
		ratio = fftval[lastpos[0]+IMDDIFF] /  fftval[lastpos[0]+SIGDIFF];
		imdtotal = 20.0 * log10 (ratio);
		ratio = fftval[lastpos[0]-IMDDIFF] /  fftval[lastpos[0]-SIGDIFF];
		imdtotal = imdtotal + 20.0 * log10 (ratio);
		IMDtemp[0][IMDcount[0]] = (int) (imdtotal/2.0);
		}
	else {

// IMD set to max value
		IMDtemp[0][IMDcount[0]]=65536;
		}
// for visible aux windows
  int i;
	for (i=0;i<3;++i) {
		if (aux[i]->isVisible()) {
			f = aux[i]->getFreq();
        if (f>minfreq && f<maxfreq){  // make sure a valid freq is returned
        lastpos[i+1] = (int) ((f*SAMPLES)/8000);

          if (fftval[lastpos[i+1]]+IMD_SECURITY < fftval[lastpos[i+1]+4] && fftval[lastpos[i+1]]+IMD_SECURITY < fftval[lastpos[i+1]-4]) {
			    ratio = fftval[lastpos[i+1]+12] /  fftval[lastpos[i+1]+4];
		  	  imdtotal = 20.0 * log10 (ratio);
				  ratio = fftval[lastpos[i+1]-12] /  fftval[lastpos[i+1]-4];
		  	  imdtotal = imdtotal + 20.0 * log10 (ratio);
		  	  IMDtemp[i+1][IMDcount[i+1]] = (int) (imdtotal/2.0);
          }
			  else {
          // IMD set to max value
				   IMDtemp[i+1][IMDcount[i+1]]=65536;
				  }
      }
		}
		else {

// lastpos stay at -1 and IMD is reset to max value
			lastpos[i+1] = -1;
			IMD[i+1] = 65536;
		}
	}
// average the IMD over 4 values
	for (i=0; i<4;++i) {
		if (IMDcount[i]>2) {
			IMDcount[i]=0;  // reset count
			IMD[i] = (IMDtemp[i][0] + IMDtemp[i][1] + IMDtemp[i][2] + IMDtemp[i][3]) / 4;
      } else ++IMDcount[i];
  }
	painter.end();
// paint new screen without erasing
	repaint(false);
}


void waterfall::mousePressEvent (QMouseEvent* event)
{
	int i;

/* find freq at the cursor
first test for out-of-bounds mouse click and no response
to clicks outside window */

  if (event->x() >= xmax){
   xpos = xmax;
    }
  else {
      if (event->x() <= 5){
       xpos = 5;
       }
      else {
        xpos = event->x();
        }
  }
	freq = (int) (xpos*(maxfreq-minfreq))/ width()+minfreq;
    if (event->button() == LeftButton && event->state() == 0) {
		/* Check if we should activate one of the aux windows */

		for (i=0;i<4;++i) {
			if (xtranslate[xpos] >= (unsigned int)lastpos[i]-6 && xtranslate[xpos] <= (unsigned int)lastpos[i]+6 && lastpos[i] > 0) {
				activechannel=COMM_RXCH+i;
    if(i!=0) KWin::setActiveWindow(aux[i-1]->winId());
				return;
				}
		}
		if (activechannel==COMM_RXCH) {
			commControl (COMM_RXCH, COMM_FREQ, freq*100);
			commControl (COMM_TXCH, COMM_FREQ, freq*100);
			}
		else
			commControl (activechannel, COMM_FREQ, freq*100);
	}

	if (event->button() == RightButton) {  // create new RX Channel!
		for (i=0;i<3;++i) {
			if (!aux[i]->isVisible()) {
				aux[i]->show();
		  	commControl (COMM_RXCH+i+1, COMM_MODE, MO_NORMAL);
		  	commControl (COMM_RXCH+i+1, COMM_DCDLEVEL, config.psk31.dcdLevel);
  			commControl (COMM_RXCH+i+1, COMM_QPSK, false);     // BPSK
  			commControl (COMM_RXCH+i+1, COMM_AFC, true);       // AFC on
  			commControl (COMM_RXCH+i+1, COMM_FREQ, freq*100);
				commControl (COMM_RXCH+i+1, COMM_LSB, kpsk->rxinfo.lsb); // lsb as the main channel
				aux[i]->start(); // Tell the window to start receiving!
				activechannel = COMM_RXCH+i+1;
				break;
			}
		}
	}

}

void waterfall::mouseMoveEvent (QMouseEvent* event)
{

/* find new freq
first test for out-of-bounds position & limit position
so marker is always visible */

  if (event->x() >= xmax){
   xpos = xmax;
    }
  else {
      if (event->x() <= 5){
       xpos = 5;
       }
      else {
        xpos = event->x();
        }
  }
	freq = (int) (xpos*(maxfreq-minfreq))/width()+minfreq;

	if (event->state() == LeftButton) {
		if (activechannel>COMM_RXCH) {
        commControl (activechannel, COMM_FREQ, freq*100);
				return;
			}
		else {
				commControl (COMM_RXCH, COMM_FREQ, freq*100);
				commControl (COMM_TXCH, COMM_FREQ, freq*100);
			}
	}
}

void waterfall::incrementFreq()
{
  int incr_f;
  if (activechannel == COMM_RXCH) {
       incr_f = (kpsk->rxinfo.freq/100);
        incr_f = incr_f + FREQ_INCREMENT;
        commControl (COMM_RXCH, COMM_FREQ, incr_f*100);
				commControl (COMM_TXCH, COMM_FREQ, incr_f*100);
				
			}
	else {
       incr_f = aux[activechannel-(COMM_RXCH+1)]->getFreq();
       incr_f= incr_f + FREQ_INCREMENT;
			 commControl (activechannel, COMM_FREQ, incr_f*100);

			}
}

void waterfall::decrementFreq()
{
  int decr_f;
  if (activechannel == COMM_RXCH) {
        decr_f = (kpsk->rxinfo.freq/100);
        decr_f = decr_f - FREQ_INCREMENT;
       	commControl (COMM_RXCH, COMM_FREQ, decr_f*100);
				commControl (COMM_TXCH, COMM_FREQ, decr_f*100);
				return;
			}
		else {
        decr_f = aux[activechannel-(COMM_RXCH+1)]->getFreq();
        decr_f= decr_f - FREQ_INCREMENT;
				commControl (activechannel, COMM_FREQ, decr_f*100);
			}
}


QPixmap waterfall::getLineal (double offset, bool lsb)
{
	int i, ix;

// setup item for frequency labels	
  QFont* freqfont = new QFont ("helvetica", 10, QFont::Normal);
	QFontMetrics fm(*freqfont);
	QString freqtext;



// setup lineal pixmap
  QPixmap _lineal;
	QPainter painter;
	_lineal.resize (xmax+1, FREQ_HEIGHT);
	_lineal.fill (Qt::black);

	painter.begin (&_lineal);

// first draw tick marks
	painter.setPen (Qt::white);

	for (i=0;i<=maxfreq; i=i+100) {
    ix = (i* xmax)/(maxfreq-minfreq);
		painter.drawLine ( ix, FREQ_HEIGHT-1, ix, FREQ_HEIGHT-6);
	}

// now draw the frequency labels
	painter.setPen (Qt::green);
	painter.setFont (*freqfont);

	for (i=0;i<maxfreq;i=i+400) {
      // handle offset and lsb if set in band dialog
		  if (offset){
            if (lsb){
    				freqtext.setNum (float(((offset*1000) - maxfreq + i)/1000),'f',1);
  		      }
            else
				    freqtext.setNum (float(((offset*1000) + minfreq + i)/1000), 'f', 1);
      }
        else
      // no offset or lsb
          freqtext.setNum (minfreq + i-1);
		ix = (i* xmax)/(maxfreq-minfreq) - (fm.width(freqtext)/2);
// draw frequency text

    painter.drawText (ix, FREQ_HEIGHT-9-3, freqtext);
	}
	painter.end();
	return _lineal;
}

int waterfall::getIMD (int chn)
{
	return IMD[chn-COMM_RXCH];
}

/* Initialize the Waterfall */

void waterfall::reInit()
{
	int i;

  if (config.h2oColor.color_water){
    h2oColors(); // create the colors for waterfall
  }
  else {
    mkGray(); // create grayscale
  }
	

	uchar mask_bits[] = {0x08, 0x00, 0x1c, 0x00, 0x1c, 0x00, 0x3e, 0x00, 0x3e, 0x00, 0x7f, 0x00,
                       0x7f, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x7f, 0x00,			
                       0x7f, 0x00, 0x3e, 0x00, 0x3e, 0x00, 0x1c, 0x00, 0x1c, 0x00, 0x08, 0x00
                      };
  QBitmap bitmask = QBitmap (9, 18, mask_bits, false);

	static const char* mark0_xpm[] = {
		"9 18 2 1",
		". c #000000",
		"# c #990000",
    "....#....",  // 0
		"...###...",  // 1
		"...###...",  // 2
		"..#####..",  // 3
		"..##.##..",  // 4
		".###.###.",  // 5
		".##...##.",  // 6
		"###...###",  // 7
		"##.....##",  // 8
		"##.....##",  // 9
		"###...###",  // 10
		".##...##.",  // 11
		".###.###.",  // 12
		"..##.##..",  // 13
		"..#####..",  // 14
		"...###...",  // 15
		"...###...",  // 16
		"....#....",  // 17
		};

	marker = QPixmap(mark0_xpm);

	static const char* mark1_xpm[] = {
		"9 18 2 1",
		". c #000000",
		"# c #FF0000",
    "....#....",  // 0
		"...###...",  // 1
		"...###...",  // 2
		"..#####..",  // 3
		"..##.##..",  // 4
		".###.###.",  // 5
		".##...##.",  // 6
		"###...###",  // 7
		"##.....##",  // 8
		"##.....##",  // 9
		"###...###",  // 10
		".##...##.",  // 11
		".###.###.",  // 12
		"..##.##..",  // 13
		"..#####..",  // 14
		"...###...",  // 15
		"...###...",  // 16
		"....#....",  // 17
		};

	activemarker = QPixmap(mark1_xpm);

	marker.setMask (bitmask);
	activemarker.setMask (bitmask);
	for (i=0;i<3;++i)
		aux[i]->reInit();
	slotNewBand (config.logbook.band);
}

/** This creates a new lineal if the band is changed */

void waterfall::slotNewBand (int item)
{
	int i;
	config.logbook.band=item;
	if (item != -1)
		lineal = getLineal (config.band.offset[item], config.band.lsb[item]);
  else
		lineal = getLineal (0, 0);
	// set usb/lsb on all active channels
	
	commControl (COMM_RXCH, COMM_LSB, config.band.lsb[item]);
	commControl (COMM_TXCH, COMM_LSB, config.band.lsb[item]);

	for (i=0;i<3;++i) {
		if (aux[i]->isVisible())
			commControl (COMM_RXCH+i+1, COMM_LSB, config.band.lsb[item]);
	}
}

void waterfall::h2oColors()  // technicolor waterfall
{
	int i;
	for(i=0; i<1; i++)
   {watercolor[i]=QColor(config.h2oColor.zero); }
  for(; i<17; i++)
   {watercolor[i]=QColor(config.h2oColor.vlow); }
  for(; i<34; i++)
   {watercolor[i]=QColor(config.h2oColor.low); }
  for(; i<51; i++)
   {watercolor[i]=QColor(config.h2oColor.mid); }
 for(; i<68; i++)
   {watercolor[i]=QColor(config.h2oColor.hmid); }
  for(; i<97; i++)
   {watercolor[i]=QColor(config.h2oColor.high); }
  for(; i<100; i++)
   {watercolor[i]=QColor(config.h2oColor.vhigh); }

}

/* This is taken from twpsk */

void waterfall::mkGray()
{
  int i,r,g,b;
  for (i=0; i<100; i++) {
  // 100 intensity steps from 55,55,55 to 255,255,255 in steps 2
    r=g=b=i*2+55;
    watercolor[i] = QColor(r,g,b);
  }
}

/* Clear aux window text*/
void waterfall::slotClrCh1Rx()
{
  aux[0]->luc->clear();
}
void waterfall::slotClrCh2Rx()
{
  aux[1]->luc->clear();
}
void waterfall::slotClrCh3Rx()
{
  aux[2]->luc->clear();
}
