/***************************************************************************
                          gttextwidget.cpp  -  description
                             -------------------
    begin                : Don Sep 14 18:17:08 CEST 2000
    copyright            : (C) 2000 by Luc Langehegermann
    email                : lx2gt@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
/* Some consideration has been given to rewriting this widget using a
   standard Qt text widget like QTextEdit. As of July 2002, it was found
   that QtextEdit is a fine basis of a text editor and simplifies many
   functions but includes some problems areas such as autoscrolling on
   insert(). It has also believe that this large class significantly slows
   program response times producing garbled receive. */

#include <stdio.h>
#include <stdlib.h>
#include <qtimer.h>
#include <qclipboard.h>
#include <qapplication.h>
#include "gttextwidget.h"
#include "globals.h"

GTTextWidget::GTTextWidget(QWidget *parent, int l) : QGridView(parent)
{
  setWFlags(WResizeNoErase|WRepaintNoErase);
	line.setAutoDelete(true);
  scrolltimer = NULL;
  firstline = -1;
	lastline = -1;
	maxlines=l;
	// set some initial flags
	font = QFont::defaultFont();
	setFont (font);
	const QColorGroup & g = colorGroup();
	currentcol = 0;
	textcolor[0]=g.text();
	textcolor[1]=g.text();
	setNumCols(1);
	setNumRows(0);

	setBackgroundMode (PaletteBase);
	setFrameStyle (QFrame::WinPanel|QFrame::Sunken);
	show();

	vscroll = (QScrollBar*) verticalScrollBar();
  ((QScrollBar *)vscroll)->setSteps(cellHeight(), ((QScrollBar *)vscroll)->pageStep()-cellHeight());

	blinktimer = new QTimer (this);
	QObject::connect (blinktimer, SIGNAL(timeout()), SLOT(slotCursorBlink()));

	setEditEnabled (false);
	setCallCheckEnabled(false);
	newLine();
	QObject::connect (this, SIGNAL (keyPressed(char)), this, SLOT (slotKey(char)));
  // remove the horizontal scrollbar
  setHScrollBarMode(QScrollView::AlwaysOff);
}

GTTextWidget::~GTTextWidget()
{
}

static QPixmap *buffer = 0;

static void cleanupMLBuffer()
{
    delete buffer;
    buffer = 0;
}

static QPixmap* getCacheBuffer( QSize sz )
{
  if ( !buffer ) {
		qAddPostRoutine( cleanupMLBuffer );
		buffer = new QPixmap;
  	}

    if ( buffer->width() < sz.width() || buffer->height() < sz.height() )
			buffer->resize( sz );
    return buffer;
}

/** Overloaded member of QGridView that paints a line of text to the widget
    line is a QPtrList object, linked list of lines of text */

void GTTextWidget::paintCell (QPainter* p, int row, int)
{
	QString text;
	bool color;
	bool marked;
	QPainter painter;
	const QColorGroup & g = colorGroup();
	QFontMetrics fm(font);
	int y = fm.ascent();
	int x = 2;
  QRect updateR = cellRect();
  QPixmap *buffer = getCacheBuffer( updateR.size() );

  buffer->fill ( g.base() );

	if ((int) (line.count()-1) >= row) {
		// set text, color and whether marked to the entry in line indexed by row
    text = line.at(row)->text();
		color = line.at(row)->color();
		marked = line.at(row)->marked();

	  painter.begin( buffer );
		painter.translate( -updateR.left(), -updateR.top() );

		painter.setFont (font);
	
		if (marked) {
			painter.setPen (g.highlightedText() );
			painter.fillRect (updateR.left(), updateR.top(), fm.width(text)+3, fm.height(), g.brush( QColorGroup::Highlight ));
			// paint the  highlighted line
      painter.drawText (x, y, text);
		}
		else {
      // if not marked use regular color and paint line
			painter.setPen (textcolor[color]);
			painter.drawText (x, y, text);
		}
	/* Draw the cursor if we are in the last line */

		if ((cursoron == true) && (edited == true) && (row == (int) (line.count()-1)) ) {
			painter.drawLine (updateR.left()+fm.width(text)+2, updateR.top(), updateR.left()+fm.width(text)+2, fm.height()-2);
		}
	  painter.end();
	}
  p->drawPixmap( updateR.left(), updateR.top(), *buffer,
			 0, 0, updateR.width(), updateR.height() );
}

void GTTextWidget::newLine()
{
	bool add_line = true;
// scrollbar at bottom (unten_bleiben)
  bool unten_bleiben = false;

  if ((vscroll->value() == vscroll->maxValue()) || (vscroll->maxValue() == 0))
    unten_bleiben = true;
	if (!isVisible())
		unten_bleiben = true;


	if (line.count()+1 >= maxlines) {
      // run out of available lines, remove the first line
			line.removeFirst();
      // new text line construct to line linked list
			line.append (new GTTextLine (currentcol));
			repaint(false);

      if (!unten_bleiben)
      {
         if (vscroll->value() == 0)
         {
            repaint(false);
         }
         else
         {
            vscroll->subtractLine();
         }
      }
   }
   else
   {
/* we have not run out of lines, simply add new construct
   and update the number of rows */
      add_line = true;
			line.append (new GTTextLine (currentcol));
			setNumRows (line.count());
   }
// if scrollbar a bottom, scroll to show new line
	if (unten_bleiben) {
		int t = contentsHeight();
		int v = visibleHeight();
		if (t>v) {
			if (add_line)
				scrollBy (0,(t-v));
			else
				scrollBy (0, cellHeight());
		}
	}
}

void GTTextWidget::appendChar (char c, bool col, bool rep)
{
  QFontMetrics fm(font);
  int pos;
  char b[2];
  QString character;
  QString lastword;

  // do some filtering... if character < 31 and not \b or \n return

  if (c < 31 && c!='\n' && c!='\b') return;
  
  b[1] = 0;
  b[0] = c;
  character = QString (b);

  if (col != currentcol) {
		currentcol=col;
		if (callCheckEnabled && currentcol == false)
			doCallCheck();
		newLine();
		repaintCell (line.count()-2, 0, false);
		if (character == "\n") return;
  }

// Handle new line character
  if (character == "\n") {
		if (callCheckEnabled && currentcol == false)
			doCallCheck();
		newLine();
		repaintCell (line.count()-2, 0, false);
		return;
		}
	if (fm.width(line.last()->text()) >= visibleWidth()-vscroll->width()) {
		pos = line.last()->text().findRev (' ');
	  lastword = line.last()->text().mid(pos, 99999);
		line.last()->setText( line.last()->text().left (pos));
		if (callCheckEnabled && currentcol == false)
			doCallCheck();
		newLine();
		repaintCell (line.count()-2, 0, false);
	  line.last()->setText (lastword.mid (1, 99999));
	}
// Handle Backspace character
	if (character == "\b") {
    if (line.last()->text().length() != 0)
			line.last()->setText (line.last()->text().left (line.last()->text().length()-1));
		else {
			if (line.count() != 1) {
				line.removeLast();
				repaintCell (line.count(), 0, false);
				}
			}
		}
	else {
		line.last()->setText (line.last()->text() + character);
		}
	if (rep) repaintCell (line.count()-1, 0, false);
	if (character == " " && callCheckEnabled && currentcol == false)
		doCallCheck();
}
// Overloaded member of QWidget
void GTTextWidget::resizeEvent (QResizeEvent* e)
{
  QGridView::resizeEvent (e);
	setCellWidth (width()-20);
}

void GTTextWidget::appendString (char* text, bool signal, bool col)
{
	unsigned int i;
	for (i=0;i<strlen(text);++i) {
		appendChar (text[i], col, false);
  }
	if (signal)
		emit textAppended (text);
  repaintCell (line.count()-1, 0, false);
}

void GTTextWidget::appendString (QString text, bool signal)
{
	if (!text.isNull()) {
		char* s = strdup (text.latin1());
		appendString (s, signal);
		free (s);
	}
}

void GTTextWidget::setFont (QFont f)
{
	font = f;
	QFontMetrics fm (font);
	setCellHeight (fm.height());
	repaint(false);
}

void GTTextWidget::clearAllMarks()
{
	unsigned int i;
	for (i=0;i<line.count();++i) {
		line.at(i)->setMarked (false);
    updateCell(i,0);
  }
  repaint(false);
	firstline=-1;
	lastline=-1;
}
// Overloaded member of QScrollView
void GTTextWidget::contentsMousePressEvent( QMouseEvent *e)
{
	int row;
	int len;
	QFontMetrics fm(font);
// set beginning of marked area with left button
	if (e->button() == LeftButton) {
   	row = rowAt(e->pos().y());
  if (row != -1 && (int) line.count()>row){

	    len = fm.width(line.at(row)->text())+5;
     }
		else {
			len =-1;
     }

		if ((row >= firstline) && (row <= lastline) && (len != -1) && (e->pos().x() < len)) {
	 		QDragObject *d = new QTextDrag( QString(getMarkedText()), this );
     	d->dragCopy();
			}
		else {

			clearAllMarks();
    	marking = true;
    	mark_new = true;
    	markdirection = 0;
    	lastmarkline = -1;
			}

		}
// paste from clipboard with middle button
	if (e->button() == RightButton) {
		if (QApplication::clipboard()->text().latin1() != NULL) {
			char* tmp = strdup (QApplication::clipboard()->text().latin1());
			appendString (tmp);
			free (tmp);
			}
		}
}
// Overloaded member of QScrollView
void GTTextWidget::contentsMouseMoveEvent (QMouseEvent* e)
{
  int i,row;
  if (marking)
  {
     row = rowAt(e->pos().y());
     if (row != lastmarkline)
     {
        lastmarkline = row;
        if (row != -1)
        {
           if (mark_new || firstline == -1)
           {
              firstline = row;
              lastline = row;
           }
            if (markdirection == 0)
              if (firstline <= row)
                 markdirection = MARKDIR_TOPDOWN;
              else
                 markdirection = MARKDIR_DOWNTOP;
            if (row < firstline)
              markdirection = MARKDIR_DOWNTOP;

           if (markdirection == MARKDIR_TOPDOWN)
           {
              // Gucken, ob gerade eine oder mehrere Zeilen wieder freigegeben
              // wurden.
              if (lastline > row)
              {
                 // Ja! Alles zwischen firstline (incl) und row(excl.) freigeben
                 for (i=row+1; i<=lastline; i++)
                 {
                    // deals with no text on line equivalent to i out of range
		                if(line.at(i)==NULL){
                        return;
                      }
                    line.at(i)->setMarked( false );
                    updateCell(i,0);
                 }
                 lastline = row;
              }
               // Gucken, ob wir gerade ein paar Zeilen markiert haben
              if ((row > firstline) || mark_new)
              {
                 for (i=firstline; i<=row; i++)
                 {
                    if(i <= (int) line.count()){
                      // deals with no text on line equivalent to i out of range
		                if(line.at(i)==NULL){
                        return;
                      }
                      line.at(i)->setMarked(true);
                      updateCell(i, 0);
                    }
                 }
                 lastline = row;
              }
           }
            // Gucken, ob wir ganz unten mit unserer Markierung angelangt
           // sind. Wenn ja wird das ganze etwas heruntergescrollt
           //  rowAt(contentsY() - visibleHeight()  equals last visible row
           if (row == rowAt(contentsY()+visibleHeight()))
           {
              // scroll down (unten)
              scrolltimer = new QTimer(this);
              connect(scrolltimer, SIGNAL(timeout()), SLOT(slotTimerUnten()));
              scrolltimer->start(MARKTIMER, true);
           }
           else
              if (scrolltimer != NULL)
                 if (markdirection == MARKDIR_TOPDOWN)
                    if (scrolltimer->isActive())
                       scrolltimer->stop();

           // Gucken, ob wir ganz oben mit unserer Markierung angelangt
           // sind. Wenn ja wird das ganze etwas heraufgescrollt
           if (row != 0)
           // rowAT(contentsY()  equals topCell
              if (row == rowAt(contentsY()))
              {
                 // scroll up (oben)
                 scrolltimer = new QTimer(this);
                 connect(scrolltimer, SIGNAL(timeout()), SLOT(slotTimerOben()));
                 scrolltimer->start(MARKTIMER, true);
              }
              else
                 if (scrolltimer != NULL)
                    if (markdirection == MARKDIR_DOWNTOP)
                       if (scrolltimer->isActive())
                          scrolltimer->stop();

           // Gucken, ob wir gerade ueber dem oberen Ende 'rausscrollen
           if (markdirection == MARKDIR_DOWNTOP)
           {
              // Oben soll was markiert werden
              if (row < firstline)
              {
                 for (i=row; i<firstline; i++)
                 {
                  // deals with no text on line equivalent to i out of range
		                if(line.at(i)==NULL){
                        return;
                      }
                      line.at(i)->setMarked( true );
                      updateCell(i,0);
                 }
                 firstline = row;
              }
               // Wieder nach unten scrollen? -> alles obere "demarkieren"
              if (row > firstline)
               {
                  for (i=firstline; i<row; i++)
                  {
                  // deals with no text on line equivalent to i out of range
		                if(line.at(i)==NULL){
                        return;
                      }
                      line.at(i)->setMarked( false );
                      updateCell(i,0);
                  }
                  firstline = row;
               }
            }

            if (firstline == lastline)
               markdirection = MARKDIR_TOPDOWN;

            mark_new = false;
         }
      }

   }
  repaint(false);
	
}
// scroll up (oben) slot
void GTTextWidget::slotTimerOben()
{
	int row = rowAt(contentsY());

	if (row > 0)
	{

     // topcell is rowAt(contentsY()
		scrollBy(0,-cellHeight());
		line.at(row)->setMarked( true );
		updateCell(row, 0);
		firstline = row;
  }
  update();
   if ((scrolltimer != NULL) && (row != 0))
      scrolltimer->start(MARKTIMER, true);
}
// scroll down (unten) slot
void GTTextWidget::slotTimerUnten()
{
  // rowAt(contentsY() + visibleHeight()) is last visible row
	unsigned int row = rowAt(contentsY() + visibleHeight());

	if (row < line.count()-1)
	{
		scrollBy(0,+cellHeight());
		line.at(row)->setMarked( true );
		updateCell(row, 0);
		lastline = row;
	}

	if ((scrolltimer != NULL) && ((unsigned int)row < line.count()-1))
		scrolltimer->start(MARKTIMER, true);
}
// Overloaded member of QScrollView
void GTTextWidget::contentsMouseReleaseEvent (QMouseEvent* e)
{
   if (e->button() == LeftButton)
   {
      //end of marking
      marking = false;
      if (firstline > -1)
      {
         if (scrolltimer != NULL)
         {
            scrolltimer->stop();
            delete scrolltimer;
            scrolltimer = NULL;
         }
         // copy marked text to clipboard
         QApplication::clipboard()->setText(getMarkedText());
      }

   }

}

void GTTextWidget::setEditEnabled (bool ed)
{
	if (ed) {
		setFocusPolicy (StrongFocus);
		setAcceptDrops (true);
	    cursoron=false;
		edited=true;
		setCursor (ibeamCursor);
	}
	else {
		setFocusPolicy (NoFocus);
		setAcceptDrops (false);
		cursoron=false;
		edited=false;
		setCursor (arrowCursor);
	}
	
}

void GTTextWidget::slotCursorBlink()
{
    cursoron = !cursoron;
    repaintCell( line.count()-1, 0, false);
}

void GTTextWidget::focusInEvent( QFocusEvent * )
{
	if ( !blinktimer->isActive() )
		blinktimer->start( QApplication::cursorFlashTime() / 2, FALSE );
	cursoron = TRUE;
	repaintCell( line.count()-1, 0, false);
}

void GTTextWidget::focusOutEvent ( QFocusEvent * )
{
	blinktimer->stop();
	if ( cursoron) {
		cursoron = false;
		repaintCell( line.count()-1, 0, false);
		}
}

/* Overloaded QWidget member to limit editing keys and allow only
   limited ASCII keys to be sent to the server an transmitted */
void GTTextWidget::keyPressEvent( QKeyEvent *e )
{
  // only if edit is enabled, otherwise window is receive only
	if (edited) {
     // These functions move the window up by one row
     // when the scroll bar arrow up is pressed
		if (e->key() == Key_Up) {
    // topcell is rowAt(contentsY()
			scrollBy(0,-cellHeight());
			repaint (false);
			e->accept();
			return;
			}
     // These functions move the window down by one row
     // when the scroll bar arrow down is pressed
		if (e->key() == Key_Down) {
    // topcell is rowAt(contentsY()
			scrollBy(0,cellHeight());
			repaint (false);
			e->accept();
			return;
			}

		int key = e->ascii();
		cursoron=true;

			if (key == '\n' || key == '\r') {
				appendChar ('\n', currentcol);
        emit keyPressed ('\n');
        e->accept();
				return;
				}		

			if (key == '\b') {
				appendChar ('\b', currentcol);
				emit keyPressed ('\b');
        e->accept();
				return;
        }

			if (key >= 31 && key <= 255) {
				appendChar (key, currentcol);
				emit keyPressed (key);
        e->accept();
				return;
				}
	}
	e->ignore();
}
// Overloaded QWidget member
void GTTextWidget::dragEnterEvent(QDragEnterEvent* event)
{
  event->accept(QTextDrag::canDecode(event));
}
// Overloaded QWidget member
void GTTextWidget::dropEvent(QDropEvent* event)
{
	QString text;
	if ( QTextDrag::decode(event, text)) {
		char* tmp = strdup (text.latin1());
		appendString (tmp);
		free (tmp);
	}
		
}

void GTTextWidget::slotKey (char k)
{
	char b[2];
	b[1] = 0;
	b[0] = k;
	emit textAppended (b);
}

QString GTTextWidget::getMarkedText()
{
	int i;
	QString tmpstring;
	for (i=firstline;i<=lastline;i++)
	{
    if((int) line.count() <= i)
// deals with no text on line equivalent to i out of range
		if(line.at(i)==NULL){return tmpstring="";}
  		tmpstring = tmpstring + line.at(i)->text();
		if (firstline != lastline)
			tmpstring = tmpstring + "\n";
	}
	return tmpstring;
}
      // These functions move the window up by one page
     // when the scroll bar is pressed
   void GTTextWidget::slotPageUp()
{
   // contentsY() - visibleHeight()  equals the y pos of bottom of view

		((QScrollBar *)vscroll)->setValue(contentsY()+visibleHeight());
}
     // These functions move the window down by one page
     // when the scroll bar is pressed
  void GTTextWidget::slotPageDown()
{

		((QScrollBar *)vscroll)->setValue(contentsY()-visibleHeight());
}

void GTTextWidget::setColors (QColor col1, QColor col2)
{
	textcolor[0]=col1;
	textcolor[1]=col2;
	repaint();
}
/** Makes the Call check, but only if we use the 'transmit' Color. This is an Digimode specific functions, normal programs should'nt use it */

void GTTextWidget::doCallCheck()
{
	int pos;
	unsigned int i;
	QString text = line.last()->text();
	if (text.isNull()) return; // return if an invalid string!
	text = text.upper();

	/* Seek the " DE " in the string */

	pos = text.findRev (" DE ");
	if (pos==-1) return; /* none found */
	pos = pos + 4;

	text = text.mid (pos, 12);	
	text = text.stripWhiteSpace();
	pos = text.find (" ");
	if (pos >0)
  	text = text.left (pos);

	/* Okay, now check if the call contains at least a number and a char */

	int numnumber = text.contains (QRegExp ("[0-9]"));
	int numchar = text.contains (QRegExp ("[A-Z]"));
	if (numchar == 0 || numnumber == 0)
		return;


	/* look if we have only valid characters */

	for (i=0;i<text.length();++i) {
		char c = text.at(i).latin1();
		if (!((c>= 47 && c<= 57) || (c>=65 && c<=90)))
			return;
	}

  /* now check if we are reading our own call from a poor macro*/

  if (config.personals.call != "" && text.contains(config.personals.call, FALSE) >0) return;

  /* emit call if we have more the 3 good characters */
	if (text.length() >= 3) emit newCallsign (text);
}

void GTTextWidget::setCallCheckEnabled (bool s)
{
	callCheckEnabled=s;
}

/* This erases all the text */

void GTTextWidget::clear()
{
	line.clear();
	setNumRows (0);
	newLine();
	repaint();
}
