//
// C++ Interface: kpgdebugger
//
// Description: 
//
//
// Author: Lumir Vanek <lvanek@users.sourceforge.net>, (C) 2007
//
// Copyright: See COPYING file that comes with this distribution
//
//
#ifndef KPGDEBUGGER_H
#define KPGDEBUGGER_H

#include "kpgdebuggerbase.h"

#include <qpixmap.h>

#include <kparts/factory.h> // KPart Factory
#include <kate/view.h>      // Katepart view

#include "../kpgconnectioninthread.h"
#include "kpgdebuggingthread.h"
#include "kpgfunctiondebuginfo.h"

class KFind;
class KXMLGUIFactory;

class KPGConnection;
class KPGDatabase;
class KPGFunction;
class KPGDebuggerChildView;
class KPoGreView;


/**
  * PL pg/SQL debugger window
  *
  * @author Lumir Vanek <lvanek@users.sourceforge.net>
  */
class KPGDebugger : public KPGDebuggerBase, virtual public KXMLGUIClient
{
	Q_OBJECT
public:
	KPGDebugger(KPGDebuggerChildView *, KPoGreView *, KParts::Factory *, KXMLGUIFactory *, const PGSTD::string &, const QString &, KPGFunction *, const QString&, pqxx::oid);
    ~KPGDebugger();

	// Add yourself and Kate view to GUI factory
    void addToGuiFactory();
    
    // Remove yourself and Kate view from GUI factory
    void removeFromGuiFactory();
			
    // Set function source code
    void setEditorText(const QString &);
    
    // Get function source code
    const QString editorText() const;
    
    // Returns QTable with result
    QTable *tableResult() const { return m_pTableResult; }
    
    // Signalize, if result contain at least one row
    bool resultEmpty() const;
    
    // Is all rows fetched from resultset do m_pTableResult ?
    bool fetchedAllRows() const { return m_nTotalRows == m_nFetchedRows;  }
    
	// Returns true, if breakpoint is set to given line
	bool isBreakpointSet(unsigned int) const;
	
	// Returns PostgreSQL row identifier of debugged function
	pqxx::oid oidFunction() const { return m_oidFunction; }
	
	// Returns PostgreSQL row identifier of debugged function
	pqxx::oid oidDatabase() const { return m_oidDatabase; }
	
	// Returns name of the parent server
	const QString & serverName() const { return m_strServerName; }
	
	// Attach given function to debugger
	void attachFunction(pqxx::oid);
	
	// Return state of debugger
	KPGDebuggingThread::EState state() const { return m_threadDebugger.state(); }
	
	// Get Map of function sources and another info, for cache it
	const MapFunctionDebugInfo & mapFunctionDebugInfo() const { return m_threadDebugger.mapFunctionDebugInfo(); }
    
    // Returns TRUE if the debugger thread is running; otherwise returns FALSE.
    bool running() const { return  m_threadDebugger.running(); }
	
protected:	
	
	// Create Kate part view
	Kate::View* createKatePart(KParts::Factory*);
	
	// Display SQL result
	void displayResult();
	
	// Display variables
	bool displayVariables(const KPGVariableList &, bool);
	
	// Display stack
	void displayStack(const KPGStackList &);
	
	// Display active/reached/execution breakpoints marks, obtained from debugger frontend
	void displayBreakpoints(const KPGBreakpointList &);
	
    // Display disabled breakpoints marks, stored in MapFunctionDebugInfo
    void displayBreakpoints();
    
	// Receive event from KPGConnectionInThread
	virtual void customEvent(QCustomEvent *);
	
	// Enables GUI actions
	void enableGui();
	
	// Disable GUI actions
	void disableGui();
	
	// Clear all breakpoints marks in editor
	void clearBreakpointMarks() const;
		
	// Execute debugged function
	void executeFunction();
    
    // Fetch next X rows from large result
    void fetchNext();
    
    // Fetch all rows from large result  
    void fetchAll();

    // Find first occurence of text in result table
    void findInResultFirst(QStringList &);
        
    // Find next occurence of text in result table
    void findInResultNext();
    
    virtual void virtual_hook(  int id, void* data );
			
protected:
	    
	// Debugger actions
    KAction* m_pActRun;
	KAction* m_pActStop;
	KAction* m_pActStepOver;
	KAction* m_pActStepInto;
	KAction* m_pActToggleBreakPoint;
    KAction* m_pActFetchNext;
    KAction* m_pActFetchAll;
    KAction* m_pActFindInResult;
    KAction* m_pActFindNextInResult;
    
    // Common clipboard actions for result table
    KAction* m_pActResultCopyCell;
    KAction* m_pActResultCopyRow;
    KAction* m_pActResultCopyTableCsv;
    KAction* m_pActResultCopyTableXml;
    
    // Common clipboard actions for variables table
    KAction* m_pActVariablesCopyCell;
    KAction* m_pActVariablesCopyRow;
    KAction* m_pActVariablesDepositValue;
    
    // True, if this is in GUI factory
    bool m_bIsAddedToGuiFactory;
    
    // GUI factory
    KXMLGUIFactory *m_pXmlGuiFactory;
    
    // True, if user not entered function parameters yet
    bool m_bParametersNotEntered;

    // Connection options
	PGSTD::string m_strConnectionOptions;
    
    // Database function to execute
    QString m_strFunctionName;
    
    // Function namespace
    QString m_strNamespaceName;
        	
	// List of argument informations, available since PostgreSQL 8.1
    ListFunctionArguments m_listArguments;
	
	// Function returns set ?
	bool m_bReturnSet;
	   
	// Kate part editor
    Kate::View* m_pKateView;
	
	// Thread that run debugger queries on background
	KPGDebuggingThread m_threadDebugger;
	
	// Thread that run queries on background - debugged function
	KPGConnectionInThread m_connInThreadFunction;
	
	// Resultset to display - debugged function
	pqxx::result m_pqxxResultFunction;
				
	// Number of rows in result
	unsigned int m_nTotalRows; 
	
	// Number of fetched rows from result
	unsigned int m_nFetchedRows; 
	
	// Number of columns in result
	unsigned int m_nTotalCols; 
    
    // For searching in data tables and result tables
    QStringList m_listOfResultQuerySearchHistory;
	
	// PostgreSQL row identifier of debugged function
	pqxx::oid m_oidFunction;
		
	// PostgreSQL row identifier of database, from wich is debugged function
	pqxx::oid m_oidDatabase;
	
	// Name of the parent server
	QString m_strServerName;
	
	// PostgreSQL row identifier of currently debugged function
	pqxx::oid m_oidFunctionCurrent;
	
	// Line number of current line (0. based)
	unsigned int m_uiLineCurrent;
    
    // List of variables. Caching this list allow to highligth changed varibles by pixmapVariableChanged
    KPGVariableList m_listVariables;
    
    // Changed variable marker
    QPixmap pixmapVariableChanged;
    
private:
	int max(int a, int b) { return a > b ? a : b; }
	int min(int a, int b) { return a < b ? a : b; }
		
protected slots:
	
	// Process notice from connection in thread - debugger
	void slotProcessNoticeDebugger(const QString &);
	
	// Process notice from connection in thread - debugged function
    void slotProcessNoticeFunction(const QString &);
	
	//--- Debugger actions
    void slotExecute();
    void slotStop();
    void slotFetchNext();
    void slotFetchAll();
    void slotFindInResult();
    void slotFindNextInResult();
    void slotStepOver();
    void slotStepInto();
    void slotToggleBreakPoint();
    void slotVariablesDepositValue();
    
    //--- Common clibpoard actions for result QTable
    void slotCopyResultCell();
    void slotCopyResultRow();
    void slotCopyResultTableCsv();
    void slotCopyResultTableXml();
    
    //--- Common clibpoard actions for result QTable
    void slotCopyVariablesCell();
    void slotCopyVariablesRow();
    
    // Display popup menu for result table
    void slotContextMenuResultRequested(int, int, const QPoint &);
    
    // Display popup menu for variables table
    void slotContextMenuVariablesRequested(int, int, const QPoint &);
    
    // Find next occurence of text
    void slotFindInResultNext();
    
    // Highligth found text
    void slotHighlight( const QString &, int, int);
    
private:
    // A generic implementation of the "find" function
    KFind *m_pFind;
    
    int m_iRowToSearch;
    int m_iColToSearch;
};

#endif
