/***************************************************************************
 *   Copyright (C) 2008-2011 by Marcel Hasler                              *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This file is part of KOSD.                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation, either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program. If not, see <http://www.gnu.org/licenses/>.  *
 ***************************************************************************/

#include "kosd/kosd.h"

#include "kosdkeysd.h"
#include "kosdversion.h"
#include "mixer.h"
#include "powermanager.h"

#include <QDebug>

#include <KAboutData>
#include <KConfigGroup>
#include <KGlobalAccel>
#include <KLocale>
#include <KPluginFactory>
#include <KSharedConfig>

K_PLUGIN_FACTORY(KOsdKeysDFactory, registerPlugin<KOsdKeysD>();)
K_EXPORT_PLUGIN(KOsdKeysDFactory("kosd_keys"))

KOsdKeysD::KOsdKeysD(QObject* parent, const QList<QVariant>&)
    : KDEDModule(parent)
{
    KLocale::setMainCatalog("kosd");

    KAboutData aboutData("kosd_keys", "kosd", ki18n("KOSD"), KOSD_VERSION, ki18n("KOSD"), KAboutData::License_GPL,
                         ki18n("(C) 2008-2011 Marcel Hasler"), KLocalizedString(), 0, 0);
    aboutData.addAuthor(ki18n("Marcel Hasler"), ki18n("Author and Maintainer"), "mahasler@gmail.com");
    KComponentData componentData(&aboutData);

    connect(this, SIGNAL(moduleRegistered(const QDBusObjectPath&)), this, SLOT(parseConfiguration()));

    m_actions = new KActionCollection(this, componentData);
    m_actions->setConfigGlobal(true);

    // Add volume up
    KAction *newAct = m_actions->addAction("increaseVolume", this);
    newAct->setText(i18n("Increase Volume"));
    newAct->setGlobalShortcut(KShortcut("Volume Up"), KAction::DefaultShortcut | KAction::ActiveShortcut);
    connect(newAct, SIGNAL(triggered()), this, SLOT(increaseVolume()));

    // Add volume down
    newAct = m_actions->addAction("decreaseVolume", this);
    newAct->setText(i18n("Decrease Volume"));
    newAct->setGlobalShortcut(KShortcut("Volume Down"), KAction::DefaultShortcut | KAction::ActiveShortcut);
    connect(newAct, SIGNAL(triggered()), this, SLOT(decreaseVolume()));

    // Add volume mute
    newAct = m_actions->addAction("muteVolume", this);
    newAct->setText(i18n("Mute Volume"));
    newAct->setGlobalShortcut(KShortcut("Volume Mute"), KAction::DefaultShortcut | KAction::ActiveShortcut);
    connect(newAct, SIGNAL(triggered()), this, SLOT(muteVolume()));

    // Add show volume
    newAct = m_actions->addAction("showVolume", this);
    newAct->setText(i18n("Show Volume"));
    newAct->setGlobalShortcut(KShortcut(), KAction::DefaultShortcut | KAction::ActiveShortcut);
    connect(newAct, SIGNAL(triggered()), this, SLOT(showVolume()));

    // Add increase brightness
    newAct = m_actions->addAction("increaseBrightness", this);
    newAct->setText(i18n("Increase Brightness"));
    newAct->setGlobalShortcut(KShortcut(), KAction::DefaultShortcut | KAction::ActiveShortcut);
    connect(newAct, SIGNAL(triggered()), this, SLOT(increaseBrightness()));

    // Add decrease brightness
    newAct = m_actions->addAction("decreaseBrightness", this);
    newAct->setText(i18n("Decrease Brightness"));
    newAct->setGlobalShortcut(KShortcut(), KAction::DefaultShortcut | KAction::ActiveShortcut);
    connect(newAct, SIGNAL(triggered()), this, SLOT(decreaseBrightness()));

    // Add show brightness
    newAct = m_actions->addAction("showBrightness1", this);
    newAct->setText(i18n("Show Brightness"));
    newAct->setGlobalShortcut(KShortcut("Monitor Brightness Up"), KAction::DefaultShortcut | KAction::ActiveShortcut);
    connect(newAct, SIGNAL(triggered()), this, SLOT(showBrightness()));

    // Add show brightness
    newAct = m_actions->addAction("showBrightness2", this);
    newAct->setText(i18n("Show Brightness"));
    newAct->setGlobalShortcut(KShortcut("Monitor Brightness Down"), KAction::DefaultShortcut | KAction::ActiveShortcut);
    connect(newAct, SIGNAL(triggered()), this, SLOT(showBrightness()));

    // Add show battery charge
    newAct = m_actions->addAction("showBatteryCharge", this);
    newAct->setText(i18n("Show Battery Charge"));
    newAct->setGlobalShortcut(KShortcut("Battery"), KAction::DefaultShortcut | KAction::ActiveShortcut);
    connect(newAct, SIGNAL(triggered()), this, SLOT(showBattery()));
}


KOsdKeysD::~KOsdKeysD()
{
}


void KOsdKeysD::increaseVolume()
{
    int volume = Mixer::increaseVolume();
    showVolume(volume, false);
}


void KOsdKeysD::decreaseVolume()
{
    int volume = Mixer::decreaseVolume();

    bool muted = false;
    if (volume <= 0)
    {
        Mixer::setMuted(true);
        muted = true;
    }

    showVolume(volume, muted);
}


void KOsdKeysD::muteVolume()
{
    bool muted = Mixer::toggleMuted();
    showVolume(Mixer::volume(), muted);
}


void KOsdKeysD::showVolume()
{
    unsigned int volume = Mixer::volume();
    bool muted = Mixer::isMuted();
    showVolume(volume, muted);
}


void KOsdKeysD::increaseBrightness()
{
    PowerManager::increaseBrightness();
    showBrightness();
}


void KOsdKeysD::decreaseBrightness()
{
    PowerManager::decreaseBrightness();
    showBrightness();
}


void KOsdKeysD::showBrightness()
{
    if (m_useSvg)
        showSvgBrightness();
    else
        showPixmapBrightness();
}
    

void KOsdKeysD::showBattery()
{
    if (m_useSvg)
        showSvgBattery();
    else
        showPixmapBattery();
}


void KOsdKeysD::parseConfiguration()
{
    KSharedConfigPtr config = KSharedConfig::openConfig("kosd_keysrc");
    KConfigGroup group(config, "General");
    m_useSvg = (group.readEntry("UseSvg", 1) == 1);

    QString controlName = group.readEntry("MixerControl", "Master");
    Mixer::setControlName(controlName);
}


void KOsdKeysD::showVolume(unsigned int volume, bool muted)
{
    if (m_useSvg)
        showSvgVolume(volume, muted);
    else
        showPixmapVolume(volume, muted);
}


void KOsdKeysD::showPixmapVolume(unsigned int volume, bool muted)
{
    if (muted)
        KOsd::showCustom("audio-volume-muted", i18n("Volume Muted"), 0);
    else
    {
        QString pixmap;
        if (volume >= 75)
            pixmap = "audio-volume-high";
        else if (volume >= 25)
            pixmap = "audio-volume-medium";
        else
            pixmap = "audio-volume-low";

        KOsd::showCustom(pixmap, i18n("Volume<br>%1%", volume), volume);
    }
}


void KOsdKeysD::showSvgVolume(unsigned int volume, bool muted)
{
    KOsd::showVolume(volume, muted);
}


void KOsdKeysD::showPixmapBrightness()
{
    int brightness = PowerManager::brightness();
    KOsd::showCustom("preferences-desktop-display",  i18n("Brightness<br>%1%", brightness), brightness);
}


void KOsdKeysD::showSvgBrightness()
{
    int brightness = PowerManager::brightness();
    KOsd::showBrightness(brightness);
}


void KOsdKeysD::showPixmapBattery()
{
    int charge = PowerManager::batteryCharge();
    bool plugged = PowerManager::acAdapterPlugged();

    QString pixmap;
    if (charge > 95)
        pixmap = plugged ? "battery-charging" : "battery-100";
    else if (charge > 80)
        pixmap = plugged ? "battery-charging-080" : "battery-080";
    else if (charge > 50)
        pixmap = plugged ? "battery-charging-060" : "battery-060";
    else if (charge > 20)
        pixmap = plugged ? "battery-charging-040" : "battery-040";
    else if (charge > 10)
        pixmap = plugged ? "battery-charging-low" : "battery-low";

    KOsd::showCustom(pixmap, i18n("Battery Charge<br>%1%", charge), charge);
}


void KOsdKeysD::showSvgBattery()
{
    int charge = PowerManager::batteryCharge();
    bool plugged = PowerManager::acAdapterPlugged();
    KOsd::showBattery(charge, plugged);
}

#include "kosdkeysd.moc"
