//                       -*- mode: C++ -*-
//
// Copyright(C) 2005,2006,2007 Stefan Siegl <stesie@brokenpipe.de>
// Copyright(C) 2006 Martin Albrecht <malb@informatik.uni-bremen.de>
// Copyright(C) 2007 Christian Dietrich <stettberger@brokenpipe.de>
// kopete_silc - silc plugin for kopete messenger
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef KOPETESILC_SILCBUDDYCONTACT_H
#define KOPETESILC_SILCBUDDYCONTACT_H

#include <kaction.h>
#include <kactionclasses.h>
#include <qdatetime.h>

#include "silccontact.h"

class SilcBuddyAttributes;
class SilcChannelContact;

/**
 * @brief Reimplementation of Kopete::Contact class, used for SILC Users
 *
 * Every user of the connected SILC network will have it's very own
 * SilcBuddyContact class associated as soon as he or she is known to
 * kopete_silc.  This is as soon as he or she joins a joined chat session,
 * incoming private message, etc.
 * 
 * @author Stefan Siegl <ssiegl@gmx.de>
 */
class SilcBuddyContact : public SilcContact {
  Q_OBJECT;

public:

  SignatureStatus verifySignature(SilcTK::SilcMessagePayload payload);

  SilcBuddyContact(SilcAccount *account,
		   const QString &nickname,
		   const QString &fingerprint,
		   Kopete::MetaContact *meta,
		   const QString &icon = QString::null);
  virtual ~SilcBuddyContact();


  /**
   * @brief return a pointer to the first associated client entry
   */
  inline SilcTK::SilcClientEntry clientEntry(unsigned int i = 0)
  {
    if(_clientEntries.empty() || i > _clientEntries.count()) return NULL;
    return _clientEntries[i];
  }

  inline const SilcTK::SilcClientEntry clientEntry(unsigned int i = 0) const 
  {
    if(_clientEntries.empty() || i > _clientEntries.count()) return NULL;
    return _clientEntries[i];
  }

  const SilcTK::SilcClientEntry clientEntry(SilcChannelContact *ch) const;
  
  SilcTK::SilcClientEntry clientEntry(SilcChannelContact *ch);

  KDE_DEPRECATED inline const unsigned int clientEntriesCount(void) const
  {
    return _clientEntries.count();
  }

  /**
   * @brief reset the list of associated client entries, add the provided one
   */
  void setClientEntry(SilcTK::SilcClientEntry e);

  /**
   * @brief add the provided client entry to the list of associated entries
   */
  void addClientEntry(SilcTK::SilcClientEntry e);

  /**
   * @brief remove the provided client entry from the list of associated ones
   */
  void removeClientEntry(SilcTK::SilcClientEntry e);


  inline const QString fingerprint(void) const 
  {
    return contactId().mid(1);
  }

  bool fpTrusted(void) const
  {
    return _fpTrusted;
  }

  void setFpTrusted(bool trusted);

  inline const QStringList &channelList(void) const
  {
    return _channels;
  }

  inline bool allowRichText(void) const { return _allowRichText; };
  inline void setAllowRichText(bool v) { _allowRichText = v; };

  void setChannelList(QStringList &list) 
  {
    _channels = list;
  }

  static QString convFingerprint(const char *);

  static void silc_private_message(SilcTK::SilcClient client,
				   SilcTK::SilcClientConnection conn,
				   SilcTK::SilcClientEntry sender,
				   SilcTK::SilcMessagePayload payload,
				   SilcTK::SilcMessageFlags flags,
				   const unsigned char *message,
				   SilcTK::SilcUInt32 message_len);

  static const QString publicKeyPath(QString fp);
  const QString publicKeyPath(void) const;
  static bool havePublicKey(QString fp);
  bool havePublicKey(void);
  bool whoami(void);
  void updateWhois(QString username, QString realname);

  /**
   * @brief either add or remote the buddy from the watch list
   */
  bool watchme(bool watch);
  static bool watchme_callback(SilcTK::SilcClient client,
                 SilcTK::SilcClientConnection conn,
                 SilcTK::SilcCommand command,
                 SilcTK::SilcStatus status,
                 SilcTK::SilcStatus error,
                 void *context,
                 va_list ap);

  inline bool watched(void) const  { return _watched; };

  virtual void serialize(QMap<QString, QString>&, QMap<QString, QString>&);

  virtual QPtrList<KAction> *customContextMenuActions(Kopete::ChatSession *);

  virtual void sendFile(const KURL &sourceURL = KURL(),
			const QString &fileName = QString::null,
			uint fileSize = 0L);

  virtual void slotUserInfo(void);

  void mimeAlternateToMsg( Kopete::Message &, SilcTK::SilcMime, bool) const;

  /* this data is provided by WHOIS */
  QString _username;
  QString _realname;
  QStringList _channels;
  QDateTime _update;

  
  /**
   * @brief send file as mime to contact
   */
  virtual void sendFileAsMime(const QString &fileName);

  /**
   * @brief SilcBuddyAttributes of this buddy
   */
  SilcBuddyAttributes *attributes;

signals:
  void signalWhois(QString nickname, QString username, QString realname);

public slots:
  void slotIsOp(void);
  void slotKick(void);

protected slots:
  /**
   * @brief send private message to the SILC network
   */
  virtual void slotSendMessage(Kopete::Message &, Kopete::ChatSession *);

private slots:
  /**
   * @brief reset _clientEntry to NULL if we went offline
   */
  void slotOnlineStatusChanged(Kopete::Contact *contact,
			       const Kopete::OnlineStatus &status,
			       const Kopete::OnlineStatus &oldStatus);

  void slotPropertyChanged(Kopete::Contact *contact, const QString &key,
			   const QVariant &oldValue,
			   const QVariant &newValue);

private:
  QValueList<SilcTK::SilcClientEntry> _clientEntries;
  bool _fpTrusted;
  bool _watched;
  bool _allowRichText;

  Kopete::ChatSession *activeManager;
  KToggleAction *actionIsOp;
  KAction *actionKick;
};

/**
 * Pending Buddies have this format
 *
 */

class SilcBuddyContactData {
  public:
  QString nickname;
  QString finger;
  Kopete::MetaContact *meta;
  SilcAccount *account; 

  SilcBuddyContactData(SilcAccount * account, 
                       QString nickname,
                       QString finger,
                       Kopete::MetaContact *meta);
  inline QString &nickName() { return nickname; }

};

#endif // KOPETESILC_SILCBUDDYCONTACT_H
