/***************************************************************************
                          mymoneyscheduled.h
                             -------------------
    copyright            : (C) 2000-2002 by Michael Edwardes
    email                : mte@users.sourceforge.net
                           Javier Campos Morales <javi_c@users.sourceforge.net>
                           Felix Rodriguez <frodriguez@users.sourceforge.net>
                           John C <thetacoturtle@users.sourceforge.net>
                           Thomas Baumgart <ipwizard@users.sourceforge.net>
                           Kevin Tambascio <ktambascio@users.sourceforge.net>
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef MYMONEYSCHEDULED_H
#define MYMONEYSCHEDULED_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// ----------------------------------------------------------------------------
// QT Includes

#include <qstringlist.h>
#include <qmap.h>
#include <qdatetime.h>


// ----------------------------------------------------------------------------
// Project Includes

#include "mymoneytransaction.h"
#include "mymoneyaccount.h"
#include <kmymoney/export.h>

/**
  * @author Michael Edwardes
  */

/**
  * This class represents a schedule. (A series of bills, deposits or
  * transfers).
  *
  * @short A class to represent a schedule.
  * @see MyMoneyScheduled
  */
class KMYMONEY_EXPORT MyMoneySchedule {
public:
  /**
    * This enum is used to describe all the possible schedule frequencies.
    * The special entry, OCCUR_ANY, is used to combine all the other types.
    */
  enum occurenceE { OCCUR_ANY=0, OCCUR_ONCE=1, OCCUR_DAILY=2, OCCUR_WEEKLY=4, OCCUR_FORTNIGHTLY=8,
                    OCCUR_EVERYOTHERWEEK=16, OCCUR_MONTHLY=32, OCCUR_EVERYFOURWEEKS=64,
                    OCCUR_EVERYOTHERMONTH=128, OCCUR_EVERYTHREEMONTHS=256,
                    OCCUR_TWICEYEARLY=1024, OCCUR_EVERYOTHERYEAR=2048, OCCUR_QUARTERLY=4096,
                    OCCUR_EVERYFOURMONTHS=8192, OCCUR_YEARLY=16384
                     };

  /**
    * This enum is used to describe the schedule type.
    */
  enum typeE {  TYPE_ANY=0, TYPE_BILL=1, TYPE_DEPOSIT=2, TYPE_TRANSFER=4, TYPE_LOANPAYMENT=5 };

  /**
    * This enum is used to describe the schedule's payment type.
    */
  enum paymentTypeE { STYPE_ANY=0, STYPE_DIRECTDEBIT=1, STYPE_DIRECTDEPOSIT=2,
                      STYPE_MANUALDEPOSIT=4, STYPE_OTHER=8,
                      STYPE_WRITECHEQUE=16 };

  /**
    * This enum is used by the auto-commit functionality.
    *
    * Depending upon the value of m_weekdayOption the schedule can
    * be entered on a different date
  **/
  enum weekendOptionE { MoveFriday=0, MoveMonday=1, MoveNothing=2 };

  /**
    * Standard constructor
    */
  MyMoneySchedule();

  /**
    * Constructor for initialising the object.
    *
    * Please note that the optional fields are not set and the transaction
    * MUST be set before it can be used.
    */
  MyMoneySchedule(const QString& name, typeE type, occurenceE occurence, paymentTypeE paymentType,
        const QDate& startDate, const QDate& endDate, bool fixed, bool autoEnter);

  /**
    * Standard destructor
    */
  ~MyMoneySchedule() {
  }

  /**
    * Simple get method that returns the occurence frequency.
    *
    * @return occurenceE The instance frequency.
    */
  occurenceE occurence(void) const { return m_occurence; }

  /**
    * Simple get method that returns the schedule type.
    *
    * @return typeE The instance type.
    */
  typeE type(void) const { return m_type; }

  /**
    * Simple get method that returns the schedule startDate.
    *
    * @return QDate The instance startDate.
    */
  QDate startDate(void) const { return m_startDate; }

  /**
    * Simple get method that returns the schedule paymentType.
    *
    * @return paymentTypeE The instance paymentType.
    */
  paymentTypeE paymentType(void) const { return m_paymentType; }

  /**
    * Simple get method that returns true if the schedule is fixed.
    *
    * @return bool To indicate whether the instance is fixed.
    */
  bool isFixed(void) const { return m_fixed; }

  /**
    * Simple get method that returns true if the schedule will end
    * at some time.
    *
    * @return bool Indicates whether the instance will end.
    */
  bool willEnd(void) const { return m_endDate.isValid(); }

  /**
    * Simple get method that returns the number of transactions remaining.
    *
    * @return int The number of transactions remaining for the instance.
    */
  int transactionsRemaining(void) const;

  /**
    * Simple get method that returns the schedule end date.
    *
    * @return QDate The end date for the instance.
    */
  const QDate endDate(void) const { return m_endDate; }

  /**
    * Simple get method that returns true if the transaction should be
    * automatically entered into the register.
    *
    * @return bool Indicates whether the instance will be automatically entered.
    */
  const bool autoEnter(void) const { return m_autoEnter; }

  /**
    * Simple get method that returns the transaction data for the schedule.
    *
    * @return MyMoneyTransaction The transaction data for the instance.
    */
  const MyMoneyTransaction& transaction(void) const { return m_transaction; }

  /**
    * Simple method that returns the schedule id.
    *
    * @return QString The instances id.
    */
  const QCString id(void) const { return m_id; }

  /**
    * Simple method that returns the schedule last payment.
    *
    * @return QDate The last payment for the instance.
    */
  const QDate lastPayment(void) const { return m_lastPayment; }

  /**
    * Get the weekendOption that determines how the schedule check code
    * will enter transactions that occur on a weekend.
    *
    * This not used by MyMoneySchedule but by the support code.
  **/
  weekendOptionE weekendOption(void) const { return m_weekendOption; }

  /**
    * Simple method that sets the frequency for the schedule.
    *
    * @param occ The new occurence (frequency).
    * @return none
    */
  void setOccurence(occurenceE occ);

  /**
    * Simple method that sets the type for the schedule.
    *
    * @param type The new type.
    * @return none
    */
  void setType(typeE type);

  /**
    * Simple method that sets the start date for the schedule.
    *
    * @param date The new start date.
    * @return none
    */
  void setStartDate(const QDate& date);

  /**
    * Simple method that sets the payment type for the schedule.
    *
    * @param type The new payment type.
    * @return none
    */
  void setPaymentType(paymentTypeE type);

  /**
    * Simple method to set whether the schedule is fixed or not.
    *
    * @param fixed boolean to indicate whether the instance is fixed.
    * @return none
    */
  void setFixed(bool fixed);

  /**
    * Simple method that sets the transaction for the schedule.
    *
    * @param transaction The new transaction.
    * @return none
    */
  void setTransaction(const MyMoneyTransaction& transaction);

  /**
    * Simple set method to set the end date for the schedule.
    *
    * @param date The new end date.
    * @return none
    */
  void setEndDate(const QDate& date);

  /**
    * Simple set method to set whether this transaction should be automatically
    * entered into the journal whenever it is due.
    *
    * @param autoenter boolean to indicate whether we need to automatically
    *                  enter the transaction.
    * @return none
    */
  void setAutoEnter(bool autoenter);

  /**
    * Simple set method to set the schedule's id.  DO NOT USE, EVER!
    *
    * @param id The instances id.
    * @return none
    *
    * @note DO NOT USE, EVER IN APPLICATION CODE! This method is for test purposes only!
    */
  void setId(const QCString& id)
    { m_id = id; }

  /**
    * Simple set method to set the schedule's last payment.
    *
    * @param date The instances last payment date.
    * @return none
    */
  void setLastPayment(const QDate& date);

  /**
    * Set the weekendOption that determines how the schedule check code
    * will enter transactions that occur on a weekend. The following values
    * are valid:
    *
    * - MoveNothing: don't modify date
    * - MoveFriday: modify the date to the previous friday
    * - MoveMonday: modify the date to the following monday
    *
    * If an invalid option is given, the option is set to MoveNothing.
    *
    * @param option See list in description
    * @return none
    *
    * @note This not used by MyMoneySchedule but by the support code.
    **/
  void setWeekendOption(const weekendOptionE option);

  /**
    * Validates the schedule instance.
    *
    * Makes sure the paymentType matches the type and that the required
    * fields have been set.
    *
    * @param id_check if @p true, the method will check for an empty id.
    *                 if @p false, this check is skipped. Default is @p true.
    *
    * @return If this method returns, all checks are passed. Otherwise,
    *         it will throw a MyMoneyException object.
    *
    * @exception MyMoneyException with detailed error information is thrown
    *            in case of failure of any check.
    */
  void validate(bool id_check=true) const;

  /**
    * Calculates the date of the next payment.
    *
    * @param refDate the reference date from which the next payment
    *                date will be calculated (defaults to current date)
    *
    * @return QDate The date the next payment is due. This date is
    *         always past @a refDate.  In case of an error or
    *         if there is no more payment then an empty/invalid QDate()
    *         will be returned.
    */
  QDate nextPayment(const QDate& refDate = QDate::currentDate()) const;

  /**
    * Calculates the dates of the payment over a certain period of time.
    *
    * An empty list is returned for no payments or error.
    *
    * @param startDate The start date for the range calculations
    * @param endDate The end date for the range calculations.
    * @return QValueList<QDate> The dates on which the payments are due.
    */
  QValueList<QDate> paymentDates(const QDate& startDate, const QDate& endDate) const;

  /**
    * Returns the instances name
    *
    * @return The name
    */
  const QString name(void) const { return m_name; }

  /**
    * Changes the instance name
    *
    * @param nm The new name
    * @return none
    */
  void setName(const QString& nm);

  bool operator ==(const MyMoneySchedule& right);

  bool operator <(const MyMoneySchedule& right);

  const MyMoneyAccount account(int cnt = 1) const;
  const MyMoneyAccount transferAccount(void) const { return account(2); };
  QDate dateAfter(int transactions) const;

  bool isOverdue() const;
  bool isFinished() const;
  bool hasRecordedPayment(const QDate&) const;
  void recordPayment(const QDate&);
  QValueList<QDate> recordedPayments(void) const { return m_recordedPayments; }

private:
  /**
    * This method forces the day of the passed @p date to
    * be the day of the start date of this schedule kept
    * in m_startDate. It is internally used when calculating
    * the payment dates over several periods.
    *
    * @param date reference to QDate object to be checked and adjusted
    */
  void fixDate(QDate& date) const;

private:
  /// Its occurence
  occurenceE m_occurence;

  /// Its type
  typeE m_type;

  /// The date the schedule commences
  QDate m_startDate;

  /// The payment type
  paymentTypeE m_paymentType;

  /// Can the amount vary
  bool m_fixed;

  /// The, possibly estimated, amount plus all other relevant details
  MyMoneyTransaction m_transaction;

  /// The last transaction date if the schedule does end at a fixed date
  QDate m_endDate;

  /// Enter the transaction into the register automatically
  bool m_autoEnter;

  /// The internal id.
  QCString m_id;

  /// Internal date used for calculations
  QDate m_lastPayment;

  /// The name
  QString m_name;

  /// The recorded payments
  QValueList<QDate> m_recordedPayments;

  /// The weekday option
  weekendOptionE m_weekendOption;
};

#endif
