/***************************************************************************
 *   Copyright (C) 2004 by Marco Gulino                                    *
 *   marco.gulino@gmail.com                                                *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <kabc/addressbook.h>
#include <klocale.h>
#include <qdatetime.h> 
#include <math.h>

#include "alphabets.h"
#include "c_sms.h"
#include "gsm.h"
extern GSM* d_gsm;

#include <iostream>
using namespace std;

c_SMS::c_SMS(QObject *parent, const char *name)
 : QObject(parent, name)
{
	inList=false;
}

c_SMS::c_SMS(QStringList numbers, QStringList text)
{
	sl_Numbers=numbers;
	sl_TextList=text;
	inList=false;
}

c_SMS::c_SMS(QString data, QString memSlot, bool toDecode, bool pdu, QObject *parent, const char *name)
 : QObject(parent, name)
{
	inList=false;
	initSMS(data, memSlot, toDecode, pdu);
}


c_SMS::~c_SMS()
{
}


#include "c_sms.moc"


/*!
    \fn c_SMS::initSMS(QString data)
 */
void c_SMS::initSMS(QString data, QString memslot, bool toDecode, bool pdu)
{
	setMemSRC( memslot );
	s_ID=memslot + "-%1";
	s_ID=s_ID.arg ( data.section(',', 0, 0).toInt(), 3);
	i_Type=-1;
	if ( pdu)
	{
		i_Type=data.section(',',1,1).toInt();
		QString sms_center, dc;
		int i=0;
		s_Header=data.section('\n',0,0).stripWhiteSpace();
		QString s_data=data.section('\n',1).stripWhiteSpace();
		// Finding SMS Center informations
		i=s_data.left(2).toInt(NULL, 16);
		s_data=s_data.mid(2);
		if(i>0)
		{
			sms_center = c_SMS::getPDUNum( s_data.mid(2,i*2) );
			s_data=s_data.mid(i*2);
		}
		s_data=s_data.mid(2); /* removing SMS_Submit or SMS_Deliver octet */ /// TODO Do something with this
		if( i_Type==STO_SENT || i_Type==STO_UNSENT )
		{
// 			kdDebug() << "Switching to outgoing SMS mode for PDU decoding" << endl;
			s_data=s_data.mid(2); /* TP-Message-Reference */ /// TODO Do something with this
			i=s_data.left(2).toInt(NULL, 16); /* Finding dest numbers */ /// TODO Enable multi-numbers support
			s_data=s_data.mid(2);
			if(i>0)
			{
				if(i%2) i++;
				sl_Numbers = c_SMS::getPDUNum( s_data.mid(2,i) );
				s_data=s_data.mid(i+2);
			} else s_data=s_data.mid(2);
			s_data=s_data.mid(2); /* Removing Protocol Identifier (TP-PID) */ /// TODO Do something with this
			dc=s_data.left(2); // Data coding scheme
			s_data=s_data.mid(2);
			s_data=s_data.mid(2); /* TP-Validity-Period. */ /// TODO Do something with this
		}
		else
		{
// 			kdDebug() << "Switching to incoming SMS mode for PDU decoding" << endl;
			i=s_data.left(2).toInt(NULL, 16);
			s_data=s_data.mid(2); /* Finding sender number */
			if(i>0)
			{
				if(i%2) i++;
				sl_Numbers = c_SMS::getPDUNum( s_data.mid(2,i) );
				s_data=s_data.mid(i+2);
			} else s_data=s_data.mid(2);
			dc=s_data.mid(2,2); // Data coding scheme
			s_data=s_data.mid(4);
			// Timestamp handling is here
			smsDateString=c_SMS::getPDUDate( s_data.left(14) );
			s_data=s_data.mid(14);
		}
		sl_TextList=decodePDUText( s_data, dc );
	} else
	{
		if( data.section( ',', 1, 1).contains( "REC READ" ) >0 ) i_Type = c_SMS::REC_READ;
		if( data.section( ',', 1, 1).contains( "REC UNREAD") >0 ) i_Type = c_SMS::REC_UNREAD;
		if( data.section( ',', 1, 1).contains( "STO SENT") >0 ) i_Type = c_SMS::STO_SENT;
		if( data.section( ',', 1, 1).contains( "STO UNSENT" ) >0 ) i_Type = c_SMS::STO_UNSENT;
		sl_Numbers=QStringList::split( ' ', data.section('\n',0,0).section( ',', 2, 2).replace("\"","").stripWhiteSpace() );
		if(toDecode)
			sl_TextList=QStringList( GSM::decodeText(data.right( data.length() - data.find( '\n' ) ).stripWhiteSpace() ) );
		else 
			sl_TextList=QStringList( data.right( data.length() - data.find( '\n' ) ).stripWhiteSpace() );
		
		if(i_Type==c_SMS::REC_UNREAD) notified=false; else notified=true;
		// Now we need to find date/time, if present...
		for(int i=0; i<=data.section('\n',0,0).contains(','); i++)
		{
			if( data.section(',',i,i).contains('/')>0 ) 
			{
				smsDateString=data.section('\n',0,0).section(',',i,i+1).replace('\"',"").stripWhiteSpace();
			}
		}
	}
	if(smsDateString.isEmpty() ) smsDateString=i18n("<no date>");
}

c_SMS* c_SMS::findSMS( QPtrList<c_SMS>* smsList, int id)
{
    QPtrListIterator<c_SMS> it( *smsList );
	if (it.isEmpty() ) return NULL;
	it.toFirst();
    c_SMS *curSMS;
    while ( (curSMS = it.current()) != 0 ) {
		if(curSMS->id() == id && curSMS!=NULL) return curSMS;
        ++it;
    }
    return NULL;
}

QStringList c_SMS::getParsedNumList()
{
	QStringList parsedNumList;
    for ( QStringList::Iterator it = sl_Numbers.begin(); it != sl_Numbers.end(); ++it ) {
		parsedNumList+=d_gsm->getNameFromNumber(*it);
	}

	return parsedNumList;
}


/*!
    \fn c_SMS::toString()
 */
QString c_SMS::toSpeechString()
{
	QString out;
	switch( i_Type ){
	case REC_READ:
		out+=i18n("Received SMS");
		break;
	case REC_UNREAD:
		out+=i18n("Unread SMS");
		break;
	case STO_SENT:
		out+=i18n("Sent SMS");
		break;
	case STO_UNSENT:
		out+=i18n("Stored SMS");
		break;
	}
	out+=".\n";
	out+=i18n("Id") + QString(": ") + s_ID + "\n";
	out+=i18n("Date") + QString(": ") + 
// QDateTime.fromString(smsDateString)
smsDateString.replace("<","").replace(">","").replace("/"," ").replace(":"," ") 
+ "\n";
	out+=i18n("Numbers") + ": ";
	for(QStringList::Iterator it=sl_Numbers.begin(); it!=sl_Numbers.end(); ++it )
	{
		out+=d_gsm->getNameFromNumber( *it ) + " ";
		for(unsigned int i=0; i< (*it).length(); i++)
			out+=(*it).at(i) + " ";
		if( *it!=sl_Numbers.last() ) out+=", ";
	}
	out+="\n";
	out+=sl_TextList.join("\n");
	return out;
}


QString c_SMS::toString()
{
	QString out;
	switch( i_Type ){
	case REC_READ:
		out+=i18n("Received SMS");
		break;
	case REC_UNREAD:
		out+=i18n("Unread SMS");
		break;
	case STO_SENT:
		out+=i18n("Sent SMS");
		break;
	case STO_UNSENT:
		out+=i18n("Stored SMS");
		break;
	}
	out+=".\n";
	out+=i18n("Id") + QString(": ") + s_ID + "\n";
	out+=i18n("Date") + QString(": ") + smsDateString + "\n";
	out+=i18n("Numbers") + ": ";
	for(QStringList::Iterator it=sl_Numbers.begin(); it!=sl_Numbers.end(); ++it )
	{
		out+=d_gsm->getNameFromNumber( *it ) + " <" + *it + ">";
		if( *it!=sl_Numbers.last() ) out+=", ";
	}
	out+="\n";
	out+=sl_TextList.join("\n");
	return out;
}



/*!
    \fn c_SMS::getPDUNum(QString numString)
 */
const QString c_SMS::getPDUNum(QString numString)
{
	QString out="+";
	for(unsigned int i=0; i < numString.length(); i+=2)
	{
		out+=numString.at(i+1);
		out+=numString.at(i);
	}
	out.replace("F","");
	return out;
}


/*!
    \fn c_SMS::getPDUDate(QString strDate)
 */
const QString c_SMS::getPDUDate(QString strDate)
{
	int y, m, d, hh, mm, ss;
	y=strDate.at(1).digitValue()*10 + strDate.at(0).digitValue();
	if(y<70) y+=2000; else y+=1900;
	m=strDate.at(3).digitValue()*10 + strDate.at(2).digitValue();
	d=strDate.at(5).digitValue()*10 + strDate.at(4).digitValue();
	hh=strDate.at(7).digitValue()*10 + strDate.at(6).digitValue();
	mm=strDate.at(9).digitValue()*10 + strDate.at(8).digitValue();
	ss=strDate.at(11).digitValue()*10 + strDate.at(10).digitValue();
	return QString("%1/%2/%3 %4:%5:%6").arg(d).arg(m).arg(y).arg(hh).arg(mm).arg(ss);
}


/*!
    \fn c_SMS::strToBits(const char* string)
 */
const char* c_SMS::strToBits(QString strNum)
{
	QString out_str="%1";
	out_str=out_str.arg( strNum.toInt(0,16), 8, 2 ).replace(" ","0");
	char* binArray=new char[8];
	for(int i=0; i<8; i++)
		binArray[i]=out_str.at(/*7-*/i).digitValue();
	return binArray;
}


/*!
    \fn c_SMS::bitsToChar(bool* bits)
 */
const QChar c_SMS::bitsToChar(const char* bits)
{
	char out=0;
	for(int i=6; i>=0; i--)
	{
		out+=( bits[6-i]* (int) pow(2.0,i) );
	}
	return out;
}


const QString c_SMS::getFrom7bits(QString pduText, unsigned int length)
{
	kdDebug() << "SMS 7bit coded" << endl;
	char oldBits[7];
	char tempBits[16];
	char newBits[7];
	QString out;
	memset(newBits, -1, 7);
	for(unsigned int i=0; i*2<pduText.length(); i++)
	{
		memcpy( tempBits,strToBits( pduText.mid(i*2, 2) ), 8 );
		if(i%7 >0) memcpy( &tempBits[8], oldBits, i%7);
		memcpy( oldBits, tempBits, /*(i+1)%7 */ 7);
		memcpy( newBits, &tempBits[(i%7)+1] , 7);
		out+=bitsToChar( newBits );
		if(i>0 && (i+1)%7==0 && out.length() < length ) out+=bitsToChar(oldBits);
	}
	return fromGSMAlphabet( out );
}

/*!
    \fn c_SMS::getFrom8bits(QString strText)
 */
const QString c_SMS::getFrom8bits(QString strText)
{
	QString out;
	kdDebug() << "SMS 8bit coded" << endl;
	for(unsigned int i=0; i*2<strText.length(); i++)
	{
		out+=QChar( strText.mid(i*2,2).toInt(0, 16) );
	}
	return out;
}


/*!
    \fn c_SMS::getFrom16bits(QString strText)
 */
const QString c_SMS::getFrom16bits(QString strText)
{
	kdDebug() << "SMS 16bit coded" << endl;
	QString out;
	for(unsigned int i=0; i*4<strText.length(); i++)
	{
		out+=QChar( strText.mid(i*4,4).toInt(0, 16) );
	}
	return out;
}


/*!
    \fn c_SMS::decodePDUText(QString text, QString encoding)
 */
const QString c_SMS::decodePDUText(QString text, QString encoding)
{
// 	kdDebug() << "Encoding str=" << encoding << endl;
	if(encoding.toInt(0,16) == 0 )
	{
		// try to auto detect sms encoding by size field
		if ( ( (float) text.mid(2).length() / (float) text.left(2).toInt(0,16) ) >= 2.1 )
			return getFrom16bits( text.mid(2) );
		else return getFrom7bits ( text.mid(2), text.left(2).toInt(0,16) );
	}
	const char* enc=strToBits(encoding);
	QString encstr;
	for(int i=0; i<8; i++) encstr+=QString::number( (int)enc[i] ) + "|";
	if( enc[4]==0 && enc[5]==1 )
		return getFrom8bits(text.mid(2) );
	if( enc[4]==1 && enc[5]==0 )
		return getFrom16bits(text.mid(2) );
	return getFrom7bits(text.mid(2), text.left(2).toInt(0,16) );
}


/*!
    \fn c_SMS::fromGSMAlphabet(QString src)
 */
const QString c_SMS::fromGSMAlphabet(QString src)
{
	QString out;
	int found;
	for(unsigned int i=0; i<src.length(); i++)
	{
		found=0;
		if( src.at(i).unicode() == 0x001B && i+1 < src.length() )
		{
			for( unsigned int j=0; j< sizeof(extended7bitIndexes); j++)
				if( extended7bitIndexes[j]== src.at(i+1).unicode() ) found=j+1;
		}
		if(found) out+=QChar( alphabet7bit[ 127 + found ] );
		else out+=QChar( alphabet7bit[ src.at(i).unicode() ] );
		
		if(found) i++;
	}
	return out;
}


/*!
    \fn c_SMS::getPDUStream()
 */
const QString c_SMS::getPDUStream()
{
	int text_bits;
	if( i_Type == REC_READ || i_Type == REC_UNREAD ) return NULL; // We probably don't need to translate incoming sms.
	QString out;
	out+="00 11 00"; // SMS Message Sender number + SMS-Submit + TP-Message Reference
	out+=QString("%1").arg( sl_Numbers.first().replace("+","").length(), 2, 16 ).replace(" ","0");
	if(sl_Numbers.first().at(0) == '0') out+="81"; else out+="91";
	out+=toPDUNum( sl_Numbers.first().replace("+","") ).upper();
	out+="00";
	if( is7BitString( sl_TextList.join("") ) ) 
	{
		kdDebug() << "7Bit text\n";
		out+="00";
		text_bits=7;
		out+="FF";
		out+= textTo7bits( sl_TextList.join("") );
	} else 
	{
		kdDebug() << "16Bit text\n";
		out+="08";
		text_bits=16;
		out+="FF";
		out+=QString("%1").arg(sl_TextList.join("").length()*2,2,16 ).replace(" ","0");
		for(unsigned int i=0; i<sl_TextList.join("").length(); i++)
			out+=QString("%1").arg( sl_TextList.join("").at(i).unicode(), 4, 16 ).replace(" ","0");
	}
	out=out.replace(" ","");
	
	return out.upper();
}


/*!
    \fn c_SMS::toPDUNum
 */
const QString c_SMS::toPDUNum(QString num)
{
	QString out;
	if(num.length() % 2 == 1 ) num+="F";
	for(unsigned int i=0; i<num.length(); i+=2)
		out+=num.mid(i+1,1) + num.mid(i,1);
	return out;
}


/*!
    \fn c_SMS::textTo7bits(QString text)
 */
QString c_SMS::textTo7bits(QString text)
{
	kdDebug() << "SMS 7bit recoded" << endl;
	char oldBits[7];
	char tempBits[8];
	char newBits[7];
	int result;
	QString out, temp;
	memset(newBits, 0, sizeof(newBits) );
	memset(tempBits, 0, sizeof(tempBits) );
	memset(oldBits, 0, sizeof(oldBits) );
	for(unsigned int i=0; i<text.length(); i++) 
		temp += get7bitChar( text.at(i) );
		kdDebug() << "8BIT Code:" << temp << endl;
#define y i*2
		for(long i= (temp.length()/2)-1; i>-1; i-- )
		{
			result=0;
			memcpy(newBits, &strToBits(temp.mid(y,2) )[1], 7); // Here we get the bit stream of our character
			memcpy(&tempBits[(i%8)+1], newBits, sizeof(newBits) - (i%8) ); // we copy the new bit array into the new one
			memcpy(tempBits, oldBits, (i%8)+1 ); // now copy the bits from the last cycle
			memset(oldBits, 0, (i+1)%8 ); memcpy(oldBits, &newBits[sizeof(newBits) - (i%8)], i%8 );
			for(int z=0; z<8; z++) result+=(int) pow( 2.0 ,z)*tempBits[7-z];
			if( i%8 != 7 ) out.prepend( QString("%1").arg( result, 2, 16 ).replace(" ","0") );
		}
// 	for(unsigned int i=0; i<text.length(); i++)
// 	{
// 		memcpy( tempBits,strToBits( text.mid(i*2, 2) ), 8 );
// 		if(i%7 >0) memcpy( &tempBits[8], oldBits, i%7);
// 		memcpy( oldBits, tempBits, /*(i+1)%7 */ 7);
// 		memcpy( newBits, &tempBits[(i%7)+1] , 7);
// 		out+=bitsToChar( newBits );
// 	}
	out= out.upper().prepend(QString("%1").arg(text.length(),2,16).replace(" ","0") );
	return out;
}
