/***************************************************************************
                          chatmaster.cpp  -  description
                             -------------------
    begin                : Sat Jan 18 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "chatmaster.h"

#include <kdebug.h>

#include "../contact/contact.h"
#include "../dialogs/networkwindow.h"
#include "../network/chatinformation.h"
#include "../network/msnswitchboardconnection.h"
#include "../currentaccount.h"
#include "../kmessdebug.h"
#include "../kmess.h"
#include "chatwindow.h"

// The constructor
ChatMaster::ChatMaster(QObject *parent)
 : QObject(parent, "ChatMaster"),
   initialized_(false)
{
}



// The destructor
ChatMaster::~ChatMaster()
{
  disconnected();

#ifdef KMESSDEBUG_CHATMASTER
  kdDebug() << "DESTROYED ChatMaster " << endl;
#endif
}



// A chat window was destroyed
void ChatMaster::chatWindowDestroyed(QObject *chatWindow)
{
#ifdef KMESSDEBUG_CHATMASTER
  kdDebug() << "ChatMaster::chatWindowDestroyed: removing chat window from list." << endl;
#endif

  // Remove from the list
  chatWindows_.remove( static_cast<ChatWindow*>(chatWindow) );
}



// The user has disconnected, so close all open chats
void ChatMaster::disconnected()
{
  if(chatWindows_.count() > 0)
  {
#ifdef KMESSDEBUG_CHATMASTER
    kdDebug() << "Destroying ChatMaster: Closing open chat windows" << endl;
#endif

    // Go through all the chat windows
    for ( ChatWindow *chatWindow = chatWindows_.first(); chatWindow; chatWindow = chatWindows_.next() )
    {
      // Close+delete the chat window
      // (note that the window also has the WDestructiveClose flag set)
      chatWindow->close(true);
    }

    // Clear the chat window collection
    chatWindows_.setAutoDelete(false);
    chatWindows_.clear();
  }
}



// Forward a new chat signal from a chat window
void ChatMaster::forwardNewChat(const Contact *contact, QString message, ChatWindow *chatWindow)
{
#ifdef KMESSTEST
  ASSERT( chatWindow != 0 );
#endif
#ifdef KMESSDEBUG_CHATMASTER
  kdDebug() << "ChatMaster - Forwarding the new chat signal." << endl;
#endif
  emit newChat( contact, message, chatWindow );
}



// Initialize the class
bool ChatMaster::initialize()
{
  if ( initialized_ )
  {
    kdDebug() << "ChatMaster already initialized." << endl;
    return false;
  }

  initialized_ = true;
  return true;
}



// Start a chat
void ChatMaster::startChat( ChatInformation *chatInfo )
{
#ifdef KMESSTEST
  ASSERT( chatInfo != 0 );
#endif
  ChatWindow *chatWindow;
  // Look through the chat windows for an exclusive chat with the contact given
  for ( chatWindow = chatWindows_.first(); chatWindow; chatWindow = chatWindows_.next() )
  {
    if ( chatWindow->isExclusiveChatWithContact( chatInfo->getContactHandle() ) )
    {
      // This chat window is already a chat with the contact, so reopen it.
      chatWindow->startChat( chatInfo );

      return;
    }
  }
  // Create a new chat window
#ifdef KMESSDEBUG_CHATMASTER
  kdDebug() << "ChatMaster - Creating a new chat window." << endl;
#endif
  chatWindow = new ChatWindow( kapp->mainWidget(), "chatwindow" );
  connect( chatWindow, SIGNAL(           destroyed(QObject*)  ) ,
           this,       SLOT  ( chatWindowDestroyed(QObject*)  ) );
  connect( chatWindow, SIGNAL(          newChat(const Contact*, QString, ChatWindow*) ),
           this,       SLOT  (   forwardNewChat(const Contact*, QString, ChatWindow*) ) );
  chatWindows_.append( chatWindow );
  if ( !chatWindow->initialize() )
  {
    kdDebug() << "ChatMaster - Couldn't initialize the new chat window." << endl;
    return;
  }

#ifdef KMESS_NETWORK_WINDOW
  NetworkWindow *nw= ((KMess *)parent())->getNetworkWindow();
  MsnSwitchboardConnection *sc= chatWindow->getSwitchboardConnection();
  connect(sc, SIGNAL(messageSent(const QString &)),
          nw, SLOT(addOutgoingServerMessage(const QString &)));
  connect(sc, SIGNAL(messageReceived(const QString &)),
          nw, SLOT(addIncomingServerMessage(const QString &)));
#endif

  chatWindow->startChat( chatInfo );
}

#include "chatmaster.moc"
