/**************************************************************************/
/*  Klavaro - a flexible touch typing tutor                               */
/*  Copyright (C) 2005 - 2008  Felipe Castro                              */
/*                                                                        */
/*  This program is free software, licensed under the terms of the GNU    */
/*  General Public License as published by the Free Software Foundation,  */
/*  which is also included in this package, in a file named "COPYING".    */
/**************************************************************************/

/*
 * Functions to implement and manage the keyboard editing operations
 */
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>

#include "support.h"
#include "interface.h"
#include "main.h"
#include "callbacks.h"
#include "keyboard.h"

/*
 * Variables
 */
extern GtkWidget *window_main_;
extern GtkWidget *window_tutor_;

GtkWidget *window_keyboard_ = NULL;
GtkWidget *window_hints_ = NULL;

const gunichar vowels[] = {
	L'a', L'e', L'i', L'o', L'u',
	(gunichar) 1072, (gunichar) 1077, (gunichar) 1080, (gunichar) 1086,
	(gunichar) 1091,
	(gunichar) 945, (gunichar) 949, (gunichar) 953, (gunichar) 959,
	(gunichar) 965,
	L'\0'
};

struct
{
	gchar *name;
	gunichar lochars[4][KEY_LINE_LEN + 1];
	gunichar upchars[4][KEY_LINE_LEN + 1];
	gboolean modified_status;
	gchar button_label[32];
} keyb;

/*******************************************************************************
 * Interface functions
 */
gchar *
keyb_get_name ()
{
	return (keyb.name);
}

void
keyb_set_name (const gchar * name)
{
	g_free (keyb.name);
	keyb.name = g_strdup (name);
}

void
keyb_init_name (const gchar * name)
{
	keyb.name = g_strdup (name);
}

gunichar
keyb_get_lochars (gint i, gint j)
{
	return (keyb.lochars[i][j]);
}

gunichar
keyb_get_upchars (gint i, gint j)
{
	return (keyb.upchars[i][j]);
}

gboolean
keyb_get_modified_status ()
{
	return (keyb.modified_status);
}

void
keyb_set_modified_status (gboolean new_status)
{
	keyb.modified_status = new_status;
}

/**********************************************************************
 * Read the character sets (keyb.lochars[] & keyb.upchars[])
 * for the keyboard currently selected.
 */
void
keyb_get_layout ()
{
	gint i;
	gchar *tmp_name = NULL;
	gchar tmp_str[6 * KEY_LINE_LEN + 1];
	glong n_itens;
	gunichar *uchs;
	FILE *fh;
	GtkWidget *wg;

	/*
	 * Search at home
	 */
	tmp_name = g_strconcat (main_get_user_dir (), keyb.name, ".kbd", NULL);
	fh = (FILE *) g_fopen (tmp_name, "r");
	if (fh == NULL)
	{
		/*
		 * Search at data
		 */
		g_free (tmp_name);
		tmp_name = g_strconcat (main_get_data_path (), keyb.name, ".kbd", NULL);
		fh = (FILE *) g_fopen (tmp_name, "r");
	}
	g_free (tmp_name);

	/* Success */
	if (fh)
	{
		for (i = 0; i < 4; i++)
		{
			fgets (tmp_str, 6 * KEY_LINE_LEN + 1, fh);
			tmp_str[6 * KEY_LINE_LEN] = '\0';
			uchs = g_utf8_to_ucs4_fast (tmp_str, -1, &n_itens);
			if (n_itens > KEY_LINE_LEN)
				g_error ("invalid keyboard layout: %s\n"
					 "invalid line: %i\n"
					 "invalid number of chars: %li", keyb.name, i + 1, n_itens);
			memcpy (keyb.lochars[i], uchs, (n_itens - 1) * sizeof (gunichar));
			g_free (uchs);
			for (; n_itens < KEY_LINE_LEN; n_itens++)
				keyb.lochars[i][n_itens] = L' ';
		}
		for (i = 0; i < 4; i++)
		{
			fgets (tmp_str, 6 * KEY_LINE_LEN + 1, fh);
			tmp_str[6 * KEY_LINE_LEN] = '\0';
			uchs = g_utf8_to_ucs4_fast (tmp_str, -1, &n_itens);
			if (n_itens > KEY_LINE_LEN)
				g_error ("invalid keyboard layout: %s\n"
					 "invalid line: %i\n"
					 "invalid number of chars: %li", keyb.name, i + 5, n_itens);
			memcpy (keyb.upchars[i], uchs, (n_itens - 1) * sizeof (gunichar));
			g_free (uchs);
			for (; n_itens < KEY_LINE_LEN; n_itens++)
				keyb.upchars[i][n_itens] = L' ';
		}
		fclose (fh);

		keyb.modified_status = FALSE;
	}
	/*
	 * Recursively try defaults
	 */
	else
	{
		if (strcmp (keyb.name, DEFAULT_KEYBOARD) == 0)
			g_error ("couldn't open the default keyboard layout.");

		wg = lookup_widget (window_main_, "entry_keyboard");
		if (strcmp (keyb.name, "") == 0)
		{
			g_free (keyb.name);
			keyb.name = gtk_editable_get_chars (GTK_EDITABLE (wg), 0, -1);
			keyb_get_layout ();
			return;
		}

		g_message ("couldn't find the keyboard layout: \"%s\"\n"
			   " Opening the default one: \"%s\"", keyb.name, DEFAULT_KEYBOARD);
		gtk_entry_set_text (GTK_ENTRY (wg), DEFAULT_KEYBOARD);
		g_free (keyb.name);
		keyb.name = g_strconcat (DEFAULT_KEYBOARD, NULL);
		keyb_get_layout ();
		return;
	}
}

/**********************************************************************
 * Test if chr is a vowel 
 */
gboolean
keyb_is_vowel (gunichar chr)
{
	gint i;

	for (i = 0; vowels[i] != L'\0'; i++)
		if (g_unichar_tolower (chr) == vowels[i])
			return (TRUE);
	return (FALSE);
}

/**********************************************************************
 * Get the set of available vowels of the keyboard
 */
gint
keyb_get_vowels (gunichar * vows)
{
	gint i;
	gint j;
	gint k = 0;

	for (i = 0; i < 4; i++)
		for (j = 0; j < KEY_LINE_LEN; j++)
		{
			if (keyb_is_vowel (keyb.lochars[i][j]))
				vows[k++] = keyb.lochars[i][j];
			if (k == 20)
				break;
		}
	if (k == 0)
		for (k = 0; k < 5; k++)
			vows[k] = keyb.lochars[2][k];
	return (k);
}

/**********************************************************************
 * Get the set of available consonants of the keyboard
 */
gint
keyb_get_consonants (gunichar * consonants)
{
	gint i, j;
	gint k = 0;
	gunichar chr;

	for (i = 0; i < 4; i++)
		for (j = 0; j < KEY_LINE_LEN; j++)
		{
			chr = keyb.lochars[i][j];
			if (g_unichar_isalpha (chr) && (!keyb_is_vowel (chr)))
				consonants[k++] = chr;

			chr = g_unichar_tolower (keyb.upchars[i][j]);
			if (g_unichar_isalpha (chr) && (!keyb_is_vowel (chr))
			    && (chr != keyb.lochars[i][j]))
				consonants[k++] = chr;
		}
	return (k);
}

/**********************************************************************
 * Get the set of available symbols of the keyboard
 */
gint
keyb_get_symbols (gunichar * symbols)
{
	gint i, j;
	gint k = 0;
	gunichar chr;

	for (i = 0; i < 4; i++)
		for (j = 0; j < KEY_LINE_LEN; j++)
		{
			chr = keyb.lochars[i][j];
			if (g_unichar_ispunct (chr))
				symbols[k++] = chr;

			chr = keyb.upchars[i][j];
			if (g_unichar_ispunct (chr))
				symbols[k++] = chr;
		}
	return (k);
}

/**********************************************************************
 * Save the custom keyboard layout created by the user
 */
void
keyb_save_new_layout ()
{
	gint i;
	gchar *tmp_name = NULL;
	FILE *fh;

	assert_user_dir ();
	tmp_name = g_strconcat (main_get_user_dir (), keyb.name, ".kbd", NULL);
	fh = (FILE *) g_fopen (tmp_name, "w");
	g_free (tmp_name);

	for (i = 0; i < 4; i++)
	{
		tmp_name = g_ucs4_to_utf8 (keyb.lochars[i], KEY_LINE_LEN - 1, NULL, NULL, NULL);
		fprintf (fh, "%s\n", tmp_name);
		g_free (tmp_name);
	}
	for (i = 0; i < 4; i++)
	{
		tmp_name = g_ucs4_to_utf8 (keyb.upchars[i], KEY_LINE_LEN - 1, NULL, NULL, NULL);
		fprintf (fh, "%s\n", tmp_name);
		g_free (tmp_name);
	}
	fclose (fh);

	gtk_widget_hide (lookup_widget (window_keyboard_, "label_modified"));
	if (strcmp (keyb.name, ".tmp") == 0)
		gtk_widget_show (lookup_widget (window_keyboard_, "label_saved_tmp"));
	else
		gtk_widget_hide (lookup_widget (window_keyboard_, "label_saved_tmp"));
	gtk_widget_set_sensitive (lookup_widget (window_keyboard_, "label_kb_load"), TRUE);
	keyb.modified_status = FALSE;
}

/**********************************************************************
 * Remove custom keyboard layout created by the user
 */
void
keyb_remove_user_layout ()
{
	gchar *aux;
	gchar *tmp_name;
	GtkWidget *wg;

	wg = lookup_widget (window_keyboard_, "entry_kb_edit");
	aux = gtk_editable_get_chars (GTK_EDITABLE (wg), 0, -1);

	tmp_name = g_strconcat (main_get_user_dir (), aux, ".kbd", NULL);
	g_unlink (tmp_name);
	g_free (tmp_name);

	if (strcmp (keyb.name, aux) == 0)
	{
		g_free (keyb.name);
		wg = lookup_widget (window_main_, "entry_keyboard");
		keyb.name = gtk_editable_get_chars (GTK_EDITABLE (wg), 0, -1);
		keyb_get_layout ();
		keyb_update_virtual_layout ();
	}
	keyb_fill_layout_list (0);

	g_free (aux);
}


/**********************************************************************
 * Update the virtual keyboard accordingly to its character set and
 * shift key state.
 */
void
keyb_update_virtual_layout ()
{
	gint i, j;
	gchar lab_name[10];
	gchar ut8[7];
	gunichar uch;
	GtkWidget *wg;
	gboolean tog_state;

	strcpy (lab_name, "lab");
	wg = lookup_widget (window_keyboard_, "toggle_shift1");
	tog_state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wg));
	for (i = 0; i < 4; i++)
	{
		for (j = 0; j < KEY_LINE_LEN - (i == 0 ? 1 : (i == 1 ? 2 : 3)); j++)
		{
			uch = tog_state ? keyb.upchars[i][j] : keyb.lochars[i][j];
			if (g_unichar_isalpha (uch)
			    && g_unichar_tolower (keyb.upchars[i][j]) == keyb.lochars[i][j])
				uch = g_unichar_toupper (uch);
			snprintf (lab_name + (i + 3), 3, "%i", j + 1);
			if ((wg = lookup_widget (window_keyboard_, lab_name)))
			{
				ut8[g_unichar_to_utf8 (uch, ut8)] = '\0';
				gtk_label_set (GTK_LABEL (wg), ut8);
			}
		}
		lab_name[i + 3] = '_';
	}
}

/**********************************************************************
 * Reads the lists of available keyboard layouts
 */
void
keyb_fill_layout_list (gint entry_type)
{
	gboolean kb_name_exists;
	gint i;
	gint name_len;
	gchar tmp_end[5];
	gchar tmp_name[MAX_KEYBOARDS][21];
	GDir *dir = NULL;
	gchar *dentry = NULL;
	GList *files = NULL;
	GtkWidget *wg = NULL;

	if (entry_type == 1)
	{
		/*
		 * Reads the list of original files
		 */
		dir = g_dir_open (main_get_data_path (), 0, NULL);
		if (dir == NULL)
			g_error ("could not find the data directory:/n %s/n",
				 main_get_data_path ());
	}
	else if (entry_type == 2 || entry_type == 3)
	{
		/*
		 * Reads the list of custom files
		 */
		assert_user_dir ();
		dir = g_dir_open (main_get_user_dir (), 0, NULL);
	}
	else if (entry_type == 0)
	{
		/*
		 * Reads the list of custom and original files
		 */
		keyb_fill_layout_list (3);
		keyb_fill_layout_list (2);
		keyb_fill_layout_list (1);
		return;
	}

	kb_name_exists = FALSE;
	i = 0;
	while (i < MAX_KEYBOARDS && ((dentry = g_strdup (g_dir_read_name (dir))) != NULL))
	{
		name_len = strlen (dentry);
		if (name_len > 20 || name_len < 5)
		{
			g_free (dentry);
			continue;
		}

		strcpy (tmp_end, dentry + name_len - 4);
		if (strcmp (tmp_end, ".kbd"))
		{
			g_free (dentry);
			continue;
		}

		strcpy (tmp_name[i], dentry);
		tmp_name[i][name_len - 4] = '\0';

		if (entry_type == 3 && strcmp (tmp_name[i], ".tmp") == 0)
		{
			g_free (dentry);
			continue;
		}

		if (strcmp (tmp_name[i], keyb.name) != 0)
		{
			files = g_list_insert_sorted (files, tmp_name[i], comp_str);
			i++;
		}
		else
			kb_name_exists = TRUE;

		g_free (dentry);
	}
	g_dir_close (dir);
	if (kb_name_exists)
		files = g_list_insert (files, keyb.name, 0);
	else
		files = g_list_insert (files, "", 0);

	callbacks_shield_set (TRUE);
	if (entry_type == 1)
	{
		wg = lookup_widget (window_keyboard_, "combo_kb_main");
		gtk_combo_set_popdown_strings (GTK_COMBO (wg), files);
	}
	else if (entry_type == 2)
	{
		wg = lookup_widget (window_keyboard_, "combo_kb_user");
		gtk_combo_set_popdown_strings (GTK_COMBO (wg), files);
	}
	else if (entry_type == 3)
	{
		wg = lookup_widget (window_keyboard_, "combo_kb_edit");
		gtk_combo_set_popdown_strings (GTK_COMBO (wg), files);
	}
	callbacks_shield_set (FALSE);

	g_list_free (files);

	/*
	 * Updates the "Layout to be selected" field
	 */
	wg = lookup_widget (window_keyboard_, "label_kb_name");
	gtk_label_set (GTK_LABEL (wg), keyb.name);

}

/**********************************************************************
 * Create a popup window to get a real key pressed and
 * write the name of the virtual one.
 */
void
keyb_edit_key (GtkButton * button)
{
	gchar pix_name_aux[32] = "hands_0.xpm";
	gchar *tmp_name;
	GtkWidget *wg;
	static GtkWidget *popup_keyedit = NULL;

	strcpy (keyb.button_label, gtk_widget_get_name (GTK_WIDGET (button)));

	/*
	 * Tutoring mode
	 */
	if (window_tutor_ != NULL)
	{
		hints_get_from_label (pix_name_aux, keyb.button_label);
		tmp_name = find_pixmap_file (pix_name_aux);
		wg = lookup_widget (window_keyboard_, "pix_hands");
		gtk_image_set_from_file (GTK_IMAGE (wg), tmp_name);
		g_free (tmp_name);
		return;
	}

	/*
	 * Editing mode
	 */
	keyb.button_label[0] = 'l';
	keyb.button_label[1] = 'a';
	keyb.button_label[2] = 'b';
	if (popup_keyedit == NULL)
		popup_keyedit = create_popup_keyedit ();
	wg = lookup_widget (popup_keyedit, "entry_popup_key");
	callbacks_shield_set (TRUE);
	gtk_entry_set_text (GTK_ENTRY (wg), "");
	callbacks_shield_set (FALSE);
	gtk_widget_show (popup_keyedit);
}

/**********************************************************************
 * Apply the key pressed to the virtual keyboard
 * and to the upper or lower character sets
 */
void
keyb_change_key (gunichar real_key)
{
	gint key_lin, key_col;
	gunichar str_char;
	gchar tmp_utf8[7];
	gboolean tog_state;
	GtkWidget *wg;

	str_char = g_unichar_toupper (real_key);
	wg = lookup_widget (window_keyboard_, keyb.button_label);
	tmp_utf8[g_unichar_to_utf8 (str_char, tmp_utf8)] = '\0';
	gtk_label_set (GTK_LABEL (wg), tmp_utf8);

	for (key_lin = 0; keyb.button_label[key_lin + 3] == '_'; key_lin++);
	key_col = atoi (keyb.button_label + (key_lin + 3)) - 1;

	wg = lookup_widget (window_keyboard_, "toggle_shift1");
	tog_state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (wg));
	if (tog_state)
	{
		keyb.upchars[key_lin][key_col] = str_char;
		if (str_char >= L'A' && str_char <= L'Z')
			keyb.lochars[key_lin][key_col] = g_unichar_tolower (str_char);
	}
	else
	{
		keyb.lochars[key_lin][key_col] = g_unichar_tolower (str_char);
		if (str_char >= L'A' && str_char <= L'Z')
			keyb.upchars[key_lin][key_col] = str_char;
	}

	if (keyb.modified_status == FALSE)
	{
		keyb.modified_status = TRUE;
		gtk_widget_hide (lookup_widget (window_keyboard_, "label_saved_tmp"));
		gtk_widget_show (lookup_widget (window_keyboard_, "label_modified"));
		gtk_widget_set_sensitive (lookup_widget (window_keyboard_, "label_kb_load"), FALSE);
		wg = lookup_widget (window_keyboard_, "notebook_kb");
		gtk_notebook_set_page (GTK_NOTEBOOK (wg), 1);
	}
}

/*******************************************************************************
 * Get an utf8 string for the par symbol
 */
gchar *
keyb_get_utf8_paragraph_symbol ()
{
	static gchar parsym[7];
	static gboolean is_initialized = FALSE;

	if (is_initialized == FALSE)
	{
		is_initialized = TRUE;
		parsym[g_unichar_to_utf8 (UPSYM, parsym)] = '\0';
	}
	return (parsym);
}

/*******************************************************************************
 * A hack to (not) switch the note tab
 */
gboolean
keyb_force_edit_tab (gpointer data)
{
	GtkWidget *wg = NULL;

	gdk_beep ();
	if (window_keyboard_ == NULL)
		return FALSE;

	wg = lookup_widget (window_keyboard_, "notebook_kb");
	gtk_notebook_set_current_page (GTK_NOTEBOOK (wg), 1);
	return FALSE;
}

/*******************************************************************************
 * Initialize the hints mapping array
 */

gchar hints[4][KEY_LINE_LEN + 1];
gboolean hints_is_initialized = FALSE;

void
hints_init ()
{
	gint i;
	gchar *tmp_name;
	FILE *fh;

	if (hints_is_initialized == TRUE)
		return;

	hints_is_initialized = TRUE;
	tmp_name = g_strconcat (main_get_data_path (), "fingers_position.txt", NULL);
	fh = (FILE *) g_fopen (tmp_name, "r");
	if (fh)
	{
		for (i = 0; i < 4; i++)
			fgets (hints[i], KEY_LINE_LEN + 1, fh);
		fclose (fh);
		hints_set_tips ();
	}
	else
		g_warning ("couldn't open the file:\n %s", tmp_name);
	g_free (tmp_name);
}

gchar *
hints_string_from_charcode (gchar charcode)
{
	gchar *fingerhint = NULL;

	switch (charcode)
	{
	case '1':
		fingerhint = g_strdup (_("small finger"));
		break;
	case '2':
		fingerhint = g_strdup (_("ring finger"));
		break;
	case '3':
		fingerhint = g_strdup (_("middle finger"));
		break;
	case '4':
		fingerhint = g_strdup (_("index finger"));
		break;
	case '5':
		fingerhint = g_strdup (_("thumbs"));
		break;
	case '6':
		fingerhint = g_strdup (_("index finger"));
		break;
	case '7':
		fingerhint = g_strdup (_("middle finger"));
		break;
	case '8':
		fingerhint = g_strdup (_("ring finger"));
		break;
	case '9':
		fingerhint = g_strdup (_("small finger"));
		break;
	default:
		fingerhint = g_strdup ("???");
	}
	return (fingerhint);
}

void
hints_set_tips ()
{
	gint i, j;
	gchar *but_ini;
	gchar *tmp;
	GtkWidget *wg;
	GtkTooltipsData *tt;

	for (i = 0; i < 4; i++)
	{
		if (i == 0)
			but_ini = g_strdup ("but");
		else if (i == 1)
			but_ini = g_strdup ("but_");
		else if (i == 2)
			but_ini = g_strdup ("but__");
		else
			but_ini = g_strdup ("but___");

		for (j = 0; j < (i == 0 ? 14 : (i == 1 ? 13 : 12)); j++)
		{
			tmp = g_strdup_printf ("%s%i", but_ini, j + 1);
			wg = lookup_widget (window_keyboard_, tmp);
			g_free (tmp);
			tt = gtk_tooltips_data_get (wg);
			tmp = hints_string_from_charcode (hints[i][j]);
			gtk_tooltips_set_tip (tt->tooltips, wg, tmp, NULL);
			g_free (tmp);
		}
		g_free (but_ini);
	}
}

/**********************************************************************
 * Map the character to the file which shows the finger associated with that key
 */
void
hints_get_from_char (gchar * file_name, gunichar character)
{
	gint i;
	gint j;

	strcpy (file_name, "hands_9.xpm");
	if (character == UPSYM)
		return;
	strcpy (file_name, "hands_5.xpm");
	if (character == L' ')
		return;

	hints_init ();

	for (i = 3; i >= 0; i--)
		for (j = 0; j < 15; j++)
			if (character == keyb.lochars[i][j])
			{
				file_name[6] = hints[i][j];
				return;
			}

	for (i = 3; i >= 0; i--)
		for (j = 0; j < 15; j++)
			if (character == keyb.upchars[i][j])
			{
				file_name[6] = hints[i][j];
				return;
			}
	file_name[6] = '0';
}

/**********************************************************************
 * Map the label to the file which shows the finger associated with that key
 */
void
hints_get_from_label (gchar * file_name, gchar * label)
{
	gint key_lin;
	gint key_col;
	gchar *tailptr;

	if (g_strrstr (label, "but_space") != NULL)
	{
		strcpy (file_name, "hands_5.xpm");
		return;
	}
	if (g_strrstr (label, "but") == NULL)
		return;

	strcpy (file_name, "hands_0.xpm");

	for (key_lin = 0; label[key_lin + 3] == '_'; key_lin++);
	key_col = strtol (label + (key_lin + 3), &tailptr, 0) - 1;

	if (*tailptr != '\0')
		return;

	hints_init ();

	file_name[6] = hints[key_lin][key_col];
}

/**********************************************************************
 * Update the image of the window_hints_
 */
void
hints_update_from_char (gunichar character)
{
	gchar file_name[32];
	GtkWidget *wg;

	if (window_hints_ == NULL)
		return;

	wg = lookup_widget (window_hints_, "pixmap_hints");
	hints_get_from_char (file_name, character);
	gtk_image_set_from_file (GTK_IMAGE (wg), find_pixmap_file (file_name));
}
