/*
 * kwin_skulpture.cpp - Skulpture window decoration for KDE
 *
 * Copyright (c) 2008 Christoph Feck <christoph@maxiom.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include "kwin_skulpture.h"

#include <KDE/KConfig>
#include <KDE/KConfigGroup>
#include <KDE/KLocale>

#include <QtCore/QCoreApplication>
#include <QtCore/QLibrary>
#include <QtCore/QPluginLoader>
#include <QtGui/QStyle>
#include <QtGui/QStyleFactory>
#include <QtGui/QStyleOption>
#include <QtGui/QStylePainter>
#include <QtGui/QStylePlugin>


/*-----------------------------------------------------------------------*/
/**
 * The Skulpture window decoration for KDE actually is a wrapper.
 * It uses Qt's MDI window decoration. As such, it is very lightweight.
 *
 */

extern "C" KDE_EXPORT KDecorationFactory *create_factory()
{
	return new SkulptureDecorationFactory();
}


/*-----------------------------------------------------------------------*/
/*
 * When KWin is already running, it doesn't notice that
 * a new style has been installed, and QStyleFactory will
 * not find the new style.
 *
 * This function works around this Qt problem by manually
 * loading the style.
 *
 */

static QStyle *loadStyle(const QString &styleName)
{
    QLibrary library;
    Q_FOREACH (const QString &path, QCoreApplication::libraryPaths()) {
        library.setFileName(path + QLatin1String("/styles/") + styleName);
        if (library.load()) {
            QString libName = library.fileName();
            library.unload();
            QPluginLoader plugin(libName);
            QStyleFactoryInterface *factory = qobject_cast<QStyleFactoryInterface*>(plugin.instance());
            if (factory) {
                return factory->create(styleName);
            }
        }
    }
    return 0;
}


/*-----------------------------------------------------------------------*/

QtMdiDecorationButton::QtMdiDecorationButton(ButtonType type, KCommonDecoration *parent)
	: KCommonDecorationButton(type, parent)
{
	init();
}


void QtMdiDecorationButton::init()
{
	setAttribute(Qt::WA_PaintOnScreen, false);
	setAttribute(Qt::WA_NoSystemBackground, true);
	setAutoFillBackground(false);
	setFocusPolicy(Qt::NoFocus);
	setAttribute(Qt::WA_OpaquePaintEvent, false);
	setAttribute(Qt::WA_Hover, true);
}


QtMdiDecorationButton::~QtMdiDecorationButton()
{
	/* */
}


void QtMdiDecorationButton::reset(unsigned long /*changed*/)
{
	/* NOTE: must be implemented, because it is declared pure */
}


void QtMdiDecorationButton::initStyleOption(QStyleOptionButton &opt)
{
        const QtMdiDecoration *deco = (const QtMdiDecoration *) decoration();

	opt.init(this);
	if (isDown()) {
		opt.state |= QStyle::State_Selected;
	} else {
		opt.state &= ~QStyle::State_Selected;
	}
	if (decoration()->isActive()) {
		opt.state |= QStyle::State_Active;
	} else {
		opt.state &= ~QStyle::State_Active;
	}
        opt.palette.setColor(QPalette::Window, deco->options()->color(KCommonDecoration::ColorTitleBar, deco->isActive()));
}


void QtMdiDecorationButton::paintEvent(QPaintEvent */* event */)
{
	const QtMdiDecoration *deco = (const QtMdiDecoration *) decoration();
	QPainter painter(this);
	QStyleOptionButton opt;
	initStyleOption(opt);
        const int iconSize = parentWidget()->style()->pixelMetric(QStyle::PM_SmallIconSize, &opt, this);
        QIcon::Mode iconMode = QIcon::Normal;

        if (isDown()) {
            iconMode = QIcon::Selected;
        } else if (opt.state & QStyle::State_MouseOver) {
            iconMode = QIcon::Active;
        }
	QStyle::StandardPixmap sp;
	switch (type()) {
		case OnAllDesktopsButton: sp = QStyle::StandardPixmap(QStyle::SP_CustomBase + (deco->isOnAllDesktops() ? 2 : 1)); break;
		case MenuButton:	sp = QStyle::SP_TitleBarMenuButton; break;
		case AboveButton:	sp = QStyle::StandardPixmap(QStyle::SP_CustomBase + (deco->keepAbove() ? 6 : 4)); break;
		case BelowButton:	sp = QStyle::StandardPixmap(QStyle::SP_CustomBase + (deco->keepBelow() ? 7 : 5)); break;
		case HelpButton:	sp = QStyle::SP_TitleBarContextHelpButton; break;
		case MinButton:		sp = QStyle::SP_TitleBarMinButton; break;
		case MaxButton:	sp = deco->maximizeMode() == KDecoration::MaximizeFull ? QStyle::SP_TitleBarNormalButton : QStyle::SP_TitleBarMaxButton; break;
		case ShadeButton:	sp = deco->isShade() ? QStyle::SP_TitleBarUnshadeButton : QStyle::SP_TitleBarShadeButton; break;
		case CloseButton:	sp = QStyle::SP_TitleBarCloseButton; break;
		default:			sp = QStyle::SP_CustomBase; break;
	}

        opt.palette.setColor(QPalette::Text, deco->options()->color(KCommonDecoration::ColorFont, opt.state & QStyle::State_Active));
        const int shift = opt.state & QStyle::State_Selected ? 1 : 0;
        const bool useIcon = (sp == QStyle::SP_TitleBarMenuButton) && deco->useIcon();
        const QIcon icon = useIcon ? deco->icon().pixmap(iconSize, iconSize) : parentWidget()->style()->standardIcon(sp, &opt, this);
	painter.setOpacity(opt.state & QStyle::State_Active ? 1.0 : 0.7);
	icon.paint(&painter, rect().adjusted(shift, shift, shift, shift), Qt::AlignCenter, iconMode);
}


/*-----------------------------------------------------------------------*/

QtMdiDecoration::QtMdiDecoration(KDecorationBridge *bridge, KDecorationFactory *factory, QStyle *s)
	: KCommonDecoration(bridge, factory), style(s)
{
        /* */
}


QtMdiDecoration::~QtMdiDecoration()
{
	/* */
}


QString QtMdiDecoration::visibleName() const
{
	return i18n("Sculpture");
}


void QtMdiDecoration::initStyleOption(QStyleOption &opt)
{
	opt.init(widget());
	opt.fontMetrics = QFontMetrics(options()->font(isActive()));
	if (isActive()) {
		opt.state |= QStyle::State_Active;
	} else {
		opt.state &= ~QStyle::State_Active;
	}
}


bool QtMdiDecoration::decorationBehaviour(DecorationBehaviour behaviour) const
{
	switch (behaviour) {
		case DB_MenuClose:
		case DB_WindowMask:
		case DB_ButtonHide:
			return true;
		default:
			return KCommonDecoration::decorationBehaviour(behaviour);
	}
}


void QtMdiDecoration:: updateWindowShape()
{
	QStyleOptionTitleBar option;
	initStyleOption(option);
	QStyleHintReturnMask mask;
	if ((widget()->style())->styleHint(QStyle::SH_WindowFrame_Mask, &option, widget(), &mask)) {
		setMask(mask.region);
	} else {
		setMask(QRegion(widget()->rect()));
	}
}


int QtMdiDecoration::layoutMetric(LayoutMetric lm, bool respectWindowState, const KCommonDecorationButton *button) const
{
    bool border = !(maximizeMode() == MaximizeFull && !options()->moveResizeMaximizedWindows());

    switch (lm) {
        case LM_BorderLeft:
        case LM_BorderRight:
        case LM_BorderBottom:
            return border ? borderWidth : 0;
        case LM_TitleBorderLeft:
        case LM_TitleBorderRight:
            return border ? 4 : 0;
        case LM_TitleEdgeLeft:
        case LM_TitleEdgeRight:
            return border ? 8 : 0;
        case LM_TitleEdgeTop:
            return border ? 2 : 0;
        case LM_TitleEdgeBottom:
            return 1;
        case LM_TitleHeight:
            return titleHeight + 2;
        case LM_ButtonHeight:
            return border ? titleHeight : titleHeight + 3;
        case LM_ButtonWidth:
            return titleHeight;
        case LM_ButtonSpacing:
            return 0;
        case LM_ExplicitButtonSpacer:
            return 2;
        case LM_ButtonMarginTop:
            return 0;
    }
    return KCommonDecoration::layoutMetric(lm, respectWindowState, button);
}


KCommonDecorationButton *QtMdiDecoration::createButton(ButtonType type)
{
	return new QtMdiDecorationButton(type, this);
}


void QtMdiDecoration::init()
{
	QWidget wid;
        if (style) {
        	wid.setStyle(style);
        }
	if (wid.style()) {
		QStyleOptionTitleBar option;
		option.init(&wid);
		option.fontMetrics = QFontMetrics(options()->font(true));
		borderWidth = wid.style()->pixelMetric(QStyle::PM_MdiSubWindowFrameWidth, &option, 0);
		titleHeight = wid.style()->pixelMetric(QStyle::PM_TitleBarHeight, &option, 0);
		noBorder = wid.style()->styleHint(QStyle::SH_TitleBar_NoBorder, &option, 0);
		autoRaise = wid.style()->styleHint(QStyle::SH_TitleBar_AutoRaise, &option, 0);
	} else {
		borderWidth = 4;
		titleHeight = 16;
		noBorder = false;
		autoRaise = false;
	}
	KConfig configFile(QLatin1String("kwinskulpturerc"));
	KConfigGroup conf(&configFile, "General");
	coloredFrame = conf.readEntry("UseTitleBarBorderColors", false);
	contrastFrame = conf.readEntry("UseExtraContrastBorder", false);
        showIcon = conf.readEntry("UseApplicationIcon", true);
	switch (options()->preferredBorderSize(factory())) {
		case BorderTiny:		borderWidth = 2; break;
		default:
		case BorderNormal:		borderWidth = 3; break;
		case BorderLarge:		borderWidth = 4; break;
		case BorderVeryLarge:	borderWidth = 5; break;
		case BorderHuge:		borderWidth = titleHeight / 2; break;
		case BorderVeryHuge:	borderWidth = titleHeight * 3 / 4; break;
		case BorderOversized:	borderWidth = titleHeight; break;
	}
	if (contrastFrame) {
		borderWidth += 1;
	}
	KCommonDecoration::init();
	if (wid.style()) widget()->setStyle(wid.style());
	widget()->setAutoFillBackground(false);
	widget()->setAttribute(Qt::WA_NoSystemBackground, true);
	widget()->setAttribute(Qt::WA_OpaquePaintEvent, true);
	widget()->setAttribute(Qt::WA_PaintOnScreen, false);
//        widget()->window()->setAttribute(Qt::WA_NoSystemBackground, true);
}


static inline QColor blend_color(const QColor &c0, const QColor &c1, qreal blend)
{
	int b = int(0.5 + 256.0 * blend);
	b = qMin(256, qMax(0, b));
	QRgb rgba0 = c0.rgba();
	QRgb rgba1 = c1.rgba();
	return QColor(
		qRed(rgba0) + (((qRed(rgba1) - qRed(rgba0)) * b) >> 8),
		qGreen(rgba0) + (((qGreen(rgba1) - qGreen(rgba0)) * b) >> 8),
		qBlue(rgba0) + (((qBlue(rgba1) - qBlue(rgba0)) * b) >> 8),
		qAlpha(rgba0) + (((qAlpha(rgba1) - qAlpha(rgba0)) * b) >> 8)
	);
}


void QtMdiDecoration::paintEvent(QPaintEvent */*event */)
{
	bool border = !(maximizeMode() == MaximizeFull && !options()->moveResizeMaximizedWindows());
//	QWidget *w = widget();
	QStylePainter painter(widget());

	// draw the title bar
	QStyleOptionTitleBar option;
	initStyleOption(option);
	option.titleBarState = option.state;
	option.subControls = QStyle::SC_TitleBarLabel; // | QStyle::SC_TitleBarSysMenu;
	if (noBorder) {
		option.rect.setHeight(titleHeight);
	} else {
		option.rect.adjust(4, 4, -4, -4);
		option.rect.setHeight(titleHeight);
	}
	if (!border) {
		option.rect.adjust(-6, 0, 6, 0);
	}
	option.text = QString();
#if 0
	option.icon = decoration()->icon();
	option.titleBarFlags = Qt::WindowSystemMenuHint;
#else
	option.titleBarFlags = 0;
#endif
//	option.palette = options()->palette(ColorTitleBar, isActive());
	if (coloredFrame && border) {
		painter.fillRect(widget()->rect(), options()->color(ColorTitleBar, isActive()));
#if 0
		QColor color0 = option.palette.color(QPalette::Window);
		QColor color1 = options()->color(ColorTitleBar, isActive());
		QRect r = widget()->rect();
		for (int i = 0; i < borderWidth; ++i) {
			painter.setPen(blend_color(color1, color0, i / (borderWidth - 1.0)));
			painter.drawRect(r.adjusted(0, 0, -1, -1));
			r.adjust(1, 1, -1, -1);
		}
#endif
	} else {
		painter.fillRect(widget()->rect(), option.palette.color(QPalette::Window));
	}
	QFont font = options()->font(isActive());
//	font.setBold(false);
//	font.setItalic(true);
//	font.setPointSizeF(font.pointSizeF() / 1.19);
	painter.setFont(font);
	option.palette.setColor(QPalette::Highlight, options()->color(ColorTitleBar, true));
	option.palette.setColor(QPalette::HighlightedText, options()->color(ColorFont, true));
	option.palette.setColor(QPalette::Window, options()->color(ColorTitleBar, false));
	option.palette.setColor(QPalette::WindowText, options()->color(ColorFont, false));
//	option.rect = painter.style()->subControlRect(QStyle::CC_TitleBar, &option, QStyle::SC_TitleBarCloseButton, widget());
	painter.drawComplexControl(QStyle::CC_TitleBar, option);
//painter.save(); painter.setCompositionMode(QPainter::CompositionMode_Source); painter.fillRect(QRect(140, 10, 30, 10), QColor(200, 100, 200, 100)); painter.restore();
	// draw the title caption
	// TODO use MDI code
	painter.save();
	int captionWidth = width() - buttonsLeftWidth() - buttonsRightWidth() - (border ? 16 : 4);
	option.text = option.fontMetrics.elidedText(caption(), Qt::ElideMiddle, captionWidth);
	QRect labelRect = QRect((border ? 8 : 2) + buttonsLeftWidth(), 0, captionWidth, titleHeight + 4);
	painter.setPen(QColor(0, 0, 0, 25));
	painter.drawText(labelRect.adjusted(1, 1, 1, 1), Qt::AlignLeft | Qt::AlignVCenter | Qt::TextSingleLine, option.text);
        QColor color = options()->color(ColorFont, isActive());
	if (!(option.state & QStyle::State_Active)) {
             color.setAlpha((color.alpha() * 180) >> 8);
        }
	painter.setPen(color);
	painter.drawText(labelRect, Qt::AlignLeft | Qt::AlignVCenter | Qt::TextSingleLine, option.text);
	painter.restore();

	// draw the frame
	if (border) {
		QStyleOptionFrame frameOptions;
		initStyleOption(frameOptions);
		if (coloredFrame) {
			frameOptions.palette.setColor(QPalette::Window, options()->color(ColorTitleBar, isActive()));
		}
		frameOptions.lineWidth = borderWidth;
		if (contrastFrame) {
			QBrush outerLineFill = QColor(170, 170, 170);
		//	QBrush outerLineFill = Qt::red;

			painter.fillRect(frameOptions.rect.adjusted(0, 0, 0, -frameOptions.rect.height() + 1), outerLineFill);
			painter.fillRect(frameOptions.rect.adjusted(0, frameOptions.rect.height() - 1, 0, 0), outerLineFill);
			painter.fillRect(frameOptions.rect.adjusted(0, 0, -frameOptions.rect.width() + 1, 0), outerLineFill);
			painter.fillRect(frameOptions.rect.adjusted(frameOptions.rect.width() - 1, 0, 0, 0), outerLineFill);
			frameOptions.rect.adjust(1, 1, -1, -1);
		}
		painter.drawPrimitive(QStyle::PE_FrameWindow, frameOptions);
	}
}


/*-----------------------------------------------------------------------*/

SkulptureDecorationFactory::SkulptureDecorationFactory()
{
    style = loadStyle(QLatin1String("skulpture"));
}


SkulptureDecorationFactory::~SkulptureDecorationFactory()
{
	/* */
}


KDecoration *SkulptureDecorationFactory::createDecoration(KDecorationBridge *bridge)
{
        return (new QtMdiDecoration(bridge, this, style))->decoration();
}


bool SkulptureDecorationFactory::reset(unsigned long changed)
{
	resetDecorations(changed);
	return true;
}


QList<SkulptureDecorationFactory::BorderSize> SkulptureDecorationFactory::borderSizes() const
{
	return QList<BorderSize>()
			<< BorderTiny
			<< BorderNormal
			<< BorderLarge
			<< BorderVeryLarge
			<< BorderHuge
			<< BorderVeryHuge
			<< BorderOversized;
}


bool SkulptureDecorationFactory::supports(Ability ability) const
{
	switch (ability)
	{
		case AbilityAnnounceButtons:
		case AbilityButtonMenu:
		case AbilityButtonHelp:
		case AbilityButtonShade:
		case AbilityButtonMinimize:
		case AbilityButtonMaximize:
		case AbilityButtonClose:
		case AbilityButtonSpacer:
                        return true;
                // the following three are only supported with Skulpture Qt style
		case AbilityButtonOnAllDesktops:
		case AbilityButtonAboveOthers:
		case AbilityButtonBelowOthers:
			return true;
		case AbilityAnnounceColors:
		case AbilityColorTitleBack:
		case AbilityColorTitleFore:
			return true;
		default:
			return false;
	}
}


/*-----------------------------------------------------------------------*/

#include "kwin_skulpture.moc"


